package com.sonicsw.mf.framework.agent;

import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;

public class ContainerStateNotifier
extends Thread
{
    private Agent m_agent;
    private IComponentContext m_context;
    private long m_notificationInterval;
    private boolean m_run = true;  // flag to indicate we should send notifications of container state

    private boolean DEBUG = false;
    private boolean DEBUG_THREADS = false;

    /*
     * package-only constructor
     */
    public ContainerStateNotifier(Agent agent, IComponentContext context)
    {
        super("Container State Notifier");
        setDaemon(true);

        m_agent = agent;
        m_context = context;
        m_notificationInterval = (long)getInitialNotificationInterval();

        if (DEBUG)
        {
            System.out.println("ContainerStateNotifier: m_notificationInterval = " + m_notificationInterval);
        }
    }

    @Override
    public void run()
    {
        while (m_run)
        {
            if (m_notificationInterval > 0)
            {
                // send the directed notification
                m_agent.sendContainerStateNotification();
                if (DEBUG_THREADS)
                {
                    System.out.println("ContainerStateNotifier: sent container state notification...");
                }
            }

            // sleep for the specified interval
            synchronized(this)
            {
                try
                {
                    wait(m_notificationInterval);
                }
                catch (InterruptedException ie)
                {
                    if (DEBUG_THREADS)
                    {
                        System.out.println("ContainerStateNotifier.run: notification thread interrupted during wait(interval)...");
                    }
                }
            }
        }//while (m_run)
    }

    /**
     * Stops all notification activity and cleans up
     */
    synchronized void cleanup()
    {
        m_run = false;

        // notify the polling thread to wake up
        synchronized(this)
        {
            notifyAll();
        }
    }

    private long getInitialNotificationInterval()
    {
        long notificationInterval = (long)(IContainerConstants.STATUS_POLL_INTERVAL_DEFAULT * 800);  // 80% of default polling interval, converted to milliseconds

        try
        {
            IElement containerConfig = (IElement) m_context.getConfiguration(true);  //"true" - register interest in updates

            // return the default value
            if (containerConfig == null)
            {
                return notificationInterval;
            }

            IAttributeSet monitoringAttrs = (IAttributeSet)containerConfig.getAttributes().getAttribute("MONITORING");
            if (monitoringAttrs != null)
            {
                Integer pollInterval = (Integer)monitoringAttrs.getAttribute("STATUS_POLL_INTERVAL");

                if (DEBUG)
                {
                    System.out.println("ContainerStateNotifier: pollInterval = " + pollInterval);
                }

                if (pollInterval != null)
                {
                    if (DEBUG)
                    {
                        System.out.println("ContainerStateNotifier: poll interval value = " + pollInterval.longValue());
                    }
                    notificationInterval = (long)(pollInterval.longValue() * 800);   // 80% of polling interval, converted to milliseconds
                }
            }
        }
        catch(Exception e)
        {
            if (DEBUG)
            {
                System.out.println("ContainerStateNotifier.getNotificationInterval: exception while retrieving polling interval, e = " + e.getMessage());
            }
        }

        if (DEBUG)
        {
            System.out.println("ContainerStateNotifier: return notification interval = " + notificationInterval);
        }

        return notificationInterval;
    }

    void setNotificationInterval(long notificationInterval)
    {
        if (DEBUG)
        {
            System.out.println("ContainerStateNotifier.setNotificationInterval: old notification interval = " + m_notificationInterval + ", new notification interval = " + notificationInterval);
        }

        synchronized(this)
        {
            m_notificationInterval = notificationInterval;

            notifyAll();
        }
    }

    long getNotificationInterval()
    {
        return m_notificationInterval;
    }
}