package com.sonicsw.mf.framework.agent;

import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.PrintWriter;
import java.io.RandomAccessFile;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;

import com.sonicsw.mx.util.IEmptyArray;
import com.sonicsw.mx.util.LoaderInputStream;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.impl.ExecUtility;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.IFrameworkComponent;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.runtime.IDirectoryServiceProxy;

public final class ContainerUtil
{
    private static final String CACHE_PWD_FILE = "p_file";

    private static final String FRAMEWORK_COMPONENT_INTERFACE_NAME = IFrameworkComponent.class.getName();

    private static final ThreadLocal<SimpleDateFormat> SIMPLE_DATE_FORMAT_THREAD_LOCAL = new ThreadLocal<SimpleDateFormat>() {
        @Override
        protected SimpleDateFormat initialValue() {
            return new SimpleDateFormat("yy/MM/dd HH:mm:ss");
        }
    };

    static boolean isFrameworkComponent(Class componentClass)
    {
        Class[] interfaces = componentClass.getInterfaces();
        for (int i = 0; i < interfaces.length; i++)
        {
            if (interfaces[i].getName().equals(FRAMEWORK_COMPONENT_INTERFACE_NAME))
            {
                return true;
            }
        }

        // try the super class
        Class superClass = componentClass.getSuperclass();
        if (superClass == null || superClass.getName().startsWith("java."))
        {
            return false;
        }

        return isFrameworkComponent(superClass);
    }

    /**
     * Check to see if there are any additional paths we should add to the framework class loader
     */
    static void updateDefaultClassLoader(IContainer container, URLClassLoader classLoader, String classpath)
    throws Exception
    {
        if (classpath.length() == 0)
        {
            return;
        }

        URL[] urls = classLoader.getURLs();
        HashSet set = new HashSet();
        for (int i = 0; i < urls.length; i++)
        {
            if (urls[i].getProtocol().equals("file"))
            {
                String path = urls[i].getPath();
                if (path.indexOf(IMFDirectories.MF_ARCHIVE_DIR) > 0)
                {
                    set.add(new File(path.substring(1)));
                }
            }
        }

        StringTokenizer st = new StringTokenizer(classpath, File.pathSeparator);
        tokenLoop:
        while (st.hasMoreElements())
        {
            // if the protocol is not "file" simply add to the classloader
            URL url = new URL(st.nextToken());
            if (url.getProtocol().equals("file"))
            {
                // if the url does not point to something in the archive directory simply add to the class loader
                String path = url.getPath();
                if (path.indexOf(IMFDirectories.MF_ARCHIVE_DIR) > -1)
                {
                    File file = new File(path.substring(1));
                    if (set.contains(file))
                     {
                        continue; // its already on the classpath
                    }

                    // else search through and see if there is already a different version of the jar
                    // on the classpath
                    File dir = file.getParentFile();
                    String fileVersion = file.getName();
                    String fileName = fileVersion.substring(fileVersion.indexOf('.') + 1);
                    fileVersion = fileVersion.substring(0, fileVersion.indexOf('.'));

                    Iterator iterator = set.iterator();
                    while (iterator.hasNext())
                    {
                        File existingFile = (File)iterator.next();
                        if (existingFile.equals(file))
                         {
                            continue tokenLoop; // its already there
                        }

                        File existingDir = existingFile.getParentFile();
                        if (!existingDir.equals(dir))
                         {
                            continue; // not the same dir so don't do further comparison
                        }

                        String existingFileVersion = existingFile.getName();
                        String existingFileName = existingFileVersion.substring(existingFileVersion.indexOf('.') + 1);
                        existingFileVersion = existingFileVersion.substring(0, existingFileVersion.indexOf('.'));

                        if (existingFileName.equals(fileName))
                        {
                            if (!existingFileVersion.equals(fileVersion))
                            {
                                container.logMessage(null, "Component will be loaded using older code:" + IContainer.NEWLINE + IContainer.NEWLINE +
                                                           "An older version of the cached archive \"" + fileName +
                                                           "\" already exists on the container's classpath; " +
                                                           "the container must be restarted to correct this." + IContainer.NEWLINE, Level.WARNING);
                                continue tokenLoop;
                            }
                        }
                    }
                    set.add(file);
                }
            }

            Class classLoaderClass = classLoader.getClass();
            Method appendURLMethod = classLoaderClass.getMethod("appendURL", new Class[] { URL.class });
            appendURLMethod.invoke(classLoader, new Object[] { url });
        }
    }

    /**
     * Transfer a given object into the context of a different class loader
     */
    static Object transferObject(Object obj, ClassLoader loader)
    {
        // Serialize the parameter
        ByteArrayOutputStream arrayOut = new ByteArrayOutputStream(5120);

        try
        {
            ObjectOutputStream out = new ObjectOutputStream(arrayOut);
            out.writeObject(obj);
            out.flush();
        }
        catch (Exception e)
        {
            // can't serialize so return the original object
            return obj;
        }

        // Start deserialization
        try
        {
            ByteArrayInputStream arrayIn = new ByteArrayInputStream(arrayOut.toByteArray());
            LoaderInputStream in = new LoaderInputStream(arrayIn, loader);
            return in.readObject();
        }
        catch (Exception e)
        {
            // can't deserialize so return the original object
            return obj;
        }
    }

    public static  String extractDSConfigIDFromContainerConfig(IElement containerConfig)
    {
        IAttributeSet containerAttrs = containerConfig.getAttributes();
        IAttributeSet componentsAttrs = (IAttributeSet)containerAttrs.getAttribute(IContainerConstants.COMPONENTS_ATTR);
        if (componentsAttrs == null)
        {
            return null;
        }

        Object[] components = componentsAttrs.getAttributes().entrySet().toArray();

        for (int i = 0; i < components.length; i++)
        {
            Map.Entry componentEntry = (Map.Entry) components[i];
            String componentName = (String) componentEntry.getKey();

            if (componentName.equals(IDirectoryServiceProxy.GLOBAL_ID))
            {
                IAttributeSet compSet = (IAttributeSet) componentEntry.getValue();
                return ((Reference)compSet.getAttribute(IContainerConstants.CONFIG_REF_ATTR)).getElementName();
            }

        }
        return null;
    }

    public static IDirElement[] importConfigurations(File configFile, String encryptedPassword)
    	throws Exception
    {
    	RandomAccessFile raf = new RandomAccessFile(configFile, "r");
        byte[] bytes = new byte[(int)raf.length()];
        raf.read(bytes);
        raf.close();

        // check if a password is provided (indicates that the config file is encrypted)
        if (encryptedPassword != null)
        {
            bytes = decryptBytes(bytes, encryptedPassword);
        }

        return ElementFactory.importElementsFromXML(new String(bytes), null);
    }

    // convenience method so we can extract one element from the file. It will be used in cases
    // where we want the DS element out of a DS boot file that contains replication connections as well
    // returns the first element of the given type
    public static IDirElement importConfiguration(File configFile, String encryptedPassword, String elType)
    	throws Exception
    {
    	IDirElement[] elements = importConfigurations(configFile, encryptedPassword);
    	for (int i=0; i<elements.length; i++)
        {
            if (elements[i].getIdentity().getType().equals(elType))
            {
                return elements[i];
            }
        }
    	return null;
    }

    public static IDirElement importConfiguration(File configFile, String encryptedPassword)
    throws Exception
    {
        RandomAccessFile raf = new RandomAccessFile(configFile, "r");
        byte[] bytes = new byte[(int)raf.length()];
        raf.read(bytes);
        raf.close();

        // check if a password is provided (indicates that the config file is encrypted)
        if (encryptedPassword != null)
        {
            bytes = decryptBytes(bytes, encryptedPassword);
        }

        return ElementFactory.importFromXML(new String(bytes), null);
    }

    public static void encryptAndStore(String content, String filePath, String encryptedPassword) throws Exception
    {
        File storeFile = new File(filePath);
        if (encryptedPassword != null)
        {
            String clearPwd = new String(PBE.decrypt(encryptedPassword.getBytes())); // will decrypt the password if it is encrypted
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            PrintWriter writer = new PrintWriter(out);
            writer.println(content);
            writer.close();
            byte[] encryptedData = PBE.encrypt(out.toByteArray(), clearPwd);
            storeFile.delete();
            RandomAccessFile encFile = new RandomAccessFile(storeFile, "rw");
            encFile.write(encryptedData);
            encFile.close();
        }
        else
        {
            PrintWriter writer = new PrintWriter(new BufferedWriter(new FileWriter(storeFile, false)), true);
            writer.println(content);
            writer.close();
        }
    }

    public static byte[] decryptBytes(byte[] bytes, String encryptedPassword) throws EncryptionException
    {
        String password = new String(PBE.decrypt(encryptedPassword.getBytes())); // will decrypt the password if it is encrypted
        return PBE.decrypt(bytes, password);
    }

    public static String createLogMessage(String id, String message, int severityLevel)
    {
        return createLogMessage(null, id, message, severityLevel);
    }

    public static String createLogMessage(String containerID, String id, String message, int severityLevel)
    {
        StringBuffer timestampedMessage = new StringBuffer();
        timestampedMessage.append('[');
        timestampedMessage.append(SIMPLE_DATE_FORMAT_THREAD_LOCAL.get().format(new Date(System.currentTimeMillis())));
        timestampedMessage.append("]");
        if (containerID != null)
        {
            timestampedMessage.append(" {");
            timestampedMessage.append(containerID);
            timestampedMessage.append('}');
        }
        if (id != null)
        {
            timestampedMessage.append(" ID=");
            timestampedMessage.append(id);
        }
        timestampedMessage.append(" (");
        timestampedMessage.append(Level.LEVEL_TEXT[severityLevel]);
        timestampedMessage.append(") ");
        timestampedMessage.append(message);

        return timestampedMessage.toString();
    }

    public static String appendThrowableToMessage(String message, Throwable exception)
    {
        StringBuffer buffer = new StringBuffer(message);
        buffer.append(IContainer.NEWLINE);
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        PrintWriter writer = new PrintWriter(out, true);
        printStackTrace(exception, writer);
        byte[] data = out.toByteArray();
        buffer.append(new String(data));
        return buffer.toString();
    }

    public static String printStackTrace(Throwable exception, String message)
    {
        return ExecUtility.printStackTrace(exception, message);
    }

    static void printStackTrace(Throwable exception, PrintWriter writer)
    {
        ExecUtility.printStackTrace(exception, writer);
    }

    public static boolean isCausedByTimeout(Throwable exception)
    {
        if (exception instanceof com.sonicsw.mf.comm.ConnectTimeoutException ||
            exception instanceof com.sonicsw.mf.comm.InvokeTimeoutCommsException ||
            exception instanceof com.sonicsw.mf.comm.InvokeTimeoutException)
        {
            return true;
        }


        Class exceptionClass = exception.getClass();

        try
        {
            Method getTargetExceptionMethod = exceptionClass.getMethod("getTargetException", IEmptyArray.EMPTY_CLASS_ARRAY);
            Throwable targetException = (Throwable)getTargetExceptionMethod.invoke(exception, IEmptyArray.EMPTY_OBJECT_ARRAY);
            if (targetException != null)
            {
                return isCausedByTimeout(targetException);
            }
        }
        catch (NoSuchMethodException nsme)
        {} // ignore
        catch (IllegalAccessException iae)
        {} // ignore
        catch (InvocationTargetException ite)
        {} // ignore

        try
        {
            Method getTargetErrorMethod = exceptionClass.getMethod("getTargetError", IEmptyArray.EMPTY_CLASS_ARRAY);
            Throwable targetError = (Throwable)getTargetErrorMethod.invoke(exception, IEmptyArray.EMPTY_OBJECT_ARRAY);
            if (targetError != null)
            {
                return isCausedByTimeout(targetError);
            }
        }
        catch (NoSuchMethodException nsme)
        {} // ignore
        catch (IllegalAccessException iae)
        {} // ignore
        catch (InvocationTargetException ite)
        {} // ignore

        try
        {
            Method getLinkedExceptionMethod = exceptionClass.getMethod("getLinkedException", IEmptyArray.EMPTY_CLASS_ARRAY);
            Throwable linkedException = (Throwable)getLinkedExceptionMethod.invoke(exception, IEmptyArray.EMPTY_OBJECT_ARRAY);
            if (linkedException != null)
            {
                return isCausedByTimeout(linkedException);
            }
        }
        catch (NoSuchMethodException nsme)
        {} // ignore
        catch (IllegalAccessException iae)
        {} // ignore
        catch (InvocationTargetException ite)
        {} // ignore

        try
        {
            Method getCauseMethod = exceptionClass.getMethod("getCause", IEmptyArray.EMPTY_CLASS_ARRAY);
            Throwable cause = (Throwable)getCauseMethod.invoke(exception, IEmptyArray.EMPTY_OBJECT_ARRAY);
            if (cause != null)
            {
                return isCausedByTimeout(cause);
            }
        }
        catch (NoSuchMethodException nsme)
        {} // ignore
        catch (IllegalAccessException iae)
        {} // ignore
        catch (InvocationTargetException ite)
        {} // ignore

        try
        {
            Method getActualExceptionMethod = exceptionClass.getMethod("getActualException", IEmptyArray.EMPTY_CLASS_ARRAY);
            Throwable cause = (Throwable)getActualExceptionMethod.invoke(exception, IEmptyArray.EMPTY_OBJECT_ARRAY);
            if (cause != null)
            {
                return isCausedByTimeout(cause);
            }
        }
        catch (NoSuchMethodException nsme)
        {} // ignore
        catch (IllegalAccessException iae)
        {} // ignore
        catch (InvocationTargetException ite)
        {} // ignore

        return false;

    }

    public static void storeCachePassword(File cacheDirectory, String password) throws Exception
    {
        File storageFile = new File(cacheDirectory, CACHE_PWD_FILE);

        if (password == null)
        {
            if (storageFile.exists())
            {
                storageFile.delete();
            }
            return;
        }

        ObjectOutputStream storageStream = new ObjectOutputStream(new FileOutputStream(storageFile));
        storageStream.writeObject(new PBEString().encrypt(password));
        storageStream.close();
    }

    public static String retrieveCachePassword(File cacheDirectory, boolean delFile) throws Exception
    {
        File storageFile = new File(cacheDirectory, CACHE_PWD_FILE);
        if (!storageFile.exists())
        {
            return null;
        }
        ObjectInputStream storageStream = new ObjectInputStream(new FileInputStream(storageFile));
        String pwd = new PBEString().decrypt((String)storageStream.readObject());
        storageStream.close();
        if (delFile)
        {
            storageFile.delete();
        }
        return pwd;
    }

    public static Properties readProperties(java.io.InputStream inputStream) throws IOException
    {
        java.io.BufferedReader reader = new java.io.BufferedReader (new java.io.InputStreamReader(inputStream));
        Properties props = new Properties();

        while (true)
        {
            String line = reader.readLine();
            if (line == null)
            {
                break;
            }

            int eqIndex = line.indexOf('=');
            if (eqIndex == -1)
            {
                continue;
            }
            String key = line.substring(0, eqIndex);
            String value = line.substring(eqIndex+1);

            props.setProperty(key, value);

        }

       reader.close();
       return props;
    }
}

