package com.sonicsw.mf.framework.agent;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import com.sonicsw.mf.common.archive.IDescriptorTags;

/**
 * Wraps the information provided by a Sonic archive's descriptor, hiding the parsing and
 * structure of the META-INF/sonic.xml file.
 */
public class ExpandedSonicArchive
{
    /*
    // for testing
    public static void main(String[] args)
    {
        try
        {
            new ExpandedSonicArchive(args[0]);
        } catch(Exception e) { e.printStackTrace(); }
    }
    */

    private String m_name;
    private File m_rootDirectory;
    private ArrayList m_privateClasspath;
    private HashMap m_sharedLibraries;
    private ArrayList m_globalClasspath;
    private ArrayList m_bootClasspath;
    private ArrayList m_bootAppendClasspath;
    private ArrayList m_bootPrependClasspath;
    private ArrayList m_launchClasspath;
    private ArrayList m_dependsOn;

    private static final String[] EMPTY_STRING_ARRAY = new String[0];
    public static final URL[] EMPTY_URL_ARRAY = new URL[0];

    /**
     * For use by the container to evaluate classpaths based on the meta data described in the
     * expanded archive's sonic.xml descriptor.
     *
     * @param archive The root of the expanded archive
     * @throws ParserConfigurationException
     * @throws IOException
     */
    public ExpandedSonicArchive(File archive)
    throws ParserConfigurationException, SAXException, IOException
    {
        m_rootDirectory = archive;
        if (!m_rootDirectory.exists())
        {
            throw new IOException("Archive path does not exist: " + archive);
        }
        if (!m_rootDirectory.isDirectory())
        {
            throw new IOException("Archive path is not a directory: " + archive);
        }

        File sonicXML = new File(m_rootDirectory, "META-INF/sonic.xml");
        if (!sonicXML.exists())
        {
            throw new IOException("Archive descriptor does not exist: " + sonicXML.toString());
        }

        DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
        Document document = builder.parse(sonicXML);

        NodeList nodes = document.getDocumentElement().getChildNodes();
        for (int i = nodes.getLength() - 1; i >= 0; i--)
        {
            Node node = nodes.item(i);
            if (node.getNodeType() == Node.ELEMENT_NODE)
            {
                if (((Element)node).getTagName().equals(IDescriptorTags.NAME_TAG))
                {
                    m_name = node.getFirstChild().getNodeValue();
                    if (m_name == null || m_name.length() < 1)
                    {
                        throw new IOException("Invalid " + IDescriptorTags.NAME_TAG + " name");
                    }
                }
                else
                if (((Element)node).getTagName().equals(IDescriptorTags.PRIVATE_CLASSPATH_TAG))
                {
                    m_privateClasspath = parseClasspath(node.getChildNodes());
                }
                else
                if (((Element)node).getTagName().equals(IDescriptorTags.SHARED_LIBRARY_TAG))
                {
                    if (m_sharedLibraries == null)
                    {
                        m_sharedLibraries = new HashMap();
                    }
                    parseSharedLibrary(node.getChildNodes());
                }
                else
                if (((Element)node).getTagName().equals(IDescriptorTags.GLOBAL_CLASSPATH_TAG))
                {
                    m_globalClasspath = parseClasspath(node.getChildNodes());
                }
                else
                if (((Element)node).getTagName().equals(IDescriptorTags.LAUNCH_CLASSPATH_TAG))
                {
                    m_launchClasspath = parseClasspath(node.getChildNodes());
                }
                else
                if (((Element)node).getTagName().equals(IDescriptorTags.BOOT_CLASSPATH_TAG))
                {
                    m_bootClasspath = parseClasspath(node.getChildNodes());
                }
                else
                if (((Element)node).getTagName().equals(IDescriptorTags.BOOT_APPEND_CLASSPATH_TAG))
                {
                    m_bootAppendClasspath = parseClasspath(node.getChildNodes());
                }
                else
                if (((Element)node).getTagName().equals(IDescriptorTags.BOOT_PREPEND_CLASSPATH_TAG))
                {
                    m_bootPrependClasspath = parseClasspath(node.getChildNodes());
                }
                else
                if (((Element)node).getTagName().equals(IDescriptorTags.DEPENDS_ON_TAG))
                {
                    m_dependsOn = parseDependsOn(node.getChildNodes());
                }
            }
        }
    }

    public String getName()
    {
        return m_name;
    }

    public URL[] getPrivateClasspath()
    {
        if (m_privateClasspath == null)
        {
            return EMPTY_URL_ARRAY;
        }
        return (URL[])m_privateClasspath.toArray(EMPTY_URL_ARRAY);
    }

    public String[] getSharedLibraryNames()
    {
        if (m_sharedLibraries == null)
        {
            return EMPTY_STRING_ARRAY;
        }
        return (String[])m_sharedLibraries.keySet().toArray(EMPTY_STRING_ARRAY);
    }

    public ArrayList getSharedLibraryList(String sharedLibraryName)
    {
        ArrayList list =  (ArrayList)m_sharedLibraries.get(sharedLibraryName);
        if (list == null)
        {
            return new ArrayList();
        }
        else
        {
            return list;
        }
    }

    public URL[] getSharedLibraryClasspath(String sharedLibraryName)
    {
        ArrayList classpath = (ArrayList)m_sharedLibraries.get(sharedLibraryName);
        if (classpath == null)
        {
            return EMPTY_URL_ARRAY;
        }
        return classpath == null ? null : (URL[])classpath.toArray(EMPTY_URL_ARRAY);
    }

    public URL[] getGlobalClasspath()
    {
        if (m_globalClasspath == null)
        {
            return EMPTY_URL_ARRAY;
        }
        return (URL[])m_globalClasspath.toArray(EMPTY_URL_ARRAY);
    }

    public URL[] getLaunchClasspath()
    {
        if (m_launchClasspath == null)
        {
            return EMPTY_URL_ARRAY;
        }
        return (URL[])m_launchClasspath.toArray(EMPTY_URL_ARRAY);
    }

    public URL[] getBootClasspath()
    {
        if (m_bootClasspath == null)
        {
            return EMPTY_URL_ARRAY;
        }
        return (URL[])m_bootClasspath.toArray(EMPTY_URL_ARRAY);
    }

    public URL[] getBootAppendClasspath()
    {
        if (m_bootAppendClasspath == null)
        {
            return EMPTY_URL_ARRAY;
        }
        return (URL[])m_bootAppendClasspath.toArray(EMPTY_URL_ARRAY);
    }

    public URL[] getBootPrependClasspath()
    {
        if (m_bootPrependClasspath == null)
        {
            return EMPTY_URL_ARRAY;
        }
        return (URL[])m_bootPrependClasspath.toArray(EMPTY_URL_ARRAY);
    }

    public String[] getDependsOn()
    {
        if (m_dependsOn == null)
        {
            return EMPTY_STRING_ARRAY;
        }
        return (String[])m_dependsOn.toArray(EMPTY_STRING_ARRAY);
    }
    
    private ArrayList parseClasspath(NodeList nodes)
    throws IOException
    {
        ArrayList classpath = new ArrayList();

        int numberOfNodes = nodes.getLength();
        for (int i = 0; i < numberOfNodes; i++)
        {
            Node node = nodes.item(i);
            if (node.getNodeType() == Node.ELEMENT_NODE && ((Element)node).getTagName().equals(IDescriptorTags.PATH_ELEMENT_TAG))
            {
                File pathElement = new File(m_rootDirectory, node.getFirstChild().getNodeValue());
                if (!pathElement.exists())
                {
                    throw new IOException("Archive content does not exist: " + pathElement.toString());
                }
                classpath.add(pathElement.toURL());
            }
        }

        return classpath;
    }

    private void parseSharedLibrary(NodeList nodes)
    throws IOException
    {
        String sharedLibraryName = null;
        ArrayList sharedLibraryClasspath = null;

        for (int i = nodes.getLength() - 1; i >= 0; i--)
        {
            Node node = nodes.item(i);
            if (node.getNodeType() == Node.ELEMENT_NODE)
            {
                if (((Element)node).getTagName().equals(IDescriptorTags.SHARED_LIBRARY_NAME_TAG))
                {
                    sharedLibraryName = node.getFirstChild().getNodeValue();
                }
                else
                if (((Element)node).getTagName().equals(IDescriptorTags.SHARED_LIBRARY_CLASSPATH_TAG))
                {
                    sharedLibraryClasspath = parseClasspath(node.getChildNodes());
                }
            }
        }

        if (sharedLibraryName == null || sharedLibraryName.length() < 1)
        {
            throw new IOException("Invalid " + IDescriptorTags.SHARED_LIBRARY_NAME_TAG);
        }

        // An empty shared library is valid but not useful
        m_sharedLibraries.put(sharedLibraryName, sharedLibraryClasspath == null ? new ArrayList() : sharedLibraryClasspath);
    }

    private ArrayList parseDependsOn(NodeList nodes)
    throws IOException
    {
        ArrayList dependsOn = new ArrayList();

        int numberOfNodes = nodes.getLength();
        for (int i = 0; i < numberOfNodes; i++)
        {
            Node node = nodes.item(i);
            if (node.getNodeType() == Node.ELEMENT_NODE && ((Element)node).getTagName().equals(IDescriptorTags.SHARED_LIBRARY_NAME_TAG))
            {
                String sharedLibraryName = node.getFirstChild().getNodeValue();
                if (sharedLibraryName == null || sharedLibraryName.length() < 1)
                {
                    throw new IOException("Invalid " + IDescriptorTags.SHARED_LIBRARY_NAME_TAG);
                }
                dependsOn.add(sharedLibraryName);
            }
        }

        return dependsOn;
    }
}
