package com.sonicsw.mf.framework.agent;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.Properties;
import java.util.Set;

import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.runtime.IFileDescriptor;
import com.sonicsw.mf.common.runtime.IRemoteCallResult;
import com.sonicsw.mf.common.runtime.IRemoteExecResult;
import com.sonicsw.mf.common.runtime.impl.ExecUtility;
import com.sonicsw.mf.common.runtime.impl.FileDescriptor;
import com.sonicsw.mf.common.runtime.impl.RemoteCallResult;
import com.sonicsw.mf.common.runtime.impl.RemoteCallResultWithData;
import com.sonicsw.mf.common.util.ZipSerializer;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.IHostManager;

import progress.message.net.ProgressInetAddress;

class HostManager implements IHostManager
{
    /**
     * System property override the reported hostnames for this machine. Value
     * is a comma separated list.
     */
    private static final String SONICSW_MF_HM_HOSTNAMES_PROPERTY = "sonicsw.mf.hm.hostnames";

    private static final File   SONIC_CONTAINERS_DIR = new File(System.getProperty(IContainer.SONIC_CONTAINERS_DIR_PROPERTY));
    private static final String LAUNCH_CONTAINER_SCRIPT = "launchcontainer";
    private static final String CONTAINER_LAUNCH_FAILED_TEXT = "com.sonicsw.mf.framework.agent.HostManager.launchContainer failed: ";
    private static final String LOG_EXIT_TEXT = "Exiting...";
    private static final String FILELOG_SUFFIX = ".log";

    private final ContainerDS m_directory;
    private final String m_containerCanonicalName;
    private final String m_domainName;
    private final ContainerImpl m_containerImpl;


    HostManager(String domainName, String containerCanonicalName, ContainerImpl containerImpl)
    {
        m_containerImpl = containerImpl;
        m_directory = m_containerImpl.getDS();
        m_domainName = domainName;
        m_containerCanonicalName = containerCanonicalName;
    }

    @Override
    public IRemoteCallResult  installLauncher(String targetSonicHome)
    {
        if (IContainer.CURRENT_LAUNCHER_VERSION ==  null)
        {
            return new RemoteCallResult(false, "com.sonicsw.mf.framework.agent.HostManager.installLauncher failed: This container was not started from launcher install");
        }

        try
        {
            new com.sonicsw.mf.framework.agent.ci.LauncherFilesInstaller().installLauncherFiles(new File(System.getProperty(IContainer.MF_CONTAINER_CAR_ROOT_PROPERTY)), new File(targetSonicHome));
            return new RemoteCallResult(true, null);
        }
        catch (Throwable t)
        {
            return new RemoteCallResult(false, ContainerUtil.printStackTrace(t, "com.sonicsw.mf.framework.agent.HostManager.installLauncher failed:"));
        }
    }

    @Override
    public IRemoteCallResult  setupContainer(String containerDSPath, Properties unconfiguredProps, String INIEncryptionPassword)
    {
        if (IContainer.CURRENT_LAUNCHER_VERSION ==  null)
        {
            return new RemoteCallResult(false, "com.sonicsw.mf.framework.agent.HostManager.setupContainer failed: This container was not started from launcher install");
        }

        try
        {
            File launcherFilesPath = new File(Agent.SONICSW_HOME, IContainer.LAUNCHER_DIR);
            IElement containerConfig = m_directory.getElementByLogicalName(m_containerCanonicalName, containerDSPath);
            File destDir = ContainerSetup.setupContainer(containerDSPath, containerConfig, launcherFilesPath.getAbsolutePath(), unconfiguredProps, null, INIEncryptionPassword);
            return new RemoteCallResult(true, destDir.getCanonicalPath());

        }
        catch (Throwable t)
        {
            return new RemoteCallResult(false, ContainerUtil.printStackTrace(t, "com.sonicsw.mf.framework.agent.HostManager.setupContainer failed:"));
        }
    }

    //Setup a container in the same sonic home as this container
    @Override
    public IRemoteCallResult  setupContainer(Properties properties, String INIEncryptionPassword)
    {
        return setupContainerInternal(Agent.SONICSW_HOME, IContainer.LAUNCHER_DIR, properties, INIEncryptionPassword);
    }

    //Setup a container in a given sonic home using launcherVesrion. If launcherVesrion is null uses this container's launcher version.
    @Override
    public IRemoteCallResult  setupContainer(String sonicHome, String launcherVesrion, Properties properties, String INIEncryptionPassword)
    {
        String launcherDirectory = (launcherVesrion != null) ? IContainer.LAUNCHERS_ROOT_DIR + launcherVesrion : IContainer.LAUNCHER_DIR;
        return setupContainerInternal(sonicHome, launcherDirectory, properties, INIEncryptionPassword);
    }

    private IRemoteCallResult  setupContainerInternal(String sonicHome, String launcherDirectory, Properties properties, String INIEncryptionPassword)
    {
        if (IContainer.CURRENT_LAUNCHER_VERSION ==  null)
        {
            return new RemoteCallResult(false, "com.sonicsw.mf.framework.agent.HostManager.setupContainer failed: This container was not started from launcher install");
        }

        try
        {
            File launcherFilesPath = new File(sonicHome, launcherDirectory);
            File destDir = ContainerSetup.setupContainer(launcherFilesPath.getAbsolutePath(), properties, INIEncryptionPassword);
            return new RemoteCallResult(true, destDir.getCanonicalPath());
        }
        catch (Throwable t)
        {
            return new RemoteCallResult(false, ContainerUtil.printStackTrace(t, "com.sonicsw.mf.framework.agent.HostManager.setupContainer failed:"));
        }
    }

    @Override
    public IRemoteCallResult deleteFiles(String pathOfdirectoryOrFile, Boolean deleteContent0)
    {
        File fileToDelete = new File(pathOfdirectoryOrFile);
        boolean deleteContent = deleteContent0 != null && deleteContent0.booleanValue();

        if (!fileToDelete.exists())
        {
            return new RemoteCallResult(true, null);
        }

        if (!deleteContent && fileToDelete.isDirectory() && fileToDelete.listFiles().length > 0)
        {
            return new RemoteCallResult(false, fileToDelete.getAbsolutePath() + " is not empty");
        }

        if (deleteFile(fileToDelete))
        {
            return new RemoteCallResult(true, null);
        }
        else
        {
            return new RemoteCallResult(false, "Failed to delete " + fileToDelete.getAbsolutePath());
        }

    }
    @Override
    public IRemoteCallResult launchContainer(String containerName, Boolean launchAsWindowsService)
    {

        return launchContainerInternal(SONIC_CONTAINERS_DIR, containerName, launchAsWindowsService);
    }

    @Override
    public IRemoteCallResult launchContainer(String sonicHome, String containerName, Boolean launchAsWindowsService)
    {
        try
        {
            return launchContainerInternal(ContainerSetup.findSonicContainersDir(new File(sonicHome)), containerName, launchAsWindowsService);
        }
        catch (Throwable t)
        {
            return new RemoteCallResult(false, ContainerUtil.printStackTrace(t, "com.sonicsw.mf.framework.agent.HostManager.launchContainer failed:"));
        }
    }

    private IRemoteCallResult launchContainerInternal(File containersDir, String containerName, Boolean launchAsWindowsService0)
    {
        boolean launchAsWindowsService = launchAsWindowsService0 == null ? false : launchAsWindowsService0.booleanValue();
        File workingDir = new File(containersDir, m_domainName  + "." + containerName);

        if (!workingDir.exists() || !workingDir.isDirectory())
        {
            return new RemoteCallResult(false, workingDir.getAbsolutePath() + " does not exist");
        }

        if (!workingDir.isDirectory())
        {
            return new RemoteCallResult(false, workingDir.getAbsolutePath() + " is not a directory");
        }

        File launchScript = null;
        String outputFileName = "/dev/null";
        if (launchAsWindowsService)
        {
            if (!ContainerSetup.ON_WINDOWS)
            {
                return new RemoteCallResult(false,  "The container can be launched as a Windows Service on Windows only");
            }
            launchScript = new File(workingDir, ContainerSetup.WINDOWS_SERVICE_SCRIPT);
        }
        else if (ContainerSetup.ON_WINDOWS)
        {
            launchScript = new File(workingDir, LAUNCH_CONTAINER_SCRIPT+".bat");
        }
        else
        {
            launchScript = new File(workingDir, ContainerSetup.UNIX_BATCH_LAUNCHER_SCRIPT);

            if (!Boolean.getBoolean("sonic.mf.no_batch_output"))
            {
                //Temporary until full console output policu is finalized
                outputFileName = new File(workingDir, containerName + ".out").getAbsolutePath();
            }
        }

        if (!launchScript.exists())
        {
            return new RemoteCallResult(false, launchScript.getAbsolutePath() + " does not exist");
        }

        ArrayList<String> exeCmd = new ArrayList<String>();

        if (launchAsWindowsService)
        {
            exeCmd.add("cmd.exe");
            exeCmd.add("/C");
            exeCmd.add("\"" + launchScript.getAbsolutePath() +"\"");
            exeCmd.add("/start");
        }
        else if (ContainerSetup.ON_WINDOWS)
        {
            exeCmd.add("cmd.exe");
            exeCmd.add("/C");
            exeCmd.add("start");
            // add another cmd /C here because just using start will implicitly do cmd /K (e.g. not close the prompt).
            exeCmd.add("cmd.exe");
            exeCmd.add("/C");
            exeCmd.add("\"" + launchScript.getAbsolutePath() +"\"");
        }
        else
        {
            exeCmd.add(launchScript.getAbsolutePath());
            exeCmd.add(outputFileName);
        }

        IRemoteExecResult result = remoteExec(exeCmd.toArray(new String[exeCmd.size()]), null, workingDir.getAbsolutePath(), null);
        if (!result.isSuccessful())
        {
            return new RemoteCallResult(false, CONTAINER_LAUNCH_FAILED_TEXT + result.getOutput());
        }

        if (result.getExitCode() != 0)
        {
            return new RemoteCallResult(false, CONTAINER_LAUNCH_FAILED_TEXT + ExecUtility.execResultOutputToString(result));
        }

        return new RemoteCallResult(true, null);
    }

    @Override
    public IRemoteCallResult removeContainer(String containerName)
    {
        String fullContainerName = m_domainName  + "." + containerName;
        File workingDir = new File(SONIC_CONTAINERS_DIR, fullContainerName);
        String logNameBasename = fullContainerName + FILELOG_SUFFIX;

        File logDir = getContainerLogDirectory(workingDir, logNameBasename);
        if (!logDir.isAbsolute())
        {
            // logDir is relative to the working dir
            logDir = new File(workingDir, logDir.getPath());
        }
        File logFile = new File(logDir, logNameBasename);

        StringBuilder output = new StringBuilder();

        try
        {
            com.sonicsw.mf.framework.agent.ci.LauncherContainerDriver.shutdownLocalContainer(workingDir, true, true);
        }
        catch (Exception e)
        {
            return new RemoteCallResult(false, "com.sonicsw.mf.framework.agent.HostManager.removeContainer failed in shutdownLocalContainer: " + e.toString());
        }

        if (ContainerSetup.ON_WINDOWS)
        {
            String winserviceScriptName = new File(workingDir, ContainerSetup.WINDOWS_SERVICE_SCRIPT).getAbsolutePath();
            final String[] waitCmd = new String[]{winserviceScriptName, "/wait_for_stop"};
            remoteExec(waitCmd, null, workingDir.getAbsolutePath(), null);

            final String[] removeCmd = new String[]{winserviceScriptName, "/remove"};
            IRemoteExecResult removeResult = remoteExec(removeCmd, null, workingDir.getAbsolutePath(), null);
            if (removeResult.isSuccessful())
            {
                // intentionally using the default platform encoding to form the string
                output.append(new String(removeResult.getProcessStdOutput()));
            }
            else
            {
                // add stacktrace
                output.append("winserve /remove failed with:\n");
                output.append(removeResult.getOutput());
                output.append('\n');
            }
        }

        boolean ok = deleteFile(workingDir);
        if (!ok)
        {
            output.append("Failed to delete directory \"" + workingDir.getAbsolutePath() + "\"");
        }

        return new RemoteCallResult(ok, output.length() > 0 ? output.toString() : null);
    }

    private File getContainerLogDirectory(File workingDir,
            String logNameBasename)
    {
        File logDir = null;
        File logLocationFile = new File(workingDir, IContainer.LOG_LOCATION_FILE);
        if (logLocationFile.exists())
        {
            try
            {
                BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(logLocationFile)));
                try
                {
                    String line = reader.readLine();
                    if (line != null)
                    {
                        line = line.trim();
                        if (line.length() > 0)
                        {
                            logDir = new File(line);
                        }
                    }
                }
                finally
                {
                    reader.close();
                }
            }
            catch (IOException e)
            {
              // If cannot get the log directory from LOG_LOCATION_FILE assume the default location
            }

        }

        if (logDir == null)
        {
            logDir = new File(workingDir, logNameBasename);
        }
        return logDir;
    }


    @Override
    public Properties getJVMProperties()
    {
        return System.getProperties();
    }

    @Override
    public IRemoteCallResult bytesToFile(byte[] src, String destinationFilePath, Boolean rewriteIfExists0)
    {
        boolean rewriteIfExists = rewriteIfExists0 == null ? false : rewriteIfExists0.booleanValue();

        try
        {
            File destinationFile = new File (destinationFilePath);
            if (destinationFile.exists() && !rewriteIfExists)
            {
                return new RemoteCallResult(false, "com.sonicsw.mf.framework.agent.HostManager.bytesToFile failed: " + destinationFilePath + " already exists");
            }
            if (destinationFile.exists())
            {
                deleteObsolete(destinationFile);
                if (destinationFile.exists())
                {
                    return new RemoteCallResult(false, "com.sonicsw.mf.framework.agent.HostManager.bytesToFile failed: Could not delete" + destinationFilePath);
                }
            }

            new ZipSerializer(destinationFile.getParentFile()).expandZipFromBytes(src, destinationFile.getName());
            return new RemoteCallResult(true, null);


        }
        catch (Throwable t)
        {
            return new RemoteCallResult(false, ContainerUtil.printStackTrace(t, "com.sonicsw.mf.framework.agent.HostManager.bytesToFile failed to unzip into \"" + destinationFilePath + "\":"));
        }

    }

    @Override
    public RemoteCallResultWithData fileToBytes(String srcFilePath)
    {
        try
        {
            File srcFile = new File (srcFilePath);
            if (!srcFile.exists())
            {
                return new RemoteCallResultWithData(false, "com.sonicsw.mf.framework.agent.HostManager.fileToBytes failed: " + srcFilePath + " does not exist", null);
            }

            byte[] zipBytes = new ZipSerializer(srcFile.getParentFile()).zipFileOrDir(srcFile.getName());

            return new RemoteCallResultWithData(true, null, zipBytes);
        }
        catch (Throwable t)
        {
            return new RemoteCallResultWithData(false, ContainerUtil.printStackTrace(t, "com.sonicsw.mf.framework.agent.HostManager.fileToBytes failed to zip \"" + srcFilePath + "\":"), null);
        }

    }

    @Override
    public IRemoteCallResult downloadFileFromDS(String dsFilePath, String destinationFilePath, Boolean rewriteIfExists0, Boolean expandZip0)
    {
        boolean expandZip = expandZip0 == null ? false : expandZip0.booleanValue();
        boolean rewriteIfExists = rewriteIfExists0 == null ? false : rewriteIfExists0.booleanValue();
        if (expandZip)
        {
            return downloadAndExpand(dsFilePath,  destinationFilePath,  rewriteIfExists);
        }
        else
        {
            return downloadToFile(dsFilePath,  destinationFilePath,  rewriteIfExists);
        }
    }

    private IRemoteCallResult downloadAndExpand(String dsFilePath, String destinationPath, boolean rewriteIfExists)
    {
        File destinationFile = new File (destinationPath);
        if (destinationFile.exists())
        {
            if (rewriteIfExists)
            {
                deleteObsolete(destinationFile);
            }
            else
            {
                return new RemoteCallResult(false, "com.sonicsw.mf.framework.agent.HostManager.downloadFileFromDS failed: " + destinationPath + " already exists");
            }

            if (destinationFile.exists())
            {
                return new RemoteCallResult(false, "com.sonicsw.mf.framework.agent.HostManager.downloadFileFromDS failed: Could not delete" + destinationPath);
            }

        }

        File absolutePathFile = new File(destinationFile.getAbsolutePath());
        File hostDir = absolutePathFile.getParentFile();
        String fileName = absolutePathFile.getName();
        File tmpDownloadFile = new File(hostDir, fileName + ".tmp");
        tmpDownloadFile.delete();

        IRemoteCallResult result =  downloadFileFromDSInternal(dsFilePath, tmpDownloadFile);
        if (!result.isSuccessful())
        {
            return result;
        }

        try
        {
            new ZipSerializer(hostDir).expandZip(tmpDownloadFile, destinationFile);
        }
        catch (Throwable t)
        {
            return new RemoteCallResult(false, ContainerUtil.printStackTrace(t, "com.sonicsw.mf.framework.agent.HostManager.downloadFileFromDS failed to unzip into \"" + absolutePathFile.getPath() + "\": "));
        }
        finally
        {
            tmpDownloadFile.delete();
        }
        return new RemoteCallResult(true, null);
    }


    private IRemoteCallResult downloadToFile(String dsFilePath, String destinationFilePath, boolean rewriteIfExists)
    {
        File destinationFile = new File (destinationFilePath);

        if (destinationFile.exists() && !rewriteIfExists)
        {
            return new RemoteCallResult(false, "com.sonicsw.mf.framework.agent.HostManager.downloadFileFromDS failed: " + destinationFilePath + " already exists");
        }
        if (destinationFile.exists() && destinationFile.isDirectory())
        {
            return new RemoteCallResult(false, "com.sonicsw.mf.framework.agent.HostManager.downloadFileFromDS failed: " + destinationFilePath + " is a directory");
        }
        if (destinationFile.exists() && !destinationFile.delete())
        {
            return new RemoteCallResult(false, "com.sonicsw.mf.framework.agent.HostManager.downloadFileFromDS failed: Could not delete" + destinationFilePath);
        }

        return downloadFileFromDSInternal(dsFilePath, destinationFile);
    }

    private void createParents(File filePath) throws IOException
    {
        File parentPath = filePath.getParentFile();
        if (!parentPath.exists() && !parentPath.mkdirs())
        {
            throw new IOException(" \"" + parentPath.getAbsolutePath() + "\" does not exist and cannot be created");
        }
    }

    private IRemoteCallResult downloadFileFromDSInternal(String dsFilePath, File destinationFile)
    {
        RandomAccessFile dest = null;
        boolean success = false;
        IRemoteCallResult remoteCallResult = null;
        try
        {
            createParents(destinationFile);
            dest = new RandomAccessFile(destinationFile, "rw");

            int offset = 0;
            while (true)
            {
                IBlob blob = m_directory.getFSBlob(dsFilePath, false, offset);
                byte[] bytes = blob.getBlobBytes();
                if (bytes == null || bytes.length == 0)
                {
                    break;
                }
                dest.write(bytes);
                offset += bytes.length;
            }
            success = true;
            remoteCallResult = new RemoteCallResult(true, null);
        }
        catch (Throwable t)
        {
            return new RemoteCallResult(false, ContainerUtil.printStackTrace(t, "com.sonicsw.mf.framework.agent.HostManager.downloadFileFromDS failed:"));
        }
        finally
        {
            if (dest != null)
            {
                try
                {
                    dest.close();
                    if (!success)
                    {
                        destinationFile.delete();
                    }
                }
                catch (Throwable t)
                {
                    remoteCallResult = new RemoteCallResult(false, ContainerUtil.printStackTrace(t, "com.sonicsw.mf.framework.agent.HostManager.downloadFileFromDS close failed:"));
                }
            }
        }
        return remoteCallResult;
    }

    private void deleteObsolete(File fileToDelete)
    {
      if (fileToDelete.isDirectory())
      {
          String[] files = fileToDelete.list();
          for (int i=0; i < files.length; i++)
        {
            deleteObsolete(new File(fileToDelete, files[i]));
        }
          fileToDelete.delete();
      }
    else
    {
        fileToDelete.delete();
    }
    }

    @Override
    public IFileDescriptor[] listDirectory(String dirPath)
    {
        File dir = new File(dirPath);
        if (!dir.exists() || !dir.isDirectory())
        {
            return null;
        }

        //Must explicitly set an absolute path so that the result will not be relative to the client's working directory
        File[] files = dir.listFiles();
        IFileDescriptor[] resultList = new IFileDescriptor[files.length];
        for (int i = 0; i < files.length; i++)
        {
            resultList[i] = new FileDescriptor (new File(files[i].getAbsolutePath()));
        }

        return resultList;
    }

    @Override
    public IRemoteExecResult remoteExec(String[] cmdarray, String[] envp, String workDirPath, byte[] inputBytes)
    {
         return ExecUtility.exec(cmdarray, envp, workDirPath, inputBytes);
    }

    //Delete a directory and all its content
    private boolean deleteFile(File fileToDelete)
    {
        boolean ok = true;
        if (fileToDelete.isDirectory())
        {
            String[] files = fileToDelete.list();
            for (int i=0; i < files.length; i++)
            {
                if (!deleteFile(new File(fileToDelete, files[i])))
                {
                    ok = false;
                }
            }
            if (!fileToDelete.delete())
            {
                ok = false;
            }
        }
        else
        {
            if (!fileToDelete.delete())
            {
                ok = false;
            }
        }

        return ok;
    }

    @Override
    public String[] getAllHostnamesAndIPs(Boolean includeLoopback)
    {
        final String hostnames = System.getProperty(SONICSW_MF_HM_HOSTNAMES_PROPERTY);
        if (null != hostnames) {
            return hostnames.split(",");
        } else {
            Set<String> hostnamesAndIps = ProgressInetAddress.getAllHostnamesAndIps(includeLoopback.booleanValue());
            return hostnamesAndIps.toArray(new String[hostnamesAndIps.size()]);
        }
    }

}
