package com.sonicsw.mf.framework.agent;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.jms.BytesMessage;
import javax.jms.Connection;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.management.JMException;
import javax.management.JMRuntimeException;
import javax.management.Notification;
import javax.management.ObjectName;
import javax.management.RuntimeOperationsException;

import com.sonicsw.mx.util.IEmptyArray;
import com.sonicsw.mx.util.LoaderInputStream;

import com.sonicsw.mf.comm.IConnectorClient;
import com.sonicsw.mf.comm.IExceptionListener;
import com.sonicsw.mf.comm.IGlobalComponentListener;
import com.sonicsw.mf.comm.jms.ConnectorClient;
import com.sonicsw.mf.comm.jms.DurableConnector;
import com.sonicsw.mf.common.IComponent;
import com.sonicsw.mf.common.IConsumer;
import com.sonicsw.mf.common.MFException;
import com.sonicsw.mf.common.MFServiceNotActiveException;
import com.sonicsw.mf.common.MgmtMsgTooBigException;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.runtime.IContainerExitCodes;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.impl.CanonicalName;
import com.sonicsw.mf.common.util.ObjectNameHelper;
import com.sonicsw.mf.framework.IConnectorServer;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.IFrameworkComponent;
import com.sonicsw.mf.framework.IPermissionsManager;
import com.sonicsw.mf.framework.security.PermissionsManager;

import progress.message.client.EUserAlreadyConnected;

/**
 * The JMSConnectorServer class is responsible for mapping MF management communications
 * over JMS. It has knowledge of how to map request/reply and notifications over JMS.
 * It provides the following additional functionality over its superclass:
 *
 *   - it is capable of receiving operation requests, executing the operation and
 *     replying the operation return value
 *   - it adds the ability to invoke an operation via a request without waiting
 *     for a reply (valid for operations that return a void).
 *
 * Clients of the JMSConnectorServer are:
 *
 *   - the MF container
 *   - JMX/JMS connector server
 *
 * The JMSConnectorServer transparently provides a level of QoS beyond that of regular
 * SonicMQ JMS messaging. Clients of the JMSConnectorServer are unaware of temporary
 * loss (and re-establishment) of connection.
 */
public class JMSConnectorServer
extends ConnectorClient
implements MessageListener, IExceptionListener
{
    private static final int MF_SUBJECT_ROOT_LENGTH = MF_SUBJECT_ROOT.length();

    // vector of request handlers
    private Vector m_requestHandlers = new Vector();

    // table of global subscriptions
    private HashMap m_globalSubscriptions = new HashMap();

    private ContainerImpl m_container;
    private IConnectorServer m_connectorDelegate;

    private TaskScheduler m_taskScheduler;

    private boolean m_closePending = false;

    static long NOTIFICATION_TTL;
    private static final long NOTIFICATION_TTL_DEFAULT = DurableConnector.DURABLE_SUBSCRIPTION_TTL; // no point in defaulting to something more than this
    private static final String NOTIFICATION_TTL_PROPERTY = "sonicsw.mf.notificationTTL";

    private boolean TRACE = false;

    static
    {
        String notificationTTL = System.getProperty(NOTIFICATION_TTL_PROPERTY);
        NOTIFICATION_TTL = notificationTTL == null ? NOTIFICATION_TTL_DEFAULT : Long.parseLong(notificationTTL);
    }

    /**
     * Allocates a JMSConnectorServer object and associates it with the given
     * container name.
     */
    JMSConnectorServer(ContainerImpl container)
    {
        super(container.getContainerIdentity());
        m_container = container;
        m_taskScheduler = (TaskScheduler)m_container.getTaskScheduler();
        super.setExceptionListener(this);
        super.setConnectionListener(m_container);
        super.m_inContainer = true;  // indicate that ConnectorClient instance resides in an MF container
    }

    /**
     * @see com.sonicsw.mf.comm.IExceptionListener#onException(Exception)
     */
    @Override
    public void onException(Exception exception)
    {
        if (TRACE)
        {
            System.out.println("JMSConnectorServer.onException: isConnected = " + this.isConnected() + ", exception msg = " + exception.getMessage());
        }

        // The exception will be an indication of a
        // permanent failure - the container will be shut down...
        if (!m_container.isClosing())
        {
            if (exception instanceof JMSException && ((JMSException)exception).getLinkedException() instanceof EUserAlreadyConnected)
            {
                logMessage('"' + m_container.getContainerIdentity().getCanonicalName() + "\" appears to be running elsewhere", Level.SEVERE);
                m_container.shutdown(IContainerExitCodes.CONTAINER_ALREADY_RUNNING_EXIT_CODE);
            }
            else
            {
                logMessage("Permanent management connection failure, trace follows...", exception, Level.SEVERE);

                if (m_container.isBooted())
                {
                    m_container.m_agent.shutdown(IContainerExitCodes.COMMS_FAILURE_EXIT_CODE);
                }
                else
                {
                    m_container.shutdown(IContainerExitCodes.COMMS_FAILURE_EXIT_CODE);
                }
            }
        }
    }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#close()
     */
    @Override
    public void close()
    {
        // remove all the global subscriptions now
        IConsumer[] globalSubscriptions = (IConsumer[])m_globalSubscriptions.values().toArray(new IConsumer[0]);
        for (int i = 0; i < globalSubscriptions.length; i++)
        {
            if (globalSubscriptions[i] != null)
            {
                globalSubscriptions[i].close();
            }
        }

        // remove the main subscription
        IConsumer subscription = super.m_subscription;
        if (subscription != null)
        {
            subscription.close();
        }

        super.close();

        // remove any pending requests from the request handlers' list(s)
        m_requestHandlers.removeAllElements();
    }

    @Override
    public void closePending()
    {
        m_closePending = true;

        super.closePending();

        // Subscriptions to global topics will be removed during the
        // invocation of the "close" method, rather than here.  Thus,
        // outstanding requests that need to be completed prior to
        // shutdown may be handled even after "closePending" is called.
    }

    /**
     * Return a connected MF connection.
     * @return
     */
    Connection getConnection() {
        return m_durableConnector.getConnection(); //call DurableConnector.getMFConnection()
    }

    /**
     * Adds a handler for request messages.
     *
     * @param handler A handler that provides a dynamic invocation method to handle
     *                a request.
     *
     * @return        A consumer object on which close() should be called to
     *                remove and cleanup on behalf of the request handler.
     *
     * @see com.sonicsw.mf.common.IConsumer#close()
     */
    IConsumer addRequestHandler(ContainerImpl.RequestHandler handler)
    {
        RequestHandlerDelegate delegate = new RequestHandlerDelegate(handler);
        m_requestHandlers.addElement(delegate);
        return delegate;
    }

    IConnectorServer getConnectorServer()
    {
        synchronized(this)
        {
            if (m_connectorDelegate == null)
            {
                // inner class stops casting
                m_connectorDelegate = new IConnectorServer()
                {
                    @Override
                    public long getConnectTimeout() { return JMSConnectorServer.this.getConnectTimeout(); }
                    @Override
                    public long getSocketConnectTimeout() { return JMSConnectorServer.this.getSocketConnectTimeout(); }
                    @Override
                    public long getRequestTimeout() { return JMSConnectorServer.this.getRequestTimeout(); }
                    @Override
                    public void setConnectTimeout(long timeout) { JMSConnectorServer.this.setConnectTimeout(timeout); }
                    @Override
                    public void setSocketConnectTimeout(long timeout) { JMSConnectorServer.this.setSocketConnectTimeout(timeout); }
                    @Override
                    public void setRequestTimeout(long timeout) { JMSConnectorServer.this.setRequestTimeout(timeout); }
                    @Override
                    public boolean isConnected() { return JMSConnectorServer.this.isConnected(); }
                };
            }
        }

        return m_connectorDelegate;
    }

    /**
     * Remove a handler of requests.
     *
     * @param delegate The delegate that represents the request handler.
     *
     * @see #addRequestHandler(IRequestHandler, String)
     */
    private void removeRequestHandler(RequestHandlerDelegate delegate)
    {
        m_requestHandlers.removeElement(delegate);
    }

    /**
     * Global components are those MF components that can be globally
     * addressed within the domain and do not require definition of
     * the container in which they reside. Examples of global components
     * are the Directory Service and the Agent Manager.
     *
     * An instance name should be provided when there are identifiable
     * instances of the same global component, e.g. "PRIMARY" and "BACKUP".
     *
     * This method should be called to have the JMSConnectorServer listen for
     * requests for for the given global component instance.
     */
    void addGlobalComponentSupport(String globalID, String instanceID, IGlobalComponentListener globalComponentListener)
    throws JMSException
    {
        String pseudoContainerID = getPseudoContainerID(globalID, instanceID);

        IConsumer subscription = (IConsumer)m_globalSubscriptions.remove(pseudoContainerID);

        if (subscription != null)
        {
            subscription.close();
        }

        while (!(m_closePending || m_container.isClosing()))
        {
            try
            {
                String subject = MF_SUBJECT_ROOT + "*." + super.m_domainName + '.' + pseudoContainerID;
                String subscriptionName = super.m_domainName + '/' + pseudoContainerID;
                subscription = super.m_durableConnector.subscribe(subject, subscriptionName, this, globalComponentListener);
                break;
            }
            catch(javax.jms.IllegalStateException e)
            {
                // this should be due to the connection currently being closed, so we should be able to retry
                try { Thread.sleep(1000); } catch (InterruptedException ie) { } // wait a small time before retrying
                
                if ((JMSConnectorServer.super.m_traceMask & IConnectorClient.TRACE_CONNECTION_FAILURES) > 0 && (JMSConnectorServer.super.m_traceMask & IComponent.TRACE_VERBOSE) > 0)
                {
                    JMSConnectorServer.this.logMessage("Retrying to create global subscription for '" + globalID + "' due to... ", e, Level.TRACE);
                }
            }
        }
        
        if (m_closePending || m_container.isClosing())
        {
            return;
        }
        
        if (TRACE)
        {
            System.out.println("JMSConnectorServer.addGlobalComponentSupport: subscription = " + subscription + ", pseudoContainerID = " + pseudoContainerID + ", instanceID = " + instanceID);
        }

        m_globalSubscriptions.put(pseudoContainerID, subscription);
    }

    /**
     * This method should be called to have the JMSConnectorServer stop listening
     * for requests for the given global component.
     *
     * An instance name should be provided when there are identifiable
     * instances of the same global component, e.g. "PRIMARY" and "BACKUP".
     *
     * @see #addGlobalComponentSupport(String, String)
     */
    void removeGlobalComponentSupport(String globalID, String instanceID)
    {
        String pseudoContainerID = getPseudoContainerID(globalID, instanceID);

        IConsumer subscription = (IConsumer)m_globalSubscriptions.remove(pseudoContainerID);
        if (subscription != null)
        {
            subscription.close();
        }
    }

    /**
     * Send a notification to the directed destination (a specific remote listener).
     */
    void sendDirectedNotification(String commsType, String destination, int listenerHash, Notification notification, Object handback)
    {
        boolean isMFNotification = notification instanceof INotification;
        
        boolean isTraceable = !isMFNotification || !(((INotification)notification).getEventName().equals(Agent.LOG_MESSAGE_NOTIFICATION_TYPE));
        
        // this next section of code purely relates to tracing
        int traceMask = 0;
        ObjectName source = null;
        String forwardedBy = null;
        CanonicalName forwarderName = null;
        AbstractMBean mBean = null;
        
        if (isTraceable)
        {
            source = (ObjectName)notification.getSource();
            if (isMFNotification)
            {
                forwardedBy = (String)((INotification)notification).getAttributes().get("ForwardedBy");
            }
            if (forwardedBy != null)
            {
                forwarderName = new CanonicalName(forwardedBy);
                mBean = m_container.getMBeanNonBlocking(forwarderName.getComponentName());
            }
            else
            {
                if (isMFNotification)
                {
                    mBean = m_container.getMBeanNonBlocking(source.getKeyProperty("ID"));
                }
            }
            
            if (mBean != null)
            {
                IComponent component = mBean.m_component;
                if (component != null)
                {
                    traceMask = component.getTraceMask().intValue();
                }
            }
        }

        try
        {
            // build the notification message and its properties
            BytesMessage message = new progress.message.jimpl.BytesMessage();
            message.setStringProperty(JMS_COMMS_TYPE_PROPERTY, commsType);
            message.setIntProperty(JMS_LISTENER_ID_PROPERTY, listenerHash);
            message.setShortProperty(JMS_CONTENT_TYPE_PROPERTY, NOTIFICATION_CONTENT_TYPE);

            // set the body (content)
            ByteArrayOutputStream arrayOut = new ByteArrayOutputStream(5120);
            ObjectOutputStream out = new ObjectOutputStream(arrayOut);
            out.writeObject(notification);
            out.writeObject(handback);
            out.flush();
            message.writeInt(arrayOut.size());
            message.writeBytes(arrayOut.toByteArray());
            out.close();

            // create an object array that contains initial state information
            String requestID = "3";  // dummy requestID
            Object[] reqArr = super.buildRequestArray(destination,requestID);

            if (isTraceable && (traceMask & IComponent.TRACE_NOTIFICATIONS) > 0 && (traceMask & IComponent.TRACE_VERBOSE) > 0)
            {
                if (forwardedBy == null)
                {
                    m_container.logMessage(isMFNotification ? source.getKeyProperty("ID") : source.getCanonicalName(), "Sending notification [" + notification.getType() + "] to JMX client " + getJMXClientHostAndID(destination), Level.TRACE);
                }
                else
                {
                    m_container.logMessage(forwarderName.getComponentName(), "Forwarding notification [" + notification.getType() + "] to JMX client " + getJMXClientHostAndID(destination), Level.TRACE);
                }
            }

            // publish the message
            super.m_durableConnector.publish(destination, message, super.m_requestTimeout, NOTIFICATION_TTL, reqArr);
        }
        catch(Throwable e)
        {
            if (isTraceable && (traceMask & IComponent.TRACE_NOTIFICATIONS) > 0 && (traceMask & IComponent.TRACE_VERBOSE) > 0)
            {
                if (forwardedBy == null)
                {
                    m_container.logMessage(isMFNotification ? source.getKeyProperty("ID") : source.getCanonicalName(), "Failed to send notification [" + notification.getType() + "] to JMX client " + getJMXClientHostAndID(destination) + ", trace follows", e, Level.TRACE);
                }
                else
                {
                    m_container.logMessage(forwarderName.getComponentName(), "Failed to forward notification [" + notification.getType() + "] to JMX client " + getJMXClientHostAndID(destination) + ", trace follows", e, Level.TRACE);
                }
            }
            
            return;
        }
    }
    
    String getJMXClientHostAndID(String destination)
    {
        StringBuffer sb = new StringBuffer("(host=");
        destination = destination.substring(destination.indexOf(".JMXCLIENT.") + 11);
        int index = destination.indexOf('.');
        
        sb.append(destination.substring(0, index).replace('_', '.'));
        sb.append(", id=");
        sb.append(destination.substring(index + 1));
        sb.append(')');
        
        return sb.toString();
    }

    @Override
    public void logMessage(String message, int severityLevel)
    {
        m_container.logMessage(null, message, severityLevel);
    }

    @Override
    public void logMessage(String message, Throwable exception, int severityLevel)
    {
        m_container.logMessage(null, message, exception, severityLevel);
    }

    //
    // MessageListener interface
    //

    /**
     * Receive an rpc request/reply mapped to a JMS message and remap to an invocation
     * on the handler.
     */
    @Override
    public void onMessage(final Message rpcMessage)
    {
        // ignore duplicates
        if (isRedelivered(rpcMessage))
        {
            return;
        }

        // if this is not a request then pass to the super class
        short contentType;
        try
        {
            contentType = rpcMessage.getShortProperty(JMS_CONTENT_TYPE_PROPERTY);
        }
        catch(JMSException e)
        {
            logMessage("Bad MF JMS message content type received .. dropping message, trace follows...", e, Level.WARNING);
            return;
        }

        // if the container is in the midst of closing, do not process any more
        // requests; replies, on the other hand, should still be handled [as they might
        // be part of the shutdown process].
        if ((m_closePending) && (contentType == REQUEST_CONTENT_TYPE))
        {
            return;
        }

        if (contentType != REQUEST_CONTENT_TYPE)
        {
            super.onMessage(rpcMessage);
            return;
        }

        // else this is a request message, and this class knows how to deal with requests

        // get the list of handlers
        RequestHandlerDelegate[] handlers = null;
        synchronized(m_requestHandlers)
        {
            handlers = new RequestHandlerDelegate[m_requestHandlers.size()];

            if (handlers.length > 0)
            {
                m_requestHandlers.copyInto(handlers);
            }
        }

        // determine the userID
        String userID = null;
        try
        {
            userID = rpcMessage.getStringProperty("JMSXUserID");
        }
        catch(JMSException e)
        {
            logMessage("Failed to obtain associated UserID .. dropping message, trace follows...", e, Level.WARNING);
            return;
        }

        String commsType = null;
        try
        {
            commsType = rpcMessage.getStringProperty(JMS_COMMS_TYPE_PROPERTY);
            // if this is internal comms and the messages came from an external client then the client
            // is attempting to spoof internal comms
            if (!IContainer.PRE75_SECURITY && commsType.equals(IContainer.INTERNAL_COMMS_TYPE))
            {
                if (m_container.m_agent != null) // there is a small window on startup that we cannot check because the m_agent has not been set
                {
                    IPermissionsManager permissionsManager = m_container.m_agent.getPermissionsManager();
                    if (permissionsManager != null && ((PermissionsManager)permissionsManager).isPermissionsCheckingEnabled())
                    {
                        String topic = rpcMessage.getJMSDestination().toString();
                        int index = topic.indexOf("::");
                        if (index > -1)
                        {
                            topic = topic.substring(index + 2);
                        }
                        if (!topic.startsWith(ConnectorClient.MF_SUBJECT_ROOT + "MFCLIENT."))
                        {
                            StringBuffer sb = new StringBuffer("Attempted impersonation (");
                            if (userID != null && userID.length() > 0)
                            {
                                sb.append("user=").append(userID).append(", ");
                            }
                            StringTokenizer st = new StringTokenizer(topic, ".");
                            st.nextToken();
                            st.nextToken();
                            sb.append("clientType=").append(st.nextToken());
                            sb.append(") ... dropping management request");
                            logMessage(sb.toString(), Level.SEVERE);
                            return;
                        }
                    }
                }
            }
        }
        catch(JMSException e)
        {
            logMessage("Bad MF JMS message comms type received .. dropping message, trace follows...", e, Level.SEVERE);
            return;
        }

        // if the message has a TTL, get it
        long expirationTime = 0;
        try
        {
            expirationTime = rpcMessage.getJMSExpiration();
        }
        catch (JMSException e)
        {
            // don't need to do anything - just use the default TTL value...
        }

        // determine the role
        String role = null;
        try
        {
            role = rpcMessage.getJMSDestination().toString();
            role = role.substring(MF_SUBJECT_ROOT_LENGTH);
            role = role.substring(0, role.indexOf('.'));
        }
        catch(JMSException e)
        {
            logMessage("Bad MF JMS message destination received .. dropping message, trace follows...", e, Level.WARNING);
            return;
        }

        for (int i = 0; i < handlers.length; i++)
        {
            if (handlers[i].commsType.equals(commsType))
            {
                final RequestHandlerDelegate handler = handlers[i];
                final String requestorRole = role;
                final String requestorUserID = userID;
                Runnable invoker = new Runnable()
                {
                    @Override
                    public void run()
                    {
                        ((TaskScheduler.ExecutionThread)Thread.currentThread()).setRole(requestorRole);
                        ((TaskScheduler.ExecutionThread)Thread.currentThread()).setUserID(requestorUserID);
                        handler.onMessage(rpcMessage);
                    }
                };
                // incoming requests should be given a higher priority
                //m_taskScheduler.scheduleTask(invoker, true);
                m_taskScheduler.scheduleTask(invoker, true, expirationTime);  // JP Morgan-related Performance Enhancements [see Section 2.3 of the Feature Summary for Performance Enhancements in vobs_specs]
                return;
            }
        }
    }

    /**
     * Class acts as a structure to hold a MF request handler's details and
     * act as a delegate to the handler.
     *
     * This class includes the mapping between JMS based requests and a neutral
     * invoke format.
     */
    public final class RequestHandlerDelegate
    implements MessageListener, IConsumer
    {
        ContainerImpl.RequestHandler handler;
        String commsType;

        private RequestHandlerDelegate(ContainerImpl.RequestHandler handler)
        {
            this.handler = handler;
            this.commsType = handler.getHandlerCommsType();
        }

        /**
         * Receive a request mapped to a JMS message and remap to an invocation
         * on the handler.
         */
        @Override
        public void onMessage(Message message)
        {
            long onMessageStart = 0;
            long onMessageBeforeInvoke = 0;
            long onMessageAfterInvoke = 0;
            long onMessageBeforePublish = 0;
            long onMessageAfterPublish = 0;
            if (TRACE)
            {
                onMessageStart = System.currentTimeMillis();
                onMessageBeforeInvoke = onMessageStart;
                onMessageAfterInvoke = onMessageStart;
                onMessageBeforePublish = onMessageStart;
                onMessageAfterPublish = onMessageStart;
            }
            String replyToSubject = null;
            boolean isOneway = false;
            long replyTimeout = JMSConnectorServer.super.m_requestTimeout;
            ObjectName objectName = null;
            String invokeMethod = null;
            String[] signature = null;
            Object[] params = null;

            BytesMessage requestMessage = (BytesMessage)message;
            BytesMessage reply = null;

            try
            {
                replyToSubject = requestMessage.getStringProperty(JMSConnectorServer.JMS_REPLY_TO_PROPERTY);

                try
                {
                    isOneway = requestMessage.getBooleanProperty(JMSConnectorServer.JMS_ONEWAY_REQUEST_PROPERTY);
                } catch(NullPointerException npe) { } // defaults to false

                if (!isOneway)
                {
                    try
                    {
                        replyTimeout = requestMessage.getLongProperty(JMSConnectorServer.JMS_REQUEST_TIMEOUT_PROPERTY);
                    } catch(NumberFormatException nfe) { } // defaults to container request timeout
                }

                String target = requestMessage.getStringProperty(JMSConnectorServer.JMS_REQUEST_TARGET_PROPERTY);
                if (target != null)
                {
                    objectName = new ObjectName(target);
                    
                    // fixup the target name (in case the container name included any global component and FT aspects)
                    objectName = ObjectNameHelper.translateGlobalObjectName(objectName, m_container.getContainerIdentity().getContainerName());
                }

                invokeMethod = requestMessage.getStringProperty(JMSConnectorServer.JMS_REQUEST_OPERATION_PROPERTY);

                if ((JMSConnectorServer.super.m_traceMask & TRACE_REQUEST_REPLY) > 0)
                {
                    StringBuffer msg = new StringBuffer("Request received from " + replyToSubject);
                    if ((JMSConnectorServer.super.m_traceMask & IComponent.TRACE_VERBOSE) > 0)
                    {
                        msg.append(", details...");
                        msg.append("\n\ttarget = ").append((objectName == null ? "MBeanServer" : objectName.getCanonicalName()));
                        msg.append("\n\trequest = ").append(invokeMethod);
                    }
                    JMSConnectorServer.this.logMessage(msg.toString(), Level.TRACE);
                }
            }
            catch(Throwable e)
            {
                if ((JMSConnectorServer.super.m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                {
                    JMSConnectorServer.this.logMessage("Failed to unmarshal request, trace follows...", e, Level.TRACE);
                    if (e instanceof InvocationTargetException)
                    {
                        JMSConnectorServer.this.logMessage("Target exception follows...", ((InvocationTargetException)e).getTargetException(), Level.TRACE);
                    }
                }
                else
                {
                    JMSConnectorServer.this.logMessage("Failed to unmarshal MF request", Level.SEVERE);
                }
                return;
            }

            IComponent component = null;
            if (objectName != null && ObjectNameHelper.isMFComponentName(objectName)) // else its a JMX request directed directly to the MBeanServer
            {
                String id = objectName.getKeyProperty("ID");
                while (component == null) // loop to cover the window between getting the delegate and the component possibly being nulled
                {
                    // get the delegate (waits if the delegate is being reloaded)
                    AbstractMBean mBean = JMSConnectorServer.this.m_container.getMBean(id);
                    if (JMSConnectorServer.this.m_container.isClosing())
                    {
                        return;
                    }
                    if (mBean == null)
                     {
                        break; // allowing it to fall through
                    }
                    component = mBean.getManagedComponent();
                }
            }

            try
            {
                if (requestMessage.readBoolean()) // its an empty payload
                {
                    signature = IEmptyArray.EMPTY_STRING_ARRAY;
                    params = IEmptyArray.EMPTY_OBJECT_ARRAY;
                }
                else
                {
                    byte[] bytes = new byte[requestMessage.readInt()];
                    requestMessage.readBytes(bytes);
                    ByteArrayInputStream arrayIn = new ByteArrayInputStream(bytes);
                    ObjectInputStream in = null;
                    if (component != null && !(component instanceof IFrameworkComponent) && invokeMethod.equals("invoke"))
                    {
                        in = new LoaderInputStream(arrayIn, component.getClass().getClassLoader());
                    }
                    else
                    {
                        in = new ObjectInputStream(arrayIn);
                    }
                    signature = (String[])in.readObject();
                    params = (Object[])in.readObject();
                    in.close();
                }
            }
            catch(Throwable e)
            {
                if ((JMSConnectorServer.super.m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                {
                    JMSConnectorServer.this.logMessage("Failed to unmarshal request, trace follows...", e, Level.TRACE);
                    if (e instanceof InvocationTargetException)
                    {
                        JMSConnectorServer.this.logMessage("Target exception follows...", ((InvocationTargetException)e).getTargetException(), Level.TRACE);
                    }
                }
                else
                {
                    JMSConnectorServer.this.logMessage("Failed to unmarshal MF request", Level.SEVERE);
                }
                return;
            }

            String operationName = invokeMethod;
            if (params.length > 0)
            {
                for (int i = 0; i < params.length; i++)
                {
                    if (params[i] instanceof ObjectName)
                    {
                        params[i] = ObjectNameHelper.translateGlobalObjectName((ObjectName)params[i], m_container.getContainerIdentity().getContainerName());
                    }
                }
            }
            if (invokeMethod.equals("invoke"))
            {
                operationName = (String)params[1];
            }


            Object returnValue = null;
            try
            {
                // if its not oneway, do some preparitory work to speed up the replying when the invoke returns
                if (!isOneway)
                {
                    // build the reply and its properties
                    reply = new progress.message.jimpl.BytesMessage();
                    reply.setJMSCorrelationID(requestMessage.getJMSCorrelationID());
                    reply.setStringProperty(JMSConnectorServer.JMS_COMMS_TYPE_PROPERTY, requestMessage.getStringProperty(JMSConnectorServer.JMS_COMMS_TYPE_PROPERTY));
                    reply.setShortProperty(JMSConnectorServer.JMS_CONTENT_TYPE_PROPERTY, JMSConnectorServer.REPLY_CONTENT_TYPE);
                    reply.setStringProperty(JMSConnectorServer.JMS_REQUEST_TARGET_PROPERTY, objectName == null ? null : objectName.getCanonicalName());
                    reply.setStringProperty(JMSConnectorServer.JMS_REQUEST_OPERATION_PROPERTY, operationName);
                    reply.setLongProperty(JMSConnectorServer.JMS_REQUEST_RECEIVED_PROPERTY, System.currentTimeMillis());
                }

                if (TRACE)
                {
                    onMessageBeforeInvoke = System.currentTimeMillis();
                }
                returnValue = this.handler.invoke(objectName, invokeMethod, params, signature);
                if (TRACE)
                {
                    onMessageAfterInvoke = System.currentTimeMillis();
                }
                if (isOneway)
                {
                    return;
                }
            }
            catch(Throwable e)
            {
               if (JMSConnectorServer.this.m_container.isClosing())
            {
                return;
            }
                if ((JMSConnectorServer.super.m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                {
                    JMSConnectorServer.this.logMessage("Failed \"" + operationName + "\" on " + (objectName == null ? "MBeanServer" : objectName.getCanonicalName()) + ", trace follows...", e, Level.TRACE);
                }
                returnValue = e;

                while (returnValue != null && returnValue instanceof java.lang.reflect.InvocationTargetException)
                {
                    Throwable targetException = ((java.lang.reflect.InvocationTargetException)returnValue).getTargetException();
                    if (targetException != null)
                    {
                        returnValue = targetException;
                    }
                    else
                    {
                        break;
                    }
                }

                // special cases to fix defects
                if (returnValue instanceof JMException || returnValue instanceof JMRuntimeException)
                {
                    // Sonic00021349/Sonic00021350
                    if (replyToSubject.indexOf(".JNDICLIENT.") > 0)
                    {
                        // this is coded into JNDI client reply-to (but no constant defined)
                        returnValue = e.getCause();
                    }
                    else
                    // Sonic00025955
                    if (returnValue instanceof RuntimeOperationsException && ((RuntimeOperationsException)returnValue).getTargetException() instanceof MFServiceNotActiveException)
                    {
                        returnValue = ((RuntimeOperationsException)returnValue).getTargetException();
                    }
                }

                // there will be nothing reported .. switch tracing on to see the exception (TRACE_FAILURES)
                if (isOneway)
                {
                    return;
                }
            }

            try
            {
                // set the reply body (content)
                reply.setLongProperty(JMSConnectorServer.JMS_REQUEST_REPLIED_PROPERTY, System.currentTimeMillis());
                reply.writeBoolean(returnValue == null); // write true if its an empty payload
                if (returnValue != null)
                {
                    final int size = returnValue instanceof IBlob ? IBlob.BLOB_CHUNK_SIZE + 5120 : 5120;
                    ByteArrayOutputStream arrayOut = new ByteArrayOutputStream(size);
                    ObjectOutputStream out = new ObjectOutputStream(arrayOut);
                    out.writeObject(returnValue);
                    out.flush();
                    reply.writeInt(arrayOut.size());
                    reply.writeBytes(arrayOut.toByteArray());
                    out.close();
                }

                // build an object array with the initial state information (used for the "publish" invocation)
                String requestID = "2";  // dummy requestID
                Object[] reqArr = JMSConnectorServer.super.buildRequestArray(replyToSubject,requestID);

                // publish the reply
                if ((JMSConnectorServer.super.m_traceMask & TRACE_REQUEST_REPLY) > 0)
                {
                    JMSConnectorServer.this.logMessage("Reply sent to " + replyToSubject, Level.TRACE);
                }
                if (TRACE)
                {
                    onMessageBeforePublish = System.currentTimeMillis();
                }
                
                try
                {
                    JMSConnectorServer.super.m_durableConnector.publish(replyToSubject, reply, replyTimeout, replyTimeout, reqArr);
                }
                catch(MgmtMsgTooBigException mmtbe)
                {
                    // log exception
                    String targetObj = (objectName == null) ? "MBeanServer" : objectName.getCanonicalName();
                    if ((JMSConnectorServer.super.m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                    {
                        JMSConnectorServer.this.logMessage("Reply too large to send for \"" + operationName + "\" on " + targetObj + ", trace follows...", mmtbe, Level.TRACE);
                    }
                    else
                    {
                        JMSConnectorServer.this.logMessage("Reply too large to send for \"" + operationName + "\" on " + targetObj + ":" + mmtbe.toString(), Level.SEVERE);
                    }
                    
                    // build distinct exception for caller
                    MFException returnException =
                        new MFException("Request to " + targetObj + " for \"" + operationName
                                                   + "\" generated response which exceeded maximum size for management messages");
                    
                    // attempt to rebuild and republish reply with exception in place of original return object
                    // so caller knows what's happened and isn't left waiting.  The headers are left as currently set.
                    reply.clearBody();  // clear existing content and ensure message is writeable (MessageSizeValidator.writeMsgToFile() may have made it read-only)
                    reply.writeBoolean(false);  // payload is not empty
                    ByteArrayOutputStream arrayOut = new ByteArrayOutputStream(5120);
                    ObjectOutputStream out = new ObjectOutputStream(arrayOut);
                    out.writeObject(returnException);  // write exception
                    out.flush();
                    reply.writeInt(arrayOut.size());
                    reply.writeBytes(arrayOut.toByteArray());
                    out.close();
                    
                    JMSConnectorServer.super.m_durableConnector.publish(replyToSubject, reply, replyTimeout, replyTimeout, reqArr);
                }
                
                if (TRACE)
                {
                    onMessageAfterPublish = System.currentTimeMillis();
                }
            }
            catch(Throwable e)
            {
                if (JMSConnectorServer.this.m_container.isClosing())
                {
                    return;
                }

                if ((JMSConnectorServer.super.m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                {
                    JMSConnectorServer.this.logMessage("Failed to marshal reply to \"" + operationName + "\" on " + (objectName == null ? "MBeanServer" : objectName.getCanonicalName()) + ", trace follows...", e, Level.TRACE);
                }
                else
                {
                    JMSConnectorServer.this.logMessage("Failed to marshal reply to \"" + operationName + "\" on " + (objectName == null ? "MBeanServer" : objectName.getCanonicalName()) + ":" + e.toString(), Level.SEVERE);
                }
                return;
            }
            if (TRACE)
            {
                long onMessageFinished = System.currentTimeMillis();
                System.out.println("Task's onMessage time for " + invokeMethod + " in thread " + Thread.currentThread().toString() +
                        "\n\tunmarshall " + (onMessageBeforeInvoke - onMessageStart) +
                        "\n\tinvoke " + (onMessageAfterInvoke - onMessageBeforeInvoke) +
                        "\n\tmarshall " + (onMessageBeforePublish - onMessageAfterInvoke) +
                        "\n\tpublish " + (onMessageAfterPublish - onMessageBeforePublish) +
                        "\n\ttotal " + (onMessageFinished - onMessageStart)
                );
            }
        }

        //
        // IConsumer interface
        //

        @Override
        public void close()
        {
            JMSConnectorServer.this.removeRequestHandler(this);
        }
    }
}
