package com.sonicsw.mf.framework.agent;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.StringTokenizer;

import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.common.IBlobSource;
import com.sonicsw.mf.common.ILogger;
import com.sonicsw.mf.common.MFException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.VersionOutofSyncException;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.agent.cache.CacheException;
import com.sonicsw.mf.framework.agent.cache.IConfigCache;
import com.sonicsw.mf.framework.agent.cache.IConfigCacheView;
import com.sonicsw.mf.framework.agent.cache.LatestVersionMissingException;
import com.sonicsw.mf.framework.agent.cache.PersistentCacheException;

public class LocalFileManager
{
    private static final String SONIC_HOME_PROTOCOL = "sonichome:///";
    private static final int SONIC_HOME_PROTOCOL_LENGTH = SONIC_HOME_PROTOCOL.length();
    private static final String SONIC_HOME_PROTOCOL_PREFIX = SONIC_HOME_PROTOCOL.substring(0, SONIC_HOME_PROTOCOL_LENGTH - 1);
    private static final int SONIC_HOME_PROTOCOL_PREFIX_LENGTH = SONIC_HOME_PROTOCOL_PREFIX.length();
    private static final String SONICSW_HOME = System.getProperty(IContainer.SONICSW_HOME_PROPERTY);


    private IConfigCache m_configCache;
    private IConfigCacheView m_configCacheView;
    private IBlobSource m_blobSource;
    private ILogger m_logger;
    private String m_containerID;

    private int m_traceMask = 0;


    private boolean DEBUG = false;

    // use this constructor when using the local cached files through a container
    // containerID == null when this is used without a container
    public LocalFileManager(IBlobSource blobSource, IConfigCache cache, ILogger logger, String containerID)
        throws CacheException
    {
          m_blobSource = blobSource;
          m_configCache = cache;
          m_configCacheView = cache.getCacheView();
          m_logger = logger;
          m_containerID = containerID;
    }

    void setBlobSource(IBlobSource blobSource)
    {
         m_blobSource = blobSource;
    }

    public File getLocalFile(String locations, String logicalName) throws MFException, CacheException
    {
        return getLocalFile(locations, logicalName, null);
    }

    public File getLocalFile(String locations, String logicalName, String[] dsLogicalPath) throws MFException, CacheException
    {
        File theFile = null;
        LogicalFile file = null;
        StringTokenizer locTokenizer = new StringTokenizer(locations, ";");
        while (locTokenizer.hasMoreElements())
        {
            String location = locTokenizer.nextToken();

            if (DEBUG)
            {
                System.out.println("LocalFileManager.getLocalFile trying location == " + location + " logicalName == " + logicalName);
            }
            file = localFileFromLocation(location, logicalName, false, dsLogicalPath);
            if (file != null)
            {
                if (DEBUG)
                {
                    System.out.println("LocalFileManager.getLocalFile returning file " + file.getFileName() +
                                                  " found at location " + location);
                }
                // we assume getLocalFile is used for archives that are cached, unlike localFileFromLocation which could
                // get called for a classpath entry that is not cached (location == null)
                return new File(file.getFileName());
            }
            // The exceptions that might be thrown here:
            // if the persistent cache is not setup, if there's a problem copying the file to the cache
            // or (DirectoryServiceException) from lower level DS calls. If the DS is not
            // available, localFileFromLocation will return null (given a location, otherwise it will
            // try to find the most recent but not latest version)
        }
        if (file == null)
        {
            // try to get the most recently cached file
            try
            {
                if (DEBUG)
                {
                    System.out.println("LocalFileManager.getLocalFile about to call getAvailableFileByLogicalName for logicalName " + logicalName);
                }
                String useLogicalName = logicalName;
                if (!useLogicalName.startsWith("/"))
                {
                    useLogicalName = "/" + useLogicalName;
                }
                theFile = m_configCacheView.getAvailableFileByLogicalName(useLogicalName);
                if (DEBUG)
                {
                    if (theFile == null)
                    {
                        System.out.println("getAvailableFileByLogicalName in LocalFileManager.getLocalFile returned null");
                    }
                    else
                    {
                        System.out.println("getAvailableFileByLogicalName in LocalFileManager.getLocalFile returned " + theFile.getAbsolutePath());
                    }
                }
            }
            catch (Exception e)
            {
                if (DEBUG)
                {
                    System.out.println("LocalFileManager.getLocalFile caught exception, TRACE FOLLOWS");
                    e.printStackTrace();
                }
                throw new PersistentCacheException(e.toString());
            }
        }

        return theFile;
    }
    // This method is used from several different code paths, and so has the merged logic of several previous
    // container file parsing methods. The "nativeFile"
    // argument is true when the container is processing the NATVIE_LIBRARIES attribute in its config. Native files,
    // when cached, need to end up in the native_libraries directory.
    public LogicalFile localFileFromLocation(String location0, String logicalName0, boolean nativeFile, String[] dsLogicalPath)
        throws MFException, CacheException
    {
        LogicalFile file = null;
        // We'll set dsLogicalPath later if we find it in the DS or the cache
        if (dsLogicalPath != null)
        {
            dsLogicalPath[0] = null;
        }

        String location =  substituteSonicHome(location0);
        String logicalName = substituteSonicHome(logicalName0);

        if (DEBUG)
        {
            System.out.println("LocalFileManager.localFileFromLocation looking for logicalName " + logicalName + " location == " + location);
        }
        // is logicalName a file on disk
        String sonicFS = IContainer.DS_CLASSPATH_PROTOCOL.substring(0, IContainer.DS_CLASSPATH_PROTOCOL.length() - 1);
        String testSonicFS = logicalName;
        if (location != null)
        {
            // fix location. We add a trailing slash if the location doesn't end in a slash
            // to allow legal URL + logical name combinations. If there is a location specified,
            // we assume logicalName does not have a slash as the first letter. Allowed:
            // location == <protocol>:// logicalName == a/b/c  (we add the extra "/")
            // location == <protocol>:///lib logicalName == a/b/c (we add the extra "/")
            // location === <protocol>://Domain1 logicalName == a/b/c (we add the extra "/")
            if (!location.endsWith("/") || location.endsWith("://"))
            {
                location = location + "/";
            }
            testSonicFS = location + logicalName;
        }

        // test the sonicfs:/// syntax
        String firstNCharacters = null;
        if (testSonicFS.length() >= sonicFS.length())
        {
            firstNCharacters = testSonicFS.substring(0, sonicFS.length());
        }
        if (DEBUG)
        {
            System.out.println("LocalFileManager.localFileFromLocation comparing " + firstNCharacters + " to " + sonicFS);
        }
        boolean isSonicFS = (firstNCharacters != null) && firstNCharacters.equals(sonicFS);
        if ((isSonicFS || location != null) && m_configCache.getCacheRootDirectory() == null)
        {
            throw new MFException("Component classpath cannot include Directory Service based archives as configuration cache is not enabled.");
        }
        // if location was specified, and it wasn't sonicfs, we deal with it separately
        if (location != null && !isSonicFS)
        {
            return getFileFromOtherURL(location, logicalName);
        }
        // if location is not specified, then the file is either sonicfs or a local disk file
        if (location == null && (firstNCharacters == null || !firstNCharacters.equals(sonicFS)))
        {
            if (DEBUG)
            {
                System.out.println("LocalFileManager.localFileFromLocation trying URL from logicalName, without caching " + logicalName);
            }
            try
            {
                URL url = new URL(logicalName);
                if (url.getProtocol().equals("file"))
                {
                    if (new File(url.getFile()).isDirectory() && !logicalName.endsWith("/"))
                    {
                        logicalName += '/';
                        url = new URL(logicalName);
                    }
                }
                file = new LogicalFile(logicalName, url);
            }
            catch (MalformedURLException e)
            {
                try
                {
                    // it's a file. get a file URL
                    File fileForUrl = new File(logicalName);
                    file = new LogicalFile(logicalName, fileForUrl.toURL());
                }
                catch (MalformedURLException e2)
                {
                   //we're really not expecting an exception here. file:// can go in front of anything
                   throw new MFException("Unable to create a URL for " + logicalName);
                }
            }
            return file;
        }

        // SONICFS:
        // go find the archived file
        //strip off sonicfs:/// or sonicfs://<domain>/ using the URL constructor
        String fileName = null;
        try
        {
            //Use a file URL to extract the file part since the the sonicfs handler is not yet always established
            fileName = new URL("file" +  testSonicFS.substring("sonicfs".length())).getFile();
        }
        catch (MalformedURLException urlEx)
        {
            throw new MFException("Invalid sonicfs URL: " + testSonicFS);
        }
        if (dsLogicalPath != null)
        {
            dsLogicalPath[0] = fileName;
        }
        File blobFile = null;
        // first try to get the requested configuration from the cache
        try
        {
            if (DEBUG)
            {
                System.out.println("LocalFileManager.localFileFromLocation trying the cache for " + fileName);
            }
            blobFile = m_configCacheView.getFileByLogicalName(fileName, nativeFile);
        }
        catch (LatestVersionMissingException missingE)
        {
            // if it was not available, can we get it from the directory service
              if (DEBUG)
            {
                System.out.println("LocalFileManager.localFileFromLocation caught LatestVersionMissingException, trying from the DS ");
            }
              blobFile = getBlobFromDSAndCache(fileName, nativeFile, (location != null));
        }
        if (blobFile != null && location != null)
        {
            // in the sonicfs case, if location is not null, and the file from the cache
            // was previously fetched from a non-ds source, we will recache it from the DS
            IElement cachedFile = m_configCacheView.getElementByLogicalName(fileName);
            if (isNonDSFile(cachedFile))
            {
                if (DEBUG)
                {
                    System.out.println("LocalFileManager.localFileFromLocation switched sources to the DS, recaching the file " );
                }
                blobFile = getBlobFromDSAndCache(fileName, false, true);
            }
        }
        if (blobFile != null)
        {
            try
            {
                return new LogicalFile(blobFile.getCanonicalPath(), blobFile.toURL());
            }
            catch (MalformedURLException ex)
            {
                throw new MFException("Unable to create URL for " + blobFile.toString());
            }
            catch (IOException e)
            {
                throw new MFException("Unable to get canonical path for "+ blobFile.toString());
            }
        }
        return null;
    }

    private LogicalFile getFileFromOtherURL(String location, String logicalName) throws CacheException
    {
        LogicalFile file = null;
        if (DEBUG)
        {
            System.out.println("LocalFileManager.getFileFromOtherURL calling storeFile");
        }
        boolean latestCopyInTheCache = m_configCache.storeFile(location, logicalName);
        if (latestCopyInTheCache)
        {
            if (DEBUG)
            {
                System.out.println("LocalFileManager.getFileFromOtherURL found something in the cache");
            }
            String useLogicalName = logicalName;
            if (!useLogicalName.startsWith("/"))
            {
                useLogicalName = "/" + useLogicalName;
            }
            File localFile = m_configCacheView.getFileByLogicalName(useLogicalName, false);
            URL localURL;
            try
            {
                return new LogicalFile(localFile.getCanonicalPath(), localFile.toURL());
            }
            catch (MalformedURLException ex)
            {
                throw new PersistentCacheException("Unable to create URL for " + useLogicalName);
            }
            catch (IOException e)
            {
                throw new PersistentCacheException("Unable to get canonical path for "+ useLogicalName);
            }
        }
        else
        {
            if (DEBUG)
            {
                System.out.println("LocalFileManager.getFileFromOtherURL returning null");
            }
            return null;
        }
    }

     public boolean isNonDSFile(IElement fileEnvElement)
    {
        IAttributeSet topSet = fileEnvElement.getAttributes();
        IAttributeSet systemAttrs = (IAttributeSet)topSet.getAttribute(IBlob.SYSTEM_ATTRIBUTES);
        if (systemAttrs != null)
        {
            Boolean isNonDSFile = (Boolean)systemAttrs.getAttribute(IBlob.NON_DSFILE);
            return (isNonDSFile != null && isNonDSFile.booleanValue());
        }
        return false;
    }

    private File getBlobFromDSAndCache(String fileName, boolean nativeFile, boolean notFromCache)
        throws CacheException, VersionOutofSyncException, DirectoryServiceException
    {
        File blobFile = null;
        try
        {
            // The caller could not connect to the DS, it's equivalenet to InvokeTimeoutException
            if (m_blobSource == null)
            {
                throw new InvokeTimeoutException("The Directory Service is not Available.");
            }

            IBlob blob = m_blobSource.getBlobByLogicalName(m_containerID, fileName);
            if (blob == null)
            {
                return null;
            }
            // put the configuration retrieved from the DS in the cache
            // The blob returned by the DS might not be "logicalName" but an archive that
            // contains "logicalName". Put it in the cache with the correct logicalName. The cache will
            // expand it.
            IDirElement el = blob.getElement();
            String archiveName = el.getArchiveName();
            if (DEBUG)
            {
              System.out.println(
                  "LocalFileManager.getBlobFromDSAndCache, archiveName == " +
                  archiveName + " and it's logical? == " + blob.isLogicalName());
            }
            if (archiveName == null)
            {
                archiveName = fileName;
            }
            IDirElement blobEl = blob.getElement();
            if (m_configCacheView.isDoNotCache(blobEl))
            {
                blobFile = m_configCache.storeBlobTemporarily(blob);
            }
            else
            {
                m_configCache.setBlobByLogicalName(archiveName, blob, (nativeFile && (archiveName == fileName)));
                blobFile = m_configCacheView.getFileByLogicalName(fileName, nativeFile);
            }
            if (DEBUG)
            {
                System.out.println("LocalFileManager.getBlobFromDSAndCache returning " + blobFile.getAbsolutePath() + " for " + fileName);
            }
            if (m_logger != null)
            {

            	if ((m_traceMask & Agent.TRACE_CONFIGURATION_CACHE) > 0 && (m_traceMask & Agent.TRACE_DETAIL) > 0)
                {
                    m_logger.logMessage("Configuration [" + blob.getElement().getIdentity().getName() + "] added to cache", Level.TRACE);
                }

            }
        }
        catch (InvokeTimeoutException timeoutE)
        {
            // could not contact the DS, so is there an earlier version of the jar we can start with?
            if (notFromCache)
            {
                return null;
            }
            blobFile = m_configCacheView.getAvailableFileByLogicalName(fileName);
            if (blobFile != null)
            {
                  // since a component can be loaded twice on container startup (once to determine if its
                  // a framework component and then again if its not), we don't want to log the same message
                  // twice
                  String filename = blobFile.getName();
                  String version = filename.substring(0, filename.indexOf('.'));
                  filename = filename.substring(filename.indexOf('.') + 1);

                  if (m_logger != null)
                {
                    m_logger.logMessage("Failed to retrieve the latest version for\"" + filename +
                                          "\" from local archive cache; most recently cached archive (version " +
                                          version + ") will be used instead", Level.WARNING);
                }

            }
            else
            {
                throw timeoutE;
            }
        }
        return blobFile;
    }

    public static String substituteSonicHome(String url) throws MFException
    {
        if (url == null)
        {
            return null;
        }

        if (url.length() >= SONIC_HOME_PROTOCOL_PREFIX_LENGTH &&
            url.substring(0, SONIC_HOME_PROTOCOL_PREFIX_LENGTH).equalsIgnoreCase(SONIC_HOME_PROTOCOL_PREFIX))
        {
           if (url.length() == SONIC_HOME_PROTOCOL_PREFIX_LENGTH || url.charAt(SONIC_HOME_PROTOCOL_PREFIX_LENGTH)  != '/')
        {
            throw new MFException("Invalid URL: \"" + url + "\".");
        }

           if (SONICSW_HOME == null)
        {
            throw new MFException("Cannot resolve \"" + url + "\" - property \"" +  IContainer.SONICSW_HOME_PROPERTY + "\" is not set.");
        }

           String relativePath = url.substring(SONIC_HOME_PROTOCOL_LENGTH);
           return  new File (new File(SONICSW_HOME), relativePath).getAbsolutePath();
        }
        else
        {
            return url;
        }
    }

	public int geTraceMask() {
		return m_traceMask;
	}

	public void setTraceMask(int mask) {
		m_traceMask = mask;
    }


}
