package com.sonicsw.mf.framework.agent;

import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;

import javax.naming.Binding;
import javax.naming.Context;
import javax.naming.ContextNotEmptyException;
import javax.naming.InvalidNameException;
import javax.naming.Name;
import javax.naming.NameAlreadyBoundException;
import javax.naming.NameClassPair;
import javax.naming.NameNotFoundException;
import javax.naming.NameParser;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.OperationNotSupportedException;
import javax.naming.spi.NamingManager;

import com.sonicsw.mx.jndi.MFNameParser;
import com.sonicsw.mx.jndi.ObjHelper;

import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;

public class MFContext implements Context
{
    /** An instance of Name Parser for this Context.*/
    private MFNameParser m_parser = MFNameParser.getInstance();
    ContainerImpl m_container;

    String m_ctxName;
    String m_fullCtxName; //context name is full name of MF DS directory mapped to this context, such as '/_MFContext/dir1/dirOfcurrentCtx'
    private static final String CONTEXT_CLASSNAME = Context.class.getName();
    private static final String VERBOSE_TRACING = "com.sonicsw.jndi.mfcontext.verbose";
    private boolean m_verboseTracing = false;

    /**
    * Constructs an instance of MFContext on createSubcontext(), lookup(), lookupLink() calls.
    * @see
    */
    MFContext(MFContext ctx, String ctxName )
    {
        if (!ctxName.startsWith("/"))
        {
            ctxName = '/' + ctxName;
        }
        m_fullCtxName = ctx.m_fullCtxName + ctxName;
        try{ m_ctxName = (new EntityName(m_fullCtxName)).getBaseName();}
        catch(Exception e){} //should not happen
    }

    /**
    * Constructs MFContext object with the provided JNDI environment properties.
    *
    */
    public MFContext( ContainerImpl container, String ctxDirName )
    {
        m_container = container;
        init(ctxDirName);
    }


    // //////////////////////////////////////////////////////////////////////
    // Implementation-> private methods
    // /////////////////////////////////////////////////////////////////////

    private void init(String ctxDirName)
    {
        String ctxName = IMFDirectories.MF_DIR_SEPARATOR + (ctxDirName.length() == 0 ? IMFDirectories.MF_JNDI_DIR : IMFDirectories.MF_JNDI_DIR + IMFDirectories.MF_DIR_SEPARATOR + ctxDirName);
        m_fullCtxName = ctxName;
    }

   // //////////////////////////////////////////////////////////////////////
   // public Methods
   // /////////////////////////////////////////////////////////////////////


   /**
    * Retrieves the named object.
    * If <tt>name</tt> is empty, returns a new instance of this context.
    * @param name String
    *		the name of the object to look up, such as <tt>/factories/SonicTopicConnectionFactory</tt>
    * @return	the object bound to <tt>name</tt>
    * @throws	NamingException if a naming exception is encountered
    * @throws	NameNotFoundException if any part of the name is not found in the underlying MF DS storage.
    *
    * @see #lookup(Name)
    * @see #lookupLink(String)
    */
    @Override
    public final Object lookup(String name)
    throws NamingException, NameNotFoundException
    {
        Object obj = null;
        if (name == null)
        {
            return null;
        }
        if (name.length() == 0)
        {
            return this;
        }

        if (!name.startsWith("/"))
        {
            name = '/' + name;
        }
        try
        {
            String parent  = (new EntityName(name)).getParent();
            String fullDSName = (m_fullCtxName + name);
            boolean bExist = false;
            IIdentity[] list = m_container.listAll( m_fullCtxName + parent);
            for (int i = 0; i < list.length; i++)
            {
                if (fullDSName.equals(list[i].getName()))
                {
                    bExist = true;
                    if ( list[i] instanceof IDirIdentity)
                    {
                        obj = new MFContext(this, name);
                    }
                    else
                    {
                        IElement elmnt = m_container.getConfiguration(fullDSName);

                        String elmntType = elmnt.getIdentity().getType();
                        if (elmntType.equals("SerializedObject"))
                        {
                            IAttributeSet attrs = elmnt.getAttributes();
                            byte[] bytes = (byte[])attrs.getAttribute("data");
                            obj = ObjHelper.deserializeObject(bytes);
                        }
                        else if (elmntType.equals("ReferenceObject"))
                        {
                            obj = getBindedRefObject(elmnt, fullDSName);
                        }
                    }
                }
            }
            if (!bExist)
            {
                throw (new NameNotFoundException(name + " not found in the specified context"));
            }
        }
        catch(Exception ex) { throwNamingException(ex); }

        return obj;
    }

    /**
     * Retrieves the named object.
     * See {@link #lookup(String)} for details.
     * @param name Name
     *		the name of the object to look up
     */

    @Override
    public final Object lookup(Name name)
    throws NamingException, NameNotFoundException
    {
        return lookup(m_parser.convertToString(name));
    }

    // Convert an element to the Reference of a referenceable object or to the object itself if it was Serialized.
   /* public static Object elementToReference(IElement element)
      throws NamingException
    {
        String elmntType = element.getIdentity().getType();
        if (elmntType.equals("SerializedObject"))
        {
            IAttributeSet attrs = element.getAttributes();
            byte[] bytes = (byte[])attrs.getAttribute("data");
            return ObjHelper.deserializeObject(bytes);
        }
        else if (elmntType.equals("ReferenceObject"))
        {
            return convertElementToRef(element, false);
        }
        else
            throw new NameNotFoundException(element.getIdentity().getName() +
                                            " is not of a ReferenceObject or a SerializedObject types");
    }*/

    private Object getBindedRefObject(IElement elmnt, String fullDSName)
      throws NamingException
    {
        try
        {
            return NamingManager.getObjectInstance(ObjHelper.convertElementToRef(elmnt, m_verboseTracing),
                                                   m_parser.parse(fullDSName), this, null);
        }
        catch(Exception ex) { throwNamingException(ex); }
        return null;
    }

    /**
     * Binds a name to an object.
     * See {@link #bind(String)} for details.
     * @param name (Name)
     *		the name to bind
     * @param obj
     *		the object to bind
     */
    @Override
    public final void bind(Name name, Object obj)
    throws NamingException, NameAlreadyBoundException
    {
        if (name == null || name.toString().length() == 0)
        {
            throw new InvalidNameException("Cannot bind empty name");
        }
        bind(m_parser.convertToString(name), obj);
    }

    /**
     * Binds a name to an object.
     * All intermediate contexts and the target context must already exist.
     * @param name (String)
     *		the name to bind -> cannot be empty.
            The following objects types supported: Serializable, Reference and Referenceable
     * @param obj
     *		the object to bind -> can't be null
     * @throws	NameAlreadyBoundException if name is already bound
     * @throws	NamingException if a naming exception is encountered
     *
     * @see #bind(Name, Object)
     * @see #rebind(String, Object)
    */
    @Override
    public final void bind(String name, Object obj)
    throws NamingException, NameAlreadyBoundException
    {

        throw new OperationNotSupportedException("This Context does not support modifying the store");
    }

    // Used by DS handlers to convert JNDI objects to DS elements the MF JNDI client
   /* public static IDirElement objectToElement(Object obj) throws DirectoryServiceException
    {
        if (obj == null )
            throw new DirectoryServiceException("The Object can't be null");

        try
        {
            if ((obj instanceof Referenceable) || (obj instanceof javax.naming.Reference))
                return bindReferenceable("/NoNameIsNeeded", obj, false);
            else if (obj instanceof java.io.Serializable)
                return bindSerializable("/NoNameIsNeeded", obj, false);
            else
                throw new DirectoryServiceException("Can only bind Serializable, References or Referenceable objects");
        }
        catch (NamingException e)
        {
             throw new DirectoryServiceException(e.toString());
        }
    }*/


  /*  private static IDirElement bindSerializable(String s, Object obj, boolean verboseTracing)
    throws NamingException
    {
       IDirElement elmnt = null;
       try
       {
        elmnt =  ElementFactory.createElement(s, "SerializedObject", "1.0");
        IAttributeSet elmntAttributes = elmnt.getAttributes();
        elmntAttributes.setBytesAttribute("data",ObjHelper.serializeObject(obj));
        elmntAttributes.setStringAttribute("classname",obj.getClass().getName());
       }
       catch(Exception ex) { throwNamingException(ex, verboseTracing); }

       return elmnt;
    }*/

   /* private static IDirElement bindReferenceable(String s, Object obj, boolean verboseTracing)
    throws NamingException
    {
        IDirElement elmnt = null;
        javax.naming.Reference reference = null;
        if (obj instanceof Referenceable)
            reference = ((Referenceable)obj).getReference();
        else if (obj instanceof javax.naming.Reference)
            reference = (javax.naming.Reference)obj;
        try
        {
            elmnt =  ElementFactory.createElement(s, "ReferenceObject", "1.0");
            IAttributeSet elmntAttributes = elmnt.getAttributes();
            String s1;
            if ((s1 = reference.getClassName()) != null)
            {
                elmntAttributes.setStringAttribute("classname",s1);
            }
            else
                throw new NamingException("Reference has no class name");
            if ((s1 = reference.getFactoryClassName()) != null)
                elmntAttributes.setStringAttribute("factoryclassname",s1);
            if ((s1 = reference.getFactoryClassLocation()) != null)
                elmntAttributes.setStringAttribute("factorylocation",s1);

            int i = reference.size();

            if (i > 0)
            {
                IAttributeList list = elmntAttributes.createAttributeList("RefAdresses");
                for (int j = 0; j < i; j++)
                {
                    RefAddr refaddr = reference.get(j);
                    String type = refaddr.getType();
                    Object obj1 = refaddr.getContent();
                    IAttributeSet set = list.addNewAttributeSetItem();
                    if (refaddr instanceof StringRefAddr)
                    {
                        set.setStringAttribute(type, (String)(obj1 != null ? obj1 : ""));
                    }
                    else
                    {
                        set.setBytesAttribute(type, ObjHelper.serializeObject(obj));
                    }
                }
            }
       }
       catch(Exception ex) { throwNamingException(ex, verboseTracing); }

       return elmnt;
    }*/


   /**
     * Binds a name to an object, overwriting any existing binding.
     * All intermediate contexts and the target context must already exist.
     *
     * @param name (String)
     *		the name to bind; cannot be empty
     * @param obj
     *		the object to rebind; cannot be null, should be a same type as previously binded object
     * @throws	NameNotFoundException if any part of the name is not found in the underlying MF DS storage.
     * @throws	NamingException if a naming exception is encountered
     *
     * @see #rebind(Name, Object)
     * @see #bind(String, Object)
    */
   @Override
public final void rebind(String name, Object obj)
       throws NamingException, NameNotFoundException
   {

        throw new OperationNotSupportedException("This Context does not support modifying the store");
   }

    /**
     * Binds a name to an object, overwriting any existing binding.
     * See {@link #rebind(String)} for details.
     * @param name (Name)
     *		the name to bind
     * @param obj
     *		the object to rebind
     *
     */
    @Override
    public final void rebind(Name name, Object obj)
    throws NamingException
    {
         if (name == null || name.isEmpty())
        {
            throw new NamingException("Cannot unbind empty name");
        }
         rebind(m_parser.convertToString(name), obj);
    }

    /* rebinds Reference object
     ==> if new Reference contains list of addresses, then the old list will be replaced with the new one
    */
  /*  private void rebindReferenceable(IAttributeSet elmntAttributes, Object obj)
    throws NamingException
    {
        javax.naming.Reference reference = null;
        if (obj instanceof Referenceable)
            reference = ((Referenceable)obj).getReference();
        else if (obj instanceof javax.naming.Reference)
            reference = (javax.naming.Reference)obj;
        try
        {
            String s1;
            if ((s1 = reference.getClassName()) != null)
            {
                elmntAttributes.setStringAttribute("classname",s1);
            }
            else
                throw new NamingException("Reference has no class name");
            if ((s1 = reference.getFactoryClassName()) != null)
                elmntAttributes.setStringAttribute("factoryclassname",s1);
            if ((s1 = reference.getFactoryClassLocation()) != null)
                elmntAttributes.setStringAttribute("factorylocation",s1);

            IAttributeList list = (IAttributeList)elmntAttributes.getAttribute("RefAdresses");
            if (list == null)
                list = elmntAttributes.createAttributeList("RefAdresses");
            else // delete old addresses
            {
                int count = list.getCount();
                for (int i = 0; i < count; i++)
                    list.deleteAttributeItem(0);
            }

            int i = reference.size();
            if (i > 0)
            {

                for (int j = 0; j < i; j++)
                {
                    RefAddr refaddr = reference.get(j);
                    String type = refaddr.getType();
                    Object obj1 = refaddr.getContent();
                    IAttributeSet set = list.addNewAttributeSetItem();
                    if (refaddr instanceof StringRefAddr)
                    {
                        set.setStringAttribute(type, (String)(obj1 != null ? obj1 : ""));
                    }
                    else
                    {
                        set.setBytesAttribute(type, ObjHelper.serializeObject(obj));
                    }
                }
            }
       }
       catch(Exception ex) { throwNamingException(ex); }
    }*/

    /**
     * Unbinds the named object(deletes object from the underlying MF storage)
     * Removes the terminal name in <code>name</code> from the target context.
     *
     * @param name (String)
     *		the name to unbind; cannot be empty
     * @throws	NamingException if a naming exception is encountered
     * @see #unbind(Name)
     */

    @Override
    public final void unbind(String name)
    throws NamingException
    {
        throw new OperationNotSupportedException("This Context does not support modifying the store");
    }

   /**
    * Unbinds the named object(deletes object from the underlying MF storage).
    *  See {@link #unbind(String)} for details.
    * @param name (Name)
    *		the name to unbind
    */

   @Override
public final void unbind(Name name)
    throws NamingException
   {
      if (name == null || name.toString().length() == 0)
    {
        throw new NamingException("Cannot unbind empty name");
    }
      unbind(m_parser.convertToString(name));
   }

    /**
    * Binds a new name to the object bound to an old name, and unbinds
    * the old name(deletes old object location from underlying storage). Both names are relative to this context.
    * Intermediate contexts of the old name are not changed.
    * The underlyng MF storage doesn't support rename operation, therefore <tt>rename operation</tt> not guaranteed to be atomic.
    * @param oldName (String)
    *		the name of the existing binding; cannot be empty
    * @param newName (String)
    *		the name of the new binding; cannot be empty
    * @throws  NamingException if a naming exception is encountered
    *
    * @see #rename(Name, Name)
    * @see #bind(String, Object)
    * @see #rebind(String, Object)
    */

    @Override
    public final void rename(String oldname, String newname)
    throws NamingException
    {
        throw new OperationNotSupportedException("This Context does not support modifying the store");
    }



    /**
    * Binds a new name to the object bound to an old name, and unbinds
    * the old name.
    * See {@link #rename(String)} for details.
    * @param oldName (Name)
    *		the name of the existing binding; cannot be null
    * @param newName (Name)
    *		the name of the new binding; cannot be null
    */
    @Override
    public final void rename(Name oldname, Name newname)
    throws NamingException
    {
        if (oldname == null || newname == null)
        {
            throw new InvalidNameException("Cannot rename empty name");
        }
        rename(m_parser.convertToString(oldname), m_parser.convertToString(newname));
    }

    /**
     * Enumerates the names bound in the named context, along with the
     * class names of objects bound to them.
     * The contents of any subcontexts are not included.
     *
     * <p> If a binding is added to or removed from this context,
     * its effect on an enumeration previously returned is undefined.
     *
     * @param name
     *		the name of the context to list
     * @return	an enumeration of the names and class names of the
     *		bindings in this context.  Each element of the
     *		enumeration is of type <tt>NameClassPair</tt>.
     * @throws	NamingException if a naming exception is encountered
     *
     * @see #list(Name)
     * @see #listBindings(String)
     * @see javax.naming.NameClassPair
     */
    @Override
    public final NamingEnumeration list(String name)
    throws NamingException
    {
        String dsName;
        if (name.length() == 0)
        {
            dsName = m_fullCtxName; //list binded objects at this Context
        }
        else
        {
            if (!name.startsWith("/"))
            {
                name = '/' + name;
            }
            dsName = m_fullCtxName + name;
        }

        IDirIdentity[] contextList = null;
        IDirElement[] objectList = null;
        try
        {
            contextList = m_container.listDirectories(dsName);
            objectList = m_container.getAllElements(dsName);
        }
        catch(Exception ex) { throwNamingException(ex); }

        return new BindingEnumeration(contextList, objectList);
    }

    /**
     * Enumerates the names bound in the named context, along with the
     * class names of objects bound to them.
     * See {@link #list(String)} for details.
     */
    @Override
    public final NamingEnumeration list(Name name)
    throws NamingException
    {
        if (name == null)
        {
            throw new InvalidNameException("Name object cannot be null");
        }
        return list(name.toString());
    }

    /**
     * Enumerates the names bound in the named context, along with the
     * objects bound to them.
     * The contents of any subcontexts are not included.
     *
     * <p> If a binding is added to or removed from this context,
     * its effect on an enumeration previously returned is undefined.
     *
     * @param name
     *		the name of the context to list
     * @return	an enumeration of the bindings in this context.
     *		Each element of the enumeration is of type
     *		<tt>Binding</tt>.
     * @throws	NamingException if a naming exception is encountered
     *
     * @see #listBindings(Name)
     * @see #list(String)
     * @see javax.naming.Binding
     */
    @Override
    public final NamingEnumeration listBindings(String name)
    throws NamingException
    {
        String dsName;
        if (name.length() == 0)
        {
            dsName = m_fullCtxName; //list binded objects at this Context
        }
        else
        {
            if (!name.startsWith("/"))
            {
                name = '/' + name;
            }
            dsName = m_fullCtxName + name;
        }

        IDirIdentity[] contextList = null;
        IDirElement[] objectList = null;
        try
        {
            contextList = m_container.listDirectories(dsName);
            objectList = m_container.getAllElements(dsName);
        }
        catch(Exception ex) { throwNamingException(ex); }

        return new ObjectBindingEnumeration(contextList, objectList);
    }

    /**
     * Enumerates the names bound in the named context, along with the
     * class names of objects bound to them.
     * See {@link #listBindings(String)} for details.
    */

    @Override
    public final NamingEnumeration listBindings(Name name)
    throws NamingException
    {
        if (name == null)
        {
            throw new InvalidNameException("Name object cannot be null");
        }
        return listBindings(m_parser.convertToString(name));
    }

    /**
     * Destroys the named context and removes it from the underlying MF storage.
     * Intermediate contexts are not destroyed.
     * @param name
     *		the name of the context to be destroyed; cannot be empty
     * @throws	NameNotFoundException if an intermediate context does not exist
     * @throws	ContextNotEmptyException if the named context is not empty
     * @throws	NamingException if a naming exception is encountered
     *
     * @see #destroySubcontext(Name)
     */

    @Override
    public final void destroySubcontext(String name)
    throws NamingException, ContextNotEmptyException
    {
        throw new OperationNotSupportedException("This Context does not support modifying the store");
    }

    /**
    * Destroys the named context and removes it from the underlying MF storage.
    * See {@link #destroySubcontext(String)} for details.
    */
    @Override
    public final void destroySubcontext(Name name)
    throws NamingException
    {
        if (name == null || name.toString().length() == 0)
        {
            throw new NamingException("Cannot destroy subcontext with an empty name");
        }
        destroySubcontext(m_parser.convertToString(name));
    }

    /**
     * Creates and binds a new context.
     * All intermediate contexts must already exist.
     * See {@link #createSubcontext(String)} for details.
     */

    @Override
    public final Context createSubcontext(Name name)
    throws NamingException, NameAlreadyBoundException, javax.naming.CommunicationException
    {
         if (name == null || name.toString().length() == 0)
        {
            throw new NamingException("Cannot create subcontext with an empty name");
        }
        return createSubcontext(m_parser.convertToString(name));
    }

    /**
    * Creates and binds a new context. All intermediate contexts
    * must already exist.
    *
    * @param name
    *		the name of the context to create; may not be empty
    * @return	the newly created context
    *
    * @throws	NameAlreadyBoundException if name is already bound
    * @throws	NamingException if a naming exception is encountered
    * @see #createSubcontext(String)
    */

    @Override
    public final Context createSubcontext(String name)
    throws NamingException, NameAlreadyBoundException, javax.naming.CommunicationException
    {
        throw new OperationNotSupportedException("This Context does not support modifying the store");
    }

    /**
    *  No support for Name Space Federation, so this method behaves as lookup
    *  See {@link #lookup(String)} for details.
    */
    @Override
    public final Object lookupLink(String name)
    throws NamingException
    {
        // No support for Name Space Federation, so this method behaves as lookup
        return lookup(name);
    }

    /**
    * No support for Name Space Federation, so this method behaves as lookup
    * See {@link #lookup(Name)} for details.
    */
    @Override
    public final Object lookupLink(Name name)
    throws NamingException
    {
        // No support for Name Space Federation
        return lookupLink(m_parser.convertToString(name));
    }

    /**
    * Returns a Context name Parser. There is no support for Naming Federation
    * all the Name Parsers are same.
    *
    * @param name Name of the NameParser, may be null
    * @return parser Context Parser
    * @throws Naming Exception if there is any naming violation.
    */
    @Override
    public final NameParser getNameParser(String name)
    throws NamingException
    {
        return m_parser;
    }

    /**
    * Returns a Context name Parser. There is no support for Naming Federation
    * all the Name Parsers are same.
    * See {@link #getNameParser(String)} for details.
    */
    @Override
    public final NameParser getNameParser(Name name)
    throws NamingException
    {
        return m_parser;
    }

    /**
    * Composes the name of this context with a name relative to this context. Given a name (name)
    * relative to this context,and the name (prefix) of this context relative to one of its ancestors,
    * this method returns the composition of the two names using the syntax appropriate for the naming
    * system(s) involved. That is, if name names an object relative to this context, the result is the
    * name of the same object, but relative to the ancestor context. None of the names may be null.
    * <p>
    * @param name Name string relative to this Context
    * <p>
    * @param prefix Prefix string
    * <p>
    * @return result Composition of the name with prefix.
    * <p>
    * @throws NamingException If there is any naming violation
    */
    @Override
    public final String composeName(String name, String prefix)
    throws NamingException
    {
        if (name.length() == 0 || prefix.length() == 0)
        {
            throw new NamingException("Names can't be empty");
        }
        return (composeName(m_parser.parse(name), m_parser.parse(prefix))).toString();
    }

    /**
    * Composes the name of this context with a name relative to this context.
    * See {@link #composeName(String, String)} for details.
    */
    @Override
    public final Name composeName(Name name, Name prefix)
    throws NamingException
    {
        if (name == null || prefix == null)
        {
            throw new NamingException("Names can't be empty");
        }
        Name composition = (Name)prefix.clone();
        composition.addAll(name);
        return composition;
    }

    /**
    * Operation not supported
    * @throws OperationNotSupportedException
    */
    @Override
    public final Object addToEnvironment(String propName, Object propVal)
    throws OperationNotSupportedException
    {
        throw new OperationNotSupportedException("addToEnvironment");
    }

    /**
    * Operation not supported
    * @throws OperationNotSupportedException
    */
    @Override
    public final Object removeFromEnvironment(String propName)
    throws OperationNotSupportedException
    {
        throw new OperationNotSupportedException("addToEnvironment");
    }

    /**
    * Returns the current Context Environment.
    * @return m_env Context Enviroment.
    * @throws NamingException
    */
    @Override
    public final Hashtable getEnvironment()
    throws NamingException
    {
        return new Hashtable();
    }

    /**
    * This method retrieves the MF DS full name of this context
    * @return String in MF DS format.
    */
    @Override
    public String getNameInNamespace()
    throws NamingException
    {
        if (m_fullCtxName.equals(IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_JNDI_DIR))
        {
            return "";
        }
        else {
            return m_fullCtxName.substring(1 + IMFDirectories.MF_JNDI_DIR.length() + 1); // "/xxx"
        }
    }

    /**
     * Closes this context.
     * @throws	NamingException if a naming exception is encountered
    */
    @Override
    public void close()
    throws NamingException
    {
       // if (m_connMng.getConnector() != null)
       // m_connMng.close();
    }

    @Override
    protected void finalize() {
        try {
            close();
        }
        catch(NamingException _ex) { }
    }

    private void throwNamingException(Exception e)  throws NamingException
    {
        ObjHelper.throwNamingException(e, m_verboseTracing);
    }

   // ///////////////////////////////////////////////////////////////////////
   // Class for enumerating name/class pairs
   // ///////////////////////////////////////////////////////////////////////

    final class BindingEnumeration
    implements NamingEnumeration
    {
        HashSet set = new HashSet();
        Iterator m_iterator;

        BindingEnumeration(IDirIdentity[] contextList, IDirElement[] objectList)
        {
            init(contextList, objectList);
            m_iterator = set.iterator();
        }

        /**
        * check for any more elements in the present Context.
        * @return true if there are any elements available in the set.
        */

        @Override
        public boolean hasMore() throws NamingException
        {
            return hasMoreElements();
        }

        /**
        * check for any more elements in the present Context.
        * @return true if there are any elements available in the set.
        */
        @Override
        public boolean hasMoreElements()
        {
            return m_iterator.hasNext();
        }

        /**
        * returns next element in the Context. The return value is of type NameClassPair.
        *
        * @return Object NameClassPair object which represents the object name and class name pair
        * of a binding found in a context.
        */
        @Override
        public Object next() throws NamingException
        {
            return nextElement();
        }

        /**
        * returns next element in the Context. The return value is of type NameClassPair.
        *
        * @return Object NameClassPair object which represents the object name and class name pair
        * of a binding found in a context.
        */
        @Override
        public Object nextElement()
        {
            return m_iterator.next();
        }

        /**
        *  clears set
        */
        @Override
        public void close() //after close, NamingEnumeration
        {
           set.clear();
        }

        private void init(IDirIdentity[] contextList, IDirElement[] objectList)
        {
            for ( int i = 0; i < contextList.length; i++)
            {
                String fullname = contextList[i].getName();
                try
                {
                    set.add(new NameClassPair((new EntityName(fullname)).getBaseName(), CONTEXT_CLASSNAME));
                }
                catch(Exception ex) {} //should not happen
            }
            for ( int i = 0; i < objectList.length; i++)
            {
                String fullname = objectList[i].getIdentity().getName();
                IAttributeSet attrs = objectList[i].getAttributes();
                String className = (String)attrs.getAttribute("classname");
                try
                {
                    set.add(new NameClassPair((new EntityName(fullname)).getBaseName(), className));
                }
                catch(Exception ex) {} //should not happen
            }
        }
   }//

   // ///////////////////////////////////////////////////////////////////////////////////
   // Class for enumerating bindings
   // ///////////////////////////////////////////////////////////////////////////////////

   final class ObjectBindingEnumeration
   implements NamingEnumeration
   {
        HashSet set = new HashSet();
        Iterator m_iterator;

        ObjectBindingEnumeration(IDirIdentity[] contextList, IDirElement[] objectList)
        {
            init(contextList, objectList);
            m_iterator = set.iterator();
        }

        /**
        * check for any more elements in the present Context.
        * @return true if there are any elements available in the set.
        */

        @Override
        public boolean hasMore() throws NamingException
        {
            return hasMoreElements();
        }

        /**
        * returns next element in the Context. The return value is of type Binding.
        *
        * @return Object Binding object which represents the object name, class name and object
        * of a binding found in a context.
        */
        @Override
        public Object next() throws NamingException
        {
            return nextElement();
        }

        /**
        * returns next element in the Context. The return value is of type Binding.
        *
        * @return Object Binding object which represents the object name, class name and object
        * of a binding found in a context.
        */
        @Override
        public Object nextElement()
        {
            return m_iterator.next();
        }

        /**
        * check for any more elements in the present Context.
        * @return true if there are any elements available in the set.
        */
        @Override
        public boolean hasMoreElements()
        {
            return m_iterator.hasNext();
        }

        /**
        *  clears set of elements
        */
        @Override
        public void close()
        {
            set.clear();
        }

        private void init(IDirIdentity[] contextList, IDirElement[] objectList)
        {
            for ( int i = 0; i < contextList.length; i++)
            {
                String fullname = contextList[i].getName();
                try
                {
                    set.add(new NameClassPair((new EntityName(fullname)).getBaseName(), CONTEXT_CLASSNAME));
                }
                catch(Exception ex) {} //should not happen
            }
            for ( int i = 0; i < objectList.length; i++)
            {
                String fullname = objectList[i].getIdentity().getName();
                String elmntType = objectList[i].getIdentity().getType();
                IAttributeSet attrs = objectList[i].getAttributes();
                String className = (String)attrs.getAttribute("classname");
                Object obj = null;
                try
                {
                    if ( elmntType.equals("SerializedObject"))
                    {
                        obj = ObjHelper.deserializeObject((byte [])attrs.getAttribute("data"));
                    }
                    else if (elmntType.equals("ReferenceObject"))
                    {
                        obj = getBindedRefObject(objectList[i], fullname);
                    }
                    set.add(new Binding((new EntityName(fullname)).getBaseName(), className, obj));
                }
                catch(Exception ex) {} //should not happen
            }
        }

   } //
}//

