package com.sonicsw.mf.framework.agent;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.HashMap;

import com.sonicsw.mx.util.IEmptyArray;

/**
 * Notification handlers are the internal (MF based) listeners to JMX notification (as opposed to
 * notification listeners which are external JMX clients).
 *
 * @see NotificationListenerDelegate
 */
class NotificationHandlerDelegate
implements Serializable
{
    private static final long serialVersionUID = 8484344976480223320L;

    private static final short m_serialVersion = 0;
    
    private String m_destination; // this will be the target component (stringified form of jmx object name)
    
    private String[] m_handledNotificationTypes = IEmptyArray.EMPTY_STRING_ARRAY;
    
    private long m_notificationSubscriptionTimeout;
    
    private transient long m_expirationTime;
    
    private transient boolean m_isDirty = false;
    
    NotificationHandlerDelegate(String destination)
    {
        m_destination = destination;
    }
    
    String getDestination()
    {
        return m_destination;
    }
    
    synchronized void setHandledNotificationTypes(String[] handledNotificationTypes)
    {
        boolean isDirty = false;
        
        if (handledNotificationTypes.length != m_handledNotificationTypes.length)
        {
            isDirty = true;
        }
        else
        {
            // do a one for one match
            for (int i = 0; i < handledNotificationTypes.length; i++)
            {
                boolean match = false;
                for (int j = 0; j < m_handledNotificationTypes.length; j++)
                {
                    if (handledNotificationTypes[i].equals(m_handledNotificationTypes[j]))
                    {
                        match = true;
                        break;
                    }
                }
                
                if (!match)
                {
                    isDirty = true;
                    break;
                }
            }
        }
        
        if (isDirty)
        {
            m_handledNotificationTypes = handledNotificationTypes;
            isDirty = true;
        }
    }
    
    synchronized boolean handlesNotificationType(String notificationType)
    {
        for (int i = 0; i < m_handledNotificationTypes.length; i++)
        {
            if (m_handledNotificationTypes[i].equals(notificationType))
            {
                return true;
            }
            int index = m_handledNotificationTypes[i].indexOf('*');
            if (index > -1)
            {
                String prefix = m_handledNotificationTypes[i].substring(0, index - 1);
                if (notificationType.startsWith(prefix))
                {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    long getNotificationSubscriptionTimeout()
    {
        return m_notificationSubscriptionTimeout;
    }
    
    void setNotificationSubscriptionTimeout(long timeout)
    {
        if (timeout != m_notificationSubscriptionTimeout)
        {
            m_notificationSubscriptionTimeout = timeout;
            m_isDirty = true;
        }
        m_expirationTime = System.currentTimeMillis() + m_notificationSubscriptionTimeout;
    }
    
    boolean hasExpired()
    {
        return System.currentTimeMillis() > m_expirationTime;
    }

    boolean isDirty()
    {
        return m_isDirty;
    }

    byte[] toByteArray()
    throws IOException
    {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ObjectOutputStream oos = new ObjectOutputStream(baos);
        oos.writeObject(this);
        oos.close();
        
        byte[] bytes = baos.toByteArray();
        
        return bytes;
    }

    static NotificationHandlerDelegate fromByteArray(byte[] bytes)
    throws IOException, ClassNotFoundException
    {
        ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
        synchronized(bais) // performance optimization
        {
            ObjectInputStream ois = new ObjectInputStream(bais);
            return (NotificationHandlerDelegate)ois.readObject();
        }
    }

    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        synchronized (this)
        {
            // write out the (fixed) number of name/value pairs for the delegate
            stream.writeInt(4);

            stream.writeUTF("serialVersion");
            stream.writeObject(new Short(m_serialVersion));

            stream.writeUTF("destination");
            stream.writeObject(m_destination);

            stream.writeUTF("handledTypes");
            stream.writeObject(m_handledNotificationTypes);

            stream.writeUTF("timeout");
            stream.writeObject(new Long(m_notificationSubscriptionTimeout));
        }
        
        // since we only serialize when we store the delegate in the cache, it is no longer dirty
        // once we have written it
        m_isDirty = false;
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        // get the number of name/value pairs in the stream
        int count = stream.readInt();

        // put the name/value pairs in a hash map
        HashMap map = new HashMap(count);
        for (int i = 0; i < count; i++)
        {
            String key = stream.readUTF();
            Object value = stream.readObject();
            map.put(key, value);
        }

        // now map them back into the local variables based on the version
        // Note: As a general rule need to catch exceptions and either do something to set a default
        //       value or ignore
        switch (((Short)map.remove("serialVersion")).shortValue())
        {
            // case olderVersion<n> ...
            // case olderVersion<n> ...
            default: // the current version or newer versions
            {
                // NotificationListenerDelegate fields
                try { m_destination = (String)map.get("destination"); } catch (Exception e) { }
                try { m_handledNotificationTypes = (String[])map.get("handledTypes"); } catch (Exception e) { }
                try { setNotificationSubscriptionTimeout(((Long)map.get("timeout")).longValue()); } catch (Exception e) { }
                break;
            }
        }
        
        // since we only deserialize when we get the delegate from the cache, it is not dirty
        // when we first read it
        m_isDirty = false;
    }
}
