package com.sonicsw.mf.framework.agent;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.management.Notification;
import javax.management.NotificationFilter;
import javax.management.NotificationFilterSupport;
import javax.management.NotificationListener;
import javax.management.ObjectName;

import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.jmx.client.ExpressionBasedNotificationFilter;
import com.sonicsw.mf.jmx.client.JMSConstants;
import com.sonicsw.mf.mgmtapi.runtime.IAgentProxy;

/**
 * Notification listeners are the external (JMX based) listeners to JMX notification (as opposed to
 * notification handlers which are internal MF clients).
 *
 * @see NotificationHandlerDelegate
 */
final class NotificationListenerDelegate
implements NotificationListener, Serializable
{
    private static final long serialVersionUID = 8484344976480223320L;

    private static final short m_serialVersion = 0;

    private ObjectName m_objectName;

    private com.sonicsw.mf.jmx.client.NotificationListenerDelegate m_remoteListener;

    private long m_expirationTime;

    private HashMap m_handbacks = new HashMap();

    private static volatile ContainerImpl m_container;

    private transient boolean m_isDirty = true;
    
    private long m_closeTimestamp;

    NotificationListenerDelegate(ObjectName objectName, com.sonicsw.mf.jmx.client.NotificationListenerDelegate remoteListener)
    {
        m_objectName = objectName;
        m_remoteListener = remoteListener;
    }
    
    void close()
    {
        try { Thread.sleep(1); } catch(Exception e) { } // always want to be a millis later
        m_closeTimestamp = System.currentTimeMillis();
    }

    static void setContainerImpl(ContainerImpl container)
    {
        m_container = container;
    }

    String getDestination()
    {
        return m_remoteListener.m_destination;
    }

    com.sonicsw.mf.jmx.client.NotificationListenerDelegate getRemoteListener()
    {
        return m_remoteListener;
    }

    ObjectName getObjectName()
    {
        return m_objectName;
    }
    
    String getRemoteListenerKey()
    {
        return m_remoteListener.m_destination + ':' + m_remoteListener.m_listenerHash + ':' + m_objectName.getCanonicalName();
    }
    
    boolean hasExpired()
    {
        return System.currentTimeMillis() > m_expirationTime;
    }

    boolean isDirty()
    {
        return m_isDirty;
    }

    void setHandbackFilterPair(Object handback, NotificationFilter newFilter)
    {
        boolean dirty = false;

        synchronized (m_handbacks)
        {
            if (!m_handbacks.containsKey(handback))
            {
                dirty = true;
            }
            else
            {
                // we need to compare if anything changed
                NotificationFilter oldFilter = (NotificationFilter)m_handbacks.get(handback);
                if (newFilter != null || oldFilter != null)
                {
                    if (newFilter == null && oldFilter != null)
                    {
                        dirty = true;
                    }
                    else if (newFilter != null && oldFilter == null)
                    {
                        dirty = true;
                    }
                    else
                    {
                        // we have to compare to see if the filter has changed
                        if (!newFilter.getClass().equals(oldFilter.getClass()))
                        {
                            dirty = true;
                        }
                        else if (newFilter instanceof NotificationFilterSupport)
                        {
                            Vector oldEnabledTypes = ((NotificationFilterSupport)oldFilter).getEnabledTypes();
                            Vector newEnabledTypes = ((NotificationFilterSupport)newFilter).getEnabledTypes();
                            if (oldEnabledTypes.size() != newEnabledTypes.size())
                            {
                                dirty = true;
                            }
                            else
                            {
                                // check each enabled type exists in the old and new
                                Iterator iterator = oldEnabledTypes.iterator();
                                while (iterator.hasNext())
                                {
                                    if (!newEnabledTypes.contains(iterator.next()))
                                    {
                                        dirty = true;
                                        break;
                                    }
                                }
                            }
                        }
                        else if (newFilter instanceof ExpressionBasedNotificationFilter)
                        {
                            if (!newFilter.equals(oldFilter))
                            {
                                dirty = true;
                            }
                        }
                    }
                }
            }

            if (dirty)
            {
                m_handbacks.put(handback, newFilter);
                m_isDirty = true;
            }
        }
    }

    void removeHandbackFilterPair(Object handback, NotificationFilter filter)
    {
        synchronized (m_handbacks)
        {
            if (m_handbacks.remove(handback) != null)
            {
                m_isDirty = true;
            }
        }
    }

    void removeAllHandbackFilterPairs()
    {
        synchronized (m_handbacks)
        {
            m_handbacks.clear();
        }
    }

    HashMap getHandbackFilterPairs()
    {
        return m_handbacks;
    }

    void setNotificationSubscriptionTimeout(long timeout)
    {
        if (m_remoteListener != null && m_remoteListener.getNotificationSubscriptionTimeout() != timeout)
        {
            m_remoteListener.setNotificationSubscriptionTimeout(timeout);
            m_isDirty = true;
        }
        m_expirationTime = System.currentTimeMillis() + (m_remoteListener != null ? m_remoteListener.getNotificationSubscriptionTimeout() : 0);
    }

    @Override
    public void handleNotification(Notification notification, Object suppliedHandback)
    {
        if (m_closeTimestamp == 0 || notification.getTimeStamp() < m_closeTimestamp) // delegate has not closed or the notification was created before closing
        {
            try
            {
                if (System.currentTimeMillis() <= m_expirationTime)
                {
                    synchronized (m_handbacks)
                    {
                        ArrayList notifiedHandbacks = new ArrayList();
                        Iterator iterator = m_handbacks.entrySet().iterator();
                        while (iterator.hasNext())
                        {
                            Map.Entry entry = (Map.Entry)iterator.next();
                            Object handback = extractHandbackID(entry.getKey()); // gets replaced
                            // test if weve already sent to this handback, if we have skip to next filter handback pair
                            if (notifiedHandbacks.contains(handback))
                            {
                                continue;
                            }
                            NotificationFilter filter = (NotificationFilter)entry.getValue();
                            if (filter == null || filter.isNotificationEnabled(notification))
                            {
                                notifiedHandbacks.add(handback);
                                m_container.sendDirectedNotification(JMSConstants.JMX_COMMS_TYPE, m_remoteListener.m_destination, m_remoteListener.m_listenerHash, notification, handback);
                            }
                        }
                    }
                }
                else
                {
                    m_container.expireSubscription(this);
                }
            }
            catch (Throwable e)
            {
                m_container.logMessage(IAgentProxy.ID, e, Level.WARNING);
            }
        }
    }

    /**
     * A compound handback ID is a string that encapsulates both an identifier for the filter and original
     * user's handback object. This method extracts the handback identity from a tokenized string of the form:
     * <p>
     *   [delimiter][delimiter][filter instance hashcode][delimiter][handbackID]
     * <p>
     * The leading 2 delimiters allow consumers of the compound ID to identify that this is actually
     * a compound, rather than a regular handback object that happens to be a string, and the subsequent
     * delimiter allows the actual handback identifier to be stripped from the compound ID (this is
     * done in ContainerImpl).
     * <p>
     * To make it very, very unliklely that a MGMT 2.0 JMX client could provide a handback object that
     * was a string of a similar format, the delimiter used is the bell character (char)8.
     * <p>
     * If the given compoundHandbackID is null or does not match this form, then it is passed back
     * directly to the caller.
     */
    private Object extractHandbackID(Object compoundHandbackID)
    {
        String bell = (char)8 + "";

        if (compoundHandbackID == null || !(compoundHandbackID instanceof String) || !((String)compoundHandbackID).startsWith(bell + bell))
        {
            return compoundHandbackID;
        }

        StringTokenizer st = new StringTokenizer(((String)compoundHandbackID).substring(2), bell);
        if (st.countTokens() != 2)
        {
            return compoundHandbackID;
        }

        st.nextToken();

        String handbackID = st.nextToken();

        return handbackID.length() == 0 ? compoundHandbackID : handbackID;
    }

    byte[] toByteArray()
    throws IOException
    {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ObjectOutputStream oos = new ObjectOutputStream(baos);
        oos.writeObject(this);
        oos.close();
        
        byte[] bytes = baos.toByteArray();
        
        return bytes;
    }

    static NotificationListenerDelegate fromByteArray(byte[] bytes)
    throws IOException, ClassNotFoundException
    {
        ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
        synchronized(bais) // performance optimization
        {
            ObjectInputStream ois = new ObjectInputStream(bais);
            return (NotificationListenerDelegate)ois.readObject();
        }
    }

    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        synchronized (m_handbacks)
        {
            // write out the (fixed) number of name/value pairs for the delegate
            stream.writeInt(4); // Sonic00022226 - interpret this as "number of name/value pairs"....

            stream.writeUTF("serialVersion");
            stream.writeObject(new Short(m_serialVersion));

            stream.writeUTF("objectName");
            stream.writeObject(m_objectName);

            stream.writeUTF("remoteListener");
            stream.writeObject(m_remoteListener);

            stream.writeUTF("handbacks");
            stream.writeObject(m_handbacks);
        }
        
        // since we only serialize when we store the delegate in the cache, it is no longer dirty
        // once we have written it
        m_isDirty = false;
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        // get the number of name/value pairs in the stream
        int count = stream.readInt();

        // put the name/value pairs in a hash map
        HashMap map = new HashMap(count);
        for (int i = 0; i < count; i++)
        {
            String key = stream.readUTF();
            Object value = stream.readObject();
            map.put(key, value);
        }

        // now map them back into the local variables based on the version
        // Note: As a general rule need to catch exceptions and either do something to set a default
        //       value or ignore
        switch (((Short)map.remove("serialVersion")).shortValue())
        {
            // case olderVersion<n> ...
            // case olderVersion<n> ...
            default: // the current version or newer versions
            {
                // NotificationListenerDelegate fields
                try { m_objectName = (ObjectName)map.get("objectName"); } catch (Exception e) { }
                try
                {
                    m_remoteListener = ((com.sonicsw.mf.jmx.client.NotificationListenerDelegate)map.get("remoteListener"));
                    setNotificationSubscriptionTimeout(m_remoteListener.getNotificationSubscriptionTimeout());
                } catch (Exception e) { }
                try { m_handbacks = (HashMap)map.get("handbacks"); } catch (Exception e) { }
                break;
            }
        }

        // since we only deserialize when we get the delegate from the cache, it is not dirty
        // when we first read it
        m_isDirty = false;
    }
}
