package com.sonicsw.mf.framework.agent;

import com.sonicsw.mx.util.FIFO;

import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.IValueType;
import com.sonicsw.mf.common.metrics.MetricsFactory;
import com.sonicsw.mf.common.metrics.manager.IMetricsRegistrar;
import com.sonicsw.mf.common.metrics.manager.ISampledStatistic;
import com.sonicsw.mf.common.metrics.manager.IStatistic;
import com.sonicsw.mf.common.metrics.manager.IStatisticProvider;
import com.sonicsw.mf.common.metrics.manager.StatisticsFactory;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.INotificationPublisher;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.runtime.IAgentProxy;

/**
 * Each MF container has a notification publisher thread that framework components can access through the
 * IFrameworkComponentContext.
 *
 * The notification publisher thread's purpose is to provide a means to:
 *  . validate a notification (make sure it has not yet expired)
 *  . establish the subscriber list for the notification
 *  . publish the notification
 * The use of a dedicated thread to handle notifications should prevent other tasks (such as requests)
 * from being blocked, during an "alert storm" for example.
 *
 * Since the publishing via the JMSConnectorServer is [currently] synchronous, there is no advantage
 * to using a pool of threads to publish notifications.
 *
 * The notification publisher thread is a daemon thread that runs at normal Java thread priority.
 */
public class NotificationPublisher
extends Thread
implements INotificationPublisher
{
    private static final int DROPPED_NOTIFICATION_LOGGING_THRESHOLD = 500;

    private ContainerImpl m_container = null; // reference to container implementation that is hosting this notification publisher thread

    private FIFO m_notificationTasks = null; // list to hold notification runnable tasks (and their respective expiration times) that are to be validated and published
    
    private int m_queueSize;
    
    private int m_25PercentThreshold;
    private int m_50PercentThreshold;
    private boolean m_50PercentThresholdBreachLogged = false;
    private int m_75PercentThreshold;
    private boolean m_75PercentThresholdBreachLogged = false;
    
    private int m_droppedSinceLastLogged = 0;
    private boolean m_initialDropLogged = false;
    
    // metrics
    private IMetricsRegistrar m_metricsRegistrar;
    // statistics
    private IStatistic m_awaitingDispatchStatistic;
    private IStatistic m_maxAwaitingDispatchStatistic;
    private IStatistic m_droppedStatistic;

    private boolean m_isClosing = false;
    
    NotificationPublisher(ContainerImpl container)
    {
        super("Management Notification Publisher Thread");
        setDaemon(true);
        m_container = container;

        // create an FIFO instance to hold [notification task, task expiration time] pairs for notifications that are to be run (published)
        m_notificationTasks = new FIFO();
        setQueueSize(IContainerConstants.NOTIFICATION_DISPATCH_QUEUE_SIZE_DEFAULT);
    }
    
    void close()
    {
        if (!m_container.isClosing())
        {
            throw new RuntimeException("Connot close publisher unless container is closing!");
        }
        
        synchronized (m_notificationTasks)
        {
            m_isClosing = true;
            m_notificationTasks.notifyAll();
        }
    }
    
    final void setQueueSize(int newQueueSize)
    {
        if (newQueueSize < 0)
        {
            newQueueSize = IContainerConstants.NOTIFICATION_DISPATCH_QUEUE_SIZE_DEFAULT;
        }
        
        if (newQueueSize == 0)
        {
            m_50PercentThreshold = 0;
            m_50PercentThresholdBreachLogged = false;
            m_75PercentThreshold = 0;
            m_75PercentThresholdBreachLogged = false;
        }
        else
        {
            m_50PercentThreshold = newQueueSize >> 1;
            m_25PercentThreshold = m_50PercentThreshold >> 1;
            m_75PercentThreshold = m_50PercentThreshold + m_25PercentThreshold;
            
            boolean log50PercentThresholdBreached = false;
            boolean log75PercentThresholdBreached = false;
            
            synchronized(m_notificationTasks)
            {
                int currentSize = m_notificationTasks.size();
                if (currentSize > m_75PercentThreshold && !m_75PercentThresholdBreachLogged)
                {
                    log75PercentThresholdBreached = true;
                    m_50PercentThresholdBreachLogged = true;
                    m_75PercentThresholdBreachLogged = true;
                }
                if (currentSize > m_50PercentThreshold && !m_50PercentThresholdBreachLogged)
                {
                    log50PercentThresholdBreached = true;
                    m_50PercentThresholdBreachLogged = true;
                }
            }
            
            if (log50PercentThresholdBreached)
            {
                m_container.logMessage(null, "Management notifications dispatch queue is 50% full", Level.WARNING);
            }
            else
            if (log75PercentThresholdBreached)
            {
                m_container.logMessage(null, "Management notifications dispatch queue is 75% full", Level.WARNING);
            }
        }

        m_queueSize = newQueueSize;
    }
    
    static IMetricInfo[] getMetricsInfo()
    {
        // create the infos
        IMetricInfo[] infos = new IMetricInfo[3];
        infos[0] = MetricsFactory.createMetricInfo(IAgentProxy.SYSTEM_NOTIFICATIONS_AWAITINGDISPATCH_METRIC_ID, IValueType.VALUE,
                                                   "The number of management notifications awaiting dispatch.",
                                                   null, false, true, true, false, "notifications");
        infos[1] = MetricsFactory.createMetricInfo(IAgentProxy.SYSTEM_NOTIFICATIONS_MAXAWAITINGDISPATCH_METRIC_ID, IValueType.VALUE,
                                                   "The maximum number of management notifications awaiting dispatch since last metrics reset.",
                                                   null, false, true, false, false, "notifications");
        infos[2] = MetricsFactory.createMetricInfo(IAgentProxy.SYSTEM_NOTIFICATIONS_DROPPED_METRIC_ID, IValueType.VALUE,
                                                   "Number of times a management notification was dropped before dispatch to listeners. Evaluated over the last 30 minutes.",
                                                   null, false, true, false, false, "notifications");
        return infos;
    }

    void initMetrics(IMetricsRegistrar metricsRegistrar)
    {
        m_metricsRegistrar = metricsRegistrar;

        // awaiting dispatch statistic provider and statistics
        IStatisticProvider[] awaitingDispatchProviders = new IStatisticProvider[]
        {
            new IStatisticProvider()
            {
                @Override
                public void updateStatistic(ISampledStatistic statistic)
                {
                    synchronized (NotificationPublisher.this.m_notificationTasks)
                    {
                        statistic.updateValue(NotificationPublisher.this.m_notificationTasks.size());
                    }
                }
                @Override
                public void resetStatistic(ISampledStatistic statistic) { } // no implementation
            }
        };
        m_awaitingDispatchStatistic = StatisticsFactory.createStatistic(IStatistic.VALUE_MODE, false, awaitingDispatchProviders, (short)0);
        m_maxAwaitingDispatchStatistic = StatisticsFactory.createStatistic(IStatistic.MAXIMUM_MODE, false, awaitingDispatchProviders, (short)0);;

        // dropped
        m_droppedStatistic = StatisticsFactory.createStatistic(IStatistic.COUNTER_MODE, true, null, (short)1);
    }

    void enableMetrics(IMetricIdentity[] ids)
    { 
        // now enable the ones that the agent deals with directly
        for (int i = 0; i < ids.length; i++)
        {
            if (ids[i].equals(IAgentProxy.SYSTEM_NOTIFICATIONS_AWAITINGDISPATCH_METRIC_ID))
            {
                m_metricsRegistrar.registerMetric(IAgentProxy.SYSTEM_NOTIFICATIONS_AWAITINGDISPATCH_METRIC_ID, m_awaitingDispatchStatistic);
            }
            else if (ids[i].equals(IAgentProxy.SYSTEM_NOTIFICATIONS_MAXAWAITINGDISPATCH_METRIC_ID))
            {
                m_metricsRegistrar.registerMetric(IAgentProxy.SYSTEM_NOTIFICATIONS_MAXAWAITINGDISPATCH_METRIC_ID, m_maxAwaitingDispatchStatistic);
            }
            else if (ids[i].equals(IAgentProxy.SYSTEM_NOTIFICATIONS_DROPPED_METRIC_ID))
            {
                m_droppedStatistic.reset();
                m_metricsRegistrar.registerMetric(IAgentProxy.SYSTEM_NOTIFICATIONS_DROPPED_METRIC_ID, m_droppedStatistic);
            }
        }
    }

    void disableMetrics(IMetricIdentity[] ids)
    {
        // now enable the ones that the agent deals with directly
        for (int i = 0; i < ids.length; i++)
        {
            if (ids[i].equals(IAgentProxy.SYSTEM_NOTIFICATIONS_AWAITINGDISPATCH_METRIC_ID))
            {
                m_metricsRegistrar.unregisterMetric(IAgentProxy.SYSTEM_NOTIFICATIONS_AWAITINGDISPATCH_METRIC_ID);
            }
            else if (ids[i].equals(IAgentProxy.SYSTEM_NOTIFICATIONS_MAXAWAITINGDISPATCH_METRIC_ID))
            {
                m_metricsRegistrar.unregisterMetric(IAgentProxy.SYSTEM_NOTIFICATIONS_MAXAWAITINGDISPATCH_METRIC_ID);
            }
            else if (ids[i].equals(IAgentProxy.SYSTEM_NOTIFICATIONS_DROPPED_METRIC_ID))
            {
                m_metricsRegistrar.unregisterMetric(IAgentProxy.SYSTEM_NOTIFICATIONS_DROPPED_METRIC_ID);
            }
        }
    }

    @Override
    public void run()
    {
        Object[] taskArray = null;
        long expirationTime = 0;

        while (!m_isClosing)
        {
            Runnable notificationTask = null;
            Long notificationExpiration = null;
            boolean logNotificationsDeliveryResumed = false;
            
            try
            {
                // obtain a notification task to execute (or wait if no task available to execute)
                synchronized (m_notificationTasks)
                {
                    if (m_notificationTasks.isEmpty())
                    {
                        if (m_isClosing)
                        {
                            return;
                        }
                        m_notificationTasks.wait(); // wait for new notification task to be enqueued...
                        if (m_isClosing)
                        {
                            return;
                        }
                        continue;
                    }
                    else
                    {
                        taskArray = (Object[])m_notificationTasks.remove(0); // get the oldest task on the list
                        int currentSize = m_notificationTasks.size();
                        if (m_queueSize > 0)
                        {
                            if (currentSize < m_75PercentThreshold)
                            {
                                if (m_initialDropLogged)
                                {
                                    m_initialDropLogged = false;
                                    logNotificationsDeliveryResumed = true;
                                }
                            }
                            if (currentSize < m_50PercentThreshold)
                            {
                                m_75PercentThresholdBreachLogged = false;
                            }
                            if (currentSize < m_25PercentThreshold)
                            {
                                m_50PercentThresholdBreachLogged = false;
                            }
                        }
                    }
                }
                if (logNotificationsDeliveryResumed)
                {
                    m_container.logMessage(null, "...normal management notifications delivery resumed", Level.WARNING);
                }
            }
            catch (Exception e)
            {
                if (m_container.isClosing())
                {
                    return;
                }
                else
                {
                    m_container.logMessage("Exception while management processing notifications, trace follows... ", e, Level.SEVERE);
                    continue;
                }
            }
            
            notificationTask = (Runnable)taskArray[0];
            notificationExpiration = (Long)taskArray[1];

            // execute the notification task (or discard the task if it has already expired)
            try
            {
                if ((notificationExpiration != null) && (notificationTask != null))
                {
                    expirationTime = notificationExpiration.longValue();
                    if (isExpired(expirationTime))
                    {
                        if (m_droppedStatistic != null)
                        {
                            // there is a small window in which this may not have been created yet
                            m_droppedStatistic.updateValue(1);
                        }
                    }
                    else {
                        notificationTask.run(); // run the task if its time-to-live has not yet passed...
                    }
                }
            }
            catch (Exception e)
            {
                if (m_container.isClosing())
                {
                    return;
                }
                else
                {
                    m_container.logMessage("Failed to send management notification, trace follows... ", e, Level.SEVERE);
                }
            }
        } // "while (!m_container.isClosing())"
    }

    @Override
    public void enqueueNotificationTask(Runnable notificationTask, long expirationTime)
    {
        if (notificationTask != null)
        {
            Object[] taskArray = new Object[2];
            taskArray[0] = (Object)notificationTask;
            taskArray[1] = (Object)new Long(expirationTime);
            
            boolean log50PercentThresholdBreached = false;
            boolean log75PercentThresholdBreached = false;
            boolean logInitialNotificationDrop = false;
            boolean logNotificationsDropped = false;

            synchronized (m_notificationTasks)
            {
                if (m_queueSize > 0)
                {
                    int currentSize = m_notificationTasks.size();
                    if (currentSize >= m_queueSize)
                    {
                        m_notificationTasks.remove(0);
                        m_droppedSinceLastLogged++;
                        if (m_droppedStatistic != null)
                        {
                            // there is a small window in which this may not have been created yet
                            m_droppedStatistic.updateValue(1);
                        }
                        if (m_droppedSinceLastLogged == 1 && !m_initialDropLogged)
                        {
                            m_initialDropLogged  = true;
                            logInitialNotificationDrop = true;
                        }
                        if (m_droppedSinceLastLogged == DROPPED_NOTIFICATION_LOGGING_THRESHOLD)
                        {
                            logNotificationsDropped = true;
                            m_droppedSinceLastLogged = 0;
                        }
                    }
                    else
                    if (currentSize > m_75PercentThreshold && !m_75PercentThresholdBreachLogged)
                    {
                        log75PercentThresholdBreached = true;
                        m_50PercentThresholdBreachLogged = true;
                        m_75PercentThresholdBreachLogged = true;
                    }
                    else
                    if (currentSize > m_50PercentThreshold && !m_50PercentThresholdBreachLogged)
                    {
                        log50PercentThresholdBreached = true;
                        m_50PercentThresholdBreachLogged = true;
                    }
                }
                m_notificationTasks.add(taskArray);
                m_notificationTasks.notifyAll();
            }
            
            if (log50PercentThresholdBreached)
            {
                m_container.logMessage(null, "Management notifications dispatch queue is 50% full", Level.WARNING);
            }
            else
            if (log75PercentThresholdBreached)
            {
                m_container.logMessage(null, "Management notifications dispatch queue is 75% full", Level.WARNING);
            }
            else
            if (logInitialNotificationDrop)
            {
                m_container.logMessage(null, "Management notifications will be dropped due to slow/stopped delivery...", Level.WARNING);
            }
            else
            if (logNotificationsDropped)
            {
                m_container.logMessage(null, DROPPED_NOTIFICATION_LOGGING_THRESHOLD + " management notifications dropped due to slow/stopped delivery", Level.WARNING);
            }
        }
    }

    private boolean isExpired(long expirationTime)
    {
        boolean expired = false;

        if (System.currentTimeMillis() > expirationTime)
        {
            expired = true;
        }

        return expired;
    }
}