package com.sonicsw.mf.framework.agent;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import com.sonicsw.mf.common.runtime.IContainerExitCodes;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.IContainer;

public final class PBE
{
    // we use reflection to avoid progress.message.crypto classes being loaded if they are never used
    // - ContainerImpl has a reference to this class and could cause such classes to be loaded even
    // though they would never be used

    private static Class m_pbeClass;
    private static Method m_encryptMethod;
    private static Method m_decryptMethod;

    static
    {
        try
        {
            m_pbeClass = Class.forName("progress.message.crypto.PBETool");
        }
        catch(ClassNotFoundException e)
        {
            logMessage("Encryption classes not found on classpath", e, Level.SEVERE);
            ContainerImpl.getContainer().shutdown(IContainerExitCodes.INVALID_CMD_LINE_EXIT_CODE);
        }

        try
        {
            m_encryptMethod = m_pbeClass.getMethod("readStream", new Class[] { byte[].class, String.class });
            m_decryptMethod = m_pbeClass.getMethod("readPBEStream", new Class[] { byte[].class, String.class });
        }
        catch(NoSuchMethodException e)
        {
            logMessage("Failed to determine crypto routines", e, Level.SEVERE);
            shutdown(IContainerExitCodes.INVALID_CMD_LINE_EXIT_CODE);
        }
    }

    public static byte[] encrypt(byte[] bytes, String password)
    throws EncryptionException
    {
        try
        {
            return (byte[])m_encryptMethod.invoke(null, new Object[] { bytes, password });
        }
        catch(InvocationTargetException e)
        {
            logMessage("Encryption failure, trace follows...", e.getTargetException(), Level.SEVERE);
            throw new EncryptionException("Failed to encrypt data");
        }
        catch(IllegalAccessException e)
        {
            logMessage("Encryption failure, trace follows...", e, Level.SEVERE);
            throw new EncryptionException("Failed to encrypt data");
        }
    }

    public static byte[] decrypt(byte[] bytes, String password)
    throws EncryptionException
    {
        try
        {
            return (byte[])m_decryptMethod.invoke(null, new Object[] { bytes, password });
        }
        catch(InvocationTargetException e)
        {
            logMessage("Decryption failure, trace follows...", e.getTargetException(), Level.SEVERE);
            throw new EncryptionException("Failed to decrypt data");
        }
        catch(IllegalAccessException e)
        {
            logMessage("Decryption failure, trace follows...", e, Level.SEVERE);
            throw new EncryptionException("Failed to decrypt data");
        }
    }

    public static byte[] decrypt(byte[] bytes)
    throws EncryptionException
    {
        Object pbeString = null;
        try
        {
            pbeString = Class.forName("com.sonicsw.mf.framework.agent.PBEString").newInstance();
        }
        catch(ClassNotFoundException e)
        {
            logMessage("Encryption classes not found on classpath", e, Level.SEVERE);
            shutdown(IContainerExitCodes.INVALID_CMD_LINE_EXIT_CODE);
        }
        catch(InstantiationException e)
        {
            logMessage("Decryption failure, trace follows...", e, Level.SEVERE);
            throw new EncryptionException("Failed to decrypt data");
        }
        catch(IllegalAccessException e)
        {
            logMessage("Decryption failure, trace follows...", e, Level.SEVERE);
            throw new EncryptionException("Failed to decrypt data");
        }

        Method decryptMethod = null;
        try
        {
            decryptMethod = pbeString.getClass().getDeclaredMethod("decrypt", new Class[] { byte[].class });
        }
        catch(NoSuchMethodException e)
        {
            logMessage("Failed to determine crypto routines", e, Level.SEVERE);
            shutdown(IContainerExitCodes.INVALID_CMD_LINE_EXIT_CODE);
        }

        try
        {
            return (byte[])decryptMethod.invoke(pbeString, new Object[] { bytes });
        }
        catch(InvocationTargetException e)
        {
            if (e.getTargetException() instanceof IOException || e.getTargetException() instanceof OutOfMemoryError)
             {
                return bytes; // assume its already in the clear
            }
            logMessage("Decryption failure, trace follows...", e.getTargetException(), Level.SEVERE);
            throw new EncryptionException("Failed to decrypt data");
        }
        catch(IllegalAccessException e)
        {
            logMessage("Decryption failure, trace follows...", e, Level.SEVERE);
            throw new EncryptionException("Failed to decrypt data");
        }
    }

    private static void logMessage(String message, int severity)
    {
        logMessage(message, null, severity);
    }

    private static void logMessage(String message, Throwable e, int severity)
    {
        IContainer container = null;
        try
        {
          container = ContainerImpl.getContainer();
        }
        catch (NoClassDefFoundError noClass)
        {
           // keep going with the logging. container is null and in fact, some of the classes a container
           // depends on are not loaded.
           // for instance, when PBE is used from the SMC to decrypt the ds.xml during the connection, if an invalid
           // password is entered, we'll land here.
        }
        if (container == null) 
        {
            System.out.println('(' + Level.LEVEL_TEXT[severity] + ')' + message);
            e.printStackTrace(System.out);
        }
        else
        {
            container.logMessage(null, message, e, severity);
        }
    }

    private static void shutdown(int exitCode)
    {
        IContainer container = ContainerImpl.getContainer();
        if (container == null)
        {
            Runtime.getRuntime().halt(exitCode);
        }
        else
        {
            container.shutdown(exitCode);
        }
    }
}
