package com.sonicsw.mf.framework.agent;

import javax.jms.Connection;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageListener;
import javax.jms.MessageProducer;
import javax.jms.Session;
import javax.jms.TextMessage;
import javax.jms.Topic;

import com.sonicsw.mf.common.runtime.IFaultTolerantState;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.NonRecoverableStateChangeException;
import com.sonicsw.mf.common.runtime.RecoverableStateChangeException;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.util.StateManager;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;

import progress.message.jclient.DeliveryMode;

class PingThread extends Thread implements MessageListener {

    private final static String MESSAGE_TYPE = "MESSAGE_TYPE";
    private final static short REQUEST_MESSAGE = 1;
    private final static short REPLY_MESSAGE   = 2;

    private final static String SENDER_FT_STATE = "SENDER_FT_STATE"; 

    private StateManager m_FTStateMgr;
    private ContainerImpl m_containerImpl;

    private Connection m_pingConnection;
    private Session m_pingSession;
    private String m_pingTopicName;
    private String m_pingTopicName4peer;
    private Topic m_pingTopic;
    private Topic m_pingTopic4peer;
    private MessageProducer m_sender;
    private MessageConsumer m_receiver;

    private volatile int m_pingInterval;
    private volatile int m_pingTimeout;
    private String m_ftRole;
    private boolean m_backupStartActive;
 
    private Object m_reqRepLock = new Object(); //lock object to synchronize request/reply
    private volatile String m_requestID;
    private volatile boolean m_hasReplied = false;
    private volatile Message m_reply = null; //the reply message to the request
    private volatile long m_lastPeerPingRequest;

    private volatile boolean m_isSuspended = false; //for suspendActive operation


    PingThread(Connection connection, String topicName, String topicName4peer, int faultDetectInterval, int faultDetectTimeout,
               String ftRole, boolean backupStartActive, StateManager stateMgr, ContainerImpl containerImpl)
    throws JMSException {
        super("PingThread on topic " + topicName);
        m_containerImpl = containerImpl;
        m_FTStateMgr = stateMgr;

        m_pingInterval = faultDetectInterval;
        m_pingTimeout = faultDetectTimeout;
        m_ftRole = ftRole;
        m_backupStartActive = backupStartActive;
        m_pingConnection = connection;
        m_pingTopicName = topicName;
        m_pingTopicName4peer = topicName4peer;

        setupPingChannel();
    }
    
    Connection getConnection()
    {
        return m_pingConnection;
    }

    private synchronized void setupPingChannel() throws JMSException {
        cleanupPingChannel(); //cleanup old channel if any

        if (m_pingConnection == null)
        {
            throw new JMSException("fault detect connection is null");
        }

        m_pingSession = m_pingConnection.createSession(false, Session.AUTO_ACKNOWLEDGE);

        //send to peer ping topic
        m_pingTopic4peer = m_pingSession.createTopic(m_pingTopicName4peer);

        // since peer topic name may change dynamically when FAULT_TOLERANCE_PEER_REF changed
        // we don't set topic at creating producer time. Topic is given at send time.
        m_sender = m_pingSession.createProducer(null);
        m_sender.setDeliveryMode(DeliveryMode.NON_PERSISTENT_REPLICATED);
        m_sender.setPriority(9);

        //receive from ping topic
        m_pingTopic = m_pingSession.createTopic(m_pingTopicName);

        m_receiver = m_pingSession.createConsumer(m_pingTopic);
        m_receiver.setMessageListener(this);
        m_pingConnection.start();
    }

    // for dynamic configuration change
    void setPingInterval(int pingInterval) { m_pingInterval = pingInterval; }

    void setPingTimeout(int pingTimeout) { m_pingTimeout = pingTimeout; }

    void  setPingTopicNameForPeer(String topicName4peer) throws JMSException {
        m_pingTopicName4peer = topicName4peer;
        if (m_pingSession != null)
        {
            m_pingTopic4peer = m_pingSession.createTopic(m_pingTopicName4peer);
        }
    }

    /**
     * Send request to peer topic and waiting for reply on this topic.
     * @return true if get reply. Otherwise, false.
     * @throws JMSException
     * @throws java.lang.InterruptedException
     */
    private Message request() throws JMSException,InterruptedException {

        TextMessage request = m_pingSession.createTextMessage();
        request.setShortProperty(MESSAGE_TYPE, REQUEST_MESSAGE);
        //set this container's FT state in the request as property SENDER_FT_STATE
        request.setShortProperty(SENDER_FT_STATE, m_FTStateMgr.getState(null) );

        while (true)
        {
            synchronized (m_reqRepLock) {
                // since peer topic may change dynamically, pass peer topic in every send method
                m_sender.send(m_pingTopic4peer, request); //send request to m_pingTopic4peer
                long startTime = System.currentTimeMillis();
                m_hasReplied = false;
                m_requestID = request.getJMSMessageID();
    
                if ((m_containerImpl.m_agent.getTraceMask().intValue() & Agent.TRACE_CONTAINER_FT) > 0
                        && (m_containerImpl.m_agent.getTraceMask().intValue() & Agent.TRACE_VERBOSE) > 0)
                {
                    m_containerImpl.logMessage(null, "FT container ping request sent, ID=" + m_requestID +
                                               ", timeout=" + m_pingTimeout, Level.TRACE);
                }

                //wait for reply after send util timeout
                m_reqRepLock.wait(m_pingTimeout * 1000); //wait to be notified by onMessage
    
                // m_hasReplied is set to true in onMessage() when the handler gets a reply
                if (m_hasReplied) {
                    return m_reply;
                } else {
                    // test if we should try again because although we timed out with no reply, it looks like we
                    // have at least received a ping request from the peer
                    if (m_lastPeerPingRequest <= System.currentTimeMillis() && m_lastPeerPingRequest >= startTime)
                    {
                        continue;
                    }
                    
                    // else return the timeout (indicated by a null reply message)
                    return null;
                }
            } //sync
        }
    }

    /**
     * In PingThread, on pingInterval, send ping request and waiting for reply
     */
    @Override
    public void run() {

        try {
            JMSException lastJMSException = null;
            
            while (!isInterrupted())
            {
                try
                {
                    //send request signal, return the reply message when get reply or null if ping timeout
                    Message reply = request();

                    // 1) performs handleReply and sending ping request sequentially should be OK, because
                    //    If no state transition, handleReply return quickly;
                    //    If there is state transition, handleReply may take longer time. However,
                    //    there is no need to send another ping in the middle of state transition.
                    // 2) if this container is suspended by suspendActive operation, don't handle reply.
                    //    but this container still request/reply ping signal so that the peer container may know this container's state.
                    if (!m_isSuspended)
                    {
                        handleReply(reply);
                    }
                    
                    // success so clear any exception we previous recorded
                    if (lastJMSException != null)
                    {
                        m_containerImpl.logMessage(null, "...FT container ping request sent", Level.WARNING);
                    }
                    lastJMSException = null;
                }
                catch (JMSException jmsE)
                {
                    if ((m_containerImpl.m_agent.getTraceMask().intValue() & Agent.TRACE_CONTAINER_FT) > 0)
                    {
                        m_containerImpl.logMessage(null, "Failed to send FT container ping request, trace follows...", jmsE, Level.TRACE);
                        m_containerImpl.logMessage(null, "retrying ...", Level.TRACE);
                    }
                    else
                    {
                        if (lastJMSException == null)
                        {
                            m_containerImpl.logMessage(null, "Failed to send FT container ping request, retrying ...", Level.WARNING);
                        }
                    }
                    
                    lastJMSException = jmsE;
                }

                sleep(m_pingInterval * 1000); //start another round of request/reply after interval time

            } //while

        } //try
        catch (InterruptedException ie)
        {
            Thread.currentThread().interrupt();
            cleanupPingChannel();
        }

    } //run

    private void handleReply(Message reply) {
        short peerFTState = -1; //Unknown state
        if (reply == null)
        {
            peerFTState = 99; //ping timeout

            if ((m_containerImpl.m_agent.getTraceMask().intValue() & Agent.TRACE_CONTAINER_FT) > 0)
            {
                m_containerImpl.logMessage(null, "FT container ping timeout", Level.TRACE);
            }
        }
        else
        {
            try
            {
                peerFTState = reply.getShortProperty(SENDER_FT_STATE);

                if ((m_containerImpl.m_agent.getTraceMask().intValue() & Agent.TRACE_CONTAINER_FT) > 0
                		&& (m_containerImpl.m_agent.getTraceMask().intValue() & Agent.TRACE_VERBOSE) > 0)
                {
                    m_containerImpl.logMessage(null, "FT container ping reply received, ID=" + reply.getJMSCorrelationID() +
                                               ", peer state=" + IFaultTolerantState.STATE_TEXT[peerFTState], Level.TRACE);
                }
            }
            catch (JMSException jmsE)
            {
                //if we get a ping, but cannot get the senderFTState from the reply,
                //leave the peerFTState as unknown which cause no state transition until next ping.
                IContainer container = ContainerImpl.getContainer();
                if ((m_containerImpl.m_agent.getTraceMask().intValue() & Agent.TRACE_CONTAINER_FT) > 0)
                {
                    container.logMessage(null, "Failed to get the FT container peer state, trace follows...", jmsE, Level.TRACE);
                    container.logMessage(null, "retrying...", jmsE, Level.TRACE);
                }
                else
                {
                    container.logMessage(null, "Failed to get the FT container peer state, retrying...", Level.WARNING);
                }
            }

        }

        try
        {
            if (m_ftRole.equalsIgnoreCase(IContainerConstants.FAULT_TOLERANCE_ROLE_PRIMARY) )
            { //This is a Primary container
                switch (m_FTStateMgr.getState(null) ) {

                    case IFaultTolerantState.STATE_WAITING: //T1: primary WAITING
                    {
                        switch (peerFTState)
                        {
                            case IFaultTolerantState.STATE_ACTIVE: //backup ACTIVE
                                // primary WAITING => STANDBY
                                m_FTStateMgr.requestStateChange(IFaultTolerantState.STATE_WAITING,
                                                                IFaultTolerantState.STATE_STANDBY,
                                                                null);
                                break;
                            case IFaultTolerantState.STATE_WAITING: //backup WAITING
                            case IFaultTolerantState.STATE_STANDBY: //backup STANDBY
                            case 99: //ping timeout
                                // primay WAITING => ACTIVE
                                m_containerImpl.setFailingOver(true);
                                m_FTStateMgr.requestStateChange(IFaultTolerantState.STATE_WAITING,
                                                                IFaultTolerantState.STATE_ACTIVE,
                                                                null);
                        }
                        break;
                    }
                    case IFaultTolerantState.STATE_ACTIVE: //T2: primary ACTIVE
                    {
                        //primary stay in ACTIVE state no matter what state backup is
                        break;
                    }
                    case IFaultTolerantState.STATE_STANDBY: //T3: primary STANDBY
                    {
                        switch (peerFTState)
                        {
                            case IFaultTolerantState.STATE_ACTIVE: //backup ACTIVE
                                //primary stay in STANDBY
                                break;
                            case IFaultTolerantState.STATE_WAITING: //backup WAITING
                            case IFaultTolerantState.STATE_STANDBY: //backup STANDBY
                            case 99: //ping timeout
                            {
                                Boolean allowFailover = null;
                                if (m_containerImpl != null) {
                                    allowFailover = m_containerImpl.getAllowFailover();
                                }
                                if ( (allowFailover != null) && allowFailover.booleanValue())
                                {
                                    // primary STANDBY => ACTIVE
                                    m_containerImpl.setFailingOver(true);
                                    m_FTStateMgr.requestStateChange(IFaultTolerantState.STATE_STANDBY,
                                                                    IFaultTolerantState.STATE_ACTIVE,
                                                                    null);
                                }
                            }
                        }
                        break;
                    }
                    default:
                }

            } // end of Primary container
            else if (m_ftRole.equalsIgnoreCase(IContainerConstants.FAULT_TOLERANCE_ROLE_BACKUP))
            { //This is a backup container
                switch (m_FTStateMgr.getState(null) ) {

                    case IFaultTolerantState.STATE_WAITING: //T1: backup WAITING
                    {
                        switch (peerFTState)
                        {
                            case IFaultTolerantState.STATE_ACTIVE: //primary ACTIVE
                                // backup WAITING => STANDBY
                                m_FTStateMgr.requestStateChange(IFaultTolerantState.STATE_WAITING,
                                                                IFaultTolerantState.STATE_STANDBY,
                                                                null);
                                break;
                            case IFaultTolerantState.STATE_WAITING: //primary WAITING
                            case IFaultTolerantState.STATE_STANDBY: //primary STANDBY
                                //backup keep WAITING
                                break;
                            case 99: //ping timeout
                                if (m_backupStartActive)
                                {
                                    // backup WAITING => ACTIVE
                                    m_containerImpl.setFailingOver(true);
                                    m_FTStateMgr.requestStateChange(IFaultTolerantState.STATE_WAITING,
                                                                IFaultTolerantState.STATE_ACTIVE,
                                                                null);
                                }
                                //if backup is not startActive, the backup keep WAITING
                        }
                        break;
                    }
                    case IFaultTolerantState.STATE_ACTIVE: //T2: backup ACTIVE
                    {
                        switch (peerFTState)
                        {
                            case IFaultTolerantState.STATE_ACTIVE: //primary ACTIVE
                                // backup ACTIVE => STANDBY
                                m_FTStateMgr.requestStateChange(IFaultTolerantState.STATE_ACTIVE,
                                                                IFaultTolerantState.STATE_STANDBY,
                                                                null);
                                break;
                            default:
                                //otherwise, backup stay in ACTIVE
                        }
                        break;
                    }
                    case IFaultTolerantState.STATE_STANDBY: //T3: backup STANDBY
                    {
                        switch (peerFTState)
                        {
                            case IFaultTolerantState.STATE_STANDBY: //primary STANDBY
                            case 99: //ping timeout
                            {
                                Boolean allowFailover = null;
                                if (m_containerImpl != null) {
                                    allowFailover = m_containerImpl.getAllowFailover();
                                }
                                if ( (allowFailover != null) && allowFailover.booleanValue())
                                {
                                    // backup STANDBY => ACTIVE
                                    m_containerImpl.setFailingOver(true);
                                    m_FTStateMgr.requestStateChange(IFaultTolerantState.STATE_STANDBY,
                                                                    IFaultTolerantState.STATE_ACTIVE,
                                                                    null);
                                }
                                break;
                            }
                            default:
                                //otherwise, backup stay in STANDBY
                        }
                        break;
                    }
                    default:
                }

            } //end of backup container
        }
        catch (RecoverableStateChangeException re)
        {
            m_containerImpl.logMessage(null, re, Level.WARNING);
        }
        catch (NonRecoverableStateChangeException nre)
        {
            m_containerImpl.logMessage(null, nre, Level.WARNING);
        }
        finally
        {
            //if this is a fail over request, the m_isFailingOver has been set to true before state change to ACTIVE.
            //otherwise, i.e. start component from SMC, the m_isFailingOver keep in false.
            //The start invocation in AbstractMBean uses m_isFailingOver flag to decide whether or not throw IllegalStateException
            //At this point, fail over process is done. Set m_isFailingOver in ContainerImpl back to false
            m_containerImpl.setFailingOver(false);
        }
    }

    /**
     * MessageListener to listen to the request/reply from peer
     * @param msg
     */
    @Override
    public void onMessage(Message msg) {
        try {
            if (msg == null)
            {
                return;
            }

            if (msg.getShortProperty(MESSAGE_TYPE) == REQUEST_MESSAGE) {
                
                m_lastPeerPingRequest = System.currentTimeMillis();
                
                //send reply
                Message request = msg;
                m_requestID = request.getJMSMessageID();

                if ((m_containerImpl.m_agent.getTraceMask().intValue() & Agent.TRACE_CONTAINER_FT) > 0
                        && (m_containerImpl.m_agent.getTraceMask().intValue() & Agent.TRACE_VERBOSE) > 0)
                {
                    m_containerImpl.logMessage(null, "FT container ping request received, ID=" + m_requestID +
                                               ", timeout=" + m_pingTimeout, Level.TRACE);
                }

                Message reply = m_pingSession.createTextMessage();
                reply.setShortProperty(MESSAGE_TYPE, REPLY_MESSAGE);

                //set this container's FT state in the reply as property SENDER_FT_STATE
                reply.setShortProperty(SENDER_FT_STATE, m_FTStateMgr.getState(null) );

                reply.setJMSCorrelationID(m_requestID);

                // since peer topic may change dynamically, pass peer topic in every send method
                m_sender.send(m_pingTopic4peer, reply);

                if ((m_containerImpl.m_agent.getTraceMask().intValue() & Agent.TRACE_CONTAINER_FT) > 0
                        && (m_containerImpl.m_agent.getTraceMask().intValue() & Agent.TRACE_VERBOSE) > 0)
                {
                    m_containerImpl.logMessage(null, "FT container ping reply sent, ID=" + m_requestID, Level.TRACE);
                }
            }

            if ( (msg.getShortProperty(MESSAGE_TYPE) == REPLY_MESSAGE) ) {
                // when get the reply, notify the sender thread
                // if this is not the reply waited by the sender, no-op
                
                String correlationID = msg.getJMSCorrelationID();

                synchronized (m_reqRepLock) {
                    if (correlationID.equals(m_requestID)) {
                        m_reply = msg;
                        m_hasReplied = true;
                        m_reqRepLock.notifyAll(); //notify waiting in request method
                    }
                } //sync
            }
        } catch (JMSException e) {
            IContainer container = ContainerImpl.getContainer();
            if ((m_containerImpl.m_agent.getTraceMask().intValue() & Agent.TRACE_CONTAINER_FT) > 0)
            {
                m_containerImpl.logMessage(null, "FT container ping failure, trace follows...", e, Level.TRACE);
                m_containerImpl.logMessage(null, "retrying...", Level.TRACE);
            }
            else
            {
                container.logMessage(null, "FT container ping failure, retrying...", Level.WARNING);
            }
        }
    } //onMessage

    /**
     * For dynamic operation. This method is called from suspender thread created in Agent
     * It transitions this container from ACTIVE to STANDBY no matter this is a primary or backup container.
     */
    synchronized void suspendActiveState()
    {
        try
        {
            // 1) the requestStateChange is a synchronized method.
            //    Therefore, if this container is in the middle of previous state transition,
            //    this requestStateChange call will be locked out until previous transition complete.
            // 2) m_isSuspended is true if requestStateChange transitions the state successfully, false if not.
            m_isSuspended = m_FTStateMgr.requestStateChange(IFaultTolerantState.STATE_ACTIVE,
                                                            IFaultTolerantState.STATE_STANDBY,
                                                            null);
            if (!m_isSuspended)
            {
                IContainer container = ContainerImpl.getContainer();
                container.logMessage(null, "Failed to suspend the container from " + IFaultTolerantState.STATE_TEXT[m_FTStateMgr.getState(null)], Level.WARNING);
            }
        }
        catch (RecoverableStateChangeException re)
        {
            m_containerImpl.logMessage(null, re, Level.WARNING);
        }
        catch (NonRecoverableStateChangeException nre)
        {
            m_containerImpl.logMessage(null, nre, Level.WARNING);
        }
    }

    synchronized void resumeAfterSuspend()
    {
        m_isSuspended = false;
    }

    @Override
    public void interrupt() {
        super.interrupt();
        cleanupPingChannel();
    }

    private synchronized void cleanupPingChannel() {

        if (m_receiver != null) {
            try {
                m_receiver.close();
            } catch (JMSException e) {
                e.printStackTrace();
            } finally {
                m_receiver = null;
            }
        }

        if (m_sender != null) {
            try {
                m_sender.close();
            } catch (JMSException e) {
                e.printStackTrace();
            } finally {
                m_sender = null;
            }
        }

        if (m_pingSession != null) {
            try {
                m_pingSession.close();
            } catch (JMSException e) {
                e.printStackTrace();
            } finally {
                m_pingSession = null;
            }
        }
    }
}