package com.sonicsw.mf.framework.agent;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.AttributeSetTypeException;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.ReadOnlyException;
import com.sonicsw.mf.common.dirconfig.DirectoryDoesNotExistException;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.UpdateDisallowedOnFailoverException;
import com.sonicsw.mf.common.dirconfig.VersionOutofSyncException;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.security.IManagementPermission;
import com.sonicsw.mf.framework.agent.cache.CacheClosedException;
import com.sonicsw.mf.framework.agent.cache.IConfigCache;
import com.sonicsw.mf.framework.agent.cache.PersistentCacheException;
import com.sonicsw.mf.framework.directory.storage.ParentDirDoesNotExistException;

final class RuntimeConfigurationManager
{
    private ContainerImpl m_container;
    private IConfigCache m_configCache;
    private ContainerDS m_directory;
    
    private Hashtable m_runtimeConfigurations = new Hashtable();
    private ArrayList m_queuedDSUpdateTasks = new ArrayList();
    
    private static final String DS_RUNTIME_TYPE = "MF_RUNTIME";
    private static final String DS_RUNTIME_TYPE_VERSION = "2.0";

    RuntimeConfigurationManager(ContainerImpl container, IConfigCache configCache, ContainerDS directory)
    {
        m_container = container;
        m_configCache = configCache;
        m_directory = directory;
        
        new DSUpdater().start();
    }
    
    IDirElement getRuntimeConfiguration(String id, String type, boolean useCacheOnly)
    {
        String configID = IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_RUNTIME_DIR + IMFDirectories.MF_DIR_SEPARATOR + m_container.getContainerIdentity().getCanonicalName() + IMFDirectories.MF_DIR_SEPARATOR + id + IMFDirectories.MF_DIR_SEPARATOR + type;

        IDirElement runtimeConfiguration = (IDirElement)m_runtimeConfigurations.get(configID);
        
        // if we don't have it in the in-memory cache, then see if its in the cache
        if (runtimeConfiguration == null)
        {
            try
            {
                runtimeConfiguration = (IDirElement)m_configCache.getCacheView().getElement(configID);
            }
            catch (CacheClosedException e)
            {
                // the cache is already closed because we are shutting down, so just return an empty element
                runtimeConfiguration = ElementFactory.createElement(configID, DS_RUNTIME_TYPE, DS_RUNTIME_TYPE_VERSION);
            }
        }
        
        // not available in cache
        if (runtimeConfiguration == null)
        {
            if (useCacheOnly) // if can only use cache then manufacture a new config
            {
                runtimeConfiguration = ElementFactory.createElement(configID, DS_RUNTIME_TYPE, DS_RUNTIME_TYPE_VERSION);
                setRuntimeConfiguration(runtimeConfiguration, useCacheOnly);
            }
            else // check the DS
            {
                if (m_container.getConnectorServer().isConnected())
                {
                    String currentUserID = TaskScheduler.getCurrentUserID();
                    try
                    {
                        if (TaskScheduler.isExecutionThread())
                        {
                            ((TaskScheduler.ExecutionThread)Thread.currentThread()).setUserID(IManagementPermission.SUPER_USER_NAME);
                        }
                        runtimeConfiguration = (IDirElement)m_directory.getElementForUpdate(configID);
                    }
                    catch (MFRuntimeException e)
                    {
                        Throwable cause = e.getCause();
                        if (cause == null || (!(cause instanceof DirectoryDoesNotExistException) && !ContainerUtil.isCausedByTimeout(cause)))
                        {
                            if (!m_container.isClosing())
                            {
                                m_container.logMessage(id, "Unexpected failure while obtaining any enabled " + type + " from the Directory Service, trace follows...", cause == null ? e : cause, Level.WARNING);
                            }
                        }
                    }
                    catch (Exception e)
                    {
                        if (!m_container.isClosing())
                        {
                            m_container.logMessage(id, "Unexpected failure while obtaining any enabled " + type + " from the Directory Service, trace follows...", e, Level.WARNING);
                        }
                    }
                    finally
                    {
                        if (TaskScheduler.isExecutionThread())
                        {
                            ((TaskScheduler.ExecutionThread)Thread.currentThread()).setUserID(currentUserID);
                        }
                    }
                }
            }
        }

        // prepare something to give back even if its null
        if (runtimeConfiguration == null)
        {
            runtimeConfiguration = ElementFactory.createElement(configID, DS_RUNTIME_TYPE, DS_RUNTIME_TYPE_VERSION);
            setRuntimeConfiguration(runtimeConfiguration, false);
        }

        runtimeConfiguration.getAttributes().getAttributes();

        return (IDirElement)runtimeConfiguration.createWritableClone();
    }

    void setRuntimeConfiguration(IDirElement element, boolean useCacheOnly)
    {
        setRuntimeConfigurationInCache(element);
        if (!useCacheOnly)
        {
            queueSetRuntimeConfigurationInDS(element);
        }
    }
    
    private void setRuntimeConfigurationInCache(IDirElement runtimeConfiguration)
    {
        try
        {
            m_runtimeConfigurations.put(runtimeConfiguration.getIdentity().getName(), runtimeConfiguration);
            try
            {
                m_configCache.deleteElement(runtimeConfiguration.getIdentity().getName());
            }
            catch (Throwable e)
            {
                if (e.getCause() == null || !(e.getCause() instanceof ParentDirDoesNotExistException))
                {
                    m_container.logMessage(null, "Cache failure, trace follows...", e, Level.WARNING);
                }
            }
            m_configCache.setElement(runtimeConfiguration);
        }
        catch (VersionOutofSyncException e) { } // can't do much about this
        catch (CacheClosedException e) { } // container is shutting down so ignore
        catch (PersistentCacheException e) { } // can't do anything about this
    }
    
    private void queueSetRuntimeConfigurationInDS(IDirElement runtimeConfiguration)
    {
        String configID = runtimeConfiguration.getIdentity().getName();
        SetRuntimeConfigurationInDSTask setRuntimeConfigurationInDSTask = new SetRuntimeConfigurationInDSTask(configID, (IDirElement)runtimeConfiguration.createWritableClone());
        synchronized (m_queuedDSUpdateTasks)
        {
            // remove any existing task for this configuration
            Iterator dsUpdateTasks = m_queuedDSUpdateTasks.iterator();
            while (dsUpdateTasks.hasNext())
            {
                DSUpdateTask dsUpdateTask = (DSUpdateTask)dsUpdateTasks.next();
                if (dsUpdateTask instanceof SetRuntimeConfigurationInDSTask && dsUpdateTask.configID.equals(configID))
                {
                    dsUpdateTasks.remove();
                }
            }
            
            m_queuedDSUpdateTasks.add(setRuntimeConfigurationInDSTask);
            m_queuedDSUpdateTasks.notifyAll();
        }
    }

    void deleteRuntimeConfiguration(String configID, boolean useCacheOnly)
    {
        deleteRuntimeConfigurationInCache(configID);
        if (!useCacheOnly)
        {
            queueDeleteRuntimeConfigurationInDS(configID);
        }
    }
    
    private void deleteRuntimeConfigurationInCache(String configID)
    {
        m_runtimeConfigurations.remove(configID);
        try
        {
            m_configCache.deleteElement(configID);
        }
        catch (Throwable e)
        {
            if (e.getCause() == null || !(e.getCause() instanceof ParentDirDoesNotExistException))
            {
                m_container.logMessage(null, "Cache failure, trace follows...", e, Level.WARNING);
            }
        }
    }
    
    private void queueDeleteRuntimeConfigurationInDS(String configID)
    {
        DeleteRuntimeConfigurationInDSTask deleteRuntimeConfigurationInDSTask = new DeleteRuntimeConfigurationInDSTask(configID);
        synchronized (m_queuedDSUpdateTasks)
        {
            // remove any existing task for this configuration
            Iterator dsUpdateTasks = m_queuedDSUpdateTasks.iterator();
            while (dsUpdateTasks.hasNext())
            {
                DSUpdateTask dsUpdateTask = (DSUpdateTask)dsUpdateTasks.next();
                if (dsUpdateTask instanceof DeleteRuntimeConfigurationInDSTask && dsUpdateTask.configID.equals(configID))
                {
                    dsUpdateTasks.remove();
                }
            }
            
            m_queuedDSUpdateTasks.add(deleteRuntimeConfigurationInDSTask);
            m_queuedDSUpdateTasks.notifyAll();
        }
    }
    
    private abstract class DSUpdateTask
    implements Runnable
    {
        private String configID;
        
        private DSUpdateTask(String configID)
        {
            this.configID = configID;
        }
    }
    
    private final class SetRuntimeConfigurationInDSTask
    extends RuntimeConfigurationManager.DSUpdateTask
    {
        IDirElement localRuntimeConfiguration;
        
        private SetRuntimeConfigurationInDSTask(String configID, IDirElement localRuntimeConfiguration)
        {
            super(configID);
            this.localRuntimeConfiguration = localRuntimeConfiguration;
        }
        
        @Override
        public void run()
        {
            while (true) // try until we have completed a successful update
            {
                if (RuntimeConfigurationManager.this.m_container.isClosing())
                {
                    return;
                }
                if (RuntimeConfigurationManager.this.m_container.getConnectorServer().isConnected())
                {
                    try
                    {
                        IDirElement runtimeConfigurationFromDS;
                        try
                        {
                            runtimeConfigurationFromDS = (IDirElement)m_directory.getElementForUpdate(localRuntimeConfiguration.getIdentity().getName());
                            if (runtimeConfigurationFromDS == null)
                            {
                                m_directory.setElement(localRuntimeConfiguration);
                            }
                            else
                            {
                                cloneRuntimeonfiguration(runtimeConfigurationFromDS, localRuntimeConfiguration);
                                m_directory.setElement(runtimeConfigurationFromDS.doneUpdate());
                            }
                        }
                        catch (MFRuntimeException e)
                        {
                            Throwable cause = e.getCause();
                            if (cause instanceof DirectoryDoesNotExistException)
                            {
                                m_directory.setElement(localRuntimeConfiguration);
                            }
                            else if (ContainerUtil.isCausedByTimeout(cause))
                            {
                                continue;
                            }
                            else if (cause instanceof UpdateDisallowedOnFailoverException)
                            {
                                m_container.logMessage(null, "Read-only Directory Service prevented storing of " + localRuntimeConfiguration.getIdentity().getName(), Level.WARNING);
                            }
                            else
                            {
                                m_container.logMessage(null, "Failed to store " + localRuntimeConfiguration.getIdentity().getName() + " in the Directory Service, trace follows...", cause, Level.WARNING);
                            }
                        }
                        return;
                    }
                    catch(Exception e) { } // continue until success
                }
                try { Thread.sleep(500); } catch (Exception e) { }
            }
        }

        private void cloneRuntimeonfiguration(IDirElement runtimeConfigurationFromDS, IElement localRuntimeConfiguration)
        {
            IAttributeSet fromDS = runtimeConfigurationFromDS.getAttributes();
            IAttributeSet toDS = localRuntimeConfiguration.getAttributes();
            
            try
            {
                // clear out the old attributes
                Iterator attributeNames = fromDS.getAttributes().keySet().iterator();
                while (attributeNames.hasNext())
                {
                    fromDS.deleteAttribute((String)attributeNames.next());
                }
                
                // now copy the updated attributes
                cloneAttributeSet(fromDS, toDS);
            }
            catch (Exception e) { e.printStackTrace(); } // not expecting to happen
        }

        private void cloneAttributeSet(IAttributeSet to, IAttributeSet from)
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
        {
            Iterator copyAttributes = from.getAttributes().entrySet().iterator();
            while (copyAttributes.hasNext())
            {
                Map.Entry entry = (Map.Entry)copyAttributes.next();
                Object value = entry.getValue();
                if (value instanceof IAttributeSet)
                {
                    IAttributeSet set = to.createAttributeSet((String)entry.getKey());
                    cloneAttributeSet(set, (IAttributeSet)value);
                }
                else if (value instanceof IAttributeList)
                {
                    IAttributeList list = to.createAttributeList((String)entry.getKey());
                    cloneAttributeList(list, (IAttributeList)value);
                }
                else
                {
                    to.setObjectAttribute((String)entry.getKey(), value);
                }
            }
        }

        private void cloneAttributeList(IAttributeList to, IAttributeList from)
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
        {
            Iterator copyItems = from.getItems().iterator();
            while (copyItems.hasNext())
            {
                Object value = copyItems.next();
                if (value instanceof IAttributeSet)
                {
                    IAttributeSet set = to.addNewAttributeSetItem();
                    cloneAttributeSet(set, (IAttributeSet)value);
                }
                else if (value instanceof IAttributeList)
                {
                    IAttributeList list = to.addNewAttributeListItem();
                    cloneAttributeList(list, (IAttributeList)value);
                }
                else
                {
                    to.addObjectItem(value);
                }
            }
        }
    }
    
    private final class DeleteRuntimeConfigurationInDSTask
    extends RuntimeConfigurationManager.DSUpdateTask
    {
        private DeleteRuntimeConfigurationInDSTask(String configID)
        {
            super(configID);
        }
        
        @Override
        public void run()
        {
            while (true) // try until we have completed a successful update
            {
                if (RuntimeConfigurationManager.this.m_container.isClosing())
                {
                    return;
                }
                if (RuntimeConfigurationManager.this.m_container.getConnectorServer().isConnected())
                {
                    try
                    {
                        try
                        {
                            m_directory.deleteElement(super.configID);
                        }
                        catch (MFRuntimeException e)
                        {
                            Throwable cause = e.getCause();
                            if (ContainerUtil.isCausedByTimeout(cause))
                            {
                                continue;
                            }
                            else
                            {
                                e.printStackTrace();
                            }
                        }
                        return;
                    }
                    catch(Exception e) { } // continue until success
                }
                try { Thread.sleep(500); } catch (Exception e) { }
            }
        }
    }
    
    private final class DSUpdater
    extends Thread
    {
        private DSUpdater()
        {
            super("Runtime configuration updater [" + RuntimeConfigurationManager.this.m_container.getContainerIdentity().getCanonicalName() + "]");
            super.setDaemon(false);
        }
        
        @Override
        public void run()
        {
            while (!RuntimeConfigurationManager.this.m_container.isClosing())
            {
                DSUpdateTask dsUpdateTask = null;
                synchronized (m_queuedDSUpdateTasks)
                {
                    if (m_queuedDSUpdateTasks.isEmpty())
                    {
                        try { m_queuedDSUpdateTasks.wait(250); } catch (InterruptedException e) { }
                    }
                    else
                    {
                        dsUpdateTask = (DSUpdateTask)m_queuedDSUpdateTasks.remove(0);
                    }
                }
                if (dsUpdateTask != null)
                {
                    dsUpdateTask.run();
                }
            }
        }
    }
}
