package com.sonicsw.mf.framework.agent;

import java.util.ArrayList;
import java.util.EmptyStackException;
import java.util.Iterator;
import java.util.Stack;

import com.sonicsw.mf.comm.IConnectorClient;
import com.sonicsw.mf.common.MFServiceNotActiveException;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.IValueType;
import com.sonicsw.mf.common.metrics.MetricsFactory;
import com.sonicsw.mf.common.metrics.manager.IMetricsRegistrar;
import com.sonicsw.mf.common.metrics.manager.ISampledStatistic;
import com.sonicsw.mf.common.metrics.manager.IStatistic;
import com.sonicsw.mf.common.metrics.manager.IStatisticProvider;
import com.sonicsw.mf.common.metrics.manager.StatisticsFactory;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.ITaskScheduler;
import com.sonicsw.mf.mgmtapi.runtime.IAgentProxy;

/**
 * Each MF container has a task scheduler that framework components can access through the
 * IFrameworkComponentContext.
 *
 * The task scheduler is intended as a way of regulating the total number of threads that
 * the container process creates on behalf of framework functionality (it does nothing to
 * control application thread use). It enables task execution threads to be pooled and
 * reused. Not all framework tasks will use the scheduler, but it is intended to be used
 * as widely as possible where activity would result in the same type of threads being
 * created over and over again.
 *
 * Clients of the scheduler request the scheduler to schedule a task and indicate if the
 * task should be executed at a high priority. High priority tasks are started ahead of
 * waiting non-high priority tasks - careful consideration should be given before specifying
 * the use of high priority.
 *
 * Task execution threads are daemon threads run at normal Java thread priority.
 */
public final class TaskScheduler
extends Thread
implements ITaskScheduler
{
    private ContainerImpl m_container;

    // If these default values change, Agent.updateRuntimeFromDeletedConfig must
    // change also. It sets the scheduler maxThreads and minThreads back to
    // default values if the MAX_THREADS and MIN_THREADS configuration attributes
    // change.
    // The test qa.mgmt.mf.container.agent.BasicAgentConfigurationTests also
    // tests these defaults, so the numbers there have to be changed as well.
    private short m_maxThreads = 50;
    private short m_minThreads = 0;

    private ArrayList<Runnable> m_normalPriorityTasks = new ArrayList<Runnable>();
    private ArrayList<Long> m_normalPriorityTaskExpirationTimes = new ArrayList<Long>();
    private ArrayList<Runnable> m_higherPriorityTasks = new ArrayList<Runnable>();
    private ArrayList<Long> m_higherPriorityTaskExpirationTimes = new ArrayList<Long>();
    private Stack<ExecutionThread> m_executionThreadCache = new Stack<ExecutionThread>();
    private int m_executingThreads = 0;

    // When the an execution thread is idle on the cache for longer than this
    // duration, the thread can be removed from the thread pool as long as the
    // min pool size is maintained
    private static final long MAX_IDLE_DURATION = 5000;

    // metrics
    private IMetricsRegistrar m_metricsRegistrar;
    // statistics
    private IStatistic m_poolSizeStatistic;
    private IStatistic m_maxPoolSizeStatistic;
    private IStatistic m_poolWaitsStatistic;

    public static final String TASK_THREAD_NAME = "Task Runner ";

    private boolean m_isClosing = false;
    private Object m_taskSchedulerLockObj = new Object();
    /**
     * Package only constructor.
     */
    TaskScheduler(ContainerImpl container)
    {
        super("Task Scheduler");
        setDaemon(true);

        m_container = container;
    }
    
    void close()
    {
        if (!m_container.isClosing())
        {
            throw new RuntimeException("Connot close task scheduler unless container is closing!");
        }
        
        synchronized (m_taskSchedulerLockObj)
        {
            m_isClosing = true;
            m_taskSchedulerLockObj.notifyAll();
        }
    }

    @Override
    public void run()
    {
        long nextObsoleteTaskCleanupTime = System.currentTimeMillis() + 5000;
        while (!m_isClosing)
        {
            Runnable task = null;
            long expirationTime = 0;

            // get a task
            synchronized(m_taskSchedulerLockObj)
            {
                if (!m_higherPriorityTasks.isEmpty())
                {
                    task = m_higherPriorityTasks.remove(0);
                    expirationTime = m_higherPriorityTaskExpirationTimes.remove(0).longValue();
                }
                else if (!m_normalPriorityTasks.isEmpty())
                {
                    task = m_normalPriorityTasks.remove(0);
                    expirationTime = m_normalPriorityTaskExpirationTimes.remove(0).longValue();
                }
                else // i.e. no task, then wait for one
                {
                    try
                    {
                        if (m_isClosing)
                        {
                            return;
                        }
                        m_taskSchedulerLockObj.wait();
                        if (m_isClosing)
                        {
                            return;
                        }
                        continue;
                    }
                    catch(InterruptedException ie) { return; }
                }
            }

            // get next task if current task is null (this is a defensive programming measure - there shouldn't be a null task on either the high priority or the normal priority list)
            if (task == null)
            {
                continue;
            }

            // get next task if current task has expired
            if ((expirationTime != 0) && (System.currentTimeMillis() > expirationTime))
            {
                // value of zero means there is no expiration time for the task
                continue;
            }
            
            if (System.currentTimeMillis() >= nextObsoleteTaskCleanupTime)
            {
                removeObsoleteTasks(m_higherPriorityTasks, m_higherPriorityTaskExpirationTimes);
                removeObsoleteTasks(m_normalPriorityTasks, m_normalPriorityTaskExpirationTimes);
                nextObsoleteTaskCleanupTime = System.currentTimeMillis() + 5000;
            }

            // get a thread to execute the task with
            ExecutionThread executionThread = null;
            while (executionThread == null)
            {
                synchronized(m_executionThreadCache)
                {
                    try
                    {
                        executionThread = m_executionThreadCache.pop();
                        if (executionThread != null)
                         {
                            m_executingThreads++;  // update count of threads currently in process of executing tasks
                        }
                    }
                    catch(EmptyStackException e)
                    {
                        if (getTotalPoolSize() == m_maxThreads) // wait until we're notified that a thread has become available
                        {
                            if (m_poolWaitsStatistic != null)
                            {
                                m_poolWaitsStatistic.updateValue(1);
                            }
                            try
                            {
                                m_executionThreadCache.wait();
                            }
                            catch(InterruptedException ie) { return; }
                        }
                        else // have not yet created maximum configured number of execution threads...
                        {
                        	// if no execution thread was obtained, then create a new one...
                            if (executionThread == null)
                            {
                                executionThread = new ExecutionThread();
                                executionThread.setName(TASK_THREAD_NAME + executionThread.hashCode());
                                executionThread.setDaemon(true);
                                executionThread.start();
                                m_executingThreads++;  // update count of threads currently in process of executing tasks
                            }
                        }
                    }
                }
            }

            // update the execution thread's task and wake it to actually execute the task
            synchronized(executionThread.getExecutionThreadLockObj())
            {
            	executionThread.m_task = task;
            	executionThread.getExecutionThreadLockObj().notifyAll();
            }
        }
    }

    private void removeObsoleteTasks(ArrayList<Runnable> taskList, ArrayList<Long> taskExpirationTimeList)
    {
        long currentTime = System.currentTimeMillis();
        ArrayList obsoleteTasks = new ArrayList();
        
        synchronized (m_taskSchedulerLockObj)
        {
            Iterator tasks = taskList.iterator();
            Iterator taskExpirations = taskExpirationTimeList.iterator();
            while (tasks.hasNext())
            {
                Object task = tasks.next();
                long expirationTime = ((Long)taskExpirations.next()).longValue();
                if (expirationTime != 0 && currentTime > expirationTime)
                {
                    tasks.remove();
                    taskExpirations.remove();
                    obsoleteTasks.add(task);
                }
            }
        }

        if (!obsoleteTasks.isEmpty())
        {
            Iterator tasks = obsoleteTasks.iterator();
            while (tasks.hasNext())
            {
                Object task = tasks.next();
                TaskScheduler.this.m_container.logMessage(null, "Dropped obsolete task " + task
                                                          + "; consider changing size of management thread pool, management request timeout or Domain Manager hardware",
                                                          Level.WARNING);

            }
        }
    }

    /**
     * Gets the maximum number of task execution threads the task scheduler will use.
     */
    int getMaxThreads() { return m_maxThreads; }

    /**
     * Set the maximum number of task threads the task scheduler will use.
     *
     * @param maxThreads Must be >20, >min threads and <= 1000.
     */
    void setMaxThreads(int maxThreads)
    {
        synchronized (m_taskSchedulerLockObj)
        {
            if (maxThreads > 1000)
            {
                throw new IllegalArgumentException("Maximum scheduler threads cannot exceed 1000.");
            }
            if (maxThreads <= m_minThreads)
            {
                throw new IllegalArgumentException("Maximum scheduler threads cannot be less than the minimum scheduler threads.");
            }
            if (maxThreads < 20)
            {
                throw new IllegalArgumentException("Maximum scheduler threads cannot be less than 20.");
            }
            m_maxThreads = (short)maxThreads;
        }
    }

    /**
     * Gets the minimum number of task execution threads the task scheduler will use
     * (i.e. the minimum thread pool size).
     */
    int getMinThreads() { return m_minThreads; }

    /**
     * Set the minimum number of task threads the task scheduler will use.
     *
     * @param minThreads Must be >= 0 and < max threads.
     */
    void setMinThreads(int minThreads)
    {
        synchronized (m_taskSchedulerLockObj)
        {
            if (minThreads < 0)
            {
                throw new IllegalArgumentException("Minimum scheduler threads cannot be less than 0.");
            }
            if (minThreads >= m_maxThreads)
            {
                throw new IllegalArgumentException("Minimum scheduler threads must be less than the maximum scheduler threads.");
            }
            m_minThreads = (short)minThreads;
        }
    }

    /**
     * Request the task scheduler to schedule the given task. If a high priority
     * is specified, the task will start execution ahead of non-high priority
     * tasks.
     */
    @Override
    public void scheduleTask(Runnable task, boolean higherPriority)
    {
        synchronized (m_taskSchedulerLockObj)
        {
            if (higherPriority)
            {
                m_higherPriorityTasks.add(task);
                m_higherPriorityTaskExpirationTimes.add(Long.valueOf(0));
            }
            else
            {
                m_normalPriorityTasks.add(task);
                m_normalPriorityTaskExpirationTimes.add(Long.valueOf(0));
            }
            m_taskSchedulerLockObj.notifyAll();  // notify the TaskScheduler thread that there is a new task to execute
        }
    }

    /**
     * Request the task scheduler to schedule the given task, and assign it an expirationTime.
     * If a high priority is specified, the task will start execution ahead of non-high priority
     * tasks.
     */
    @Override
    public void scheduleTask(Runnable task, boolean higherPriority, long expirationTime)
    {
        synchronized (m_taskSchedulerLockObj)
        {
            if (higherPriority)
            {
                m_higherPriorityTasks.add(task);
                m_higherPriorityTaskExpirationTimes.add(Long.valueOf(expirationTime));
            }
            else
            {
                m_normalPriorityTasks.add(task);
                m_normalPriorityTaskExpirationTimes.add(Long.valueOf(expirationTime));
            }
            m_taskSchedulerLockObj.notifyAll();  // notify the TaskScheduler thread that there is a new task to execute
        }
    }

    /**
     * Returns the role (if any) associated with the current thread.
     */
    public static String getCurrentRole()
    {
        Thread currentThread = Thread.currentThread();
        if (currentThread instanceof ExecutionThread)
        {
            return ((ExecutionThread)currentThread).getRole();
        }
        else
        {
            return null;
        }
    }

    /**
     * Returns the user ID (if any) associated with the current thread.
     */
    public static String getCurrentUserID()
    {
        Thread currentThread = Thread.currentThread();
        if (currentThread instanceof ExecutionThread)
        {
            return ((ExecutionThread)currentThread).getUserID();
        }
        else
        {
            return null;
        }
    }


    public static boolean isExecutionThread()
    {
        return Thread.currentThread() instanceof ExecutionThread;
    }

    static IMetricInfo[] getMetricsInfo()
    {
        // create the infos
        IMetricInfo[] infos = new IMetricInfo[3];
        infos[0] = MetricsFactory.createMetricInfo(IAgentProxy.SYSTEM_THREADS_CURRENTPOOLSIZE_METRIC_ID, IValueType.VALUE,
                                                   "Size of thread pool used to service transient management tasks.",
                                                   null, false, true, true, false, "pool threads");
        infos[1] = MetricsFactory.createMetricInfo(IAgentProxy.SYSTEM_THREADS_MAXPOOLSIZE_METRIC_ID, IValueType.VALUE,
                                                   "Maximum size of thread pool used to service transient management tasks since last metrics reset.",
                                                   null, false, true, false, false, "pool threads");
        infos[2] = MetricsFactory.createMetricInfo(IAgentProxy.SYSTEM_THREADS_POOLWAITS_METRIC_ID, IValueType.VALUE,
                                                   "Number of times transient management tasks had to wait because a pooled thread was not immediately available to service such tasks. Evaluated over the last 30 minutes.",
                                                   null, false, true, true, false, "waits");
        return infos;
    }

    void initMetrics(IMetricsRegistrar metricsRegistrar)
    {
        m_metricsRegistrar = metricsRegistrar;

        // pool size statistic provider and statistics
        IStatisticProvider[] poolSizeProviders = new IStatisticProvider[]
        {
            new IStatisticProvider()
            {
                private Object lock = new Object();
                @Override
                public void updateStatistic(ISampledStatistic statistic)
                {
                    synchronized (lock)
                    {
                        statistic.updateValue(TaskScheduler.this.getTotalPoolSize());
                    }
                }
                @Override
                public void resetStatistic(ISampledStatistic statistic) { } // no implementation
            }
        };
        m_poolSizeStatistic = StatisticsFactory.createStatistic(IStatistic.VALUE_MODE, false, poolSizeProviders, (short)0);
        m_maxPoolSizeStatistic = StatisticsFactory.createStatistic(IStatistic.MAXIMUM_MODE, false, poolSizeProviders, (short)0);;

        // pool waits
        m_poolWaitsStatistic = StatisticsFactory.createStatistic(IStatistic.COUNTER_MODE, true, null, (short)1);
    }

    void enableMetrics(IMetricIdentity[] ids)
    {
        // now enable the ones that the agent deals with directly
        for (int i = 0; i < ids.length; i++)
        {
            if (ids[i].equals(IAgentProxy.SYSTEM_THREADS_CURRENTPOOLSIZE_METRIC_ID))
            {
                m_metricsRegistrar.registerMetric(IAgentProxy.SYSTEM_THREADS_CURRENTPOOLSIZE_METRIC_ID, m_poolSizeStatistic);
            }
            else if (ids[i].equals(IAgentProxy.SYSTEM_THREADS_MAXPOOLSIZE_METRIC_ID))
            {
                m_metricsRegistrar.registerMetric(IAgentProxy.SYSTEM_THREADS_MAXPOOLSIZE_METRIC_ID, m_maxPoolSizeStatistic);
            }
            else if (ids[i].equals(IAgentProxy.SYSTEM_THREADS_POOLWAITS_METRIC_ID))
            {
                m_poolWaitsStatistic.reset();
                m_metricsRegistrar.registerMetric(IAgentProxy.SYSTEM_THREADS_POOLWAITS_METRIC_ID, m_poolWaitsStatistic);
            }
        }
    }

    void disableMetrics(IMetricIdentity[] ids)
    {
        // now enable the ones that the agent deals with directly
        for (int i = 0; i < ids.length; i++)
        {
            if (ids[i].equals(IAgentProxy.SYSTEM_THREADS_CURRENTPOOLSIZE_METRIC_ID))
            {
                m_metricsRegistrar.unregisterMetric(IAgentProxy.SYSTEM_THREADS_CURRENTPOOLSIZE_METRIC_ID);
            }
            else if (ids[i].equals(IAgentProxy.SYSTEM_THREADS_MAXPOOLSIZE_METRIC_ID))
            {
                m_metricsRegistrar.unregisterMetric(IAgentProxy.SYSTEM_THREADS_MAXPOOLSIZE_METRIC_ID);
            }
            else if (ids[i].equals(IAgentProxy.SYSTEM_THREADS_POOLWAITS_METRIC_ID))
            {
                m_metricsRegistrar.unregisterMetric(IAgentProxy.SYSTEM_THREADS_POOLWAITS_METRIC_ID);
            }
        }
    }

    /*
     *  get the total number of execution threads in the pool of execution threads; this
     *  count includes both the number of currently idle threads AND the number of
     *  threads currently executing tasks.
     *
     *  @return int  total count of threads in pool, both idle threads and executing threads
     */
    int getTotalPoolSize()
    {
    	synchronized (m_executionThreadCache)
    	{
    		return (m_executionThreadCache.size() + m_executingThreads);
    	}
    }

    public class ExecutionThread
    extends Thread
    {
        String m_userID;
        String m_role;
        Runnable m_task;
        Object m_executionThreadLockObj = new Object();
        
        private ExecutionThread()
        {
        }
        
        void setRole(String role)
        {
            synchronized (m_executionThreadLockObj)
            {
                if (this.m_role != null)
                {
                    throw new IllegalStateException("Cannot reset execution role.");
                }
                this.m_role = role;
            }
        }

        private String getRole() 
        { 
            synchronized(m_executionThreadLockObj)
            {
                return this.m_role;
            }
        }

        void setUserID(String userID)
        {
            synchronized(m_executionThreadLockObj)
            {
                this.m_userID = userID;    
            }
        }

        private String getUserID() { 
            synchronized (m_executionThreadLockObj)
            {
                return this.m_userID; 
            }
        } 

        public Object getExecutionThreadLockObj()
        {
            return m_executionThreadLockObj;
        }
        
        @Override
        public void run()
        {
            while (true)
            {
            	Runnable localTask = null;
                synchronized(m_executionThreadLockObj)
                {
                    try
                    {
                        if (this.m_task == null)
                         {
                            m_executionThreadLockObj.wait(MAX_IDLE_DURATION);  // allow time for a task to be assigned to this thread...
                        }
                        if (this.m_task == null) // not reused within max idle time so see if thread should be removed from cache of available threads...
                        {
                            synchronized(TaskScheduler.this.m_executionThreadCache)
                            {
                                // don't remove the execution thread from the a cache if the minimum
                                // cache size is not met
                                if (TaskScheduler.this.getTotalPoolSize() <= TaskScheduler.this.m_minThreads)
                                {
                                    continue;
                                }

                                // don't remove the execution thread from the cache if there are more tasks
                                // waiting to be executed than threads in the cache
                                if (TaskScheduler.this.m_executionThreadCache.size() < TaskScheduler.this.m_higherPriorityTasks.size() + TaskScheduler.this.m_normalPriorityTasks.size())
                                {
                                    continue;
                                }

                                // Remove from the cache and expire if in the cache; otherwise, was already selected for execution
                                // and must continue - Sonic00036922
                                if (TaskScheduler.this.m_executionThreadCache.remove(this))
                                {
                                    return;
                                }
                                else
                                {
                                    continue;
                                }
                            }
                        }
                        localTask = this.m_task;  // get a reference to the assigned task while in the scope of the synchronization lock...
                    }
                    catch(InterruptedException ie)
                    {
                        if ((this.m_task != null) && (TaskScheduler.this.m_executingThreads > 0))
                         {
                            // if this thread had an assigned task and was interrupted, decrement the count of executing threads before breaking out of the "while"...
                            TaskScheduler.this.m_executingThreads--;  // update count of threads currently in process of executing tasks
                        }
                        break;
                    }
                }

                // attempt to execute the task
                try
                {
                    super.setName(TASK_THREAD_NAME + this.hashCode() + " [" + localTask.getClass().getName() + ']');
                   	localTask.run();
                }
                catch(MFServiceNotActiveException e)
                {
                    if ((m_container.m_agent.m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                    {
                        TaskScheduler.this.m_container.logMessage(null, "Unhandled task failure due to inactive service, trace follows...", e, Level.TRACE);
                    }
                }
                catch(Throwable e)
                {
                    if (TaskScheduler.this.m_container.isClosing())
                    {
                        if ((TaskScheduler.this.m_container.m_agent.m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                        {
                            TaskScheduler.this.m_container.logMessage(null, "Unhandled task failure during shutdown, trace follows...", e, Level.WARNING);
                        }
                    }
                    else
                    {
                        // the task should have reported any conditions rather than throw exceptions
                        TaskScheduler.this.m_container.logMessage(null, "Unhandled task failure, trace follows...", e, Level.WARNING);
                    }
                }
                finally
                {
                	synchronized(m_executionThreadLockObj)  // make sure lock is obtained when resetting members...
                  	{
                        super.setName(TASK_THREAD_NAME + this.hashCode() + " [idle]");
                    	// unset the role and task of this runnable now that the task has been executed (or attempted)...
                        this.m_role = null;
                        this.m_task = null;
                        this.m_userID = null;
                    }
                }

                // check if the execution thread should be added back to the stack of
                // idle execution threads, or should be discarded (don't want too many idle threads)
                synchronized(TaskScheduler.this.m_executionThreadCache)
                {
                    // add thread back to pool of available threads
                    TaskScheduler.this.m_executionThreadCache.push(this);

                    // decrement the count of assigned executing threads
                    if (TaskScheduler.this.m_executingThreads > 0)
                     {
                        TaskScheduler.this.m_executingThreads--;  // update count of threads currently in process of executing tasks
                    }

                    // notify any thread waiting on the execution cache object
                    TaskScheduler.this.m_executionThreadCache.notifyAll();
                }
            }
        }
    }
}
