/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.framework.agent.cache;

import com.sonicsw.mf.framework.agent.cache.impl.ConfigCache;

/**
 * Creates a new in-memory or persistent cache object. The cache is populated with Configuration Elements if
 * it is persistent and the cache directory contains elements that were saved in a previous session.
 */
public final class ConfigCacheFactory
{
 /**
  * Creates a non persistent cache.
  *
  * @return the cahce object.
  *
  */
  public static IConfigCache createCache()
  {
      return new ConfigCache();
  }


 /**
  * Creates a persistent cache. If the directory exists and contains valid cache information
  * then elements are loaded into memory. If the directory does not exist or is empty then
  * a directory and cache files are created.
  *
  * @param dirName the name of the directory where the cache resides.
  * @param password if the persistent cache is encrypted
  *
  * @return the cahce object.
  *
  * @exception PersistentCacheException if the directory is not accessible or if the files' format is bad.
  *
  */
  public static IConfigCache createCache(String dirName, String password)
  throws PersistentCacheException
  {
       return new ConfigCache(dirName, password, false);
  }

 /**
  * Creates a new persistent cache (removes an existing one)
  *
  * @param dirName the name of the directory where the cache resides.
  * @param password if the persistent cache is encrypted
  *
  * @return the cahce object.
  *
  * @exception PersistentCacheException if the directory is not accessible.
  *
  */
  public static IConfigCache createNewCache(String dirName, String password)
  throws PersistentCacheException
  {
       return new ConfigCache(dirName, password, true);
  }

  /**
  * Creates a persistent cache. If the directory exists and contains valid cache information
  * then elements are loaded into memory. If the directory does not exist or is empty then
  * a directory and cache files are created.
  *
  * @param dirName the name of the directory where the cache resides.
  * @param password if the persistent cache is encrypted
  * @param maxBlobCacheSize maximum size of the persistent cache in bytes. LRU files are removed when
  * required to keep the cache size to this limit
  *
  * @return the cahce object.
  *
  * @exception PersistentCacheException if the directory is not accessible or if the files' format is bad.
  *
  */
  public static IConfigCache createCache(String dirName, String password, long maxBlobCacheSize)
  throws PersistentCacheException
  {
       return new ConfigCache(dirName, password, maxBlobCacheSize);
  }
}
