/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.framework.agent.cache;
import java.io.File;

import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.dirconfig.VersionOutofSyncException;

/**
 * The MF Container uses the IConfigCache interface to maintain a local persistent
 * or non persistent configuration cache. The cache contains all the configuration elements
 * needed by the components that run under this container. The IConfigCache interface also
 * provides access to the the IConfigCacheView interface - the read-only access to the cache used by
 * the container and the components.
 */
public interface IConfigCache
{
    /**
     * Returns the path to the root directory of the persistent cache. If the cache
     * is non persistent, this methods returns null.
     *
     */
    public File getCacheRootDirectory();

    /**
     * Returns the path to the native libraries directory.
     * If the cache is non persistent, this methods returns null.
     *
     */
    public File getNativeLibDirectory()
        throws CacheClosedException;

    /**
     * Returns a read-only view of the cache.
     *
     * @return the  read-only view of the cache.
     *
     * @exception IndexOutOfBoundsException if the position is invalid
     *
     */
    public IConfigCacheView getCacheView()
        throws CacheClosedException;

    /**
     * If one or more of the components are interested in the content of the entire directory then setInterestInDir is called.
     *
     * @param dirName the name of the directory
     *
     * @exception CacheClosedException if IConfigCache.close() was already called
     * @exception PersistentCacheException if the persistent cache becomes inaccessible or unwritable
     *
     */
    public void setInterestInDir(String dirName)
        throws CacheClosedException, PersistentCacheException;

    /**
     * Sets an element or apply delta to an existing element.
     *
     * When setting an element, an existing element with the same name is replaced.
     *
     * When a delta element is passed:
     * VersionOutofSyncException is thrown if the element does not exist and the IDeltaDirElement
     * object is not an INITIAL_VERSION element or if the element exists and the the IDeltaDirElement object is an INITIAL_VERSION
     * object or if the version of the IDeltaDirElement element is larger than the version of the element in the cache.
     *
     * @param element the delta element to be applied to the cache
     *
     * @exception CacheClosedException if IConfigCache.close() was already called
     * @exception PersistentCacheException if the persistent cache becomes inaccessible or unwritable
     * @exception VersionOutofSyncException if cache updates got lost or out of order
     *
     * @return the new version of the element (or delta) to report to the component. Null is returned if the cache was not updated
     */
    public IBasicElement setElement(IBasicElement element)
        throws VersionOutofSyncException, CacheClosedException, PersistentCacheException;

    /**
     * Sets a list of elements under a single directory an existing element with the same name is replaced.
     *
     *
     * @param elements the list of elements
     *
     * @exception CacheClosedException if IConfigCache.close() was already called
     * @exception PersistentCacheException if the persistent cache becomes inaccessible or not all elements are under the same directory
     * @exception VersionOutofSyncException if cache updates got lost or out of order
     *
     * @return the new version of the elements (or deltas) to report to the component. Null is returned if the cache was not updated
     */
    public IBasicElement[] setElements(IElement[] elements)
        throws VersionOutofSyncException, CacheClosedException, PersistentCacheException;

    /**
     * Deletes an element from cache. deleteElement has no effect if the element does not exist.
     *
     * @param elementName the name of the element to be deleted
     *
     * @exception CacheClosedException if IConfigCache.close() was already called
     * @exception PersistentCacheException if the persistent cache becomes inaccessible or unwritable
     *
     */
    public void deleteElement(String elementName)
        throws CacheClosedException, PersistentCacheException;

    /**
     * Deletes elements from cache. deleteElements does not throw an exception if one of the elements does not exist.
     *
     * @param elementNames the name of the elements to be deleted
     *
     * @exception CacheClosedException if IConfigCache.close() was already called
     * @exception PersistentCacheException if the persistent cache becomes inaccessible or unwritable
     *
     */
    public void deleteElements(String[] elementNames)
        throws CacheClosedException, PersistentCacheException;

    /**
     * Closes the cache and forces the memory cache into disk if it is a persistent cache.
     *
     * @param elementNames the name of the elements to be deleted
     *
     * @exception PersistentCacheException if the persistent cache is inaccessible or unwritable
     *
     */
    public void close()
        throws PersistentCacheException;

    /**
     * Used to adjust the size of the cache if the configuration has a value different from the default.
     *
     * @param newSize the new size of the cache. The method call is ignored if newSize is null
     */
    public void adjustSize(Integer newSize);

    /**
     * To set the backup version of the DS this cache works with
     *
     * @param backupVersion the backup version of the DS
     */
    public void setDSBackupVersion(Long backupVersion) throws PersistentCacheException;

    /**
     * Sets an element and associate its name with the logical (mutable) path
     *
     * When setting an element, an existing element with the same name is replaced.
     *
     * @param path the logical path
     * @param element with a storage name and atorage references
     *
     * @exception CacheClosedException if IConfigCache.close() was already called
     * @exception PersistentCacheException if the persistent cache becomes inaccessible or unwritable
     * @exception VersionOutofSyncException if cache updates got lost or out of order
     *
     * @return the element to report to the component.
     */
    public IBasicElement setElementByLogicalName(String path, IBasicElement element)
        throws CacheClosedException, PersistentCacheException, VersionOutofSyncException;

    /**
     * Sets a blob and associate its name with the logical (mutable) path
     *
     * When setting a blob, an existing blob with the same name is replaced.
     *
     * @param path the logical path
     * @param blob the blob
     *
     * @exception CacheClosedException if IConfigCache.close() was already called
     * @exception PersistentCacheException if the persistent cache becomes inaccessible or unwritable
     * @exception VersionOutofSyncException if cache updates got lost or out of order
     */
    public void setBlobByLogicalName(String path, IBlob blob, boolean isNativeFile)
        throws CacheClosedException, PersistentCacheException, VersionOutofSyncException;
    
    /**
     * Sets a blob and associate its name with the logical (mutable) path
     *
     * When setting a blob, an existing blob with the same name is replaced.
     *
     * @param path the logical path
     * @param blob the blob
     * @param isNativeFile true if the file is a native file, like a platform
     * specific dll
     * @param replaceMap true if we want the new logical/storage map of this blob
     * to replace any existing one. This is called from the first phase of
     * the container startup, to avoid exceptions when archives are deleted and
     * reimported in the DS while the container is not running.
     *
     * @exception CacheClosedException if IConfigCache.close() was already called
     * @exception PersistentCacheException if the persistent cache becomes inaccessible or unwritable
     * @exception VersionOutofSyncException if cache updates got lost or out of order
     */
    public void setBlobByLogicalName(String path, IBlob blob, boolean isNativeFile, boolean replaceMap)
        throws CacheClosedException, PersistentCacheException, VersionOutofSyncException;

    /**
     * Store a file in the cache from non-DS sources
     * @param URL Non-DS location of the file, for instance, c:/lib, http://
     * @param logicalName The logical name (and storage name) of the cache envelope element that is created for the file
     * @return true if the most recent version of the file was successfully cached, false if the file cannot be found
     * @throws PersistentCacheException if there's a problem copying the file to the cache after opening a connection to it
     */

    public boolean storeFile(String URL, String logicalName) throws PersistentCacheException;

    /**
     * Returns the logical name corresponding to storageName argument
     *
     * @param storageName The storage name of the element
     *
     * @return The logical name of the element
     */

    public String storageToLogical(String storageName);

    /**
     * Updates the logical-storage map with a new logical path.
     *
     * @param oldPath
     * @param newPath
     *
     * @return the storage name list of renamed elements in the cache
     */
    public String[] rename(String oldPath, String newPath)
        throws CacheClosedException, PersistentCacheException;

    /**
     * Deletes all the files in the native library directory of the cache. This is called every time the
     * container starts up.
     *
     * @throws CacheClosedException If the cache hasn't been initialized
     * @throws PersistentCacheException If there's a problem deleting the cached native library files
     */

    public void clearNativeLibraryDirectory() throws CacheClosedException, PersistentCacheException;

    /**
     * Store a blob temporarily so that a component can use the file. The temporary area of the cache is cleaned up
     * when the container is restarted.
     * @param blobIdentity The identity of the blob
     * @param blob The blob to be stored in the temporary area of the cache
     * @return The file in the temporary area of the cache. Null if the cache is not able to write the file
     */
    public File storeBlobTemporarily(IBlob blob) throws CacheClosedException, PersistentCacheException;
}
