/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.framework.agent.cache;
import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;

import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementIdentity;

/**
 * The IConfigCacheView interface is used by MF Container and by components for read-only access to the
 * configuration cache.
 */
public interface IConfigCacheView
{
    /**
     * Returns true if the cache is empty.
     *
     * @return true if the cache is empty
     */
    public boolean isEmpty();

    /**
     * Returns true if setInterestInDir was called for this directory
     *
     * @param dirName the name of the directory
     *
     * @return true if setInterestInDir was called for this directory
     *
     * @exception CacheClosedException if tries to access a closed cache
     *
     */
    public boolean getInterestInDir(String dirName)
        throws CacheClosedException;

    /**
     * Gets a reference to a read-only cache element
     *
     * @param elementName the name of the element
     *
     * @return a read-only reference to the element (or null if the element is not in the cache)
     *
     * @exception CacheClosedException if tries to access a closed cache
     *
     */
    public IElement getElement(String elementName)
        throws CacheClosedException;

    /**
     * Gets a reference to a read-only cache element by logical name
     *
     * @param path the logical path
     *
     * @return a read-only reference to the element (or null if the element is not in the cache)
     *
     * @exception CacheClosedException if tries to access a closed cache
     *
     */
    public IElement getElementByLogicalName(String path)
        throws CacheClosedException;

    /**
     * Returns the File in the cache directory where the blob is stored
     *
     * @param path the logical path
     * @param isNativeFile if the file is a native file, it is archived in a separate area of the cache, so we search for it in a different place
     *
     * @return File in the cache directory for the blob
     *
     * @exception CacheException if tries to access a closed cache, or there's an exception thrown from the update thread
     * @exception LatestVersionMissingException if the latest version of the file is not in the cache
     *
     */
    public File getFileByLogicalName(String path, boolean isNativeFile)
        throws CacheException, LatestVersionMissingException;

    /**
     * Returns the latest available file in the archive
     *
     * @param path the logical path
     *
     * @return File in the cache archive directory for the blob
     *
     * @exception CacheClosedException if tries to access a closed cache
     *
     */
    public File getAvailableFileByLogicalName(String path)
        throws CacheException;

    /**
     * Gets a reference to the indentity of an element in the cache.
     *
     * @param elementName the name of the element
     *
     * @return reference to the indentity of an element in the cache
     *
     * @exception CacheClosedException if tries to access a closed cache
     *
     */
    public IElementIdentity getElementIdentity(String elementName)
        throws CacheClosedException;

    /**
     * Gets a read-only reference to all the elements in the cache.
     *
     * @return a list of read-only references to the elements in the cache
     *
     * @exception CacheClosedException if tries to access a closed cache
     *
     */
    public IElement[] getAllElements()
        throws CacheClosedException;

    /**
     * Gets a read-only reference to all the elements under a directory.
     *
     * @return a list of read-only references to the elements under a directory (null if dirName is invalid).
     *
     * @exception CacheClosedException if tries to access a closed cache
     *
     */
    public IElement[] getAllElements(String dirName)
        throws CacheClosedException;
    
    /**
     * Gets a the identities of all the elements in the cache, excluding the ones 
     * named in excludeList
     *
     * @param excludeList Names of ids not to return. Used from ContainerImpl.reconcileCache
     * @return a list of the identities of all the elements in the cache
     *
     * @exception CacheClosedException if tries to access a closed cache
     *
     */
    public IElementIdentity[] getAllIdentities(ArrayList<String> excludeList)
        throws CacheClosedException;


    /**
     * Gets the list of directories we have an interest in all their elements.
     *
     * @return a list of the directory names
     *
     */
    public String[] getAllInterestDirs()
        throws CacheClosedException;

    /**
     * To get the backup version of the DS this cache works with
     *
     * @return the backup version of the DS
     */
    public Long getDSBackupVersion();

    /**
     * Gets the storage-to-logical map. Used by the container to send the map's content to the DS for reconciliation
     *
     * @returns the storage-to-logical map
     */
    public HashMap getStorageToLogicalMap();

    /**
     * Makes corrections to the map.  Used by the container to reconcile the map with DS changes.
     *
     * @param corrections the corrections table
     */
    public void applyCorrections(HashMap corrections)
        throws CacheClosedException, PersistentCacheException;

    /**
     * Returns true if this element is marked as NON_DSFILE envelope element.
     *
     * @param envEl the envelope element
     *
     * @returns  true if this element is marked as NON_DSFILE envelope element
     */
    public boolean isNonDS(IElement envEl);

    /**
     * Returns true if this element is marked as DO_NOT_CACHE envelope element.
     *
     * @param envEl the envelope element
     *
     * @returns  true if this element is marked as DO_NOT_CACHE envelope element
     */
    public boolean isDoNotCache(IElement envEl);
}
