package com.sonicsw.mf.framework.agent.cache.impl;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;

import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.agent.ContainerUtil;
import com.sonicsw.mf.framework.agent.cache.PersistentCacheException;

/**
 * Note: this class is not thread-safe.  It's used solely by ConfigCache
 * which must implement sufficient synchronization via ConfigCache.m_cacheLock.
 */
public class BlobStorageSizeManager
{

    private long m_cacheSize = 0;
    private long m_maxCacheSize = 0; // 0 means there's no limit.
    private HashMap<String, Long> m_fileList;
    private File m_fileListFile;
    private HashSet<String> m_inUse;
    private boolean spaceWarningLogged = false;;
    private int m_fileNamesPrefixLength;
    private String m_fileNamePrefix = "";

    private static final String FILE_LIST_FILE = "archive_list";
    private boolean DEBUG = false;

    BlobStorageSizeManager(String hostDirName, String domainName, String dataDirName, String password, boolean doSync,
                           long maxSize) throws PersistentCacheException
    {
    	
        m_fileNamesPrefixLength  =  hostDirName.length() + IMFDirectories.MF_ARCHIVE_DIR.length() + 1;
        m_fileNamePrefix = hostDirName + System.getProperty("file.separator") + IMFDirectories.MF_ARCHIVE_DIR;
        m_maxCacheSize = maxSize;
        if (DEBUG)
        {
            System.out.println("Initializing BlobStorageSizeManager, maxSize == " + maxSize);
        }
        m_fileListFile = new File(hostDirName, FILE_LIST_FILE);
        if (m_fileListFile.exists())
        {
            initFileList();
        }
        else
        {
            m_fileList = new HashMap<String, Long>();
        }
        m_inUse = new HashSet<String>();
    }

    // caller must hold read or write lock on ConfigCache.m_cacheLock
    void markInUse(String fileName)
    {
    	if (DEBUG)
        {
            System.out.println("BlobStorageSizeManager.markInUse " + fileName);
        }
    	
    	// MQ-34641: synchronize access to m_inUse to avoid corrupting the HashSet.
    	// markInUse() is the only BlobStorageSizeManager method that's called
    	// while holding only a ConfigCache.m_cacheLock read-lock, and therefore the
    	// only place where m_inUse could (without this synchronization) be
    	// accessed/modified concurrently. All the other methods here, including
    	// makeRoom(), the only other method that uses m_inUse, are called with
    	// the ConfigCache.m_cacheLock write-lock, so cannot be called concurrently
    	// with each other or with markInUse().
    	synchronized(m_inUse)
    	{
    	    m_inUse.add(removeFileNamePrefix(fileName));
    	}
    }

    // caller must hold write lock on ConfigCache.m_cacheLock
    void addFile(String fileName, long size)  throws PersistentCacheException
    {
        Long oldSize = m_fileList.put(removeFileNamePrefix(fileName), size);
        m_cacheSize += size;
        
        // just in case the file was already on the list
        if (oldSize != null)
        {
            m_cacheSize -= oldSize;
        }
        
        persistFileList();
        
        if (DEBUG)
        {
            System.out.println("BlobStorageSizeManager.addFile " + fileName +
                                          ", size " + size + ", new cache size " + m_cacheSize);
        }        
    }
   

    private void initFileList() throws PersistentCacheException
    {
        if (DEBUG)
        {
            System.out.println("BlobStorageSizeManager.initFileList from file: " + m_fileListFile);
        }
        try
        {
            BufferedInputStream fileIn = new BufferedInputStream(new FileInputStream(m_fileListFile));
            ObjectInputStream inS = new ObjectInputStream(fileIn);
            m_fileList = (HashMap<String, Long>)inS.readObject();
            if (DEBUG)
            {
                System.out.println("BlobStorageSizeManager.initFileList, read the file list from the stream");
            }
            fileIn.close();

            for (Long size : m_fileList.values())
            {
                m_cacheSize += size;
            }

            if (DEBUG)
            {
                System.out.println("BlobStorageSizeManager.initFileList, the size of the re-constructed cache in bytes == " +
                                       m_cacheSize);
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
            throw new PersistentCacheException(e.toString(), e);
        }
    }

    // If the space is execeeded then it stays in that state for duration of the session since the m_inUse list never shrinks
    private void spaceExceeded() 
    {
        if (!spaceWarningLogged)
        {
            System.out.println(ContainerUtil.createLogMessage(null,
                                                        "Persistent cache size is too small. You must increase the current size of " +
                                                         m_maxCacheSize + " bytes", Level.WARNING));
            spaceWarningLogged = true;
        }
        
    }

    // caller must hold write lock on ConfigCache.m_cacheLock
    void makeRoom(long size) throws PersistentCacheException
    {
        ArrayList<String> deletionList = new ArrayList<String>();
        // m_maxCacheSize == 0 means there's no limit to the cache
        if (m_maxCacheSize == 0)
        {
            return;
        }

        if (DEBUG)
        {
           System.out.println("BlobStorageSizeManager.makeRoom " + size +
        		   ", printing all files in use");
           
           for (String fileName : m_inUse)
        {
            System.out.println(fileName);
        }
        }

        Iterator<String> files = m_fileList.keySet().iterator();
        while (files.hasNext() && (size + m_cacheSize) > m_maxCacheSize)
        {
            String deleteCandidateName = null;
            while (deleteCandidateName == null && files.hasNext())
            {
                String fileName = files.next();
                if (!m_inUse.contains(fileName))
                {
                    deleteCandidateName = fileName;
                    break;
                }
            }

            if (deleteCandidateName == null)
            {
                spaceExceeded();
            }
            else
            {
                if (DEBUG)
                {
                    System.out.println("BlobStorageSizeManager.makeRoom deleting the physical file " + deleteCandidateName);
                }
                deleteArchiveFile(deleteCandidateName);
                deletionList.add(deleteCandidateName);
                m_cacheSize -= m_fileList.get(deleteCandidateName);
            }
        }
        
        if (!deletionList.isEmpty())
        {
            for (String fileName : deletionList)
            {
                m_fileList.remove(fileName);
            }
            persistFileList();
        }

        if ((size + m_cacheSize) > m_maxCacheSize)
        {
            spaceExceeded();
        }
    }

    // caller must hold write lock on ConfigCache.m_cacheLock
    void addRoom(File cachedFile) throws PersistentCacheException
    {
        String fileName = removeFileNamePrefix(ConfigCache.getCanonicalPath(cachedFile));
        Long fileSize = m_fileList.get(fileName);
        if (fileSize != null) // If the file is in the list
        {
            if (DEBUG)
            {
                System.out.println("BlobStorageSizeManager.addRoom removing " +
                    fileName + " and subtracting " + fileSize + " bytes from the size");
            }
            m_fileList.remove(fileName);
            m_cacheSize -= fileSize;
            persistFileList();
        }
    }

    private String removeFileNamePrefix(String fileName)
    {
        return fileName.substring(m_fileNamesPrefixLength);
    }

    private void deleteArchiveFile(String deleteThis)
    {
          
          File deleteFile;
          if (deleteThis.startsWith(System.getProperty("file.separator")))
        {
            deleteFile = new File(m_fileNamePrefix + deleteThis);
        }
        else
        {
            deleteFile = new File(m_fileNamePrefix + System.getProperty("file.separator") + deleteThis);
        }
          if (DEBUG)
        {
            System.out.println("BlobStorageSizeManager.deleteArchiveFile file == " + deleteFile.getAbsolutePath());
        }
          if (deleteFile.isDirectory())
          {
              String[] files = deleteFile.list();
              for (int i=0; i< files.length; i++)
            {
                deleteArchiveFile(deleteThis + System.getProperty("file.separator") + files[i]);
            }
              deleteFile.delete();
          }
          else
          {
              boolean deleted = deleteFile.delete();
              if (DEBUG)
            {
                System.out.println("BlobStorageSizeManager.deleteArchiveFile " + deleteThis + " deleted == " + deleted);
            }
          }
    }

    private void persistFileList() throws PersistentCacheException
    {
        if (DEBUG)
        {
            System.out.println("BlobStorageSizeManager.persistFileList");
        }
        try
        {
            ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(m_fileListFile));
            oos.writeObject(m_fileList);
            oos.close();
        }
        catch (IOException e)
        {
            throw new PersistentCacheException(e.toString(), e);
        }
    }
}
