/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.framework.agent.cache.impl;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeSet;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.jar.JarOutputStream;
import java.util.jar.Pack200;
import java.util.jar.Pack200.Unpacker;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import com.sonicsw.mf.common.ILogger;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.LogicalStorageNameMapper;
import com.sonicsw.mf.common.config.NameMapperPathException;
import com.sonicsw.mf.common.config.impl.Blob;
import com.sonicsw.mf.common.config.impl.Element;
import com.sonicsw.mf.common.config.impl.ElementIdentity;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.config.impl.VersionMisMatchException;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDeltaDirElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.VersionOutofSyncException;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.util.LockFile;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.agent.ContainerUtil;
import com.sonicsw.mf.framework.agent.cache.CacheClosedException;
import com.sonicsw.mf.framework.agent.cache.CacheException;
import com.sonicsw.mf.framework.agent.cache.CacheIsLocked;
import com.sonicsw.mf.framework.agent.cache.IConfigCache;
import com.sonicsw.mf.framework.agent.cache.IConfigCacheView;
import com.sonicsw.mf.framework.agent.cache.LatestVersionMissingException;
import com.sonicsw.mf.framework.agent.cache.PersistentCacheException;
import com.sonicsw.mf.framework.directory.storage.StorageException;

// Stores configuration elements. That there is no upper bound on the number of elements stored - the user of ConfigCache
// remove elements (stored in  m_elements) only when the when removed from the DS. BUT there is an upper limit to the
// size of the elements' content since m_elementCache is and LRU cache with a size limit.
public final class ConfigCache
implements IConfigCache, IConfigCacheView
{
    private static final String EXT_PACK = ".pack";
    private static final boolean DEBUG = false;
    private static final String DS_BACKUP_VERSION_ATT = "ds_backup_version";

    static final String CONTENT_DIR = "content";
    private static final String OLD_ELEMENTS_LIST = "list";
    private static final String CACHE_CONTENT = "cache_content";
    public static final String NATIVE_LIBRARIES_DIR = "/native_libraries";
    private static final String CACHE_INFO = "cache_info";
    public static final String CONFIG_CACHE = ".cache";
    public static final String CACHE_LOCK_FILE = "cache.lock";
    public static final String UPDATE_LOCK_FILE = ".lock";

    private int m_tempSuffix = 0;                      // used to create unique temp file names, for DO_NOT_CACHE files

    private HashMap<String, IDirElement> m_elements;   // the elements the cache holds
    private HashMap<String, String> m_directories;     // the list of directories in which interest has been expressed
    private final HashMap<String, Long> m_info;        // holds the DS version were working against
    private boolean m_open;                            // was the cache actually opened?

    private ElementCache m_elementCache;               // underlying element cache (manages memory & disk)
    private LogicalStorageNameMapper m_logicalMap;

    // persistence related
    private final boolean m_isPersistentCache;         // quick boolean to tell if this is a persistent cache
    private final HashSet<Thread> m_updaters;          // a table of threads currently updating the cache in some way (note synchronization is based on the ConfigCache instance)
    private final File m_cacheRootDir;                 // the root directory of the persistent cache
    private final File m_cacheListFile;                // the old file used to store the list of elements and interest dirs
    private final File m_cacheContentFile;             // the new file used to store the list of elements, interest dirs and logical to storage map
    private final File m_cacheInfoFile;                // the file used to store the DS version
    private final LockFile m_cacheLockFile;            // the global lock file for the cache
    private final PersistentBlobCache m_blobStorage;   // the part of the cache blob attachments are stored
    private final File m_cacheUpdateLockFile;          // the temporary lock file indicating an update is in progress (or was in progress when the container crashed)
    private boolean m_cacheContentIsDirty;             // flag to indicate the cache is dirty (only useful for persistent cache)
    private AsyncContentUpdater m_asyncContentUpdater; // thread used for aync updating of cache_content file
    private String m_storageError;
    private final Lock m_updaterLock;
    private final Lock m_archiveCleanerLock;
    private final BlobStorageSizeManager m_sizeManager;
    
    private final ReadWriteLock m_cacheLock = new ReentrantReadWriteLock();

    /**
     * Non-persistent constructor.
     */
    public ConfigCache()
    {
        m_isPersistentCache = false;
        m_updaters = null;
        m_elements = new HashMap<String, IDirElement>();
        m_directories = new HashMap<String, String>();
        m_logicalMap = new LogicalStorageNameMapper();
        m_info = new HashMap<String, Long>();
        m_cacheRootDir = null;
        m_cacheListFile = null;
        m_cacheContentFile = null;
        m_cacheInfoFile = null;
        m_cacheLockFile = null;
        m_cacheUpdateLockFile = null;
        m_open = true;
        m_archiveCleanerLock = null;
        m_updaterLock = null;
        m_sizeManager = null;
        m_blobStorage = null;
    }


    public ConfigCache(String cacheDirPath, String password)
    throws PersistentCacheException
    {
        this(cacheDirPath, password, 0, false);
    }

    public ConfigCache(String cacheDirPath, String password, boolean reset)
    throws PersistentCacheException
    {
        this(cacheDirPath, password, 0, reset);
    }

    public ConfigCache(String cacheDirPath, String password, long maxCacheSize)
    throws PersistentCacheException
    {
        this(cacheDirPath, password, maxCacheSize, false);
    }


    private ConfigCache(String cacheDirPath, String password, long persistentCacheMaxSize, boolean reset)
    throws PersistentCacheException
    {
        m_storageError = null;
        m_isPersistentCache = true;
        m_updaters = new HashSet<Thread>();
        m_archiveCleanerLock = new Lock();
        m_updaterLock = new Lock();
        // Makes sure the persistent cache directory exists

        m_cacheRootDir = new File(cacheDirPath);

        if (!m_cacheRootDir.exists() && !m_cacheRootDir.mkdir())
        {
            throw new PersistentCacheException("Failed to create the persistent cache directory \"" +
                                               getCanonicalPath(m_cacheRootDir) + "\".");
        }
        else if(reset)
        {
            deleteCacheDir(m_cacheRootDir);
            if (m_cacheRootDir.exists())
            {
                throw new PersistentCacheException("Could not delete obsolete cache \"" + getCanonicalPath(m_cacheRootDir) + "\"; delete it and restart the container");
            }
            if (!m_cacheRootDir.mkdir())
            {
                throw new PersistentCacheException("Failed to re-create the persistent cache directory \"" +
                                                   getCanonicalPath(m_cacheRootDir) + "\".");
            }
        }


        if (!m_cacheRootDir.canWrite())
        {
            throw new PersistentCacheException("No write permission in directory \"" + getCanonicalPath(m_cacheRootDir) + "\".");
        }

        // Create a lock on the cache
        m_cacheUpdateLockFile = new File(m_cacheRootDir, UPDATE_LOCK_FILE);
        File cacheLockFile = new File(m_cacheRootDir, CACHE_LOCK_FILE);
        m_cacheLockFile = new LockFile(cacheLockFile.getAbsolutePath());
        if (!m_cacheLockFile.lock())
        {
            throw new CacheIsLocked("Configuration cache is locked; the cache directory \"" + getCanonicalPath(m_cacheRootDir) +
                                    "\" is being used by another container which must be shutdown before starting this container.");
        }

        // Having established no other container is using the cache, we now need to check if the container crashed during
        // an update the last time it was run
        if (m_cacheUpdateLockFile.exists())
        {
            m_cacheLockFile.unlock();
            throw new PersistentCacheException("Configuration cache is corrupt; the cache directory \"" + getCanonicalPath(m_cacheRootDir) + "\" must be removed before restarting this container");
        }

        m_cacheInfoFile = new File(m_cacheRootDir, CACHE_INFO);
        m_info = readCacheInfoFile();

        // does the old "list" file exist? - if so read, write the new format and proceed
        m_cacheListFile = new File(m_cacheRootDir, OLD_ELEMENTS_LIST);
        m_cacheContentFile = new File(m_cacheRootDir, CACHE_CONTENT);
        if (m_cacheContentFile.exists())
        {
            readCacheContentFile();
        }
        else
        {
            // This will be read from file when we cache the logical-storage in persistent cache
            m_logicalMap = new LogicalStorageNameMapper();

            if (m_cacheListFile.exists())
            {
                startUpdateInternal(Thread.currentThread());
                readCacheListFile();
                m_cacheContentIsDirty = true;
                m_cacheListFile.delete();
                finishUpdateInternal(Thread.currentThread()); // this will write the new format
            }
            else
            {
                m_elements = new HashMap<String, IDirElement>();
                m_directories = new HashMap<String, String>();
            }
        }

        // Connects the elements in the list to the persistent cache
        m_elementCache = new ElementCache(null, getCanonicalPath(m_cacheRootDir), password);
        Iterator<IDirElement> iterator = m_elements.values().iterator();
        while (iterator.hasNext())
        {
            Element cacheElement = (Element)iterator.next();
            cacheElement.setCache(m_elementCache);
        }


        // creates an object to handle storage of blobs in the persistent cache
        try
        {
            m_sizeManager = new BlobStorageSizeManager(getCanonicalPath(m_cacheRootDir), CONTENT_DIR, IMFDirectories.MF_BLOBS_DIR, password, false,
                                                    persistentCacheMaxSize);
            m_blobStorage = new PersistentBlobCache(getCanonicalPath(m_cacheRootDir), CONTENT_DIR, IMFDirectories.MF_BLOBS_DIR, password, false);
        }
        catch(StorageException e)
        {
            throw new PersistentCacheException(e.toString(), e);
        }
        // cleanup obsolete versions of archives in the cache (if they exist)
        new ObsoleteArchiveRemover(m_cacheRootDir).doCleanup();

        m_open = true;
    }

    @Override
    public File getNativeLibDirectory()
    {
        if (m_cacheRootDir == null)
        {
            return null;
        }
        return new File(m_cacheRootDir, NATIVE_LIBRARIES_DIR);
    }

    @Override
    public File getCacheRootDirectory()
    {
        return m_cacheRootDir;
    }

    // Used to adjust the size of the cache if the configuration has a value different from the default
    @Override
    public void adjustSize(Integer newSize)
    {
        m_cacheLock.writeLock().lock();
        try
        {
            if (m_elementCache != null && newSize != null)
            {
                m_elementCache.adjustSize(newSize.intValue());
            }
        }
        finally
        {
            m_cacheLock.writeLock().unlock();
        }
    }

    /**
     * Read new format content file
     */
    private void readCacheContentFile()
    throws PersistentCacheException
    {
        try
        {
            BufferedInputStream fileIn = new BufferedInputStream(new FileInputStream(m_cacheContentFile));
            ObjectInputStream inS = new ObjectInputStream(fileIn);
            Object[] content = (Object[])inS.readObject();
            m_elements = (HashMap<String, IDirElement>)content[0];
            m_directories = (HashMap<String, String>)content[1];
            m_logicalMap = (LogicalStorageNameMapper)content[2];
            fileIn.close();
        }
        catch (ClassNotFoundException e)
        {
            throw new Error(e.toString(), e);
        }
        catch (IOException e)
        {
            throw new PersistentCacheException(e.toString(), e);
        }
    }

    /**
     * Read the old format content file
     */
    private void readCacheListFile()
    {
        try
        {
            BufferedInputStream fileIn = new BufferedInputStream(new FileInputStream(m_cacheListFile));
            ObjectInputStream inS = new ObjectInputStream(fileIn);
            m_elements = (HashMap<String, IDirElement>)inS.readObject();
            m_directories = (HashMap<String, String>)inS.readObject();
            fileIn.close();
        }
        catch (ClassNotFoundException e)
        {
            throw new Error(e.toString(), e);
        }
        catch (IOException e)
        {
            throw new Error(e.toString(), e);
//            return;
        }
    }

    private HashMap<String, Long> readCacheInfoFile()
    {
        try
        {
            BufferedInputStream fileIn = new BufferedInputStream(new FileInputStream(m_cacheInfoFile));
            ObjectInputStream inS = new ObjectInputStream(fileIn);
            try
            {
                return (HashMap<String, Long>)inS.readObject();
            }
            finally
            {
                fileIn.close();
            }
        }
        catch (ClassNotFoundException e)
        {
            throw new Error(e.toString(), e);
        }
        catch (IOException e)	// NOSONAR return HashMap when there is IOException.
        {
            return new HashMap<String, Long>();
        }
    }

    private void validateOpen()
    throws CacheClosedException
    {
        if (!m_open)
        {
            throw new CacheClosedException("The cache is closed.");
        }
    }

    @Override
    public IConfigCacheView getCacheView()
    throws CacheClosedException
    {
        validateOpen();
        return this;
    }

    @Override
    public IBasicElement[] setElements(IElement[] elements)
    throws CacheClosedException, PersistentCacheException, VersionOutofSyncException
    {
        validateOpen();
        startUpdateInternal(Thread.currentThread());

        m_cacheLock.writeLock().lock();
        try
        {
            createBatch();

            boolean ok = false;
            ArrayList results = new ArrayList();
            try
            {
                for (int i = 0;i < elements.length;i++)
                {
                    IDirElement element = (IDirElement)elements[i];
                    String storagePath = element.getIdentity().getName();
                    String logicalPath = null;
                    
                    // SNC00079749
                    // if this is a delete, but recreated then there will be an element with
                    // isDeleted() == true followed by another element of the same identity
                    // - in such a case look to see if the element has is logical name stored
                    // logical name map .. if it does then temporarily save the logical name
                    // and recreate the storage to logical mapping after the set (because that
                    // would have removed the mapping)
                    if (element.isDeleted() && i + 1 < elements.length)
                    {
                        if (((IDirElement)elements[i + 1]).getIdentity().getName().equals(storagePath))
                        {
                            logicalPath = storageToLogical(storagePath);
                        }
                    }
                        
                    results.add(setElementInternal(element));
                    if (logicalPath != null)
                    {
                        // reset the map then handle the second element
                        m_logicalMap.set(logicalPath, storagePath);
                        results.add(setElementInternal((IDirElement)elements[++i]));
                    }
                }

                ok = true;
                return (IBasicElement[])results.toArray(new IBasicElement[results.size()]);
            }
            finally
            {
                saveBatch(ok);
            }
        }
        finally
        {
            m_cacheLock.writeLock().unlock();
            finishUpdateInternal(Thread.currentThread());
        }
    }

    @Override
    public IBasicElement setElementByLogicalName(String path, IBasicElement element)
    throws CacheClosedException, PersistentCacheException, VersionOutofSyncException
    {
        // element will always be an IElement in DO_NOT_CACHE cases (for now, used in Eclipse dshandler)
        if ((element instanceof IElement) &&  isDoNotCache((IElement)element))
        {
            return element;
        }
        validateOpen();
        startUpdateInternal(Thread.currentThread());
        m_cacheLock.writeLock().lock();
        try
        {
            m_logicalMap.set(path, element.getIdentity().getName());
            if (element instanceof IDeltaDirElement)
            {
                element = setElementInternal( (IDeltaDirElement) element);
            }
            else
            {
                element = setElementInternal( (IDirElement) element);
            }
        }
        finally
        {
            m_cacheLock.writeLock().unlock();
            finishUpdateInternal(Thread.currentThread());
        }
        return element;
    }

    private boolean archivedFileExists(IElementIdentity fileIdentity) throws CacheException
    {
        File archivedFile = new File(getArchiveFileName(fileIdentity));
        return archivedFile.exists();
    }

    private URL createURL(String tryString) throws Exception
    {
        URL returnThis = null;
        try
        {
            if (DEBUG)
            {
                System.out.println("ConfigCache.createURL trying " + tryString);
            }
            returnThis = new URL(tryString);
        }
        catch (MalformedURLException ex)
        {
            // try making it a file:// URL
            File theFile = new File(tryString);
            if (DEBUG)
            {
                System.out.println("ConfigCache.createURL trying a FILE url, exists == " + theFile.exists());
            }
            if (theFile.exists())
            {
                returnThis = theFile.toURL();
            }
        }
        return returnThis;
    }

    @Override
    public boolean isDoNotCache(IElement envEl)
    {
        IAttributeSet topSet = envEl.getAttributes();
        IAttributeSet systemAttrs = (IAttributeSet) topSet.getAttribute(IBlob.SYSTEM_ATTRIBUTES);
        if (systemAttrs == null)
        {
          if (DEBUG)
        {
            System.out.println("ConfigCache.doNotCache returning false");
        }
          return false;
        }

        Boolean doNotCache = (Boolean) systemAttrs.getAttribute(IContainer.DO_NOT_CACHE_ATTR);
        if (doNotCache == null)
        {
          if (DEBUG)
        {
            System.out.println("ConfigCache.doNotCache returning false");
        }
          return false;
        }
        else
        {
            if (DEBUG)
            {
                System.out.println("ConfigCache.doNotCache returning " + doNotCache.booleanValue());
            }
            return doNotCache.booleanValue();
        }
    }

    @Override
    public boolean isNonDS(IElement envEl)
    {
        IAttributeSet topSet = envEl.getAttributes();
        IAttributeSet systemAttrs = (IAttributeSet) topSet.getAttribute(IBlob.SYSTEM_ATTRIBUTES);
        if (systemAttrs == null)
        {
            return false;
        }

        Boolean non_ds = (Boolean) systemAttrs.getAttribute(IBlob.NON_DSFILE);
        if (non_ds == null)
        {
            return false;
        }
        else
        {
            return non_ds.booleanValue();
        }
    }


    @Override
    public boolean storeFile(String location, String logicalName) throws PersistentCacheException
    {
        boolean refetch = false;
        if (DEBUG)
        {
            System.out.println("ConfigCache.storeFile location == " + location + " logicalName == " + logicalName);
        }
        IDirElement storeElement;
        URL url = null;
        URLConnection conn = null;
        String useLogicalName = logicalName;
        if (!logicalName.startsWith("/"))
        {
            useLogicalName = "/" + logicalName;
        }
        long newTimestamp = 0;
        long timestamp = 0;

        try
        {
            validateOpen();
            startUpdateInternal(Thread.currentThread());
            m_cacheLock.writeLock().lock();
            try
            {
                IElement envEl = getElementByLogicalName(useLogicalName);
                if (envEl != null)
                {
                    // the file is in the cache
                    if (DEBUG)
                    {
                        System.out.println("ConfigCache.storeFile found the envelope element");
                    }
                    IAttributeSet topSet = envEl.getAttributes();
                    IAttributeSet systemAttrs = (IAttributeSet) topSet.getAttribute(IBlob.SYSTEM_ATTRIBUTES);
                    Boolean non_ds = null;
                    if (systemAttrs != null)
                    {
                        non_ds = (Boolean) systemAttrs.getAttribute(IBlob.NON_DSFILE);
                    }
                    if (non_ds == null || !non_ds.booleanValue())
                    {
                        refetch = true;
                    }
                    else
                    {
                        if (DEBUG)
                        {
                            System.out.println("ConfigCache.storeFile getting the envelope element timestamp");
                        }
                        IElementIdentity id = envEl.getIdentity();
                        timestamp = id.getCreationTimestamp();
                    }
                }
                else
                {
                    refetch = true;
                }
                if (DEBUG)
                {
                    System.out.println("ConfigCache.storeFile refetch == " + refetch);
                }
                try
                {
                    url = createURL(location + useLogicalName);
                    if (url == null)
                    {
                        return false;
                    }
                    else
                    {
                        // it's a good URL. If we can't open a connection, we'll assume it couldn't be found
                      // and return false. Any other error (like from getInputStream) will throw the exception
                      // to the caller. getInputStream for a file URL throws FileNotFoundException both for
                      // non-existing files and files that are in use; to be able to distinguish between the two,
                      // we test whether the file exists or not in createURL. For non-file URLS, we just go
                      // to openConnection.
                        conn = url.openConnection();
                    }
                }
                catch (Exception ex)
                {
                    return false;
                }
                if (conn != null)
                {
                    newTimestamp = conn.getLastModified();
                    if (DEBUG)
                    {
                        System.out.println("ConfigCache.storefile comparing timestamps, newTimestamp == " + newTimestamp +
                                                      " timestamp == " + timestamp);
                    }
                    if (newTimestamp > timestamp)
                    {
                        refetch = true;
                    }
                    else if (newTimestamp < timestamp)
                    {
                        // get something printed out anyway
                        System.out.println(ContainerUtil.createLogMessage(null, "Container cache found file on disk " + logicalName + " with an older timestamp than the cached file; the file was not replaced in the cache", Level.INFO));
                    }
                    else
                    {
                        // This seems to be the only way of closing the connection after it's opened!!!!
                        conn.getInputStream().close();
                    }
                }
                if (refetch)
                {
                    if (DEBUG)
                    {
                        System.out.println("ConfigCache.storeFile starting refetch");
                    }
                    storeElement = ElementFactory.createElement(useLogicalName, "MF_FILE", "1");
                    ElementIdentity id = (ElementIdentity) storeElement.getIdentity();
                    id.setCreationTimestamp(newTimestamp);
                    // set _MF_SYSTEM_ATTRIBUTES.NON_DSFILE = true
                    IAttributeSet topSet = storeElement.getAttributes();
                    IAttributeSet systemAttrs = topSet.createAttributeSet(IBlob.SYSTEM_ATTRIBUTES);
                    systemAttrs.setBooleanAttribute(IBlob.NON_DSFILE, Boolean.TRUE);
                    InputStream urlStream = conn.getInputStream();
                    IBlob fileBlob = new Blob(storeElement, urlStream);
                    if (envEl != null)
                    {
                        if (DEBUG)
                        {
                            System.out.println("ConfigCache.storeFile deleting the old envelope element");
                        }
                        deleteElementInternal(envEl.getIdentity().getName());
                    }
                    if (DEBUG)
                    {
                        System.out.println("ConfigCache.storeFile calling setBlobByLogicalName");
                    }
                    setBlobByLogicalNameInternal(useLogicalName, fileBlob, false, false);
                    urlStream.close();
                }
            }
            finally
            {
              m_cacheLock.writeLock().unlock();
            }
        }
        catch (Exception ex)
        {
            if (DEBUG)
            {
                System.out.println("ConfigCache.storeFile caught exception, STACK TRACE follows");
                ex.printStackTrace();
            }
            if (ex instanceof PersistentCacheException)
            {
                throw (PersistentCacheException)ex;
            }
            else
            {
                throw new PersistentCacheException(ex.toString());
            }
        }
        finally
        {
            finishUpdateInternal(Thread.currentThread());
        }
        return true;
    }

    // mrd 12/08/2004 modified to deal with large file attachments. Get bytes from the blob's InputStream
    // in chunks, and append the bytes to the blob using the IStorage appendBlob method.
    @Override
    public void setBlobByLogicalName(String path, IBlob blob, boolean isNativeFile)
    throws CacheClosedException, PersistentCacheException, VersionOutofSyncException
    {
        setBlobByLogicalName(path, blob, isNativeFile, false);
    }

    @Override
    public void setBlobByLogicalName(String path, IBlob blob, boolean isNativeFile, boolean replaceMap)
    throws CacheClosedException, PersistentCacheException, VersionOutofSyncException
    {
        if (DEBUG)
        {
            System.out.println("ConfigCache.setBlobByLogicalName args " + path + " isNativeFile == " + isNativeFile);
        }
        validateOpen();

        try
        {
            startUpdateInternal(Thread.currentThread());
            m_cacheLock.writeLock().lock();
            try
            {
                setBlobByLogicalNameInternal(path, blob, isNativeFile, replaceMap);
            }
            finally
            {
                m_cacheLock.writeLock().unlock();
            }
        }
        finally
        {
            finishUpdateInternal(Thread.currentThread());
        }
    }

    // Put a blob on disk, but don't make it part of the cache. Used by LocalFileManager when it must
    // return a File object for files that are not cached,like files coming from Eclipse
    @Override
    public File storeBlobTemporarily(IBlob blob) throws PersistentCacheException, CacheClosedException
    {
        validateOpen();
        startUpdateInternal(Thread.currentThread());
        m_cacheLock.writeLock().lock();
        try
        {
            File tempFile = new File(getTempFileName(blob.getElement().getIdentity()));
            if (!tempFile.getParentFile().exists() && !tempFile.getParentFile().mkdirs())
            {
                throw new PersistentCacheException("Could not create necessary directory " + tempFile.getParent());
            }
            java.io.InputStream stream = blob.getBlobStream();
            java.io.OutputStream outStream = new FileOutputStream(tempFile, true);
            // With the new Blob constructor(IDirElement, InputStream)
            // he blob's stream can now be something other than a BlobInputStream,
            // Buffer it.
            BufferedInputStream bufInputStream = new BufferedInputStream(stream, IBlob.BLOB_CHUNK_SIZE);

            byte[] blobPiece = new byte[IBlob.BLOB_CHUNK_SIZE];
            int readIn;
            while ((readIn = bufInputStream.read(blobPiece)) != -1)
            {
                outStream.write(blobPiece, 0, readIn);
            }
            outStream.close();
            bufInputStream.close();
            // it is only now that we have stored the temporary file that we can tell how big it is and add
            // it to the size manager
            m_sizeManager.makeRoom(tempFile.length());
            m_sizeManager.addFile(getCanonicalPath(tempFile), tempFile.length());
            return tempFile;
        }
        catch (Exception e)
        {
            throw new PersistentCacheException(e.toString(), e);
        }
        finally
        {
            m_cacheLock.writeLock().unlock();
            finishUpdateInternal(Thread.currentThread());
        }
    }

    private void setBlobByLogicalNameInternal(String path, IBlob blob, boolean isNativeFile, boolean replaceMap)
    throws CacheClosedException, PersistentCacheException, VersionOutofSyncException
    {
        IDirElement element = blob.getElement();
        IElementIdentity blobIdentity = element.getIdentity();
        String name = element.getIdentity().getName();
        EntityName nameE = validateName(name);

        if (isDoNotCache(element))
        {
            return;
        }
        try
        {
            File cachedFile = new File(getArchiveFileName(blobIdentity));
            if (cachedFile.exists())
            {
                if (DEBUG)
                {
                    System.out.println("ConfigCache.setBlobByLogicalNameInternal found the file, will not recache, saving the envelope element");
                }
                Blob.markBlobState((Element)element, IBlob.COMPLETE, IBlob.LARGE_FILE_STATE);
                setElementInternal((IDirElement)element, true);
                m_logicalMap.set(path, name);
                return;
            }
            else if (DEBUG)
            {
                System.out.println("ConfigCache.setBlobByLogicalNameInternal did not find the archived file");
            }
        }
        catch (Exception e)
        {
            throw new PersistentCacheException(e.toString(), e);
        }

        m_logicalMap.set(path, name, replaceMap);

        try
        {
            int bufferSize = IBlob.BLOB_CHUNK_SIZE;

            // get the element as it is right now in the cache and mark it as INCOMPLETE
            // until the blob is done saving
            Element cachedElement = (Element)m_elements.get(name);
            if (cachedElement != null)
            {
                Blob.markBlobState(cachedElement, IBlob.INCOMPLETE, IBlob.LARGE_FILE_STATE);
                setElementInternal((IDirElement)cachedElement, true);
            }

            long startDownload;
            if (DEBUG)
            {
                startDownload = System.currentTimeMillis();
            }
            InputStream stream = blob.getBlobStream();
            int read;
            int offset = 0;
            byte[] bytes = new byte[bufferSize];
            while ((read = stream.read(bytes, 0, bufferSize)) != -1)
            {
                m_blobStorage.appendBlob(nameE, bytes, 0, read, offset);
                offset += read;
            }
            stream.close();
            if (DEBUG)
            {
                long endDownload = System.currentTimeMillis();
                System.out.println("Retrieving blob name:path=" + name + ":" + path +" took " + (endDownload - startDownload));
            }

            // If this blob is expandable, we are just going to expand the file and remove the cached blob,
            // otherwise, we will archive the blob
            if (isExpandable((Element)element) || isExpandable(m_blobStorage.blobToFile(nameE), (Element)element))
            {
                if (DEBUG)
                {
                    System.out.println("ConfigCache.setBlobByLogicalName, blob " + path + " is expandable");
                }
                expandInCache(path, blob);
            }
            else if (!archivedFileExists(blobIdentity))
            {
                File tempFile = m_blobStorage.blobToFile(nameE);
                m_sizeManager.makeRoom(tempFile.length());
                FileInputStream inStream = new FileInputStream(tempFile);
                String archiveFileName = getArchiveFileName(blobIdentity);
                m_sizeManager.addFile(archiveFileName, tempFile.length());
                // set it in the versioned blob cache
                archiveFile(archiveFileName, inStream, isNativeFile ? path : null, true);
                inStream.close();
            }
            Blob.markBlobState((Element)element, IBlob.COMPLETE, IBlob.LARGE_FILE_STATE);
            setElementInternal((IDirElement)element, true);
            // now that we're done archiving or expanding, remove the file from the "blobs" area
            m_blobStorage.deleteBlob(nameE);
        }
        catch (PersistentCacheException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new PersistentCacheException(e.toString(), e);
        }
    }

    // check the file and marks the element.
    private boolean isExpandable(File file, Element el)
    {
        boolean expandable = false;
        try
        {
            expandable = Blob.isExpandableFile(file);
            Blob.markBlobState(el, Boolean.valueOf(expandable), IBlob.EXPAND_IN_CACHE);
        }
        catch (Exception e) {
          // ignore
        }
        return expandable;
    }

    private boolean isExpandable(Element el)
    {
        IAttributeSet topSet = el.getAttributes();
        IAttributeSet systemAttrs = (IAttributeSet)topSet.getAttribute(IBlob.SYSTEM_ATTRIBUTES);
        if (systemAttrs != null)
        {
            Boolean expandableBoolean = (Boolean)systemAttrs.getAttribute(IBlob.EXPAND_IN_CACHE);
            if (DEBUG)
            {
                System.out.println("ConfigCache.isExpandable for " + el.getIdentity().getName() + " == " + expandableBoolean);
            }
            if (expandableBoolean != null)
            {
                return expandableBoolean.booleanValue();
            }
        }
        return false;
    }

    private static String getExpandedArchiveName(IElement el)
    {
        IAttributeSet topSet = el.getAttributes();
        IAttributeSet systemAttrs = (IAttributeSet)topSet.getAttribute(IBlob.SYSTEM_ATTRIBUTES);
        if (systemAttrs != null)
        {
            String expandedArchiveName = (String)systemAttrs.getAttribute(IBlob.ARCHIVE_NAME);
            return expandedArchiveName;
        }
        return null;
    }

    // this method is called by setBlobByLogicalName while still in the locked state
    // the unexpanded archive has already been put in storage. We expand the file as a ZIP
    // file rather than a JAR file so that the manifest file will get copied into the cache as well.
    private void expandInCache(String path, IBlob blob) throws Exception
    {
        IDirElement el = blob.getElement();
        File cachedFile = m_blobStorage.blobToFile(new EntityName(el.getIdentity().getName()));
        ZipInputStream zipInputStream = new ZipInputStream(new BufferedInputStream(new FileInputStream(cachedFile), IBlob.BLOB_CHUNK_SIZE));

        if (zipInputStream != null)
        {
            m_sizeManager.makeRoom(cachedFile.length()); //
            ZipEntry entry;
            while ((entry = zipInputStream.getNextEntry()) != null)
            {
                String name = entry.getName();

                InputStream input;
                boolean needsUnpack200 = needsUnpack200(name);

                name = needsUnpack200 ? name.substring(0, name.length() - EXT_PACK.length()) : name;

                if (needsUnpack200)
                {
                    input = getUnpackedInputstream(zipInputStream, entry.getSize());
                }
                else
                {
                    input = zipInputStream;
                }
                String expandedZipEntryFileName = getExpandedFileName(el.getIdentity(), name);
                if (entry.isDirectory())
                {
                    File zipDir = new File(expandedZipEntryFileName);
                    if (!zipDir.exists())
                    {
                        zipDir.mkdirs();
                    }
                }
                else
                {
                    archiveFile(expandedZipEntryFileName, input, null, false);
                }
            }

            zipInputStream.close();
            String archiveFileName = getArchiveFileName(el.getIdentity());

            //The size of the expanded archive on disk is roughly twize the size of the ZIP file
            m_sizeManager.addFile(archiveFileName, cachedFile.length()*2);

            m_sizeManager.markInUse(archiveFileName);
        }
    }

    /**
     * Get an unpacked input stream from the current zipInputStream.
     * Unfortunately Unpacker.unpack closes the input stream given that means
     * the jar gets closed after the first unpack so create an in-memory copy.
     *
     * @param zipInputStream
     * @param entrySize expected entry size or -1
     * @return
     * @throws IOException
     */
    private InputStream getUnpackedInputstream(ZipInputStream zipInputStream,
            long entrySize) throws IOException
    {
        Unpacker unpacker = Pack200.newUnpacker();
        long estimatedSize = entrySize;
        if (estimatedSize == -1)
        {
            estimatedSize = IBlob.BLOB_CHUNK_SIZE;
        }
        // if the entry size was given allocate at the right size already
        ByteArrayOutputStream os = new ByteArrayOutputStream((int)estimatedSize);
        // zipInputStream typically returns around 1k, but sometimes lots of zeros
        // do expand to much more. 16k seems reasonable enough
        byte[] buf = new byte[16*1024];
        int read;
        while (-1 != (read = zipInputStream.read(buf))) {
            os.write(buf, 0, read);
        }
        // reuse the vars to give GC the chance to claim the large byte[] back
        buf = os.toByteArray();
        os = new ByteArrayOutputStream(buf.length * 2);
        InputStream input = new ByteArrayInputStream(buf);
        buf = null;
        JarOutputStream jout = new JarOutputStream(os);
        unpacker.unpack(input, jout);
        jout.close();

        return new ByteArrayInputStream(os.toByteArray());
    }

    private boolean needsUnpack200(String name)
    {
        return name.endsWith(EXT_PACK);
    }

    @Override
    public void clearNativeLibraryDirectory() throws CacheClosedException, PersistentCacheException
    {
        validateOpen();
        if (DEBUG)
        {
            System.out.println("ConfigCache.clearNativeLibraryDirectory");
        }
        startUpdateInternal(Thread.currentThread());
        m_cacheLock.writeLock().lock();
        try
        {
            String dirName = m_cacheRootDir + NATIVE_LIBRARIES_DIR;
            File nativeLibDir = new File(dirName);
            if (nativeLibDir.exists())
            {
                File[] files = nativeLibDir.listFiles();
                for (int i=0; i<files.length; i++)
                {
                    m_sizeManager.addRoom(files[i]);
                    if (!files[i].delete())
                    {
                        throw new PersistentCacheException("Unable to delete the native library file " + files[i]);
                    }
                }
            }
            // do the same thing with the temporary area of the cache
            dirName = m_cacheRootDir + new String(new char[]{IMFDirectories.MF_DIR_SEPARATOR}) + IMFDirectories.MF_TEMPCACHE_DIR;
            File tempDir = new File(dirName);
            if (tempDir.exists())
            {
                File[] files = tempDir.listFiles();
                for (int i=0; i<files.length; i++)
                {
                    m_sizeManager.addRoom(files[i]);
                    if (!files[i].delete())
                    {
                        throw new PersistentCacheException("Unable to delete the temporary cache file " + files[i]);
                    }
                }
            }
        }
        finally
        {
            m_cacheLock.writeLock().unlock();
            finishUpdateInternal(Thread.currentThread());
        }
    }

    private File archiveFile(String fileName, InputStream blobStream, String nativeLibLogicalName, boolean archiveOrLoose)
        throws PersistentCacheException
    {
        if (DEBUG)
        {
            System.out.println("ConfigCache.archiveFile fileName == " + fileName);
        }
        File archive = new File(fileName);
        File parentDir = archive.getParentFile();
          // check the archive's parent directory exists in the archive cache .. if not create it
        if (!parentDir.exists())
        {
            if (!parentDir.mkdirs())
            {
                throw new PersistentCacheException("Failed to create local archive directory for archive: " + archive);
            }
        }

          // check if the correct version of the archive already exists
        if (!archive.exists())
        {
            // check we can write to the directory
            if (!parentDir.canWrite())
            {
                throw new PersistentCacheException("Cannot write to local archive directory for archive: " + archive);
            }
            // now write out the versioned archive
            try
            {
                OutputStream fos = new FileOutputStream(archive);
                BufferedOutputStream nativeOutStream = null;
                File nativeLibFile = null;
                if (nativeLibLogicalName != null)
                {
                    String nativeLibFileName = this.m_cacheRootDir + NATIVE_LIBRARIES_DIR + nativeLibLogicalName.substring(nativeLibLogicalName.lastIndexOf("/"));
                    if  (DEBUG)
                    {
                        System.out.println("ConfigCache.archiveFile, Native archived file name == " + nativeLibFileName);
                    }
                    nativeLibFile = new File(nativeLibFileName);
                    File nativeLibDir = nativeLibFile.getParentFile();
                    if (!nativeLibDir.exists())
                    {
                        if (!nativeLibDir.mkdirs())
                        {
                            throw new PersistentCacheException("Failed to create local directory for native libraries: " + nativeLibDir);
                        }
                    }
                    if (!nativeLibDir.canWrite())
                    {
                        throw new PersistentCacheException("Cannot write to local native Library directory for library: " + nativeLibFile);
                    }
                 if (!nativeLibFile.exists())
                 {
                     nativeOutStream = new BufferedOutputStream(new FileOutputStream(nativeLibFile), IBlob.BLOB_CHUNK_SIZE);
                     if (DEBUG)
                    {
                        System.out.println("ConfigCache.archiveFile opened native file copy stream");
                    }
                 }
                }
                fos = new BufferedOutputStream(fos, IBlob.BLOB_CHUNK_SIZE);
                byte [] buf = new byte[IBlob.BLOB_CHUNK_SIZE];
                int readIn;
                while ((readIn = blobStream.read(buf)) != -1)
                {
                    fos.write(buf, 0, readIn);
                    if (nativeOutStream != null)
                    {
                        nativeOutStream.write(buf, 0, readIn);
                    }
                }
                fos.close();
                if (nativeOutStream != null)
                {
                    nativeOutStream.close();
                    m_sizeManager.addFile(getCanonicalPath(nativeLibFile), nativeLibFile.length());
                }
            }
            catch(IOException e)
            {
                throw new PersistentCacheException("Failed to cache archive: " + archive, e);
            }
        }
        if (archiveOrLoose)
        {
            m_sizeManager.markInUse(getCanonicalPath(archive));
        }
        return archive;
    }

    private String getArchiveFileName(IElementIdentity blobIdentity, String cacheArea)  throws PersistentCacheException
    {
        String blobName = blobIdentity.getName();
        long version = blobIdentity.getVersion();
        long timestamp = blobIdentity.getCreationTimestamp();
        StringBuilder sb = null;
        try
        {
            sb = new StringBuilder(getCanonicalPath(m_cacheRootDir));
        }
        catch (Exception ex)
        {
            throw new PersistentCacheException(ex.toString(), ex);
        }
        sb.append(File.separatorChar);
        sb.append(cacheArea);

        StringTokenizer urlTokens = new StringTokenizer(blobName, "/");
        while (urlTokens.hasMoreTokens())
        {
            sb.append(File.separatorChar);
            if (urlTokens.countTokens() == 1)
            {
                sb.append(timestamp);
                sb.append('.');
                sb.append(version);
                sb.append('.');
            }
              sb.append(urlTokens.nextToken());
        }

        return sb.toString();
    }

    private String getArchiveFileName(IElementIdentity blobIdentity) throws PersistentCacheException
    {
        return getArchiveFileName(blobIdentity, IMFDirectories.MF_ARCHIVE_DIR);
    }

    private String getTempFileName(IElementIdentity blobIdentity)  throws PersistentCacheException
    {
        // don't use subdirectories in the temp area so the area can be easily cleaned up later
        String blobName = blobIdentity.getName().substring(blobIdentity.getName().lastIndexOf(IMFDirectories.MF_DIR_SEPARATOR) + 1);
        long version = blobIdentity.getVersion();
        long timestamp = blobIdentity.getCreationTimestamp();
        StringBuffer sb = null;
        try
        {
            sb = new StringBuffer(getCanonicalPath(m_cacheRootDir));
        }
        catch (Exception ex)
        {
            throw new PersistentCacheException(ex.toString(), ex);
        }
        sb.append(File.separatorChar);
        sb.append(IMFDirectories.MF_TEMPCACHE_DIR);

        StringTokenizer urlTokens = new StringTokenizer(blobName, "/");
        while (urlTokens.hasMoreTokens())
        {
            sb.append(File.separatorChar);
            if (urlTokens.countTokens() == 1)
            {
                sb.append(timestamp);
                sb.append('.');
                sb.append(version);
                sb.append('.');
            }
              sb.append(urlTokens.nextToken());
        }

        return sb.toString() + "_" + m_tempSuffix++;
    }

    private String getExpandedFileName(IElementIdentity archiveID, String fileInExpandedJar) throws PersistentCacheException
    {
        String archiveFileName = getArchiveFileName(archiveID);
        return archiveFileName + File.separator + fileInExpandedJar;
    }

    @Override
    public File getAvailableFileByLogicalName(String logicalName) throws CacheException
    {
        validateOpen();
        m_cacheLock.readLock().lock();
        try
        {
            IElement envelopeElement = getElementByLogicalName(logicalName);
    
            if (envelopeElement == null)
            {
                if (DEBUG)
                {
                    System.out.println("ConfigCache.getAvailableFileByLogicalName returning because envelope element is not found");
                }
                return null;
            }
    
            IElementIdentity blobIdentity = envelopeElement.getIdentity();
            String expandedArchiveName = getExpandedArchiveName(envelopeElement); // is logicalName the name of a file inside an expanded .JAR file
            File intendedArchiveVersion = new File(getArchiveFileName(blobIdentity));
            if (DEBUG)
            {
                System.out.println("ConfigCache.getAvailableFileByLogicalName, intendedArchiveVersion file == " +
                                              getCanonicalPath(intendedArchiveVersion));
            }
            String intendedName = intendedArchiveVersion.getName();
            int firstDotPos = intendedName.indexOf('.');
            int secondDotPos = intendedName.indexOf('.', firstDotPos + 1);
            final String archiveName = intendedName.substring(secondDotPos + 1);
            FilenameFilter filter = new FilenameFilter()
            {
                @Override
                public boolean accept(File dir, String name)
                {
                   return name.endsWith(archiveName);
                }
            };
            File[] alternativeVersions = intendedArchiveVersion.getParentFile().listFiles(filter);
            // no older archive versions stored in the cache
            if (alternativeVersions == null || alternativeVersions.length == 0)
            {
                return null;
            }
    
          // find the latest one by looking at creation timestamp and version
            File alternateVersion = null;
            for (int i = 0; i < alternativeVersions.length; i++)
            {
                int winnerFirstDot, winnerSecondDot, testFirstDot, testSecondDot;
                long winnerTimestamp, testTimestamp;
                int winnerVersion, testVersion;
                String winnerName, testName;
                if (alternateVersion != null)
                {
                    winnerName = alternateVersion.getName();
                    winnerFirstDot = winnerName.indexOf('.');
                    winnerSecondDot = winnerName.indexOf('.', winnerFirstDot + 1);
                    winnerTimestamp = Long.parseLong(winnerName.substring(0, winnerFirstDot));
                    winnerVersion = Integer.parseInt(winnerName.substring(winnerFirstDot + 1, winnerSecondDot));
                    testName = alternativeVersions[i].getName();
                    testFirstDot = testName.indexOf('.');
                    testSecondDot = testName.indexOf('.', testFirstDot + 1);
                    testTimestamp = Long.parseLong(testName.substring(0, testFirstDot));
                    testVersion = Integer.parseInt(testName.substring(testFirstDot + 1, testSecondDot));
                    if (testTimestamp > winnerTimestamp)
                    {
                        alternateVersion = alternativeVersions[i];
                    }
                    else if (testTimestamp == winnerTimestamp)
                    {
                        // timestamp is the same, try the version
                        if (testVersion > winnerVersion)
                         {
                            alternateVersion = alternativeVersions[i];
                        // otherwise don't change the current winner
                        }
                    }
                    // else the winnerTimestamp is still winning
                }
                else
                {
                    alternateVersion = alternativeVersions[i];
                }
            }
            if (alternateVersion != null)
            {
                m_sizeManager.markInUse(getCanonicalPath(alternateVersion));
            }
            if (alternateVersion != null && expandedArchiveName != null && !expandedArchiveName.equals(logicalName))
            {
                // is logicalName the name of a file inside an expanded .JAR file
                // then, we just found the parent directory. Append the file name to it
                String archiveMemberName = logicalName.substring(expandedArchiveName.length());
                alternateVersion = new File(alternateVersion, archiveMemberName);
            }
            return alternateVersion;
        }
        finally
        {
            m_cacheLock.readLock().unlock();
        }
    }

    @Override
    public HashMap getStorageToLogicalMap()
    {
        return m_logicalMap.getStorageToLogicalMap();
    }

    @Override
    public void applyCorrections(HashMap corrections)
    throws CacheClosedException, PersistentCacheException
    {
        validateOpen();
        if (corrections != null && !corrections.isEmpty())
        {
            startUpdateInternal(Thread.currentThread());
            m_cacheLock.writeLock().lock();
            try
            {
                m_logicalMap.applyCorrections(corrections);
                m_cacheContentIsDirty = true;
            }
            finally
            {
                m_cacheLock.writeLock().unlock();
                finishUpdateInternal(Thread.currentThread());
            }
        }
    }

    @Override
    public String storageToLogical(String storageName)
    {
        return m_logicalMap.storageToLogical(storageName);
    }

    @Override
    public IBasicElement setElement(IBasicElement element)
    throws CacheClosedException, PersistentCacheException, VersionOutofSyncException
    {
        validateOpen();
        startUpdateInternal(Thread.currentThread());
        m_cacheLock.writeLock().lock();
        try
        {
            if (element instanceof IDeltaDirElement)
            {
                element = setElementInternal( (IDeltaDirElement) element);
            }
            else
            {
                element = setElementInternal( (IDirElement) element);
            }
        }
        finally
        {
            m_cacheLock.writeLock().unlock();
            finishUpdateInternal(Thread.currentThread());
        }
        return element;
    }

    private IBasicElement setElementInternal(IDirElement element)
    throws CacheClosedException, PersistentCacheException
    {
        return setElementInternal(element, false);
    }

    private IBasicElement setElementInternal(IDirElement element, boolean overwrite)
    throws CacheClosedException, PersistentCacheException
    {
        if (isDoNotCache(element))
        {
            return element;
        }
        IElementIdentity elementID = element.getIdentity();
        String elementName = elementID.getName();
        long elementVersion = elementID.getVersion();

        IElement oldElement = m_elements.get(elementName);
        IElementIdentity oldID = null;
        long oldVersion = -1;
        if (oldElement != null)
        {
            oldID = oldElement.getIdentity();
            oldVersion = oldID.getVersion();
        }

        IDeltaDirElement calculatedDelta = null;
        if (element.isDeleted())
        {
            // Element already deleted
            if (oldElement == null)
            {
                return null;
            }

            // This delete is not intended for this element (probably some old delete that got here late)
            if (!oldID.equalEntity(elementID))
            {
                return null;
            }

            deleteElementInternal(elementName);
        }
        else
        {
            // We already have the same or newer version
            if (!overwrite && oldElement != null && elementVersion <= oldVersion)
            {
                return null;
            }

            // We have an old version of the element in persistent cache.
            // We want to calculate the delta between the the old version and the new
            // version so we can tell the components exactly what changed.
            // We cannot do that if the cache is not persistent since the content of old
            // version of the element is not guranteed to be in memory.
            if (oldElement != null && m_elementCache != null)
            {
                calculatedDelta = ((Element)oldElement).createDelta((Element)element);
            }

            // Elements in cache are always read-only
            ((Element)element).setReadOnly(true);
            m_elements.put(elementName, element);

            if (m_elementCache != null)
            {
                ((Element)element).cacheMe(m_elementCache);
                m_cacheContentIsDirty = true;
            }
        }

        if (calculatedDelta != null)
        {
            return calculatedDelta;
        }
        else
        {
            return element;
        }
    }

    private IDeltaDirElement setElementInternal(IDeltaDirElement delta)
    throws VersionOutofSyncException, CacheClosedException, PersistentCacheException
    {

        IElementIdentity deltaID = delta.getIdentity();
        String elementName = deltaID.getName();
        IDirElement element = m_elements.get(elementName);
        if (element == null)
        {
            throw new VersionOutofSyncException("Element '" + elementName + "' is not in the cache.");
        }

        if (!element.getIdentity().equalEntity(deltaID))
        {
            throw new VersionOutofSyncException("Element " + deltaID + " is not in the cache.");
        }

        try
        {
            Element newElement = (Element)((Element)element).createWritableClone();
            newElement.doApplyDelta(delta);

            // The delta object might have non read-only elements in it - so we must explicitly
            // turn read-only on since elements in the cache are always read-only
            newElement.setReadOnly(true);

            m_elements.put(elementName, newElement);

            if (m_elementCache != null)
            {
                newElement.cacheMe(m_elementCache);
                m_cacheContentIsDirty = true;
            }

            return delta;
        }
        catch (VersionMisMatchException e)
        {
            // If the delta is obsolete then we just ignore it
            if (e.isOldDelta())
            {
                return null;
            }
            else
            {
                throw new VersionOutofSyncException("Element " + element.getIdentity() + " is obsolete." +
                    " The current version of the element is " + deltaID);
            }
        }
    }

    @Override
    public void setInterestInDir(String dirName)
    throws CacheClosedException, PersistentCacheException
    {
        validateOpen();
        startUpdateInternal(Thread.currentThread());
        m_cacheLock.writeLock().lock();

        try
        {
            m_directories.put(dirName, dirName);
            m_cacheContentIsDirty = true;
        }
        finally
        {
            m_cacheLock.writeLock().unlock();
            finishUpdateInternal(Thread.currentThread());
        }
    }

    @Override
    public boolean isEmpty()
    {
        m_cacheLock.readLock().lock();
        try
        {
            return m_elements.size() == 0 && m_directories.size() == 0;
        }
        finally
        {
            m_cacheLock.readLock().unlock();
        }
    }

    @Override
    public void deleteElement(String elementName)
    throws CacheClosedException, PersistentCacheException
    {
        validateOpen();
        startUpdateInternal(Thread.currentThread());
        m_cacheLock.writeLock().lock();
        try
        {
            deleteElementInternal(elementName);
        }
        finally
        {
            m_cacheLock.writeLock().unlock();
            finishUpdateInternal(Thread.currentThread());
        }
    }

    @Override
    public void deleteElements(String[] elementNames)
    throws CacheClosedException, PersistentCacheException
    {
        validateOpen();
        startUpdateInternal(Thread.currentThread());
        m_cacheLock.writeLock().lock();
        try
        {
            for (int i = 0;i < elementNames.length;i++)
            {
                deleteElementInternal(elementNames[i]);
            }
        }
        finally
        {
            m_cacheLock.writeLock().unlock();
            finishUpdateInternal(Thread.currentThread());
        }
    }

    private void deleteElementInternal(String elementName)
    throws CacheClosedException, PersistentCacheException
    {
        m_elements.remove(elementName);
        m_logicalMap.deleteByStorageName(elementName);
        if (m_elementCache != null)
        {
            m_elementCache.removeAttributes(elementName);
            m_cacheContentIsDirty = true;
        }
    }

    @Override
    public void close()
    throws PersistentCacheException
    {
        if (m_isPersistentCache)
        {
            m_updaterLock.waitUntilUnlocked();
            m_archiveCleanerLock.waitUntilUnlocked();
        }
        m_cacheLock.writeLock().lock();
        try
        {
            if (!m_open)
            {
                return;
            }
            m_open = false;
            m_elementCache = null;

            if (m_isPersistentCache)
            {
                m_cacheLockFile.unlock();
            }
        }
        finally
        {
            m_cacheLock.writeLock().unlock();
        }
    }

    private void persistCacheContentFile(Object[] contents)
    throws PersistentCacheException
    {
        long start = System.currentTimeMillis();
        try
        {
            FileOutputStream fos = new FileOutputStream(m_cacheContentFile);
            BufferedOutputStream bos = new BufferedOutputStream(fos);
            ObjectOutputStream oos = new ObjectOutputStream(bos);
            oos.writeObject(contents);
            oos.flush();
            fos.getFD().sync();
            fos.close();
        }
        catch (IOException e)
        {
            throw new PersistentCacheException(e.toString(), e);
        }

        if (DEBUG)
        {
            System.out.println("ConfigCache.persistCacheContentFile(): updated cache content file (" + (System.currentTimeMillis() - start) + "ms): " + getCanonicalPath(m_cacheContentFile));
        }
    }

    private void persistCacheInfoFile()
    throws PersistentCacheException
    {
        try
        {
            FileOutputStream fos = new FileOutputStream(m_cacheInfoFile);
            BufferedOutputStream bos = new BufferedOutputStream(fos);
            ObjectOutputStream oos = new ObjectOutputStream(bos);
            oos.writeObject(m_info);
            oos.flush();
            fos.getFD().sync();
            fos.close();
        }
        catch (IOException e)
        {
            throw new PersistentCacheException(e.toString(), e);
        }
    }

    @Override
    public IElement getElementByLogicalName(String path)
    throws CacheClosedException
    {
        m_cacheLock.readLock().lock();
        try
        {
            String storageName = null;
            String logicalName = null;
            try
            {
              if (DEBUG)
            {
                System.out.println("ConfigCache.getElementByLogicalName trying to find " + path);
            }
              storageName = m_logicalMap.logicalToStorage(path);
            }
            catch (NameMapperPathException elEx)
            {
                if (DEBUG)
                {
                    System.out.println("ConfigCache.getElementByLogicalName in exception");
                    elEx.printStackTrace(System.out);
                }
                if (elEx.getMapperPathObject() instanceof String)
                {
                    storageName = (String)elEx.getMapperPathObject();
                }
                logicalName = elEx.getLogicalPath();
            }
            if (storageName == null)
            {
                if (DEBUG)
                {
                    System.out.println("ConfigCache.getElementByLogicalName, storageName == null");
                }
                return null;
            }
            IElement el = getElement(storageName);
            // find out if we have an expandable blob before we return the element that we found along the way
            if ((el != null) && (logicalName != null))
            {
                if (isExpandable((Element)el))
                {
                    // non-persistently tell the caller of the logical name currently mapped to this element.
                    try
                    {
                        Blob.markBlobState( (Element) el, logicalName, IBlob.ARCHIVE_NAME);
                    }
                    catch (Exception e)
                    {
                        if (DEBUG)
                        {
                            System.out.println("ConfigCache.getElementByLogicalName in exception, setting el to null");
                            e.printStackTrace(System.out);
                        }
                        el = null;
                    }
                }
                else
                {
                    if (DEBUG)
                    {
                        System.out.println("ConfigCache.getElementByLogicalName setting el to null");
                    }
                    el = null;
                }
            }
            if (DEBUG)
            {
                System.out.println("ConfigCache.getElementByLogicalName returning el");
            }
            return el;
        }
        finally
        {
            m_cacheLock.readLock().unlock();
        }
    }

  // mrd 12/10/2004 The blob elements stored and returned from the ConfigCache always have storage names.
  // This method is defined here because of the IChunkedStreamer interface, because some blobs are created
  // with elements with names in the logical namespace. This method should never be used for blobs obtained
  // from the config cache; the stream of such blobs will always end up calling getBlob, so, for now, this
  // method is defined to throw an error for the ConfigCache
   public IBlob getFSBlob(String elementName, boolean forUpdate, int offset)
        throws PersistentCacheException
  {
       throw new PersistentCacheException("ConfigCache blobs do not contain logical names");
  }

    @Override
    public File getFileByLogicalName(String path, final boolean isNativeFile) throws CacheException, LatestVersionMissingException
    {
        File targetArchiveFile = null;
        String archiveFileName = null; // keep the name of the archive file to be kept track in the size manager
        validateOpen();
        if (isNativeFile)
        {
            startUpdateInternal(Thread.currentThread());
            m_cacheLock.writeLock().lock();
        }
        else
        {
            m_cacheLock.readLock().lock();
        }
        try
        {
            if (DEBUG)
            {
                System.out.println("ConfigCache.getFileByLogicalName looking for " + path + ", isNativeFile == " + isNativeFile);
            }
            IElement element = getElementByLogicalName(path);
            if (element == null)
            {
                throw new LatestVersionMissingException("File is missing: " + path);
            }
            if (Blob.isIncompleteBlob( (IDirElement) element))
            {
                throw new LatestVersionMissingException("File is incomplete: " + path);
            }
            // check to see if the element is marked as an expandable archive. In that case, the archive will
            // not have been cached, but expanded.
            String expandedArchiveName = getExpandedArchiveName(element);
            if (DEBUG)
            {
                System.out.println( "ConfigCache.getFileByLogicalName expandedArchiveName == " +
                      expandedArchiveName + ", path == " + path);
            }
            boolean tryExpandedArchive = ( (expandedArchiveName != null) && !expandedArchiveName.equals(path));
            if (DEBUG)
            {
                System.out.println("ConfigCache.getFileByLogicalName, will try expanded archive == " + tryExpandedArchive);
            }

            // before we go looking for the archive file, make sure the blob hasn't been removed from the cache
            File cachedFile = new File(getArchiveFileName(element.getIdentity()));
            if (!cachedFile.exists())
            {
                throw new LatestVersionMissingException("Latest file version not found: " + path);
            }

            String targetArchiveFileName = null;
            archiveFileName = getArchiveFileName(element.getIdentity());
            if (!tryExpandedArchive)
            {
                targetArchiveFileName = archiveFileName;
            }
            else
            {
                String archiveMemberName = path.substring(expandedArchiveName.length());
                targetArchiveFileName = getExpandedFileName(element.getIdentity(), archiveMemberName);
            }

            targetArchiveFile = new File(targetArchiveFileName);
            if (!targetArchiveFile.exists())
            {
                return null;
            }

            File nativeTarget = null;
            if (!isNativeFile)
            {
                m_sizeManager.markInUse(getCanonicalPath(new File(archiveFileName)));
            }
            else
            {
                nativeTarget = new File(m_cacheRootDir + NATIVE_LIBRARIES_DIR + path.substring(path.lastIndexOf("/")));
            }

            if (isNativeFile && !nativeTarget.exists())
            {
                File nativeLibDir = nativeTarget.getParentFile();
                if (!nativeLibDir.exists())
                {
                    if (!nativeLibDir.mkdirs())
                    {
                        throw new PersistentCacheException("Failed to create local directory for native libraries: " + nativeTarget);
                    }
                }
                if (!nativeLibDir.canWrite())
                {
                    throw new PersistentCacheException("Cannot write to local native Library directory for library: " + nativeTarget);
                }
                copyFile(targetArchiveFile, nativeTarget);
                if (nativeTarget.exists())
                {
                    m_sizeManager.addFile(getCanonicalPath(nativeTarget), nativeTarget.length());
                }
            }
        }
        catch (LatestVersionMissingException missing)
        {
            throw missing;
        }
        catch (Exception e)
        {
            throw new LatestVersionMissingException(e.toString(), e);
        }
        finally
        {
            if (isNativeFile)
            {
                m_cacheLock.writeLock().unlock();
                finishUpdateInternal(Thread.currentThread());
            }
            else
            {
                m_cacheLock.readLock().unlock();
            }
        }
        return targetArchiveFile;
    }

    private void copyFile(File fromFile, File toFile) throws Exception
    {
        if (fromFile.exists())
        {
            if (DEBUG)
            {
                System.out.println("ConfigCache.copyFile beginning");
            }
            BufferedInputStream fromBufStream = new BufferedInputStream(new FileInputStream(fromFile), 500000);
            if (DEBUG)
            {
                System.out.println("ConfigCache.copyfile opened inputStream");
            }
            BufferedOutputStream toBufStream = new BufferedOutputStream(new FileOutputStream(toFile), 500000);
            if (DEBUG)
            {
                System.out.println("ConfigCache.copyFile opened output stream");
            }
            int readIn = fromBufStream.read();
            if (DEBUG)
            {
                System.out.println("ConfigCache.copyFile read first byte == " + readIn);
            }
            while (readIn != -1)
            {
                toBufStream.write(readIn);
                readIn = fromBufStream.read();
            }
            if (DEBUG)
            {
                System.out.println("ConfigCache.copyFile about to close the streams");
            }
            fromBufStream.close();
            toBufStream.close();
        }
    }

    @Override
    public IElement getElement(String elementName)
    throws CacheClosedException
    {
        m_cacheLock.readLock().lock();
        try
        {
            validateOpen();
            return m_elements.get(elementName);
        }
        finally
        {
            m_cacheLock.readLock().unlock();
        }
    }

    @Override
    public boolean getInterestInDir(String dirName)
    throws CacheClosedException
    {
        m_cacheLock.readLock().lock();
        try
        {
            validateOpen();
            return m_directories.get(dirName) != null;
        }
        finally
        {
            m_cacheLock.readLock().unlock();
        }
    }

    @Override
    public IElementIdentity getElementIdentity(String elementName)
    throws CacheClosedException
    {
        m_cacheLock.readLock().lock();
        try
        {
            validateOpen();
            return m_elements.get(elementName).getIdentity();
        }
        finally
        {
            m_cacheLock.readLock().unlock();
        }
    }

    @Override
    public IElement[] getAllElements()
    throws CacheClosedException
    {
        m_cacheLock.readLock().lock();
        try
        {
            validateOpen();
            Iterator<IDirElement> iterator = m_elements.values().iterator();
            IElement[] elements = new IElement[m_elements.size()];
            int i = 0;
            while (iterator.hasNext())
            {
                elements[i++] = iterator.next();
            }
    
            return elements;
        }
        finally
        {
            m_cacheLock.readLock().unlock();
        }
    }

    private EntityName validateName(String name)
    {
        try
        {
            return new EntityName(name);
        }
        catch (ConfigException e) // Should never happen
        {
            e.printStackTrace();
            throw new Error(e.toString());
        }
    }

    @Override
    public IElement[] getAllElements(String dirName)
    throws CacheClosedException
    {
        m_cacheLock.readLock().lock();
        try
        {
            validateOpen();
    
            EntityName directory = validateName(dirName);
    
            Iterator<IDirElement> iterator = m_elements.values().iterator();
            ArrayList<IElement> result = new ArrayList<IElement>();
    
            while (iterator.hasNext())
            {
                IElement element = iterator.next();
                EntityName name = validateName(element.getIdentity().getName());
    
                if (directory.isParent(name))
                {
                    result.add(element);
                }
            }
    
            IElement[] elements = new IElement[result.size()];
            for (int i = 0;i < result.size();i++)
            {
                elements[i] = result.get(i);
            }
            return elements;
        }
        finally
        {
            m_cacheLock.readLock().unlock();
        }
    }
    

    @Override
    public IElementIdentity[] getAllIdentities(ArrayList<String>excludeList)
    throws CacheClosedException
    {
        m_cacheLock.readLock().lock();
        try
        {
            validateOpen();
            ArrayList<IElementIdentity> elementsIdsList = new ArrayList<IElementIdentity>();
            IElement[] elements = getAllElements();
            IElementIdentity[] ids = new IElementIdentity[elements.length];
            for (int i = 0;i < elements.length;i++)
            {
                if (excludeList == null || !excludeList.contains(elements[i].getIdentity().getName()))
                {
                    elementsIdsList.add(elements[i].getIdentity());
                }
            }
    
            return (IElementIdentity[])elementsIdsList.toArray(new IElementIdentity[elementsIdsList.size()]);
        }
        finally
        {
            m_cacheLock.readLock().unlock();
        }
    }

    @Override
    public String[] getAllInterestDirs()
    throws CacheClosedException
    {
        m_cacheLock.readLock().lock();
        try
        {
            validateOpen();
            Object[] dirs = m_directories.keySet().toArray();
            String[] result = new String[dirs.length];
            for (int i = 0;i < dirs.length;i++)
            {
                result[i] = (String)dirs[i];
            }
            return result;
        }
        finally
        {
            m_cacheLock.readLock().unlock();
        }
    }

    @Override
    public void setDSBackupVersion(Long backupVersion)
    throws PersistentCacheException
    {
        m_cacheLock.writeLock().lock();
        try
        {
            Long currentVersion = m_info.get(DS_BACKUP_VERSION_ATT);
            if (currentVersion == null || !currentVersion.equals(backupVersion))
            {
                m_info.put(DS_BACKUP_VERSION_ATT, backupVersion);
                if (m_isPersistentCache)
                {
                    startUpdateInternal(Thread.currentThread());
                    try
                    {
                        persistCacheInfoFile();
                    }
                    finally
                    {
                        finishUpdateInternal(Thread.currentThread());
                    }
                }
            }
        }
        finally
        {
            m_cacheLock.writeLock().unlock();
        }
    }

    @Override
    public Long getDSBackupVersion()
    {
        m_cacheLock.readLock().lock();
        try
        {
            return m_info.get(DS_BACKUP_VERSION_ATT);
        }
        finally
        {
            m_cacheLock.readLock().unlock();
        }
    }

    @Override
    public String[] rename(String oldPath, String newPath)
    throws CacheClosedException, PersistentCacheException
    {
        startUpdateInternal(Thread.currentThread());
        m_cacheLock.writeLock().lock();
        try
        {
            String[] paths = null;
            paths = m_logicalMap.rename(oldPath, newPath);
            m_cacheContentIsDirty = true;
            return paths;
        }
        finally
        {
            m_cacheLock.writeLock().unlock();
            finishUpdateInternal(Thread.currentThread());
        }
    }

    private void startUpdateInternal(Thread updater)
    throws PersistentCacheException
    {
        if (!m_isPersistentCache)
        {
            return;
        }

        if (DEBUG)
        {
            System.out.println("ConfigCache.startUpdateInternal(): start update: " + updater);
        }

        // if this is the first updater then create the update lock
        synchronized (m_cacheUpdateLockFile)
        {
            m_updaters.add(updater);
            m_cacheUpdateLockFile.notifyAll();
            try
            {
                if (m_updaters.size() == 1 && m_asyncContentUpdater == null)
                {
                    m_cacheUpdateLockFile.createNewFile();
                    if (DEBUG)
                    {
                        System.out.println("ConfigCache.startUpdateInternal(): created update lock " +
                                                       getCanonicalPath(m_cacheUpdateLockFile));
                    }
                }
            }
            catch (IOException e)
            {
                throw new PersistentCacheException("Failed to create update lock: " + e.getMessage(), e);
            }
            
            // lock the updater lock to prevent a close() occurring in the middle of an update
            m_updaterLock.lock();
        }
    }

    private void finishUpdateInternal(Thread updater)
    throws PersistentCacheException
    {
        // if theres no persistent cache then the disk can't become corrupt!
        if (!m_isPersistentCache)
        {
            return;
        }

        if (DEBUG)
        {
            System.out.println("ConfigCache.startUpdateInternal(): finish update: " + updater);
        }

        synchronized (m_cacheUpdateLockFile)
        {
            if (m_storageError != null)
            {
                PersistentCacheException persistentCacheException = new PersistentCacheException(m_storageError);
                m_storageError = null;
                throw persistentCacheException;
            }

            m_updaters.remove(updater);
            if (m_updaters.isEmpty())
            {
                // if the cache is dirty and there is no async updater, create one
                if (m_cacheContentIsDirty && m_asyncContentUpdater == null)
                {
                    m_asyncContentUpdater = new AsyncContentUpdater();
                    m_asyncContentUpdater.start();
                }
                // the async updater will remove the lock, so if there isn't one it needs to be done from this thread
                if (m_asyncContentUpdater == null)
                {
                    m_cacheUpdateLockFile.delete();
                    m_updaterLock.unlock();
                    if (DEBUG)
                    {
                        System.out.println("ConfigCache.finishUpdateInternal(): deleted update lock: " +
                                                       getCanonicalPath(m_cacheUpdateLockFile));
                    }
                }
            }
        }
    }

    public static void cleanupCorruptCache(String cacheDirPath, ILogger logger)
    {
        cleanupCorruptCache(new File(cacheDirPath), logger);
    }

    public static void cleanupCorruptCache(File cacheDir, ILogger logger)
    {
        if (!cacheDir.exists() || !cacheDir.isDirectory())
        {
            return;
        }

        LockFile cacheLockFile =  new LockFile(cacheDir.getAbsolutePath() + File.separatorChar + CACHE_LOCK_FILE);

        // If the cache is used by another process then leave
        if (!cacheLockFile.lock())
        {
            return;
        }
        else
        {
            cacheLockFile.unlock();
        }

        File cacheUpdateLockFile =  new File(cacheDir, UPDATE_LOCK_FILE);

        if (cacheUpdateLockFile.exists())
        {
            logger.logMessage("Corrupt cache found, cleanup initiated...", Level.WARNING);
            deleteCacheDir(cacheDir);
            logger.logMessage("...cache cleanup complete", Level.WARNING);
        }

    }
    private static void deleteCacheDir(File dir)
    {
        File[] files = dir.listFiles();
        for (int i = 0; i < files.length; i++)
        {
            if (files[i].isDirectory())
            {
                deleteCacheDir(files[i]);
            }
            else
            {
                files[i].delete();
            }
        }

        dir.delete();
    }


    private final class AsyncContentUpdater
    extends Thread
    {
        private AsyncContentUpdater()
        {
            super("Cache Content Updater");
        }

        @Override
        public void run()
        {
            synchronized (ConfigCache.this.m_cacheUpdateLockFile)
            {
                // wait for one second with no updaters so that we don't write the cache content file
                // if there are more updates in the pipeline
                int i = 10; // equates to 1 second of probably no updaters
                while (i > 0)
                {
                    try { ConfigCache.this.m_cacheUpdateLockFile.wait(100);} catch(InterruptedException e) { }

                    // wait until updaters have completed
                    if (ConfigCache.this.m_updaters.size() == 0)
                    {
                        i--;
                    }
                    else
                    {
                        i = 10;
                    }
                }

                Iterator iterator = null;
                HashMap<String, String> directories = null;
                LogicalStorageNameMapper logicalMap = null;
                m_cacheLock.readLock().lock();
                try
                {
                    iterator = ((HashMap<String, IDirElement>)ConfigCache.this.m_elements.clone()).entrySet().iterator();
                    directories = (HashMap<String, String>)m_directories.clone();
                    logicalMap = (LogicalStorageNameMapper)m_logicalMap.clone();
                }
                finally
                {
                    m_cacheLock.readLock().unlock();
                }
                // take a copy of the content fields and stash these for async updating
                // note: need a deep clone of the elements
                HashMap elements = new HashMap();
                while (iterator.hasNext())
                {
                    Map.Entry entry = (Map.Entry)iterator.next();
                    Element element = (Element)((Element)entry.getValue()).createHeaderClone();
                    element.dontWriteAttributes(true);
                    elements.put(entry.getKey(), element);
                }
                Object[] cacheContent = new Object[] { elements, directories, logicalMap };
                m_cacheContentIsDirty = false;

                try
                {
                    ConfigCache.this.persistCacheContentFile(cacheContent);
                }
                catch(PersistentCacheException e)
                {
                    ConfigCache.this.storageError(e.toString());
                }

                m_asyncContentUpdater = null;

                m_cacheUpdateLockFile.delete();
                if (DEBUG)
                {
                    System.out.println("ConfigCache.AsyncContentUpdater.run(): deleted update lock: " +
                                                   getCanonicalPath(ConfigCache.this.m_cacheUpdateLockFile));
                }
                m_updaterLock.unlock();
            }
        }
    }


    /**
     * Used to save many element under a single directory efficiently
     */
    private void createBatch()
    {
        if (m_elementCache == null)
        {
            return;
        }
        m_elementCache.createBatch();
    }

    /**
     * Save the batch created by createBatch(). All the elements set since the last createBatch() will be
     * saved, at once, in the persistent cache (if it is a persistent cache).
     */
    private void saveBatch(boolean ok)
    {
        if (m_elementCache == null)
        {
            return;
        }
        m_elementCache.saveBatch(ok);
    }

    private void storageError(String msg)
    {
         m_storageError = msg;
         throw new Error(msg);
    }

    // getCanonicalPath could, in theory, fail on some systems. We use getAbsolutePath
    // if getCanonicalPath fails. The only drawback is "uglier" log messages
    static String getCanonicalPath(File file)
    {
        try
        {
            return file.getCanonicalPath();
        }
        catch (Exception e)
        {
            return file.getAbsolutePath();
        }
    }


    class ObsoleteArchiveRemover
    {
        private File m_archiveDir;
        private TreeSet<ArchiveFile> m_set = new TreeSet<ArchiveFile>();

        ObsoleteArchiveRemover(File cacheRoot)
      {
        m_archiveDir = new File(cacheRoot, IMFDirectories.MF_ARCHIVE_DIR);
         // build the sorted list of archives
        if (m_archiveDir.exists())
        {
            findArchives(m_set, m_archiveDir);
        }
      }

      void doCleanup() throws PersistentCacheException
      {
        if (!m_archiveDir.exists())
        {
            return;
        }
        m_archiveCleanerLock.lock();


        // if there is only one archive then there are no obsolete versions!
        if (m_set.size() < 2)
        {
            m_archiveCleanerLock.unlock();
            return;
        }


        // we establish the ones that can be deleted by looking at the last modified date; we attempt
        // to delete all but the one with the latest last modified date (and if any are in use, the delete
        // will simply fail)
        // note: we cannot use the higher version # to see if the file should be deleted because the archive
        //       could have been completely removed from the DS and then readded, in which case the version #
        //       would have been reset to 1
        Iterator<ArchiveFile> iterator = m_set.iterator();
        File previousFile = null;
        File previousDir = null;
        File currentFile = iterator.next();
        File currentDir = currentFile.getParentFile();
        while (iterator.hasNext()) {
          try {
            previousFile = currentFile;
            currentFile = iterator.next();

            // is this exactly the same file?
            if (currentFile.equals(previousFile))
            {
                continue;
            }

            previousDir = currentDir;
            currentDir = currentFile.getParentFile();

            // is it the same directory?
            if (!currentDir.equals(previousDir))
             {
                continue; // don't need to compare further if diff dirs
            }


            String prevFullFileName = previousFile.getName();
            int prevFirstDotPos = prevFullFileName.indexOf('.');
            int prevSecondDotPos = prevFullFileName.indexOf('.',
                prevFirstDotPos + 1);
            String previousFileName = prevFullFileName.substring(prevSecondDotPos +
                1);
            String curFullFileName = currentFile.getName();
            int curFirstDotPos = curFullFileName.indexOf('.');
            int curSecondDotPos = curFullFileName.indexOf('.', curFirstDotPos + 1);
            String currentFileName = curFullFileName.substring(curSecondDotPos +
                1);


            // is it the same archive name?
            if (currentFileName.equals(previousFileName)) {
              File obsoleteFile = null;
              long curTimestamp = Long.parseLong(curFullFileName.substring(0,
                  curFirstDotPos));
              long prevTimestamp = Long.parseLong(prevFullFileName.substring(0,
                  prevFirstDotPos));
              if (curTimestamp > prevTimestamp)
            {
                obsoleteFile = previousFile;
            }
            else if (prevTimestamp > curTimestamp) {
                obsoleteFile = currentFile;
                currentFile = previousFile;
                currentDir = previousDir;
              }
              else {
                // same creation timestamp for the element, look at the version
                int curVersion = Integer.parseInt(curFullFileName.substring(
                    curFirstDotPos + 1, curSecondDotPos));
                int prevVersion = Integer.parseInt(prevFullFileName.substring(
                    prevFirstDotPos + 1, prevSecondDotPos));
                if (curVersion > prevVersion)
                {
                    obsoleteFile = previousFile;
                }
                else if (prevVersion > curVersion) {
                  obsoleteFile = currentFile;
                  currentFile = previousFile;
                  currentDir = previousDir;
                }
              }

              // do we have something to delete?
              if (obsoleteFile != null && obsoleteFile.exists()) {
                deleteArchivedFile(obsoleteFile);
                m_set.remove(obsoleteFile);
                // have to reset the iterator since we have just removed the last and we
                // can't go back one
                iterator = m_set.iterator();
                continue;
              }
            }
          }
          catch (PersistentCacheException cacheE)
          {
        	  throw cacheE;
          }
          catch (Exception e) {
            throw new PersistentCacheException("Exception thrown while cleaning obsolete cache files: " + e.toString(), e);
          }
        }
        m_archiveCleanerLock.unlock();
      }

      // When we delete archived files, the "file" might be the top directory of an expandable jar file
      // Therefore we can't just call File.delete
      private void deleteArchivedFile(File deleteThis) throws PersistentCacheException
      {
          // If the file or directory is an item in an archive then addRoom will be a noop
          m_sizeManager.addRoom(deleteThis);

          if (deleteThis.isDirectory())
          {
              File[] files = deleteThis.listFiles();
              for (int i=0; i< files.length; i++)
            {
                deleteArchivedFile(files[i]);
            }
              deleteThis.delete();
          }
        else
        {
            deleteThis.delete();
        }
      }

      private void findArchives(Set<ArchiveFile> set, File dir) {
        File[] files = dir.listFiles();
        for (int i = 0; i < files.length; i++) {
          if (files[i].isDirectory() && !hasTimestampVersion(files[i]))
        {
            findArchives(set, files[i]);
        }
        else
        {
            set.add(new ArchiveFile(files[i]));
        }
        }
      }

      private boolean hasTimestampVersion(File checkFile)
      {
          String fileName = checkFile.getName();
          int firstDot = fileName.indexOf(".");
          if (firstDot == -1)
        {
            return false;
        }
        else
          {
              int secondDot = fileName.indexOf(".", firstDot + 1);
              return (secondDot != -1);
          }
      }
  }

    private static final class ArchiveFile
    extends File
    {
        private ArchiveFile(File file)
        {
            super(file.getAbsolutePath());
        }

        @Override
        public int compareTo(File file)
        {
            // compare directories first
            int compare = super.getParentFile().compareTo(file.getParentFile());
            if (compare != 0)
            {
                return compare;
            }

            int posntimestamp = 0;
            int posnversion = 0;

            // extract the version prefix and the remainder
            String thisArchive = super.getName();
            posntimestamp = thisArchive.indexOf('.');
            posnversion = thisArchive.indexOf('.', posntimestamp + 1);
            long thisTimestamp = Long.parseLong(thisArchive.substring(0, posntimestamp));
            int thisVersion = Integer.parseInt(thisArchive.substring(posntimestamp + 1, posnversion));
            thisArchive = thisArchive.substring(posnversion + 1);
            String otherArchive = file.getName();
            posntimestamp = otherArchive.indexOf('.');
            posnversion = otherArchive.indexOf('.', posntimestamp + 1);
            long otherTimestamp = Long.parseLong(otherArchive.substring(0, posntimestamp));
            int otherVersion = Integer.parseInt(otherArchive.substring(posntimestamp + 1, posnversion));
            otherArchive = otherArchive.substring(posnversion + 1);

            // compare the names
            compare = thisArchive.compareToIgnoreCase(otherArchive);
            if (compare != 0)
            {
                return compare;
            }

            // next look at the timestamps
            if (thisTimestamp > otherTimestamp)
            {
                return 1;
            }
            else if (thisTimestamp < otherTimestamp)
            {
                return -1;
            }
            else
            {
                // now look at the version #'s
                if (thisVersion > otherVersion)
                {
                    return 1;
                }
                // else it has to be lass than
                return -1;
            }
        }
    }
}

