/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.framework.agent.cache.impl;

import java.util.HashMap;

// An LRU based memory cache with size limit
final class ContentCache extends HashMap
{
    private Item m_leastUsed;
    private Item m_mostUsed;
    private int m_maxSize;
    private int m_usedSize;
    private boolean m_maxSuspended;

    ContentCache(int maxSize)
    {
        m_leastUsed = null;
        m_mostUsed = null;
        m_maxSize = maxSize;
        m_usedSize = 0;
        m_maxSuspended = false;
    }

    @Override
    public void clear()
    {
        super.clear();
        m_leastUsed = null;
        m_mostUsed = null;
        m_usedSize = 0;
        m_maxSuspended = false;
    }


    void put(String key, Object data, int size)
    {
        remove(key);

        int availableRoom = m_maxSize - m_usedSize;
        if (!m_maxSuspended && size > availableRoom)
        {
            if (!makeRoom(size - availableRoom))
            {
                throw new RuntimeException(key + " is too large - " + size + " bytes." +
                                           " Increase the MF container cache size from " + m_maxSize + ".");
            }
        }

        Item newItem = new Item(key, data, size);
        super.put(key, newItem);
        m_usedSize += size;
        putAtEnd(newItem);
    }

    Object get(String key)
    {
        Item item = (Item)super.get(key);

        if (item == null)
        {
            return null;
        }

        // Make the item most used
        if (item != m_mostUsed)
        {
            removeFromChain(item);
            putAtEnd(item);
        }

        return item.data;
    }

    void remove(String key)
    {
        Item item = (Item)super.remove(key);

        if (item == null)
        {
            return;
        }

        removeFromChain(item);
        m_usedSize -= item.size;
    }

    private void removeFromChain(Item item)
    {
        Item prev = item.prevItem;
        Item next = item.nextItem;

        if (prev != null)
        {
            prev.nextItem = next;
        }

        if (next != null)
        {
            next.prevItem = prev;
        }

        if (m_leastUsed == item)
        {
            m_leastUsed = next;
        }

        if (m_mostUsed == item)
        {
            m_mostUsed = prev;
        }

        item.prevItem = null;
        item.nextItem = null;
    }

    void adjustSize(int newSize)
    {
        if (newSize < m_usedSize)
        {
            m_maxSize = m_usedSize;
        }
        else
        {
            m_maxSize = newSize;
        }
    }

    // If we had to use more than the current max-size then make that the new max
    void adjustSize()
    {
        m_maxSize = (m_maxSize >= m_usedSize) ? m_maxSize : m_usedSize;
    }

    // Temporarily remove the size limit - needed when an entire array of elements is cached.
    void suspendSizeLimit()
    {
        m_maxSuspended = true;
    }

    void restoreSizeLimit()
    {
        m_maxSuspended = false;
    }


    private void putAtEnd(Item item)
    {
        // This is the first item in the list
        if (m_leastUsed == null)
        {
            m_leastUsed = m_mostUsed = item;
            return;
        }

        m_mostUsed.nextItem = item;
        item.prevItem = m_mostUsed;
        m_mostUsed = item;
    }

    private boolean makeRoom(int size)
    {
        int gain = 0;

        while (gain < size)
        {
           if (m_leastUsed == null)
        {
            return false;
        }

           gain += m_leastUsed.size;
           remove(m_leastUsed.key);
        }

        return true;

    }

    private class Item
    {
        private String key;
        private Object data;
        private int size;
        private Item nextItem;
        private Item prevItem;

        Item (String key, Object data, int size)
        {
            this.key = key;
            this.data = data;
            this.size = size;
            this.prevItem = null;
            this.nextItem = null;
        }

    }
}
