/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.framework.agent.cache.impl;

import java.util.ArrayList;

import com.sonicsw.mf.common.config.AttributeSetTypeException;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.ReadOnlyException;
import com.sonicsw.mf.common.config.impl.AttributeSet;
import com.sonicsw.mf.common.config.impl.Element;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.config.impl.IElementCache;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.runtime.IContainerExitCodes;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.agent.cache.PersistentCacheException;
import com.sonicsw.mf.framework.directory.storage.IStorage;
import com.sonicsw.mf.framework.directory.storage.ParentDirDoesNotExistException;
import com.sonicsw.mf.framework.directory.storage.StorageException;
import com.sonicsw.mf.framework.directory.storage.fs.FSStorage;

// Uses an LRU cache to maintain the content of elements
public final class ElementCache implements IElementCache
{

    private static final int CACHE_STRUCTURE_VERSION = 4;
    private static final String SYSTEM_DIRECTORY_PATH = IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_SYSTEM_DIR;
    private static final String VERSION_ELEMENT =  "version";
    private static final String VERSION_ELEMENT_PATH = SYSTEM_DIRECTORY_PATH + IMFDirectories.MF_DIR_SEPARATOR + VERSION_ELEMENT;

    public static final int CACHE_MAX_SIZE_DEFAULT = 10000000;
    public static final int CACHE_MAX_SIZE_MINIMUM =  5000000;

    private IContainer m_container;
    private ContentCache m_cache;
    private IStorage m_storage;
    private ArrayList m_newBatch;
    private ArrayList m_deletedBatch;

    private static  EntityName VERSION_ELEMENT_ENTITY_NAME = null;
    private static  EntityName SYSTEM_DIRECTORY_PATH_ENTITY_NAME = null;

    static
    {
        try
        {
            VERSION_ELEMENT_ENTITY_NAME = new EntityName(VERSION_ELEMENT_PATH);
            SYSTEM_DIRECTORY_PATH_ENTITY_NAME = new EntityName(SYSTEM_DIRECTORY_PATH);
        }
        catch (ConfigException e)
        {
            e.printStackTrace();
            throw new Error(e.toString(), e);
        }
    }


    public ElementCache(IContainer container)
    {
        m_newBatch = null;
        m_deletedBatch = null;
        m_container = container;
        m_storage = null;
        m_cache = new ContentCache(CACHE_MAX_SIZE_DEFAULT);
    }

    public ElementCache(IContainer container, String hostDir, String password) throws PersistentCacheException
    {
        m_newBatch = null;
        m_deletedBatch = null;
        m_container = container;
        try
        {
            m_storage = new FSStorage(hostDir, ConfigCache.CONTENT_DIR, IMFDirectories.MF_DATA_DIR, password, false);
        }
        catch (StorageException e)
        {
            storageError("Failed to open persistent cache", e);
        }

        // Checks the version
        try
        {
            if (!m_storage.directoryExists(SYSTEM_DIRECTORY_PATH_ENTITY_NAME))
            {
                m_storage.createDirectory(SYSTEM_DIRECTORY_PATH_ENTITY_NAME);
            }

            IDirElement versionElement = m_storage.getElement(VERSION_ELEMENT_ENTITY_NAME);
            if (versionElement == null)
            {
                versionElement = ElementFactory.createElement(VERSION_ELEMENT_PATH, "version", "2.0");
                IAttributeSet attributes = versionElement.getAttributes();
                attributes.setIntegerAttribute("VERSION", new Integer(CACHE_STRUCTURE_VERSION));
                m_storage.setElement(VERSION_ELEMENT_ENTITY_NAME, (IDirElement)versionElement.doneUpdate(), true);
            }
            else
            {
                IAttributeSet attributes = versionElement.getAttributes();
                Integer dsVersion = (Integer)attributes.getAttribute("VERSION");
                if (dsVersion.intValue() != CACHE_STRUCTURE_VERSION)
                {
                    throw new PersistentCacheException("The cache version mismatches the " +
                                                        " version of the software.\nStorage version is: " +
                                                         dsVersion + ". Software version is: " + CACHE_STRUCTURE_VERSION + ".");
                }
            }

        }
/*
        catch (EncryptionException e)
        {
            throw new PersistentCacheException("Could not access the persistent cache because of a wrong or missing encryption password.");
        }
*/
        catch (StorageException e) { throw new PersistentCacheException(e.toString(), e); }
        catch (ReadOnlyException e) { throw new Error(e.toString(), e); }
        catch (AttributeSetTypeException e) { throw new Error(e.toString(), e); }
        catch (ConfigException e) { throw new Error(e.toString(), e); }


        m_cache = new ContentCache(CACHE_MAX_SIZE_DEFAULT);
    }


    // Store the attributes for each element retrieved from persistent cache (or the DS) in memory and
    // return the attributes of the specific element requested.
    private IAttributeSet getStoredAttributes(IDirElement[] stElements, Element element)
    {
         IElementIdentity wantedElementID = element.getIdentity();
         String wantedElementName = wantedElementID.getName();

         // The element was deleted from DS and persistent cache
         if (stElements == null || stElements.length == 0)
         {
              element.setDeleted();
              return null;
         }

         IDirElement targetElement = null;
         for (int i = 0; i < stElements.length; i++)
         {
             String elementName = stElements[i].getIdentity().getName();
             IAttributeSet attributes = stElements[i].getAttributes();
             m_cache.put(elementName, attributes, ((AttributeSet)attributes).estimateSize());
             if (elementName.equals(wantedElementName))
            {
                targetElement = stElements[i];
            }
         }

         if (targetElement == null || !targetElement.getIdentity().equalEntity(wantedElementID))
         {
             element.setDeleted();
             return null;
         }

         return targetElement.getAttributes();

    }


    // Return the element's attributes from cache or from the DS
    @Override
    public synchronized IAttributeSet getAttributes(Element element)
    {
        IElementIdentity elementIdentity = element.getIdentity();
        String elementName = elementIdentity.getName();
        IAttributeSet attributes = (IAttributeSet)m_cache.get(elementName);
        if (attributes != null)
        {
            return attributes;
        }

        // The attributes are not in cache and we have a persistent storage - get them from the storage
        if (m_storage != null)
        {
            try
            {
                return getStoredAttributes(m_storage.getElements(new EntityName(elementName)), element);
            }
            catch (ConfigException e) { throw new Error("Cache configuration failure", e); }
            catch (StorageException e) { storageError("Storage failure", e); }
        }

        // The attributes are not in cache and we don't have a persistent storage - get them from the DS
        IDirElement[] dsElement = null;
        if (m_container != null)
        {
            dsElement= new IDirElement[]{(IDirElement)m_container.getConfigurationFromDS(elementName)};
        }
        else {
            return null; // That should not happen since it can happen only when there is no persistent cache
        }
                         // and we have no container only at the container startup (boot) sequence that's supported
                         // only when there is a persistent cache.
        return getStoredAttributes(dsElement, element);
    }

    // Stores the element's attributes in cache
    @Override
    public synchronized void storeAttributes(Element element)
    {
        AttributeSet attributes = (AttributeSet)element.getAttributes();

        // Store the element and its attributes in the persistent cahce (if exists)
        if (m_storage != null)
        {
            if (m_newBatch == null)
            {
                try
                {
                    m_storage.setElement(new EntityName(element.getIdentity().getName()), element, true);
                }
                catch (ConfigException e) { throw new Error("Cache configuration failure", e); }
                catch (StorageException e) { storageError("Storage failure", e); }
            }
            else
            {
                m_newBatch.add(element);
            }
        }

        // Store the attributes in the in-memory cache
        m_cache.put(element.getIdentity().getName(), attributes, attributes.estimateSize());
    }

    @Override
    public synchronized void removeAttributes(String elementName)
    {
        if (m_storage != null)
        {
            try
            {
                if (m_deletedBatch == null)
                {
                    m_storage.deleteElement(new EntityName(elementName));
                }
                else
                {
                    m_deletedBatch.add(new EntityName(elementName));
                }
            }
            catch (ParentDirDoesNotExistException e)
            {
                throw new Error("Cache directory does not exist", e);
            }
            catch (ConfigException e)
            {
                throw new Error("Cache configuration failure", e);
            }
            catch (StorageException e)
            {
                storageError("Storage failure", e);
            }
        }
        m_cache.remove(elementName);
    }


    // Used to adjust the size of the cache if the configuration has a value different from the default
    @Override
    public synchronized void adjustSize(int newSize)
    {
        m_cache.adjustSize(newSize);
    }

    private void storageError(String message, StorageException e)
    {
         if (m_container == null)
        {
            throw new Error(message, e);
        }

         m_container.logMessage(null, "Storage failure, trace follows...", e, Level.SEVERE);
         m_container.logMessage(null, "If the persistent cache appears to be corrupt then the problem can be fixed by deleting the persistent cache directory.", Level.INFO);

         m_container.shutdown(IContainerExitCodes.CACHE_FAILURE_EXIT_CODE);
    }

    // To save a batch of elements under a single directory efficiently (it's more efficient when several
    // elements are packed together in a single file).
    void createBatch()
    {
        if (m_storage == null)
        {
            return;
        }

        if (m_newBatch != null)
        {
            throw new IllegalStateException();
        }
        if (m_deletedBatch != null)
        {
            throw new IllegalStateException();
        }

        m_newBatch = new ArrayList();
        m_deletedBatch = new ArrayList();

        // We don't want elements suspended from memory cache while a batch is created
        // since their content might not be available when we'll actually save the batch
        // in the file system when we call saveBatch()
        m_cache.suspendSizeLimit();
    }

    synchronized void saveBatch(boolean ok)
    {
        if (m_storage == null)
        {
            return;
        }

        // We are going to store the batch in the file-system - we didn't evict any elements from memory
        // and we can restore the size limit. If during the creation of the batch the the max-size of the in-memory
        // cache became  greater than the max, we make the current size the max by calling adjustSize().
        m_cache.restoreSizeLimit();
        m_cache.adjustSize();

        if (m_newBatch == null)
        {
            throw new IllegalStateException();
        }
        if (m_deletedBatch == null)
        {
            throw new IllegalStateException();
        }

        if (!ok)
        {
             m_newBatch = null;
             m_deletedBatch = null;
             return;
        }

        IDirElement[] newElements = new IDirElement[m_newBatch.size()];
        m_newBatch.toArray(newElements);

        EntityName[] deletedElements = new EntityName[m_deletedBatch.size()];
        m_deletedBatch.toArray(deletedElements);

        try
        {
            m_storage.deleteElements(deletedElements);
            m_storage.setElements(newElements, true);
        }
        catch (StorageException e) { storageError("Storage failure", e); }

        m_newBatch = null;
        m_deletedBatch = null;

    }

}
