package com.sonicsw.mf.framework.agent.ci;

import java.io.File;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.HashMap;
import java.util.Locale;
import java.util.StringTokenizer;

import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.runtime.IContainerExitCodes;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.agent.ClassLoaderFactory;
import com.sonicsw.mf.framework.agent.ContainerUtil;
import com.sonicsw.mf.framework.agent.ExpandedSonicArchive;
import com.sonicsw.mf.framework.agent.LocalFileManager;
import com.sonicsw.mf.framework.agent.cache.ConfigCacheFactory;
import com.sonicsw.mf.framework.agent.cache.IConfigCache;
import com.sonicsw.mf.framework.agent.cache.IConfigCacheView;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;

/**
 * This class can be used as the main() in order to start a container from a script or exec'd from
 * either an Activation Daemon or a Windows service.
 */
public final class Agent
{
    private static final String UTIL_DLL_FILE_32 = "sonicmfUtil.dll";
    private static final String UTIL_DLL_FILE_64 = "sonicmfUtil64.dll";
    private static final boolean RUNNING_AS_WINSERVICE = System.getProperty(IContainer.WINSVC_UTILDIR_PROPERTY) != null;

    private static int m_exitCode = IContainerExitCodes.NORMAL_SHUTDOWN_EXIT_CODE;

    private static Object m_containerExitCode;
    private static Class m_exitCodeClass;

    private static native void ignoreLogoffEvent();
    public static void main(final String[] args)
    {
        try
        {
            if (args.length < 3)
            {
                throw new IllegalArgumentException("Invalid number of arguments: " + args.length);
            }
            if (args[0].length() == 0)
            {
                throw new IllegalArgumentException("<cacheDir> cannot be zero length");
            }
            if (args[1].length() == 0)
            {
                throw new IllegalArgumentException("<containerName> cannot be zero length");
            }
            if (args[2].length() == 0)
            {
                throw new IllegalArgumentException("<containerConfigID> cannot be zero length");
            }
            if (!new File(args[0]).exists())
            {
                throw new IllegalArgumentException(args[0] + " does not exist");
            }


            if (Boolean.getBoolean("sonicsw.mf.launcher_upgrade_restart"))
            {
                if (RUNNING_AS_WINSERVICE)
                {
                    WinServiceUpdate.startWinserviceUpdateProcess(args[1]);
                    System.exit(IContainerExitCodes.NORMAL_SHUTDOWN_EXIT_CODE);
                }
                System.exit(IContainerExitCodes.CONTAINER_RESTART_EXIT_CODE);
            }

            // If the container is being started as a Windows Service, then use native
            // code to establish appropriate signal handling so that user logoff events
            // don't cause the JVM to terminate.  See Sonic00037674 / SNC00067274 for more info.
            String winsvcUtilDir = System.getProperty(IContainer.WINSVC_UTILDIR_PROPERTY);  // sonicsw.mf.winsvc.utilDir
            if (winsvcUtilDir != null)
            {
                winsvcUtilDir = winsvcUtilDir.trim();
                if (winsvcUtilDir.length() > 0)
                {
                    String winsvcUtilLib = winsvcUtilDir + File.separatorChar;
                    // Choose 32-bit or 64-bit version of DLL to match the JVM
                    //  - we currently support amd64, not IA-64
                    winsvcUtilLib += System.getProperty("os.arch").toLowerCase(Locale.US).equals("amd64") ? UTIL_DLL_FILE_64 : UTIL_DLL_FILE_32;
                    System.load(winsvcUtilLib);
                    ignoreLogoffEvent();
                }
            }

            StringTokenizer containerNameTokens = new StringTokenizer(args[1], ",");
            StringTokenizer containerConfigIDTokens = new StringTokenizer(args[2], ",");

            if (containerNameTokens.countTokens() > 1)
            {
                System.setProperty(IContainer.MF_LSD_COLOCATE_PROPERTY, "true");
                System.setProperty(IContainer.MF_LSD_COLOCATE_COUNT_PROPERTY, "0");
            }

            while (containerNameTokens.hasMoreTokens())
            {
                final String containerName = containerNameTokens.nextToken();
                final String containerConfigID = containerConfigIDTokens.nextToken();

                Runnable agent = new Runnable()
                {
                    @Override
                    public void run()
                    {
                        try
                        {
                            new Agent(args[0], containerName, containerConfigID);
                        }
                        catch(Throwable e)
                        {
                            startupAbort(e);
                        }
                    }
                };
                Thread startupThread = new Thread(agent);
                if (System.getProperty(IContainer.CACHE_GENERATION_URL_PROPERTY) == null)
                {
                    startupThread.setName(IContainer.CONTAINER_BOOT_THREAD_PREFIX + " [" + containerName + "]");
                }
                else
                {
                    startupThread.setName(IContainer.CONTAINER_CACHE_GENERATION_THREAD_PREFIX + " [" + containerName + "]");
                }

                synchronized(Runtime.getRuntime())
                {
                    String containerCount = System.getProperty(IContainer.MF_LSD_COLOCATE_COUNT_PROPERTY);
                    if (containerCount != null)
                    {
                        System.setProperty(IContainer.MF_LSD_COLOCATE_COUNT_PROPERTY, new Integer(Integer.parseInt(containerCount) + 1).toString());
                    }
                }

                startupThread.start();
                startupThread.join();
            }
            int exitCode;
            synchronized(m_containerExitCode) 
            {
                while (!isExit())
                {
                    m_containerExitCode.wait();
                }
                exitCode = getExitCode();
            }
            System.exit(exitCode);
        }
        catch(Throwable e)
        {
            startupAbort(e);
        }
    }
    
    static boolean isExit() 
    {
        try {
            Method isExitCode = m_exitCodeClass.getMethod("exit");
            Boolean exitFlag = (Boolean)isExitCode.invoke(m_containerExitCode);
            return exitFlag.booleanValue();
        } catch(Exception e) {
            return true; // return true. Need to exit anyway
        }
    }
    
    private static int getExitCode() 
    {
        try {
            Method getExitCode = m_exitCodeClass.getMethod("getExitCode");
            Integer exitCode = (Integer)getExitCode.invoke(m_containerExitCode);
            return exitCode.intValue();
        } catch (Exception e) {
            return IContainerExitCodes.UNSPECIFIED_FAILURE_EXIT_CODE; // Exit anyway...
        }
    }

    private static void startupAbort(Throwable e)
    {
        e.printStackTrace();
        if (e instanceof IllegalArgumentException)
        {
            printUsage();
        }

        // if this property is set then the container was started as an NT service .. in which case
        // we need to put a delay in so the service can get the stack trace from the stderr
        if (Boolean.getBoolean("sonicsw.mf.signal"))
        {
            try { Thread.sleep(5000); } catch (Exception ex) {}
        }

        if (e instanceof IllegalArgumentException)
        {
            m_exitCode = IContainerExitCodes.INVALID_CMD_LINE_EXIT_CODE;
        }
        else
        {
            m_exitCode = IContainerExitCodes.UNSPECIFIED_FAILURE_EXIT_CODE;
        }
        try
        {
            Method setExitCode = m_exitCodeClass.getMethod("setExitCode", new Class[] {java.lang.Integer.class});
            setExitCode.invoke(m_containerExitCode, new Object[] {new Integer(m_exitCode)});
        }
        catch (Exception e2) { e2.printStackTrace();} // we tried to set the exit code but something happened. Exit anyway.
        System.exit(m_exitCode);
    }

    public Agent(final String cacheDirectory0, final String containerName, final String containerConfigID)
    throws Throwable
    {

        File archiveRoot = null;
        String classpath = null;
        Long maxCacheSize = null;

        IConfigCache configCache = null;
        IConfigCacheView configCacheView;
        IElement containerConfig;

        //In the cas of LSD, where multiple containers are colocated in the JVM,we have to derive the cache directory from the container name
        // since cacheDirectory0 is the cache directory of the last container in the list. So we have to use the parent directory of that cache +
        // containerName + ".cache".
        String cacheDirectory = Boolean.getBoolean(IContainer.MF_LSD_COLOCATE_PROPERTY) ?
                                new File(new File(cacheDirectory0).getParentFile(), containerName + ".cache").getAbsolutePath() :
                                cacheDirectory0;

        try
        {
            // open the cache
            String password = ContainerUtil.retrieveCachePassword(new File(cacheDirectory), false);
            configCache = ConfigCacheFactory.createCache(cacheDirectory, password);
            configCacheView = configCache.getCacheView();


            // get the container's configuration
            containerConfig = configCacheView.getElement(containerConfigID);
            if (containerConfig == null)
            {
                throw new IllegalArgumentException(containerConfig + " does not exist in cache");
            }
            // get the URI of the container's archive and the container's classpath
            IAttributeSet containerAttrs = containerConfig.getAttributes();
            String containerArchive = (String)containerAttrs.getAttribute(IContainerConstants.ARCHIVE_NAME_ATTR);
            String searchPath = (String)containerAttrs.getAttribute(IContainerConstants.ARCHIVE_SEARCH_PATH_ATTR);
            IAttributeSet cacheAttrs = (IAttributeSet)containerAttrs.getAttribute(IContainerConstants.CACHE_ATTR);
            if (cacheAttrs != null)
            {
                maxCacheSize = (Long)cacheAttrs.getAttribute(IContainerConstants.PERSISTENT_BLOB_CACHE_SIZE_ATTR);
            }
            classpath = (String)System.getProperty(IContainer.CONTAINER_CLASSPATH_PROPERTY, "");

            // get the container archive's root
            archiveRoot = new LocalFileManager(null, configCache, null, null).getLocalFile(searchPath, containerArchive);
            System.setProperty(IContainer.MF_CONTAINER_CAR_ROOT_PROPERTY, archiveRoot.getAbsolutePath());
        }
        finally
        {
            if (configCache != null)
            {
                try { configCache.close(); } catch(Exception e) { }
            }
        }

        // get the container archive's descriptor
        ExpandedSonicArchive archive = new ExpandedSonicArchive(archiveRoot);

        // create the initial class loader hierarchy
        ClassLoader containerClassLoader = createClassLoaderGraph(containerName, archive, classpath);

        // instantitate the container implementation
        try
        {
            Thread.currentThread().setContextClassLoader(containerClassLoader);

            Class containerClass = containerClassLoader.loadClass("com.sonicsw.mf.framework.agent.ContainerImpl");
            // The class used to communicate the exit code between Agent and ContainerImpl must be loaded with the
            // same class loader as ContainerImpl.
            Class exitCodeClass = containerClassLoader.loadClass("com.sonicsw.mf.framework.agent.ContainerExitCode");
            Constructor exitCodeConstructor = exitCodeClass.getConstructor(new Class[] {});
            Object containerExitCode = exitCodeConstructor.newInstance(new Object[]{});
            synchronized(Agent.class)
            {
                m_exitCodeClass = exitCodeClass;
                m_containerExitCode = containerExitCode;
            }
            Constructor containerConstructor = containerClass.getConstructor(new Class[] { ClassLoader.class, String.class, String.class, String.class, Long.class, Object.class });
            containerConstructor.newInstance(new Object[] { containerClassLoader, cacheDirectory, containerName, containerConfigID, maxCacheSize, containerExitCode});
        }
        catch(InvocationTargetException ite)
        {
            throw ite.getTargetException();
        }
    }

    private ClassLoader createClassLoaderGraph(String containerName, ExpandedSonicArchive archive, String classpath)
    throws Exception
    {
        // create the common class loader
        URL[] urls = ClassLoaderFactory.prependClasspath(classpath, archive.getGlobalClasspath());
        URLClassLoader globalClassLoader = ClassLoaderFactory.createGlobalLoader(containerName, urls, ClassLoader.getSystemClassLoader().getParent());

        // iterate through the shared library list creating class loaders
        HashMap sharedLibraryClassLoaders = new HashMap();
        String[] sharedLibraryNames = archive.getSharedLibraryNames();
        for (int i = 0; i < sharedLibraryNames.length; i++)
        {
            ClassLoader sharedLoader = ClassLoaderFactory.createDelegateLoader(containerName, null, sharedLibraryNames[i], archive.getSharedLibraryClasspath(sharedLibraryNames[i]), globalClassLoader);
            sharedLibraryClassLoaders.put(sharedLibraryNames[i], sharedLoader);
            // register once in the current class loader namespace (App class loader)
            ClassLoaderFactory.addDelegateLoader(containerName, "AGENT", sharedLibraryNames[i], sharedLoader);
        }

        // create the agent (container) private class loader
        ClassLoader privateClassLoader = ClassLoaderFactory.createDelegatingLoader(containerName, "AGENT", archive.getPrivateClasspath(), globalClassLoader, sharedLibraryNames);

        // several aspects must be registered in the the ClassLoaderFactory that will now be loaded in the new loader namespaces
        Class classLoaderFactoryClass = privateClassLoader.loadClass(ClassLoaderFactory.class.getName());

        // get the class loader factory class for the ClassLoaderFactory in this class loader namespace to reference
        Field classLoaderFactoryClassField = classLoaderFactoryClass.getField("m_classLoaderFactoryClass");
        ClassLoaderFactory.m_classLoaderFactoryClass = (Class)classLoaderFactoryClassField.get(null);

        // reregister the global loader
        Method setGlobalLoaderMethod = classLoaderFactoryClass.getMethod("setGlobalLoader", new Class[] { URLClassLoader.class });
        setGlobalLoaderMethod.invoke(null, new Object[] { globalClassLoader });

        // reregister the shared loaders in the agent (container) class loader namespace so they are accessible
        // from delegating loaders created from the agent class loader
        Method addDelegateLoaderMethod = classLoaderFactoryClass.getMethod("addDelegateLoader", new Class[] { String.class, String.class, String.class, ClassLoader.class });
        for (int i = 0; i < sharedLibraryNames.length; i++)
        {
            addDelegateLoaderMethod.invoke(null, new Object[] { containerName, null, sharedLibraryNames[i], (ClassLoader)sharedLibraryClassLoaders.get(sharedLibraryNames[i]) });
        }

        // reregister the private loader in the agent (container) class loader namespace
        Method addDelegatingLoaderMethod = classLoaderFactoryClass.getMethod("addDelegatingLoader", new Class[] { String.class, String.class, String[].class, ClassLoader.class });
        addDelegatingLoaderMethod.invoke(null, new Object[] { containerName, "AGENT", sharedLibraryNames, privateClassLoader });

        return privateClassLoader;
    }

    private static void printUsage()
    {
        System.err.println();
        System.err.println("Starts a manageable container.");
        System.err.println();
        System.err.println("Usage: <java> <java_params> [-Dsonicsw.mf.password=<password>] com.sonicsw.mf.Agent <cacheDir> <containerName> <containerConfigID>");       //NOSONAR field change is not required.
        System.err.println();
        System.err.println("  <cacheDir>          The path to the directory containing the cache");
        System.err.println("  <containerName>     The fully qualified name of the container (e.g.");
        System.err.println("                      \"Domain1.Container1\").");
        System.err.println("  <containerConfigID> The internal configuration identity of the container.");
        System.err.println();
        System.err.println("  <password>          Password to decrypt disk based configuration information.");

        System.err.flush();
    }
}
