// Copyright (c) 2009 Progress Software Corporation. All Rights Reserved.

package com.sonicsw.mf.framework.agent.ci;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.RandomAccessFile;
import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Properties;

import javax.management.ObjectName;

import com.sonicsw.mf.comm.IConnectorClient;
import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.ILogger;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.impl.CanonicalName;
import com.sonicsw.mf.common.runtime.impl.ContainerIdentity;
import com.sonicsw.mf.common.util.MFLogger;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.agent.ContainerSetup;
import com.sonicsw.mf.framework.agent.ContainerUtil;
import com.sonicsw.mf.framework.agent.LocalFileManager;
import com.sonicsw.mf.framework.agent.LogicalFile;
import com.sonicsw.mf.framework.agent.cache.ConfigCacheFactory;
import com.sonicsw.mf.framework.agent.cache.IConfigCache;
import com.sonicsw.mf.framework.agent.cache.IConfigCacheView;
import com.sonicsw.mf.framework.agent.cache.LatestVersionMissingException;
import com.sonicsw.mf.framework.agent.cache.PersistentCacheException;
import com.sonicsw.mf.framework.agent.cache.impl.ConfigCache;
import com.sonicsw.mf.framework.directory.DSComponent;
import com.sonicsw.mf.framework.directory.IFSStorage;
import com.sonicsw.mf.jmx.client.DirectoryServiceProxy;
import com.sonicsw.mf.jmx.client.IRemoteMBeanServer;
import com.sonicsw.mf.jmx.client.JMSConnectorAddress;
import com.sonicsw.mf.mgmtapi.config.constants.IBackupDirectoryServiceConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IDirectoryServiceConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IReplicationConnectionConstants;

public abstract class ContainerResources
{
    static final String FILELOG_SUFFIX = ".log";

    static final String CONFIGURED_JVM_OPTIONS = "_MF_CONFIGURED_JVM_OPTIONS";

    static final String PATH_SEPARATOR = System.getProperty("path.separator");

    private static final String CONTAINER_WORKING_DIR = System.getProperty("user.dir");

    private static final String SONICSW_HOME = System.getProperty(IContainer.SONICSW_HOME_PROPERTY);

    private static final String[] EMPTY_STRING_ARRAY = new String[0];

    private static final Object[] EMPTY_OBJECT_ARRAY = new Object[0];

    private static final Class[] EMPTY_CLASS_ARRAY = new Class[0];

    private static final int CONNECTION_PROP_PREFIX_LENGTH = IContainer.CONNECTION_PROP_PREFIX.length();

    private static final int CENTRAL_CONNECTION_PROP_PREFIX_LENGTH = IContainer.CENTRAL_CONNECTION_PROP_PREFIX.length();

    private static final int SYSTEM_PROP_PREFIX_LENGTH= IContainer.SYSTEM_PROP_PREFIX.length();

    private static final int REPLICATION_FAILURE_DETECTION_LAUNCH_MIN = 60; //Seconds

    static final String XARG_BOOT = "-Xbootclasspath:";

    static final String XARG_BOOT_P = "-Xbootclasspath/p:";

    static final String XARG_BOOT_A = "-Xbootclasspath/a:";

    static final String sonicfsPrefix = IContainer.DS_CLASSPATH_PROTOCOL.substring(0, IContainer.DS_CLASSPATH_PROTOCOL.length() - 1);

    private static final String CENTRAL_CONNECTION_ATTR = "CENTRAL_CONNECTION"; //TODO Use IContainerConstants when available

    private boolean m_requestToConfigureFromCache;

    private String m_containerINIFileName;

    private LaunchConnector m_connector;

    private IDirectoryAdminService m_ds;

    private LocalFileManager m_localFileManager;

    private Hashtable m_connectionAttributes;

    private String m_logFilePath;  // absolute (full directory path + filename) log file name

    private boolean m_nonDMContainerWhichConnectToRemoteDS = false;

    private boolean  m_mfDirectoryJARMissing = false;

    private boolean m_dsBootFileMissing = false;

    private File m_mfDirectoryJAR;

    private boolean m_centralConnectionUsed = false;;

    protected ILogger m_logger;

    protected String m_dsBootFileName;

    protected String m_dsArchivePath;

    protected String m_logicalArchivePath;

    private File m_archiveRoot;

    File m_actionalSDKArchive;

    File m_actionalPlugMakerArchive;

    Class m_localDirectoryServiceClass;

    Object m_localDirectoryService;

    String m_containerID;

    String m_containerName;

    String m_domainName;

    boolean m_useCachedConfig;

    // Configuration
    ExtractedContainerConfig m_extractedConfig;   // container configuration we use

    ExtractedContainerConfig m_bootExtractedConfig;   // we rememeber the boot configuration so we can compare to the DS one

    int m_traceMask;

    LocalDSParameters m_localDSParam;

    InitInfo m_initInfo;

    // Cache
    String m_cacheHostDirectoryName;

    IConfigCache m_configCache;

    IConfigCacheView m_configCacheView;

    // Resources
    SystemProperties m_systemProps;

    CARClasspath m_classpath;

    DSClasspath m_configClasspath;

    private MFLogger m_pseLogger;

    // Extracts the initialization info from XML boot or INI files and creates the cache
    protected void init(String containerINIFileName)
    throws Exception
    {
        if (IContainer.CURRENT_LAUNCHER_VERSION != null)
        {
            logMessage("Using launcher \"" + new File(new File(SONICSW_HOME), IContainer.LAUNCHER_DIR).getAbsolutePath() + "\"", Level.INFO);
        }

        //Sets the SONIC_CONTAINERS directory to be used by remote clients who call the getJVMProperties
        System.setProperty(IContainer.SONIC_CONTAINERS_DIR_PROPERTY, ContainerSetup.findSonicContainersDir(new File(SONICSW_HOME)).getAbsolutePath());

        m_containerINIFileName = containerINIFileName;
        File containerINIFile = openINIFile(containerINIFileName);

        m_requestToConfigureFromCache = Boolean.getBoolean(IContainer.CONFIGURE_FROM_CACHE_PROPERTY);
        //If there is no system property that forces configuration from cache then we check if the
        // CONFIGURE_FROM_CACHE_FILE file is present. The presence of the file means that the container has just
        // updated the cache using the central managemnt broker and therefore there is no need
        // to connect to the central management broker again.
        if (!m_requestToConfigureFromCache)
        {
            File configureFromCacheFile = new File(containerINIFile.getParentFile(), IContainer.CONFIGURE_FROM_CACHE_FILE);
            m_requestToConfigureFromCache = configureFromCacheFile.exists();
            configureFromCacheFile.delete();
        }
        if (m_requestToConfigureFromCache)
        {
            System.setProperty(IContainer.CONFIGURE_FROM_CACHE_PROPERTY, "true");
        }

        RandomAccessFile raf = new RandomAccessFile(containerINIFile, "r");
        byte[] bytes = new byte[(int)raf.length()];
        raf.read(bytes);
        raf.close();


        // check if a password is provided (indicates that the config file is encrypted)
        if (IContainer.PASSWORD != null)
        {
            bytes = ContainerUtil.decryptBytes(bytes, IContainer.PASSWORD);
        }

        boolean INIIsXMLFile = new String(bytes).startsWith("<?xml");

        if (!INIIsXMLFile && !checkHomeIsWritable())
        {
            throw new Exception("The container doesn't have write permission in \"" + SONICSW_HOME + "\"; Write permission is required for automatic upgrade of the launcher");
        }

        m_initInfo = null;
        if (INIIsXMLFile)
        {
            m_initInfo = new XmlINI(bytes);
        }
        else
        {
            m_initInfo = new PropINI(bytes);
        }

        m_extractedConfig = m_bootExtractedConfig = m_initInfo.m_config;

        if (m_extractedConfig.m_containerName == null)
        {
            throw new Exception("The format of \"" + containerINIFile.getAbsolutePath() + "\" is bad");
        }

        m_domainName = m_extractedConfig.m_domainName;
        m_traceMask = m_extractedConfig.m_deployTraceMask.intValue();
        m_containerName = createContainerFullName(m_extractedConfig);


        if (m_extractedConfig.m_hostsDS.booleanValue())
        {
            boolean launcherUsed = IContainer.CURRENT_LAUNCHER_VERSION != null;

            if (!launcherUsed && (m_dsBootFileName == null || m_dsBootFileName.length() == 0))
            {
                throw new Exception("Container \"" + m_containerName + "\" hosts the Directory Service but an XML Directory Service boot file was not passed.");
            }

            if (launcherUsed)
            {
                m_dsBootFileName = IContainer.DEFAULT_DS_BOOTFILE_NAME;

                if (!(new File(IContainer.DEFAULT_DS_BOOTFILE_NAME).exists()))
                {
                    m_dsBootFileMissing = true;
                    logMessage("\" " + m_containerName + "\" hosts the Directory Service but \"" + IContainer.DEFAULT_DS_BOOTFILE_NAME + "\" is missing", Level.WARNING);
                }

                File launcherDir = new File(SONICSW_HOME, IContainer.LAUNCHER_DIR);
                m_mfDirectoryJAR = new File(launcherDir, LauncherJARBuilder.MF_DIR_JAR_PATH);
                if (!m_mfDirectoryJAR.exists())
                {
                    m_mfDirectoryJARMissing = true;
                    logMessage("\" " + m_containerName + "\" hosts the Directory Service but \"" + m_mfDirectoryJAR.getAbsolutePath() + "\" is missing", Level.WARNING);
                }
                if (m_mfDirectoryJARMissing || m_dsBootFileMissing)
                {
                    logMessage("Trying to retrieve the missing Directory Service launch file(s) from the peer Directory Service...", Level.INFO);
                }

            }
        }

        m_localDirectoryService = null;
        m_localDSParam = null;
        if (m_dsBootFileName != null && m_extractedConfig.m_hostsDS.booleanValue() && !m_mfDirectoryJARMissing && !m_dsBootFileMissing)
        {
            m_localDSParam = new LocalDSParameters(m_dsBootFileName);
        }

        File logFile = (m_extractedConfig.m_logFileName == null) ? new File(new File(CONTAINER_WORKING_DIR), m_containerName + FILELOG_SUFFIX) :
                                                                   new File(absoluteFromRelativeToWorkDir(CONTAINER_WORKING_DIR,m_extractedConfig.m_logFileName));

        m_logFilePath = logFile.getAbsolutePath();

        initLogger(m_logFilePath, m_containerName);  // invoke concrete subclass' method to initialize the ILogger instance...

        logMessage("Fetching the resources of container \"" + m_containerName + '"', Level.INFO);
        boolean doReset = detectCacheResetRequest(containerINIFile);
        if (doReset)
        {
            logMessage("Deleting the obsolete cache of container \"" + m_containerName + "\"", Level.INFO);
        }
        cleanupIfNeeded();
        createCache(new File(m_extractedConfig.m_cacheHostDirectoryName).getAbsolutePath(), m_extractedConfig.m_cachePassword, true, doReset);
    }

    private void cleanupIfNeeded()
    {
        File detectCleanupFile = new File(IContainer.CLEAN_RESTART_FILE);

        if (!detectCleanupFile.exists())
        {
            return;
        }

        detectCleanupFile.delete();

        //The operation is not supported before the 8.0 launcher
        if (IContainer.CURRENT_LAUNCHER_VERSION == null)
        {
            return;
        }

        logMessage("Clean restart was requests. Starting Working Directory cleanup...", Level.INFO);

        File wd = new File(CONTAINER_WORKING_DIR);
        String[] wdList = wd.list();

        for (int i = 0; i < wdList.length; i++)
        {
            if (wdList[i].endsWith(".sh") || wdList[i].endsWith(".bat") || wdList[i].endsWith(".log") || wdList[i].equals(ContainerSetup.CONTAINER_INI_FILE))
            {
                continue;
            }
            File fileToDelete = new File(wd, wdList[i]);
            LauncherInstallManager.deleteObsolete(fileToDelete);
            if (!fileToDelete.exists())
            {
                logMessage("Deleted \"" + fileToDelete.getAbsolutePath() + "\"", Level.INFO);
            }
            else
            {
                logMessage("Failed to delete \"" + fileToDelete.getAbsolutePath() + "\"", Level.WARNING);
            }
        }

        logMessage("...Cleanup is done", Level.INFO);
    }

    private boolean detectCacheResetRequest(File containerINIFile)
    {
        File detectCacheRestFile = new File(containerINIFile.getParentFile(), IContainer.RESET_CACHE_FILE);
        if (detectCacheRestFile.exists())
        {
            detectCacheRestFile.delete();
            return true;
        }
        else
        {
            return false;
        }
    }


    abstract public void initLogger(String logDirectoryPath, String logFileNamePrefix)
    throws Exception;

    private void getDSAccessContinuous(IAttributeSet configAttributes, Hashtable connParams)
    throws Exception
    {
        Exception priorException = null;

        boolean loggedMsg = false;
        while (true)
        {
            try
            {
                getRemoteDSAccess(configAttributes, connParams, !loggedMsg);
                if (priorException != null)
                {
                    logMessage("...retry successful", Level.INFO);
                }
                return;
            }
            catch (Exception e)
            {
                disconnectDS();

                if (ContainerUtil.isCausedByTimeout(e))
                {
                    int traceMask = m_connector == null ? 0 : m_connector.getTraceMask();
                    if ((traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                    {
                        if ((traceMask & IConnectorClient.TRACE_DETAIL) > 0)
                        {
                            m_connector.logMessage("Timeout while communicating with the Directory Service, retrying...", e, Level.TRACE);
                        }
                        else
                        {
                            m_connector.logMessage("Timeout while communicating with the Directory Service, retrying...", Level.TRACE);
                        }
                    }
                    else
                    {
                        boolean log = false;
                        if (priorException == null)
                        {
                            log = true;
                        }
                        else if (!e.getClass().isInstance(priorException))
                        {
                            log = true;
                        }
                        else if (priorException.getMessage() != null && !priorException.getMessage().equals(e.getMessage()))
                        {
                            log = true;
                        }
                        else if (e.getMessage() != null && !e.getMessage().equals(priorException.getMessage()))
                        {
                            log = true;
                        }
                        if (log)
                        {
                            logMessage("Timeout while communicating with the Directory Service, retrying...", Level.WARNING);
                        }
                    }
                    priorException = e;
                }
                else
                {
                    throw e;
                }
            }
        }
    }

    private void getReplicatedDSAccessContinuous(IAttributeSet configAttributes, Hashtable connParams)
    throws Exception
    {
        boolean loggedAccessMsg = false;
        boolean loggedActiveMsg = false;
        String peerRole = m_localDSParam.m_isBackup ? "PRIMARY" : "BACKUP";

        while (true)
        {
            if (getReplicatedLocalDS())
            {
                if (loggedAccessMsg)
                {
                    if (loggedActiveMsg)
                    {
                        logMessage("Failed to access " + peerRole + "; Transitioning to \"Active\" state", Level.INFO);
                    }

                    logMessage("Gained access to the local active Directory Service", Level.INFO);
                }
                return;
            }
            else
            {
                if (!loggedActiveMsg)
                {
                    logMessage("The " + peerRole + " Directory Service should be active; retrieving configuration from " + peerRole, Level.INFO);
                    loggedActiveMsg = true;
                }
            }

            try
            {
                getRemoteDSAccess(configAttributes, connParams, !loggedAccessMsg);
                if (loggedAccessMsg)
                {
                    logMessage("...retry successful", Level.INFO);
                }
                return;
            }
            catch (Exception e)
            {
                disconnectRemoteDS();

                if (ContainerUtil.isCausedByTimeout(e))
                {
                    if (!loggedAccessMsg)
                    {
                        logMessage("Timeout while communicating with the Directory Service, retrying...", Level.WARNING);
                        loggedAccessMsg = true;
                    }
                }
                else
                {
                    throw e;
                }
            }
        }
    }

    private void getRemoteDSAccess(IAttributeSet configAttributes, Hashtable connParams, boolean logInfo)
    throws Exception
    {
        if (connParams != null)
        {
            m_connector = createConnector(connParams, true, logInfo);
        }
        else
        {
            IAttributeSet connectionAttrsSet = null;
            IAttributeSet centralConnectionAttrsSet = (IAttributeSet)configAttributes.getAttribute(CENTRAL_CONNECTION_ATTR);
            if (centralConnectionAttrsSet != null)
            {
                connectionAttrsSet = centralConnectionAttrsSet;
                if (!m_requestToConfigureFromCache)
                {
                    System.setProperty(IContainer.GENERATE_CACHE_FROM_CENTRAL_CONNECTION_PROPERTY, "true");
                }
            }
            else
            {
                connectionAttrsSet = (IAttributeSet)configAttributes.getAttribute(IContainerConstants.CONNECTION_ATTR);
            }
            m_connector =  createConnector(new Hashtable(connectionAttrsSet.getAttributes()), false, logInfo);
        }
        createDSProxy(null, m_connector,  m_extractedConfig.m_domainName);
    }

    protected void connectAndGetConfiguration()
    throws Exception
    {
        // If we cannot extract the configuration from the DS we try from cache
        Exception configFromDSFailure = null;
        boolean recoverableFailure = false;
        try
        {
            if (m_localDSParam == null)
            {
                m_nonDMContainerWhichConnectToRemoteDS = true;
                recoverableFailure = true;
                if (m_requestToConfigureFromCache)
                {
                    throw new ConfigureFromCacheException();
                }
                // If the configuration is not yet in the cache than use getDSAccessContinuous to keep trying
                if (m_configCacheView.getElement(IContainer.CONFIG_ID_ELEMENT_PATH) != null)
                {
                    getRemoteDSAccess((m_initInfo instanceof XmlINI) ? m_initInfo.m_attributes : null, !(m_initInfo instanceof XmlINI) ? (m_initInfo.m_properties) : null, true);
                }
                else
                {
                    getDSAccessContinuous((m_initInfo instanceof XmlINI) ? m_initInfo.m_attributes : null, !(m_initInfo instanceof XmlINI) ? (m_initInfo.m_properties) : null);
                }
            }
            else
            {
                // PSE does not have access to the component context so use java.util.logging
                // to support container logging by PSE code
                m_pseLogger = new MFLogger(m_logger, "pse");

                if (m_localDSParam.m_isFaultTolerant.booleanValue())
                {
                    createReplicatedDSProxy(m_localDSParam);
                    getReplicatedDSAccessContinuous((m_initInfo instanceof XmlINI) ? m_initInfo.m_attributes : null, !(m_initInfo instanceof XmlINI) ? (m_initInfo.m_properties) : null);
                }
                else
                {
                    createDSProxy(m_localDSParam);
                }
            }

            recoverableFailure = true;
            if (m_initInfo instanceof XmlINI)
            {
                extractConfigFromDS(m_initInfo.m_containerConfID, null, null, false);
            }
            else
            {
                extractConfigFromDS(m_initInfo.m_config.m_containerPath, m_initInfo.m_config.m_activationDaemonPath, m_initInfo.m_config.m_hostManagerPath, true);
            }
        }
        catch (Error e)
        {
            //Not Recoverable
            m_extractedConfig = null;
            throw e;
        }
        catch (Exception e)
        {
            if (recoverableFailure)
            {
                configFromDSFailure = e;
                extractConfigFromCache();
                m_useCachedConfig = true;
                System.setProperty(IContainer.CONFIGURE_FROM_CACHE_PROPERTY, "true");
                System.setProperty(IContainer.GENERATE_CACHE_FROM_CENTRAL_CONNECTION_PROPERTY, "false");
            }
            else
            {
                m_extractedConfig = null;
                throw e;
            }
        }

        //If we managed to connect to the DS but the configuration is not there we do not even try
        // to get it from the cache - it's deleted and invalid
        String contName = (m_initInfo instanceof XmlINI) ? m_containerID : m_initInfo.m_config.m_containerPath;
        if (m_extractedConfig == null)
        {
            if (configFromDSFailure == null)
            {
                throw new Exception("Container configuration " + contName + " was not found in the Directory Service.");
            }
            else
            {
                throw configFromDSFailure;
            }
        }


        m_traceMask |= m_extractedConfig.m_deployTraceMask.intValue();

        if (m_extractedConfig.m_archiveSearchPath == null)
        {
            throw new Exception("The archive search path is missing.");
        }

        storeConfiguredArchivesInCache(m_extractedConfig.m_archiveSearchPath);

        // The configuration was obtained from cache, the caller will log the failure but will kepp going
        if (configFromDSFailure != null)
        {
            throw configFromDSFailure;
        }

    }

    private File openINIFile(String containerINIFileName)
    throws Exception
    {
        File containerINIFile = new File(containerINIFileName);
        if (!containerINIFile.exists() || !containerINIFile.isFile() || !containerINIFile.canRead())
        {
            String problem = !containerINIFile.exists() ?  " not found." : " cannot be read";
            String errMessage = "File " + containerINIFileName + problem;
            logMessage(errMessage, Level.SEVERE);
            throw new Exception(errMessage);
        }
        logMessage("Open container initialization file \"" + getCanonicalPath(containerINIFile) + '"', Level.INFO);
        return containerINIFile;
    }

    // Initialize from a inside a running container - already connect to a management broker
    // and needs to generate a boot file for the container we are starting
    protected void init(ILogger logger, String domainName, String containerName, String containerID, String containerWorkDirDirectoryName, String cacheHostDirectoryName, String cachePassword, boolean persistentCache)
    throws Exception
    {
        m_requestToConfigureFromCache = false;
        m_logger = logger;
        m_domainName = domainName;
        m_containerID = containerID;
        m_containerName = createContainerFullName(m_domainName, containerName);
        createCache(absoluteFromRelativeToWorkDir(containerWorkDirDirectoryName, cacheHostDirectoryName), cachePassword, persistentCache, false);
    }

    String getNativeLibDirectory()
    throws Exception
    {
        return m_configCache.getNativeLibDirectory().getAbsolutePath();
    }

    void getConfiguration(IRemoteMBeanServer connector, IDirectoryAdminService ds)
    throws Exception
    {
        logMessage("Fetching the resources of container \"" + m_containerName + "\"...", Level.INFO);

        // If we cannot extract the configuration from the DS we try from cache
        Exception configFromDSFailure = null;
        try
        {
            createDSProxy(ds, connector, m_domainName);
            extractConfigFromDS(m_containerID, null, null, false);
        }
        catch (Exception e)
        {
            configFromDSFailure = e;
            extractConfigFromCache();
        }

        //If we managed to connect to the DS but the configuration is not there we do not even try
        // to get it from the cache - it's deleted and invalid
        if (m_extractedConfig == null)
        {
            if (configFromDSFailure == null)
            {
                throw new Exception("Container configuration " + m_containerID + " was not found in the Directory Service");
            }
            else
            {
                throw configFromDSFailure;
            }

        }

        m_traceMask = m_extractedConfig.m_deployTraceMask.intValue();

        if (m_extractedConfig.m_archiveSearchPath == null)
        {
            throw new Exception("The archive search path is missing.");
        }

        storeConfiguredArchivesInCache(m_extractedConfig.m_archiveSearchPath);

        if (configFromDSFailure != null)
        {
            throw configFromDSFailure;
        }
    }

    protected ILogger getLogger()
    {
        return m_logger;
    }

    protected String getContainerName()
    {
        return m_containerName;
    }

    public String getDomainName()
    {
        return m_domainName;
    }

    protected File getArchiveRoot()
    {
        return m_archiveRoot;
    }

    protected File getMFDirectory()
    {
        return m_mfDirectoryJAR;
    }

    protected boolean mfDirectoryJARMissing()
    {
        return m_mfDirectoryJARMissing;
    }

    protected boolean mfDSLaunchFilesMissing()
    {
        return m_mfDirectoryJARMissing || m_dsBootFileMissing;
    }

    protected String getWorkingDir()
    {
        return CONTAINER_WORKING_DIR;
    }

    protected ExtractedContainerConfig getExtractedConfig()
    {
        return m_extractedConfig;
    }

    protected boolean useCachedConfig()
    {
        return m_useCachedConfig;
    }

    protected String getLogicalArchivePath()
    {
        return m_logicalArchivePath;
    }

    protected String getJVMHome()
    throws Exception
    {
        if (m_extractedConfig.m_jvmHome != null)
        {
            LogicalFile jvmHome = m_localFileManager.localFileFromLocation(null, m_extractedConfig.m_jvmHome, false, null);
                return jvmHome.getFileName();
        }
        else
        {
            return null;
        }
    }

    protected String getLogFilePath()
    {
        return m_logFilePath;
    }

    protected SystemProperties getSystemProps()
    {
        return m_systemProps;
    }

    protected CARClasspath getClasspath()
    {
        return m_classpath;
    }

    // Configured arguments in m_extractedConfig.m_jvmArgs overwrite platform dependent arguments in platformArgs
    protected void mergePlatformDependentArgs(HashMap platformArgs)
    {
        platformArgs.putAll(m_extractedConfig.m_jvmArgs);
        m_extractedConfig.m_jvmArgs = platformArgs;
    }

    private void copyXSDToWorkingDir(String xsdFileName, String workingDir)
    throws Exception
    {
        File outFile = new File(workingDir, xsdFileName);

        if (outFile.exists())
        {
            return;
        }

        BufferedOutputStream outStream = new BufferedOutputStream(new FileOutputStream(outFile));

        File srcFile = new File(xsdFileName);
        BufferedInputStream inStream = new BufferedInputStream(new FileInputStream(srcFile));

        int readIn = 0;
        while ((readIn = inStream.read()) != -1)
        {
            outStream.write(readIn);
        }
        outStream.close();

    }

    private boolean checkHomeIsWritable()
    {
        try
        {
            File testFile = new File(SONICSW_HOME, "_test_can_write_to_fs");
            new FileOutputStream(testFile).close();
            testFile.delete();
            return true;
        }
        catch (Exception e)
        {
            return false;
        }

    }


    // If deployResources() throws InvokeTimeoutException or ConnectTimeoutException - the caller could try again.
    // Or the caller will just rely on the timeout values and giveup the first time.
    protected void deployResources()
    throws Exception
    {
        String[] combinedList = mergeClasspathLists(m_systemProps.getSonicfsResources(), m_configClasspath.getSonicfsResources());
        new PopulateCacheResources(m_ds, m_configCache, combinedList, m_logger).updateCache();
    }

    protected void mergePlatformProps(HashMap platformPropsMap, boolean dsAccessible)
    throws Exception
    {
        SystemProperties platformProps = new SystemProperties(platformPropsMap, m_logger, true);
        String[] sonicfsResources = platformProps.getSonicfsResources();
        if (dsAccessible && sonicfsResources.length > 0)
        {
            try
            {
                logMessage("Fetching platform dependent resources of container \"" + m_containerName + "\".", Level.INFO);
                new PopulateCacheResources(m_ds, m_configCache, sonicfsResources, m_logger).updateCache();
            }
            catch (Throwable t)
            {
                // We don't throw this throwable - we'll try to start the container with resources refreshment
                logMessage("Resource refreshment failed - caused by:", Level.WARNING);
                logMessage("", t, Level.WARNING);
            }
        }
         platformProps.substituteSonicfsResources(getLocalFiles(platformProps.getSonicfsResources()));
         m_systemProps.mergePlatformDependentProps(platformProps.getProperties());
    }

    private void disconnectDS()
    {
        try
        {
            disconnectRemoteDS();

            //In the case of replication, a failure to close the local DS is likely since we might gain access to the remote peer
            // as we are opening the local store (probably in standby mode). Therefore we have to be prepared for any throwable
            // and continute gracefully
            if (m_localDirectoryService != null)
            {
                Method closeMethod = m_localDirectoryServiceClass.getMethod("close", new Class[0]);
                closeMethod.invoke(m_localDirectoryService, new Object[0]);
            }
        }
        catch (Throwable t)
        {
            if ((m_traceMask & LaunchContainer.CONTAINER_LAUNCH_TRACE_MASK) != 0)
            {
                logMessage("Unsuccessful disconnect of local Directory Service after updating the cache of container " + m_extractedConfig.m_containerName + " ID: " + m_containerID, getTargetException(t), Level.TRACE);
            }
        }
    }

    // If we crated the connector here (as opposed to using an existing connector) then disconnect
    private void disconnectRemoteDS()
    {
        try
        {
            if (m_connector != null)
            {
                m_connector.disconnect();
            }
        }
        catch (Exception e)
        {
            logMessage("Unsuccessful disconnect after updating the cache of container " + m_extractedConfig.m_containerName + " ID: " + m_containerID, getTargetException(e), Level.INFO);
        }
    }

    private Throwable getTargetException(Throwable e)
    {
        if (!(e instanceof InvocationTargetException))
        {
            return e;
        }

        Throwable targetException = ((InvocationTargetException)e).getTargetException();
        if (targetException == null)
        {
            targetException = e;
        }

        if (targetException instanceof Exception)
        {
            return (Exception)targetException;
        }
        else
        {
            return new Exception(targetException.toString());
        }

    }

    private void createClasspathLists()
    throws Exception
    {
        m_systemProps = new SystemProperties(m_extractedConfig.m_systemProps, m_logger, false);
        m_configClasspath = new DSClasspath(m_extractedConfig.m_configClasspath, m_logger);
    }

    protected void releaseResources()
    {
         closeCache();
         disconnectDS();
    }

    private void closeCache()
    {
        try
        {
            if (m_configCache != null)
            {
                m_configCache.close();
            }
            m_configCache = null;
        }
        catch (PersistentCacheException e)
        {
            logMessage("Failed to close the cache, caused by: ", e, Level.WARNING);
        }
    }

    private static String absoluteFromRelativeToWorkDir(String containerWorkingDirName, String dirName)
    {
        if (containerWorkingDirName == null || dirName == null ||  new File(dirName).isAbsolute())
        {
            return dirName;
        }
        else
        {
            return getCanonicalPath(new File(containerWorkingDirName, dirName));
        }
    }

    // getCanonicalPath could, in theory, fail on some systems. We use getAbsolutePath
    // if getCanonicalPath fails. The only drawback is "uglier" log messages
    static String getCanonicalPath(File file)
    {
        try
        {
            return file.getCanonicalPath();
        }
        catch (Exception e)
        {
            return file.getAbsolutePath();
        }
    }

    static String getCanonicalPath(String fileName)
    {
        return getCanonicalPath(new File(fileName));
    }

    private void createCache(String cacheHostDirectoryName, String cachePassword, boolean persistentCache, boolean reset)
    throws Exception
    {

        m_cacheHostDirectoryName = cacheHostDirectoryName;

        //if (!persistentCache) ignore - we awlays have persistent cache

        // If the cache is corrupt then remove it
        ConfigCache.cleanupCorruptCache(cacheHostDirectoryName, m_logger);
        String cacheName = getCanonicalPath(new File(cacheHostDirectoryName));
        logMessage("Cache opened in directory \"" + cacheName + '"', Level.INFO);
        // We don't bother passing a maximum-persistent-cache-size value - the launched container will take care
        // of it later.
        if (reset)
        {
            m_configCache = ConfigCacheFactory.createNewCache(cacheHostDirectoryName, cachePassword);
        }
        else
        {
            m_configCache = ConfigCacheFactory.createCache(cacheHostDirectoryName, cachePassword);
        }

        m_configCacheView = m_configCache.getCacheView();
    }

    // Return the logical names of the file if it's a DS file so we can refresh it together with other resources
    private void storeConfiguredArchivesInCache(String searchPath)
    throws Exception
    {
        // We'll throw an exception later
        if (m_extractedConfig == null)
        {
            return;
        }

        if (m_extractedConfig.m_archiveName == null || m_extractedConfig.m_archiveName.length() == 0)
        {
            m_extractedConfig = null;
            throw new Exception("The " + IContainerConstants.ARCHIVE_NAME_ATTR + " attribute is missing from the container's configuration.");
        }
        
        if (m_extractedConfig.m_actionalSDKArchiveName == null || m_extractedConfig.m_actionalSDKArchiveName.length() == 0)
        {
            m_extractedConfig = null;
            throw new Exception("The " + IContainerConstants.ACTIONAL_SDK_ARCHIVE_NAME_ATTR + " attribute is missing from the container's configuration.");
        }
        
        m_localFileManager = new LocalFileManager(m_ds, m_configCache, m_logger, m_containerID);
        m_localFileManager.setTraceMask(m_traceMask);
        String[] tmp = new String[1];
        tmp[0] = null;
        
        m_archiveRoot = m_localFileManager.getLocalFile(searchPath, m_extractedConfig.m_archiveName, tmp);
        if (m_archiveRoot == null)
        {
            throw new Exception("Could not find \"" + m_extractedConfig.m_archiveName + "\" using search path \"" + searchPath + "\"");
        }
        m_dsArchivePath = tmp[0];

        m_logicalArchivePath = m_extractedConfig.m_archiveName;
        if  (!m_logicalArchivePath.startsWith("/"))
        {
            m_logicalArchivePath = "/" + m_logicalArchivePath;
        }
        
        m_actionalSDKArchive = m_localFileManager.getLocalFile(searchPath, m_extractedConfig.m_actionalSDKArchiveName, new String[1]);
        if (m_actionalSDKArchive == null)
        {
            String actionalSDKArchiveName = m_extractedConfig.m_actionalSDKArchiveName;
            m_extractedConfig = null;
            throw new Exception(actionalSDKArchiveName + " could not be found on the container's archive search path");
        }
        
        if (m_extractedConfig.m_actionalPlugMakerArchiveName != null && m_extractedConfig.m_actionalPlugMakerArchiveName.length() > 0)
        {
            m_actionalPlugMakerArchive = m_localFileManager.getLocalFile(searchPath, m_extractedConfig.m_actionalPlugMakerArchiveName, new String[1]);
        }
    }

    private void extractConfigFromDS(String containerID, String activationDaemonPath, String hostManagerPath, boolean logicalPath)
    throws Exception
    {
       m_extractedConfig = getContainerConfigurationFromDS(containerID, activationDaemonPath, hostManagerPath, logicalPath);
       if (m_extractedConfig != null)
    {
        createClasspathLists();
    }
    }

    private void extractConfigFromCache()
    throws Exception
    {
        m_extractedConfig = getContainerConfigurationFromCache();
        if (m_extractedConfig != null)
        {
            createClasspathLists();
        }
    }

    private ExtractedContainerConfig getContainerConfigurationFromCache()
    throws Exception
    {
        IElement configIDElement = m_configCacheView.getElement(IContainer.CONFIG_ID_ELEMENT_PATH);
        if (configIDElement == null)
        {
            return null;
        }
        IAttributeSet attributes = configIDElement.getAttributes();
        m_containerID = (String)attributes.getAttribute(IContainer._MF_CONFIG_ID_ATTR);

        IElement containerConfig = m_configCacheView.getElement(m_containerID);

        if (containerConfig == null)
        {
            return null;
        }

        m_useCachedConfig = true;
        return extractConfigInfo(containerConfig.getAttributes());
    }

    private ExtractedContainerConfig getContainerConfigurationFromDS(String containerPathOrID, String activationDaemonPath, String hostManagerPath, boolean logicalPath)
    throws Exception
    {
        if (logicalPath ||  (m_traceMask & LaunchContainer.CONTAINER_LAUNCH_TRACE_MASK) != 0)
        {
            logMessage("Getting the container configuration \"" + containerPathOrID + "\" from the Directory Service", Level.INFO);
        }

        IElement containerConfig = null;
        if (logicalPath)
        {
            // First check if this container already exists
            IElement[] elements = retryGetElementsByLogicalNames(new String[] { containerPathOrID });
            containerConfig = elements.length > 0 ? elements[0] : null;

            boolean cannotCreateNewConfig = !m_extractedConfig.m_createIfDoesNotExist.booleanValue();

            if (containerConfig == null)
            {
                if (cannotCreateNewConfig)
                {
                    throw new Exception("Could not find the configuration of container \"" + containerPathOrID + "\" and there is no CREATE_IF_DOES_NOT_EXIST=true setup in file \"" + m_containerINIFileName + "\" to allow automatic creation.");
                }

                if (m_extractedConfig.m_hostsDS.booleanValue())
                {
                    throw new Exception("Could not find the configuration of container \"" + containerPathOrID + "\" - cannot create a container that hosts the Directory Service.");
                }

                IElement[] contConfigurations = ((DirectoryServiceProxy)m_ds).getCreateContainerConfiguration(containerPathOrID, activationDaemonPath, hostManagerPath, createDefaultContainerParameters(m_extractedConfig), m_connectionAttributes);
                containerConfig = contConfigurations[0];
            }

            if (m_dsBootFileMissing)
            {
                getBootDSXML(containerConfig);
            }

            m_containerID = containerConfig.getIdentity().getName();

            if (elements.length == 0 || elements[0] == null)
            {
                logMessage("Created a new empty container \"" + containerPathOrID + "\" configuration ID \"" + m_containerID + '"', Level.INFO);
            }
        }
        else
        {
            // Find out whether the cached configuration is up to date.
            Long tmp = m_configCacheView.getDSBackupVersion();
            long cacheBackupVersion = tmp == null ? 0 : tmp.longValue();
            containerConfig = m_configCacheView.getElement(containerPathOrID);
            IElement[] retArray = retryGetElementIfUpdated(cacheBackupVersion, containerPathOrID, containerConfig != null ? containerConfig.getIdentity() : null);
            if (retArray != null)
            {
                containerConfig = retArray[0];
            }

            if (containerConfig == null)
            {
                throw new Exception("Could not find the configuration of container \"" + containerPathOrID + "\".");
            }
            m_containerID = containerPathOrID;
        }

        IElement dsConfig = null;
        if (m_localDSParam != null)
        {
            if ((m_traceMask & LaunchContainer.CONTAINER_LAUNCH_TRACE_MASK) != 0)
            {
                logMessage("Getting the Directory Service configuration \"" + m_localDSParam.m_configID + "\" from the Directory Service", Level.TRACE);
            }
            dsConfig =  m_ds.getElement(m_localDSParam.m_configID, false);
        }
        cacheConfiguration(containerConfig, dsConfig);

        m_useCachedConfig = false;
        return extractConfigInfo(containerConfig.getAttributes());
    }

    private void getBootDSXML(IElement containerConfig) throws Exception
    {
        String dsElementName = ContainerUtil.extractDSConfigIDFromContainerConfig(containerConfig);
        if (dsElementName == null)
        {
            throw new Exception("The \" " + IFSStorage.HOST_DIRECTORY_ATTRIBUTE + "\" attribute is configured in \"" + m_containerName + "\" but Directory Service component was not found");
        }

        String dsBootContent = m_ds.exportDSBootFileString(dsElementName);
        ContainerUtil.encryptAndStore(dsBootContent, IContainer.DEFAULT_DS_BOOTFILE_NAME, IContainer.PASSWORD);
        logMessage("...Retrieved the missing information and generated the " + IContainer.DEFAULT_DS_BOOTFILE_NAME + " file", Level.INFO);
    }


    private IDirElement[] retryGetElementIfUpdated(long callerBackupVersion, String elementName, IElementIdentity id) throws Exception
    {
        try
        {
            return m_ds.getElementIfUpdated(callerBackupVersion, elementName, id);
        }
        catch (Exception e)
        {
            if (ContainerUtil.isCausedByTimeout(e) && m_nonDMContainerWhichConnectToRemoteDS)
            {
                // if the configuration is in the cache then no need to retry, we'll get later from cache
                if(m_configCacheView.getElement(IContainer.CONFIG_ID_ELEMENT_PATH) != null)
                {
                    throw e;
                }

                // don't log the first failure because it would create inconsistent logging, instead leave the
                // logging to the DS proxy if it fails due to a timeout

                return ((DirectoryServiceProxy)m_ds).retryGetElementIfUpdated(callerBackupVersion, elementName, id);
            }
            else
            {
                throw e;
            }
        }
    }

    private IElement[] retryGetElementsByLogicalNames(String[] containerPathOrID) throws Exception
    {
        try
        {
            return m_ds.getElementsByLogicalNames(containerPathOrID);
        }
        catch (Exception e)
        {
            if (ContainerUtil.isCausedByTimeout(e) && m_nonDMContainerWhichConnectToRemoteDS)
            {
                // if the configuration is in the cache then no need to retry, we'll get later from cache
                if(m_configCacheView.getElement(IContainer.CONFIG_ID_ELEMENT_PATH) != null)
                {
                    throw e;
                }

                // don't log the first failure because it would create inconsistent logging, instead leave the
                // logging to the DS proxy if it fails due to a timeout

                return ((DirectoryServiceProxy)m_ds).retryGetElementsByLogicalNames(containerPathOrID);
            }
            else
            {
                throw e;
            }
        }
    }

    // Store the container configuration in the cache.
    // And store the configuration ID of the container in a well known place so that later
    // we have access to it when the only thing we know is the cache. We also use this element
    // To save in cache any extra information that is not part of the container configuration,
    // but derived from it, and will not be accessible if the DS is not accessible.
    private void cacheConfiguration(IElement containerConfig, IElement dsConfig)
    throws Exception
    {
        // Store the container configuration
        m_configCache.setElement(containerConfig);
        if (dsConfig != null)
        {
            boolean isDSFT = false;
            boolean isBackup = false;

            m_configCache.setElement(dsConfig);
            // we must cache other configurations needed to boot the DS (replication connections and the primary DS)
            IAttributeSet dsTopSet = dsConfig.getAttributes();
            IAttributeSet refs = (IAttributeSet)dsTopSet.getAttribute(IDirectoryServiceConstants.CONFIG_ELEMENT_REFERENCES_ATTR);
            if (refs != null)
            {
                // if this is the PRIMARY DS, it will contain the replication connections
                Reference connectionsRef = (Reference)refs.getAttribute(IDirectoryServiceConstants.REPLICATION_CONNECTIONS_ELEMENT_REF_ATTR);
                if (connectionsRef != null)
                {
                    isDSFT = true;
                    isBackup = false;
                    IElement connectionsEl = m_ds.getElement(connectionsRef.getElementName(), false);

                    m_configCache.setElement(connectionsEl);
                }
                Reference primaryRef = (Reference)refs.getAttribute(IBackupDirectoryServiceConstants.PRIMARY_CONFIG_ELEMENT_REF_ATTR);
                if (primaryRef != null)
                {
                    isDSFT = true;
                    isBackup = true;

                    // if this is a backup, it has a reference to the primary and that must be cached. Also
                    // replication connections element must be cached from the primary
                    IElement primaryDS = m_ds.getElement(primaryRef.getElementName(), false);
                    m_configCache.setElement(primaryDS);
                    IAttributeSet primaryAttrs = primaryDS.getAttributes();
                    IAttributeSet primaryRefs = (IAttributeSet)primaryAttrs.getAttribute(IDirectoryServiceConstants.CONFIG_ELEMENT_REFERENCES_ATTR);
                    if (primaryRefs != null)
                    {
                        // if this is the PRIMARY DS, it will contain the replication connections
                        Reference primaryConnectionsRef = (Reference)primaryRefs.getAttribute(IDirectoryServiceConstants.REPLICATION_CONNECTIONS_ELEMENT_REF_ATTR);
                        if (primaryConnectionsRef != null)
                        {
                            IElement primaryConnectionsEl = m_ds.getElement(primaryConnectionsRef.getElementName(), false);

                            m_configCache.setElement(primaryConnectionsEl);
                        }
                    }
                }
            }

            if ((m_localDSParam.m_isFaultTolerant.booleanValue() != isDSFT) || (isDSFT && (m_localDSParam.m_isBackup != isBackup)))
            {
                String errorString = "The Directory Service boot file doesn't match the Directory Service configuration. ";
                errorString += getFTMismatchString("Boot file:", m_localDSParam.m_isFaultTolerant.booleanValue(), m_localDSParam.m_isBackup) +
                               getFTMismatchString(" Configuration:", isDSFT, isBackup);

                throw new Error(errorString);
            }
        }

        IDirElement configIDElement = ElementFactory.createElement(IContainer.CONFIG_ID_ELEMENT_PATH, IContainer._MF_CI_CONFIG_ID_TYPE, IContainerConstants.DS_C_VERSION);
        IAttributeSet attributes = configIDElement.getAttributes();
        attributes.setStringAttribute(IContainer._MF_CONFIG_ID_ATTR, m_containerID);
        m_configCache.deleteElement(IContainer.CONFIG_ID_ELEMENT_PATH);
        m_configCache.setElement(configIDElement.doneUpdate());
    }

    private static String getFTMismatchString(String label, boolean ft, boolean isBackup)
    {
        return label + (ft ? "Fault Tolerant" : "Non Fault Tolerant") + (ft ? (isBackup ? " Role BACKUP. " : " Role PRIMARY. ") : ".") ;
    }

    private void createDSProxy(IDirectoryAdminService ds, IRemoteMBeanServer connector, String domainName)
    throws Exception
    {
        if (ds != null)
        {
            m_ds = ds;
        }
        else
        {
            m_ds = new DirectoryServiceProxy(connector, new ObjectName(domainName + "." + IContainer.DS_ADDRESS));
        }

        try
        {
            m_ds.getDirectoryServiceVersion(); // Verify we have DS access
        }
        catch (Exception e)
        {
            // If e is from timeout and non DM container then no need to complain; we'll complain later if we fail to get the configuration from DS and from cache
            if (!ContainerUtil.isCausedByTimeout(e) || !m_nonDMContainerWhichConnectToRemoteDS)
            {
                throw e;
            }
        }
    }

    // Use reflection since The DirectoryService classes could be absent in a deployment
    private void createDSProxy(LocalDSParameters dsParam)
    throws Exception
    {
        m_localDirectoryServiceClass = Class.forName("com.sonicsw.mf.framework.agent.ci.LocalDirectoryService");
        m_localDirectoryService = m_localDirectoryServiceClass.newInstance();

        // package the properties as a Hashtable
        Hashtable directoryEnv = new Hashtable();
        directoryEnv.put("_TRACE_MASK", new Integer(m_traceMask));
        if (dsParam.m_hostDir != null)
        {
            directoryEnv.put(IFSStorage.HOST_DIRECTORY_ATTRIBUTE, dsParam.m_hostDir);
        }
        if ((dsParam.m_encryptionPassword != null) && (dsParam.m_encryptionPassword.length() != 0))
        {
            directoryEnv.put(IFSStorage.PASSWORD_ATTRIBUTE, dsParam.m_encryptionPassword);
        }

        try
        {
            Method getDSMethod = m_localDirectoryServiceClass.getMethod("getDS", new Class[]
            { ILogger.class, String.class, Hashtable.class });
            m_ds = (IDirectoryAdminService)getDSMethod.invoke(m_localDirectoryService, new Object[]
            { m_logger, dsParam.m_domainName, directoryEnv });
        }
        catch (Exception e)
        {
            Throwable t = getTargetException(e);
            if (t instanceof Exception)
            {
                throw (Exception)t;
            }
            else
            {
                throw new Exception("getDS failed with: " + t.toString());
            }
        }

    }

    // Use reflection since The DirectoryService classes could be absent in a deployment
    private void createReplicatedDSProxy(LocalDSParameters dsParam)
    throws Exception
    {
        m_localDirectoryServiceClass = Class.forName("com.sonicsw.mf.framework.agent.ci.LocalDirectoryService");
        m_localDirectoryService = m_localDirectoryServiceClass.newInstance();

        // package the properties as a Hashtable
        Hashtable directoryEnv = new Hashtable();
        directoryEnv.put("_TRACE_MASK", new Integer(m_traceMask));
        if (dsParam.m_hostDir != null)
        {
            directoryEnv.put(IFSStorage.HOST_DIRECTORY_ATTRIBUTE, dsParam.m_hostDir);
        }
        if ((dsParam.m_encryptionPassword != null) && (dsParam.m_encryptionPassword.length() != 0))
        {
            directoryEnv.put(IFSStorage.PASSWORD_ATTRIBUTE, dsParam.m_encryptionPassword);
        }

        directoryEnv.put(IDirectoryServiceConstants.RETRY_INTERVAL_ATTR, dsParam.m_replicationRetryInterval);
        directoryEnv.put(IDirectoryServiceConstants.PING_INTERVAL_ATTR, dsParam.m_replicationPingInterval);
        directoryEnv.put(IDirectoryServiceConstants.FAILURE_DETECTION_TIMEOUT_ATTR, dsParam.m_replicationFailureDetectionTimeout);
        directoryEnv.put(IDirectoryServiceConstants.MAX_REPLICATION_LOG_SIZE_ATTR, dsParam.m_maxReplicationLogSize);
        directoryEnv.put(IDirectoryServiceConstants.REPLICATION_TIMEOUT_ATTR, dsParam.m_replicationTimeout);
        directoryEnv.put(IDirectoryServiceConstants.DUAL_ACTIVE_RESOLUTION_ATTR, dsParam.m_dualActiveResolution);
        directoryEnv.put(IReplicationConnectionConstants.REPLICATION_CONNECTIONS_ATTR, dsParam.m_replConnections);
        if (dsParam.m_replSSLParams != null)
        {
            directoryEnv.put(IDirectoryServiceConstants.SSL_PARAMETERS_ATTR, dsParam.m_replSSLParams);
        }

        try
        {
            Method openStorageMethod = m_localDirectoryServiceClass.getMethod("openReplicatedStorage", new Class[]
            { ILogger.class, String.class, Hashtable.class, Boolean.class, Boolean.class });
            openStorageMethod.invoke(m_localDirectoryService, new Object[]
            { m_logger, dsParam.m_domainName, directoryEnv, dsParam.m_isBackup ? Boolean.FALSE : Boolean.TRUE, dsParam.m_startActive });
        }
        catch (Throwable e)
        {
            Throwable t = getTargetException(e);
            if (t instanceof Exception)
            {
                throw (Exception)t;
            }
            else
            {
                throw new Exception("openReplicatedStorage failed with: " + t.toString());
            }
        }

    }

    private boolean getReplicatedLocalDS()
    throws Exception
    {
        try
        {
            Method getReplicatedDSMethod = m_localDirectoryServiceClass.getMethod("getReplicatedDS", EMPTY_CLASS_ARRAY);
            m_ds = (IDirectoryAdminService)getReplicatedDSMethod.invoke(m_localDirectoryService, EMPTY_OBJECT_ARRAY);
            return m_ds != null;
        }
        catch (Throwable e)
        {
            Throwable t = getTargetException(e);
            if (t instanceof Exception)
            {
                throw (Exception)t;
            }
            else
            {
                throw new Exception("getreplicatedDS failed with: " + t.toString());
            }
        }

    }

    private ExtractedContainerConfig extractConfigInfo(IAttributeSet containerAttrs)
    throws Exception
    {
        ExtractedContainerConfig containerConfig = new ExtractedContainerConfig();
        containerConfig.m_logFileName = (String)containerAttrs.getAttribute(IContainerConstants.LOG_FILE_ATTR);

        containerConfig.m_domainName = (String)containerAttrs.getAttribute(IContainerConstants.DOMAIN_NAME_ATTR);
        if (containerConfig.m_domainName == null)
        {
            containerConfig.m_domainName = IContainerConstants.DOMAIN_NAME_DEFAULT;
        }

        containerConfig.m_hostsDS = (Boolean)containerAttrs.getAttribute(IContainerConstants.HOSTS_DIRECTORY_SERVICE_ATTR);
        if (containerConfig.m_hostsDS == null)
        {
            containerConfig.m_hostsDS = new Boolean(IContainerConstants.HOSTS_DIRECTORY_SERVICE_DEFAULT);
        }

        containerConfig.m_containerName = (String)containerAttrs.getAttribute(IContainerConstants.CONTAINER_NAME_ATTR);
        containerConfig.m_actionalSDKArchiveName = (String)containerAttrs.getAttribute(IContainerConstants.ACTIONAL_SDK_ARCHIVE_NAME_ATTR);
        containerConfig.m_actionalPlugMakerArchiveName = (String)containerAttrs.getAttribute(IContainerConstants.ACTIONAL_PLUGMAKER_ARCHIVE_NAME_ATTR);
        containerConfig.m_configClasspath = (String)containerAttrs.getAttribute(IContainerConstants.CLASSPATH_ATTR);
        containerConfig.m_archiveSearchPath = (String)containerAttrs.getAttribute(IContainerConstants.ARCHIVE_SEARCH_PATH_ATTR);
        containerConfig.m_jvmHome = (String)containerAttrs.getAttribute(IContainerConstants.JVM_HOME_ATTR);
        containerConfig.m_archiveName = (String)containerAttrs.getAttribute(IContainerConstants.ARCHIVE_NAME_ATTR);
        containerConfig.m_configNativePath = (String)containerAttrs.getAttribute(IContainerConstants.CONTAINER_NATIVE_LIBRARY_PATH_ATTR);

        containerConfig.m_deployTraceMask = (Integer)containerAttrs.getAttribute(IContainerConstants.TRACE_MASK_ATTR);
        if (containerConfig.m_deployTraceMask == null)
        {
            containerConfig.m_deployTraceMask = new Integer(IContainerConstants.TRACE_MASK_DEFAULT);
        }

        IAttributeSet cacheAttrs = (IAttributeSet)containerAttrs.getAttribute(IContainerConstants.CACHE_ATTR);

        if (cacheAttrs != null)
        {
            containerConfig.m_cacheHostDirectoryName = (String)cacheAttrs.getAttribute(IContainerConstants.CACHE_DIRECTORY_ATTR);
        }
        if (containerConfig.m_cacheHostDirectoryName == null  || containerConfig.m_cacheHostDirectoryName.length() == 0 || containerConfig.m_cacheHostDirectoryName.equals("."))
        {
            containerConfig.m_cacheHostDirectoryName = IContainerConstants.CACHE_DIRECTORY_DEFAULT;
        }
        if (cacheAttrs != null)
        {
            containerConfig.m_cachePassword = (String)cacheAttrs.getAttribute(IContainerConstants.PASSWORD_ATTR);
        }

        Object tmp = containerAttrs.getAttribute(IContainerConstants.JVM_ARGUMENTS_ATTR);
        if (tmp == null || tmp instanceof String)
        {
             containerConfig.m_jvmArgs = new HashMap();
             if (tmp != null)
            {
                containerConfig.m_jvmArgs.put(CONFIGURED_JVM_OPTIONS, tmp);
            }

        }
        else if (tmp instanceof IAttributeSet)
        {
            containerConfig.m_jvmArgs = ((IAttributeSet)tmp).getAttributes();
        }

        IAttributeSet systemProps = (IAttributeSet)containerAttrs.getAttribute(IContainerConstants.SYSTEM_PROPERTIES_ATTR);
        if (systemProps != null)
        {
            HashMap map = systemProps.getAttributes();
            SystemProperties.connectionPropertiesFilter(map, m_centralConnectionUsed);
            containerConfig.m_systemProps = map;
        }
        else
        {
            containerConfig.m_systemProps = new HashMap();
        }
        return containerConfig;
    }

    // Used for the creation of a new container if the one specified in the INI file does not exist
    private Hashtable createDefaultContainerParameters(ExtractedContainerConfig containerConfig)
    {
        Hashtable containerParams = new Hashtable();
        containerParams.put(IContainerConstants.DOMAIN_NAME_ATTR, containerConfig.m_domainName);
        containerParams.put(IContainerConstants.CONTAINER_NAME_ATTR, containerConfig.m_containerName);
        if (containerConfig.m_logFileName != null)
        {
            containerParams.put(IContainerConstants.LOG_FILE_ATTR, containerConfig.m_logFileName);
        }
        containerParams.put(IContainerConstants.CACHE_DIRECTORY_ATTR, containerConfig.m_cacheHostDirectoryName);
        if (containerConfig.m_cachePassword != null)
        {
            containerParams.put(IContainer.CACHE_PASSWORD_ATTR, containerConfig.m_cachePassword);
        }

        return containerParams;
    }

    private ExtractedContainerConfig extractINIInfo(Properties containerAttrs)
    throws Exception
    {
        ExtractedContainerConfig containerConfig = new ExtractedContainerConfig();

        containerConfig.m_logFileName = (String)containerAttrs.get(IContainerConstants.LOG_FILE_ATTR);

        containerConfig.m_domainName = (String)containerAttrs.get(IContainerConstants.DOMAIN_NAME_ATTR);
        if (containerConfig.m_domainName == null)
        {
            containerConfig.m_domainName = IContainerConstants.DOMAIN_NAME_DEFAULT;
        }

        containerConfig.m_hostsDS = new Boolean((String)containerAttrs.get(IContainerConstants.HOSTS_DIRECTORY_SERVICE_ATTR));
        if (containerConfig.m_hostsDS == null)
        {
            containerConfig.m_hostsDS = new Boolean(IContainerConstants.HOSTS_DIRECTORY_SERVICE_DEFAULT);
        }

        containerConfig.m_containerPath = (String)containerAttrs.get(IContainer.CONTAINER_PATH_ATTR);
        if (containerConfig.m_containerPath != null)
        {
            EntityName pathE = new EntityName(containerConfig.m_containerPath);
            containerConfig.m_containerName = pathE.getBaseName();
        }
        containerConfig.m_activationDaemonPath = (String)containerAttrs.get(IContainer.ACTIVATION_DAEMON_PATH_ATTR);

        containerConfig.m_hostManagerPath = (String)containerAttrs.get(IContainer.HOST_MANAGER_PATH_ATTR);

        String maskString = (String)containerAttrs.get(IContainerConstants.TRACE_MASK_ATTR);
        if (maskString != null)
        {
            containerConfig.m_deployTraceMask = new Integer(maskString);
        }
        else
        {
            containerConfig.m_deployTraceMask = new Integer(IContainerConstants.TRACE_MASK_DEFAULT);
        }

        containerConfig.m_cacheHostDirectoryName = (String)containerAttrs.get(IContainerConstants.CACHE_DIRECTORY_ATTR);
        if (containerConfig.m_cacheHostDirectoryName == null || containerConfig.m_cacheHostDirectoryName.length() == 0 || containerConfig.m_cacheHostDirectoryName.equals("."))
        {
            containerConfig.m_cacheHostDirectoryName = IContainerConstants.CACHE_DIRECTORY_DEFAULT;
        }

        String stValue = (String)containerAttrs.get(IContainer.CREATE_IF_DOES_NOT_EXIST_ATTR);
        if (stValue == null)
        {
            stValue = "false";
        }
        containerConfig.m_createIfDoesNotExist = new Boolean(stValue);

        containerConfig.m_cachePassword = (String)containerAttrs.get(IContainer.CACHE_PASSWORD_ATTR);
        return containerConfig;
    }

    private Hashtable filterINIPrefixes(Hashtable table, boolean fromINI)
    throws Exception
    {
        String connectionPrefix = IContainer.CONNECTION_PROP_PREFIX;
        int connectionPrefixLength = CONNECTION_PROP_PREFIX_LENGTH;
        // If central connection is specified then use that
        if (fromINI)
        {
            Enumeration keys = table.keys();
            while (keys.hasMoreElements())
            {
                String key = (String)keys.nextElement();

                if (key.regionMatches(true, 0, IContainer.CENTRAL_CONNECTION_PROP_PREFIX, 0, CENTRAL_CONNECTION_PROP_PREFIX_LENGTH))
                {
                    connectionPrefix = IContainer.CENTRAL_CONNECTION_PROP_PREFIX;
                    connectionPrefixLength = CENTRAL_CONNECTION_PROP_PREFIX_LENGTH;
                    if (!m_requestToConfigureFromCache)
                    {
                        System.setProperty(IContainer.GENERATE_CACHE_FROM_CENTRAL_CONNECTION_PROPERTY, "true");
                    }
                    break;
                }
            }
        }

        Hashtable connectionFactoryAttributes = new Hashtable();
        Enumeration keys = table.keys();

        while (keys.hasMoreElements())
        {
            String key = (String)keys.nextElement();
            // Remove the prefix and put in the connection attributes table
            if (fromINI && key.regionMatches(true, 0, connectionPrefix, 0, connectionPrefixLength))
            {
                String stringValue = (String)table.get(key);
                String keyName = key.substring(connectionPrefixLength);
                Object value = stringValue;
                if (keyName.equals(IContainerConstants.REQUEST_TIMEOUT_ATTR) ||
                    keyName.equals(IContainerConstants.CONNECT_TIMEOUT_ATTR) ||
                    keyName.equals(IContainerConstants.SOCKET_CONNECT_TIMEOUT_ATTR))
                {
                    value = new Integer(stringValue);
                }

                connectionFactoryAttributes.put(keyName, value);
            }
            else if (!fromINI)
            {
                connectionFactoryAttributes.put(key,  table.get(key));
            }

        }

        return connectionFactoryAttributes;
    }

    private LaunchConnector createConnector(Hashtable connectionAttrs0, boolean fromINI, boolean logInfo)
    throws Exception
    {
        Hashtable connectionFactoryEnv = filterINIPrefixes(connectionAttrs0, fromINI);

        boolean validateNodeName = substituteGenerateCacheConnectionAttrs(connectionFactoryEnv);

        if (connectionFactoryEnv.get(IContainerConstants.CONNECTIONURLS_ATTR) == null)
        {
            connectionFactoryEnv.put(IContainerConstants.CONNECTIONURLS_ATTR, IContainer.CONNECTIONURLS_DEFAULT);
        }
        m_connectionAttributes = (Hashtable)connectionFactoryEnv.clone();
        connectionFactoryEnv.remove(IContainerConstants.MANAGEMENT_NODE_ATTR);
        connectionFactoryEnv.remove(IContainerConstants.REQUEST_TIMEOUT_ATTR);
        connectionFactoryEnv.remove(IContainerConstants.CONNECT_TIMEOUT_ATTR);
        connectionFactoryEnv.remove(IContainerConstants.SOCKET_CONNECT_TIMEOUT_ATTR);
        connectionFactoryEnv.remove(IContainerConstants.LOADBALANCING_ATTR); //Not consequential during the first phase - just use the default

        // deprecated feature
        connectionFactoryEnv.remove("TRY_DS_AM_BACKUPS_ON_FAILURE");

        JMSConnectorAddress address = new JMSConnectorAddress(connectionFactoryEnv);

        String managementNode = (String)m_connectionAttributes.get(IContainerConstants.MANAGEMENT_NODE_ATTR);
        if (managementNode != null && managementNode.length() > 0)
        {
            address.setManagementNode(managementNode);
        }

        CanonicalName canonicalName = new CanonicalName(m_containerName);
        ContainerIdentity identity = new ContainerIdentity(canonicalName, null);
        LaunchConnector connector = new LaunchConnector(identity, m_logger);
        connector.setTraceMask(m_traceMask);

        // REQUEST_TIMEOUT_ATTR is in seconds, setRequestTimeout is passed millis
        Integer requestTimeout = (Integer)m_connectionAttributes.get(IContainerConstants.REQUEST_TIMEOUT_ATTR);
        if (requestTimeout != null)
        {
            connector.setRequestTimeout(requestTimeout.intValue() * 1000);
        }

        // CONNECT_TIMEOUT_ATTR is in seconds, setConnectTimeout is passed millis
        Integer connectTimeout = (Integer)m_connectionAttributes.get(IContainerConstants.CONNECT_TIMEOUT_ATTR);
        if (connectTimeout != null)
        {
            connector.setConnectTimeout(connectTimeout.intValue() * 1000);
        }

        // SOCKET_CONNECT_TIMEOUT_ATTR is in seconds, passed to connect() in millis
        Integer socketConnectTimeout = (Integer)m_connectionAttributes.get(IContainerConstants.SOCKET_CONNECT_TIMEOUT_ATTR);
        if (socketConnectTimeout != null)
        {
            connector.setSocketConnectTimeout(socketConnectTimeout.intValue() * 1000);
        }

        if (logInfo)
        {
            logMessage("Connecting with a connection timeout of " + (connectTimeout == null ? IContainerConstants.CONNECT_TIMEOUT_DEFAULT : connectTimeout.intValue()) + " seconds", Level.INFO);
        }
        connector.connect(address, (connectTimeout == null ? IContainerConstants.CONNECT_TIMEOUT_DEFAULT : connectTimeout.intValue()) * 1000, validateNodeName);

        return connector;
    }

    private boolean substituteGenerateCacheConnectionAttrs(Hashtable connectionFactoryEnv)
    {
        // Replace with cache generation direct management connection parameters (if set)
        String directURLProp = System.getProperty(IContainer.CACHE_GENERATION_URL_PROPERTY);
        String directUSERProp = System.getProperty(IContainer.CACHE_GENERATION_USER_PROPERTY);
        String directPASSWORDProp = System.getProperty(IContainer.CACHE_GENERATION_PASSWORD_PROPERTY);

        if (directURLProp != null)
        {
            connectionFactoryEnv.put(IContainerConstants.CONNECTIONURLS_ATTR, directURLProp);
        }
        if (directUSERProp != null)
        {
            connectionFactoryEnv.put(IContainerConstants.DEFAULTUSER_ATTR, directUSERProp);
        }
        if (directPASSWORDProp != null)
        {
            connectionFactoryEnv.put(IContainerConstants.DEFAULTPASSWORD_ATTR, directPASSWORDProp);
        }

        return directURLProp != null;
    }

    void logMessage(String message, int severityLevel)
    {
        m_logger.logMessage(message,  severityLevel);
    }

    void logMessage(String message, Throwable t, int severityLevel)
    {
        m_logger.logMessage(message, t, severityLevel);
    }

    private static String createContainerFullName(ExtractedContainerConfig extractedConfig)
    {
        return createContainerFullName(extractedConfig.m_domainName,  extractedConfig.m_containerName);
    }

    private static String createContainerFullName(String domainName, String containerName)
    {
        return domainName + "." + containerName;
    }

    private String[] mergeClasspathLists(String[] l1, String[] l2)
    {
        HashSet set = new HashSet();
        for (int i = 0; i < l1.length; i++)
        {
            set.add(l1[i]);
        }
        for (int i = 0; i < l2.length; i++)
        {
            set.add(l2[i]);
        }

        if (m_dsArchivePath != null)
        {
            set.add(m_dsArchivePath);
        }

        return (String[])set.toArray(EMPTY_STRING_ARRAY);
    }

    protected void getLocalFileReferences()
    throws Exception
    {
        String cachePath = m_dsArchivePath != null  ? m_dsArchivePath : m_logicalArchivePath;
        m_classpath = new CARClasspath(m_configCacheView.getFileByLogicalName(cachePath, false));
        m_systemProps.substituteSonicfsResources(getLocalFiles(m_systemProps.getSonicfsResources()));
        String[] localFiles = getLocalFiles(m_configClasspath.getSonicfsResources());
        
        m_configClasspath.substituteSonicfsResources(localFiles);
    }

    String[] getLocalFiles(String[] sonicfsFiles)
    throws Exception
    {
        ArrayList fileList = new ArrayList();
        for (int i = 0; i < sonicfsFiles.length; i++)
        {
            File localFile = null;
            try
            {
                localFile = m_configCacheView.getFileByLogicalName(sonicfsFiles[i], false);
            }
            catch (LatestVersionMissingException e)
            {
                localFile = m_configCacheView.getAvailableFileByLogicalName(sonicfsFiles[i]);
                if (localFile != null)
                {
                    m_logger.logMessage("Could not get the latest version of \"" + ContainerResources.sonicfsPrefix + sonicfsFiles[i] + "\" from the Directory Service - use the latest cache version", Level.WARNING);
                }
            }

            if (localFile == null)
            {
                m_logger.logMessage("Could not find file \"" + ContainerResources.sonicfsPrefix + sonicfsFiles[i] + "\" in the container cache", Level.WARNING);
                fileList.add(null);
            }
            else
            {
                fileList.add(localFile.getAbsolutePath());
            }
        }

        return (String[])fileList.toArray(EMPTY_STRING_ARRAY);
    }


    class LocalDSParameters
    {
        String m_configID;

        String m_domainName;

        String m_hostDir;

        String m_encryptionPassword;

        // set to default, reset later as we parse the attributes

        boolean m_isBackup = false;

        Boolean m_isFaultTolerant = Boolean.FALSE;

        Boolean m_startActive = Boolean.FALSE;

        private Integer m_replicationRetryInterval;

        private Integer m_replicationPingInterval;

        private Integer m_replicationFailureDetectionTimeout;

        private Integer m_maxReplicationLogSize;

        private Integer m_replicationTimeout;

        private Boolean m_dualActiveResolution;

        private HashMap m_replSSLParams;

        private HashMap[] m_replConnections;

        LocalDSParameters(String dsXMLFileName)
        throws Exception
        {
            File dsXMLFile = new File(dsXMLFileName);
            logMessage("Open Directory Service boot file \"" + getCanonicalPath(dsXMLFile) + '"', Level.INFO);
            IElement[] dsConfigs = ContainerUtil.importConfigurations(dsXMLFile, IContainer.PASSWORD);
            // expecting two configs for DS FT, one for non FT DS
            IElement dsConfiguration = null;
            IElement ftConnections = null;
            for (int i = 0; i < dsConfigs.length; i++)
            {
                String elType = dsConfigs[i].getIdentity().getType();
                if (elType.equals(IDirectoryServiceConstants.DS_TYPE) || elType.equals(IBackupDirectoryServiceConstants.DS_TYPE))
                {
                    dsConfiguration = dsConfigs[i];
                }
                else if (elType.equals(IReplicationConnectionConstants.DS_TYPE))
                {
                    ftConnections = dsConfigs[i];
                }
            }
            m_configID = dsConfiguration.getIdentity().getName();

            IAttributeSet dsAttributes = dsConfiguration.getAttributes();

            m_domainName = (String)dsAttributes.getAttribute(IDirectoryServiceConstants.DOMAIN_NAME_ATTR);
            if (m_domainName == null)
            {
                m_domainName = IDirectoryServiceConstants.DOMAIN_NAME_DEFAULT;
            }

            m_hostDir = (String)dsAttributes.getAttribute(IDirectoryServiceConstants.HOST_DIRECTORY_ATTR);
            if (m_hostDir == null)
            {
                m_hostDir = IDirectoryServiceConstants.HOST_DIRECTORY_DEFAULT;
            }

           IAttributeSet fsStorage = (IAttributeSet)dsAttributes.getAttribute(IDirectoryServiceConstants.FILE_SYSTEM_STORAGE_ATTR);
           if (fsStorage != null)
        {
            m_encryptionPassword = (String)fsStorage.getAttribute(IDirectoryServiceConstants.PASSWORD_ATTR);
        }

            readReplicationParameters(dsAttributes, ftConnections);

            if (dsConfiguration.getIdentity().getType().equals(IBackupDirectoryServiceConstants.DS_TYPE))
            {
                m_isBackup = true;
            }

            if (System.getProperty(IContainer.DS_START_ACTIVE_PROPERTY, "false").equalsIgnoreCase("true"))
            {
                m_startActive = Boolean.TRUE;
            }

            //Might be switched ON for the purpose of testing - 'ffalse' os passed to avoid deleting the file
            // it will be deleted by DSComponent at phase 2
            if (!m_startActive.booleanValue())
            {
                m_startActive = new Boolean (testStartActive(false));
            }
        }

        //Allows to turn m_startActive on using a file placement in the working-directory (the TIDE environment doesn't have an easy way to
        // set system properties per single container
        private boolean testStartActive(boolean deleteFile)
        {
            String testFileName = System.getProperty("_TEST_DSReplication_startActive");
            if (testFileName == null)
            {
                return false;
            }
            java.io.File testFile = new java.io.File(testFileName);
            if (!testFile.exists())
            {
                return false;
            }

            if (deleteFile)
            {
                testFile.delete();
            }
            return true;
        }

        private void readReplicationParameters(IAttributeSet dsAttributes, IElement ftConnections)
        {
            IAttributeSet replParams = (IAttributeSet)dsAttributes.getAttribute(IDirectoryServiceConstants.REPLICATION_PARAMETERS_ATTR);

            // Create a temporary empty primaryReplAttrs if non exists in the primary config to simplify the code
            if (replParams == null)
            {
                IElement tmpEl = ElementFactory.createElement("/tmp", "tmp", "tmp");
                IAttributeSet tmpPrimaryAtts = tmpEl.getAttributes();
                try
                {
                    replParams = tmpPrimaryAtts.createAttributeSet(IDirectoryServiceConstants.REPLICATION_PARAMETERS_ATTR);
                }
                catch (Exception e){} //Could never happen
            }

            Object tmp = null;
            tmp = replParams.getAttribute(IDirectoryServiceConstants.RETRY_INTERVAL_ATTR);
            m_replicationRetryInterval = tmp == null ? new Integer(IDirectoryServiceConstants.RETRY_INTERVAL_DEFAULT) : (Integer)tmp;
            tmp = replParams.getAttribute(IDirectoryServiceConstants.PING_INTERVAL_ATTR);
            m_replicationPingInterval = tmp == null ? new Integer(IDirectoryServiceConstants.PING_INTERVAL_DEFAULT) : (Integer)tmp;
            tmp = replParams.getAttribute(IDirectoryServiceConstants.FAILURE_DETECTION_TIMEOUT_ATTR);
            m_replicationFailureDetectionTimeout = tmp == null ? new Integer(IDirectoryServiceConstants.FAILURE_DETECTION_TIMEOUT_DEFAULT) : (Integer)tmp;
            // During launch, if the peer is the active one, it will be shutting down during the transition from launching to
            // starting the container. We don't want this standby to get activated during that period. Therefore, we increase
            // the m_replicationFailureDetectionTimeout value
            if (m_replicationFailureDetectionTimeout.intValue() < REPLICATION_FAILURE_DETECTION_LAUNCH_MIN)
            {
                m_replicationFailureDetectionTimeout = new Integer(REPLICATION_FAILURE_DETECTION_LAUNCH_MIN);
            }

            tmp = replParams.getAttribute(IDirectoryServiceConstants.MAX_REPLICATION_LOG_SIZE_ATTR);
            m_maxReplicationLogSize = tmp == null ? new Integer(IDirectoryServiceConstants.MAX_REPLICATION_LOG_SIZE_DEFAULT) : (Integer)tmp;
            tmp = replParams.getAttribute(IDirectoryServiceConstants.REPLICATION_TIMEOUT_ATTR);
            m_replicationTimeout = tmp == null ? new Integer(IDirectoryServiceConstants.REPLICATION_TIMEOUT_DEFAULT) : (Integer)tmp;
            tmp = replParams.getAttribute(IDirectoryServiceConstants.DUAL_ACTIVE_RESOLUTION_ATTR);
            m_dualActiveResolution = tmp == null ? new Boolean(DSComponent.AUTOMATIC_DUAL_ACTIVE_RESOLUTION_DEFAULT) : (Boolean)tmp;
            tmp = replParams.getAttribute(IDirectoryServiceConstants.BACKUP_FAILOVER_READ_ONLY_ATTR);
            Boolean backupFailoverReadOnly = tmp == null ? new Boolean(DSComponent.BACKUP_FAILOVER_READ_ONLY_DEFAULT) : (Boolean)tmp;

            //See functional spec - if the BACKUP is readonly, we can safely prefer the PRIMARY
            if (backupFailoverReadOnly.booleanValue())
            {
                m_dualActiveResolution = Boolean.TRUE;
            }

            IAttributeSet tmpSet = (IAttributeSet)replParams.getAttribute(IDirectoryServiceConstants.SSL_PARAMETERS_ATTR);
            if (tmpSet != null)
            {
                m_replSSLParams = (HashMap)tmpSet.getAttributes().clone();
            }

            // read the replication connection parameters

            if (ftConnections != null)
            {
                IAttributeSet connectionsSet = (IAttributeSet)ftConnections.getAttributes().getAttribute(IReplicationConnectionConstants.REPLICATION_CONNECTIONS_ATTR);
                if (connectionsSet != null)
                {
                    m_isFaultTolerant = Boolean.TRUE;
                    HashMap connectionsMap = connectionsSet.getAttributes();
                    Collection connCol = connectionsMap.values();
                    Iterator connIt = connCol.iterator();
                    m_replConnections = new HashMap[connCol.size()];
                    int replConnIndex = 0;
                    while (connIt.hasNext())
                    {
                        IAttributeSet connectionInfo = (IAttributeSet)connIt.next();
                        HashMap connectionMap = new HashMap();
                        Object value = connectionInfo.getAttribute(IReplicationConnectionConstants.PROTOCOL_ATTR);
                        if (value == null)
                        {
                            value = IReplicationConnectionConstants.PROTOCOL_DEFAULT;
                        }
                        connectionMap.put(IReplicationConnectionConstants.PROTOCOL_ATTR, value);
                        value = connectionInfo.getAttribute(IReplicationConnectionConstants.PRIMARY_ADDR_ATTR);
                        if (value != null)
                        {
                            connectionMap.put(IReplicationConnectionConstants.PRIMARY_ADDR_ATTR, value);
                        }
                        value = connectionInfo.getAttribute(IReplicationConnectionConstants.PRIMARY_PORT_ATTR);
                        if (value == null)
                        {
                            value = new Integer(IReplicationConnectionConstants.PRIMARY_PORT_DEFAULT);
                        }
                        connectionMap.put(IReplicationConnectionConstants.PRIMARY_PORT_ATTR, value);
                        value = connectionInfo.getAttribute(IReplicationConnectionConstants.BACKUP_ADDR_ATTR);
                        if (value != null)
                        {
                            connectionMap.put(IReplicationConnectionConstants.BACKUP_ADDR_ATTR, value);
                        }
                        value = connectionInfo.getAttribute(IReplicationConnectionConstants.BACKUP_PORT_ATTR);
                        if (value == null)
                        {
                            value = new Integer(IReplicationConnectionConstants.BACKUP_PORT_DEFAULT);
                        }
                        connectionMap.put(IReplicationConnectionConstants.BACKUP_PORT_ATTR, value);
                        value = connectionInfo.getAttribute(IReplicationConnectionConstants.WEIGHT_ATTR);
                        if (value == null)
                        {
                            value = new Integer(IReplicationConnectionConstants.WEIGHT_DEFAULT);
                        }
                        connectionMap.put(IReplicationConnectionConstants.WEIGHT_ATTR, value);
                        m_replConnections[replConnIndex++] = connectionMap;
                    }
                }
            }
        }

    }

    private abstract class InitInfo
    {
        IAttributeSet m_attributes;

        Hashtable m_properties;

        ExtractedContainerConfig m_config;

        String m_containerConfID;
    }

    private class PropINI
    extends InitInfo
    {
        PropINI(byte[] bytes)
        throws Exception
        {
            ByteArrayInputStream stream = new ByteArrayInputStream(bytes);

            Properties initProps = ContainerUtil.readProperties(stream);

            m_config = extractINIInfo(initProps);
            m_properties = initProps;
            setINISystemProps(m_properties);
        }

        private void setINISystemProps(Hashtable table)
        {
            boolean centralConnectionUsed = false;

            HashMap systemProps = new HashMap();

            Enumeration keys = table.keys();

            while (keys.hasMoreElements())
            {
                String fullKey = (String)keys.nextElement();

                if (fullKey.regionMatches(true, 0, IContainer.SYSTEM_PROP_PREFIX, 0, SYSTEM_PROP_PREFIX_LENGTH))
                {
                    String key = fullKey.substring(SYSTEM_PROP_PREFIX_LENGTH);
                    systemProps.put(key, table.get(fullKey));
                }
                else if (fullKey.regionMatches(true, 0, IContainer.CENTRAL_CONNECTION_PROP_PREFIX, 0, CENTRAL_CONNECTION_PROP_PREFIX_LENGTH))
                {
                    centralConnectionUsed = true;
                }


            }

            m_centralConnectionUsed = centralConnectionUsed && !m_requestToConfigureFromCache;

            //If the ini file contains central connection parameters we use only central-connection-prefixed system properties,
            // but that is only if m_requestToConfigureFromCache is false. When m_requestToConfigureFromCache is true then we are already
            // done with using the central connection to update the cache, and we use the regular connection
            SystemProperties.connectionPropertiesFilter(systemProps, m_centralConnectionUsed);

            Iterator keysIt = systemProps.keySet().iterator();
            while (keysIt.hasNext())
            {
                String key = (String)keysIt.next();
                System.setProperty(key, (String)systemProps.get(key));
            }
        }
    }

    private class XmlINI
    extends InitInfo
    {
        XmlINI(byte[] bytes)
        throws Exception
        {
            IElement containerConfiguration = ElementFactory.importFromXML(new String(bytes), null);
            m_attributes = containerConfiguration.getAttributes();
            m_config = extractConfigInfo(m_attributes);
            m_containerConfID = containerConfiguration.getIdentity().getName();
        }
    }

    // This configuration is extracted from the container.xml boot file or from the
    // actual DS configuration. By incapsulating it in class, we can use the same code
    // in both cases and reconcile differences according to some rules.
    static class ExtractedContainerConfig
    implements Serializable
    {
        String m_archiveSearchPath;

        String m_domainName;

        String m_containerName;

        String m_containerPath;

        String m_actionalPlugMakerArchiveName;

        String m_actionalSDKArchiveName;

        String m_configClasspath;

        String m_activationDaemonPath;

        String m_hostManagerPath;

        String m_logFileName;

        String m_configNativePath;

        Integer m_deployTraceMask;

        String m_cacheHostDirectoryName;

        String m_cachePassword;

        Boolean m_createIfDoesNotExist;

        Boolean m_hostsDS;

        HashMap m_jvmArgs;

        String m_jvmHome;

        String m_archiveName;

        HashMap m_systemProps;

        ExtractedContainerConfig deepClone()
        {
          try
          {
               ByteArrayOutputStream out = new ByteArrayOutputStream();
               ObjectOutputStream objectOut = new ObjectOutputStream(out);
               objectOut.writeObject(this);
               byte[] bytes = out.toByteArray();
               objectOut.close();

               ByteArrayInputStream in = new ByteArrayInputStream(bytes);
               ObjectInputStream objectIn = new ObjectInputStream(in);
                return (ExtractedContainerConfig)objectIn.readObject();
          }
          catch (Exception e)
          {
                throw new Error("deepClone failed");
          }
        }

    }
}
