package com.sonicsw.mf.framework.agent.ci;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;

import com.sonicsw.mf.common.ILogger;
import com.sonicsw.mf.common.runtime.Level;

// Executes PlatformDependentArguments using the container's JVM and reads the arguments PlatformDependentArguments wrote to a file.
public final class GeneratePlatformArguments
{
    private static final String PLATFORM_DEPENDENT_ARGS_PROGRAM = "com.sonicsw.mf.framework.agent.ci.PlatformDependentArguments";
    private static final String PROPERTY_PREFIX = "-D";
    private static final int PROPERTY_OFFSET = PROPERTY_PREFIX.length();

    private HashMap m_properties;
    private HashMap m_arguments;


    public GeneratePlatformArguments(String jvmHome, String[] classpathList, String outputFile, ILogger logger) throws Exception
    {
        // Try first to run with the "-server" option; if fails run without and don't put "-server" in the container
        // command line even if this JVM supports -server (the failure indicates that the server module was not installed)

        for (int i = 0; i < 2; i++)
        {
            if (generateArguments(jvmHome, classpathList, outputFile, logger, i == 0 ? true : false))
            {
                break;
            }
        }
    }

    boolean generateArguments(String jvmHome, String[] classpathList, String outputFile, ILogger logger, boolean serverOption)
            throws Exception
    {
        ArrayList jvmCommand = generateCcommand(jvmHome, classpathList, outputFile, serverOption);

        ContainerProcess process = new ContainerProcess(jvmCommand, PLATFORM_DEPENDENT_ARGS_PROGRAM,
                                                        System.getProperty("user.dir"), logger);
        process.exec();
        int exitCode = process.waitFor();

        if (exitCode != 0)
        {
            if (serverOption)
            {
                return false;
            }

            String errorMsg = "\"" + PLATFORM_DEPENDENT_ARGS_PROGRAM + "\" [" + jvmCommand + "] execution has failed";
            String execError = process.getError();
            if (execError != null && execError.length() > 0)
            {
                errorMsg += ": " + execError;
            }
            logger.logMessage(errorMsg, Level.SEVERE);
            throw new Exception(errorMsg);
        }

        String execError = process.getError();
        if (execError != null && execError.length() > 0)
        {
            logger.logMessage(execError, Level.INFO);
        }
        String execOutput = process.getOutput();
        if ( execOutput!= null && execOutput.length() > 0)
        {
            logger.logMessage(execOutput, Level.INFO);
        }

        m_properties = new HashMap();
        m_arguments = new HashMap();
        readProperties(outputFile, serverOption);
        new File(outputFile).delete();
        return true;
    }

    public HashMap getJVMArguments()
    {
        return m_arguments;
    }

    public HashMap getSystemProprties()
    {
        return m_properties;
    }

    private ArrayList generateCcommand(String jvmHome, String[] classpathList, String outputFile, boolean serverOption)
    {
        ArrayList commandList = new ArrayList();
        commandList.add(jvmHome + File.separatorChar + "bin" +  File.separatorChar + "java" + (File.pathSeparator.equals(";") ? ".exe" : ""));
        if (serverOption)
        {
            commandList.add("-server");
        }
        commandList.add("-cp");
        String classpath = StartContainerCommand.createClasspathString(classpathList);
        String devClasspath = System.getProperty("sonicsw.mf.devPrivateClasspath");
        if (devClasspath != null)
        {
            classpath = devClasspath + File.pathSeparator + classpath;
        }
        commandList.add(classpath);
        commandList.add(PLATFORM_DEPENDENT_ARGS_PROGRAM);
        commandList.add(outputFile);
        return commandList;
    }

    private void readProperties(String outfileName, boolean serverOption) throws Exception
    {
        BufferedReader reader = new BufferedReader (new InputStreamReader(new FileInputStream(outfileName)));

        while (true)
        {
            String line = reader.readLine();
            if (line == null)
            {
                break;
            }

            int eqIndex = line.indexOf('=');
            if (eqIndex == -1)
            {
                continue;
            }
            String key = line.substring(0, eqIndex);
            String value = line.substring(eqIndex+1);

            if (key.startsWith(PROPERTY_PREFIX))
            {
                m_properties.put(key.substring(PROPERTY_OFFSET), value);
            }
            else
            {
                // If the -server option is overidden by the serverOption (serverOption is false) then
                // we we don't include it in m_arguments even if that platform supports it. That could happen when
                // the JVM installation does not contain the server module.
                if (!key.equals("-server") || serverOption)
                {
                    m_arguments.put(key, value);
                }
            }
        }

       reader.close();
    }
}

