package com.sonicsw.mf.framework.agent.ci;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.net.URL;
import java.net.URLConnection;
import java.net.UnknownHostException;

import com.sonicsw.mf.common.ILogger;
import com.sonicsw.mf.common.runtime.Level;

/**
 * Provides helper methods to determine the internal and external host
 * a container is started on.
 */
class HostHelper
{
    private ILogger m_logger;
    private boolean m_skipEC2;
    private boolean m_traceEC2;
    private int m_timeout;
    
    HostHelper(ILogger logger, boolean skipEC2, boolean traceEC2, int timeout)
    {
        m_logger = logger;
        m_skipEC2 = skipEC2;
        m_traceEC2 = traceEC2;
        m_timeout = timeout;
    }

    // Get the private host name 
    // If the user has already provided an private host value we use this.  If not,
    // we first test if this is EC2 by using an http service available on its AMIs .. if not then use
    // InetAddress (but note that could involve a reverse name lookup which can be slow)
    String getPrivateHost()
    {
        String result = queryEC2Metadata("local-hostname");
        if (result == null)
        {
            try
            {
                result = InetAddress.getLocalHost().getCanonicalHostName();
            }
            catch (UnknownHostException e)
            {
                m_logger.logMessage("Unable to determine private host name (defaulting to \"localhost\"), trace follows...", e, Level.WARNING);
                result = "localhost";
            }
        }
        
        return result;
    }

    // Get the public host name 
    // If the user has already provided an public host value we use this.  If not,
    // we test if this is EC2 by using an http service available on its Linux AMIs .. if not then
    // at the moment we have no way to evaluate an public hostname, but might consider
    // some pluggable mechanism in the future
    String getPublicHost()
    {
        String result = queryEC2Metadata("public-hostname");
        return result == null ? "" : result;
    }
    
    // VMs running in EC2 provide an in-built http server that
    // exposes EC2-specific configuration information
    String queryEC2Metadata(String attribute)
    {
        String result = null;

        if (m_skipEC2)
        {
            if (m_traceEC2)
            {
                m_logger.logMessage("Skipping EC2 metadata query for " + attribute, Level.TRACE);
            }
            return result;
        }
        
        String urlstr = "http://169.254.169.254/latest/meta-data/" + attribute; // NOSONAR (MUTE SONAR RULE "IP addresses should not be hardcoded")
        BufferedReader metadataStream = null;
        try
        {
            URL ec2url = new URL(urlstr);
            URLConnection metadata = ec2url.openConnection();
            metadata.setConnectTimeout(m_timeout);
            metadata.setReadTimeout(m_timeout);
            metadataStream = new BufferedReader(new InputStreamReader(metadata.getInputStream()));
            String line;
            while ((line = metadataStream.readLine()) != null)
             {
                result = (line.length() > 0) ? line : result;  // only interested in last (non-empty) line of response
            }
            
            if (m_traceEC2)
            {
                m_logger.logMessage("EC2 metadata query '" + urlstr + "' returned '" + result + "'", Level.TRACE);
            }
        }
        catch (Throwable t)
        {
            result = null;

            // assume subsequent EC2 metadata access will also fail (i.e. we're not on EC2),
            // except in the case of FileNotFoundException which may simply indicate that
            // the requested attribute is not available.
            if (!(t instanceof FileNotFoundException))
            {
                m_skipEC2 = true;
            }
            
            if (m_traceEC2)
            {
                m_logger.logMessage("EC2 metadata query '" + urlstr + "' failed, trace follows...", t, Level.TRACE);
            }
        }
        finally
        {
            if (metadataStream != null)
            {
                try
                {
                    metadataStream.close();
                }
                catch (Throwable t2)
                {
                }
            }
        }
        
        return result;
    }

}
