package com.sonicsw.mf.framework.agent.ci;

import java.io.IOException;
import java.lang.reflect.Method;
import java.util.Set;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.IntrospectionException;
import javax.management.InvalidAttributeValueException;
import javax.management.JMRuntimeException;
import javax.management.ListenerNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanRegistrationException;
import javax.management.NotCompliantMBeanException;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.QueryExp;
import javax.management.ReflectionException;
import javax.management.RuntimeOperationsException;

import com.sonicsw.mf.comm.jms.ConnectorClient;
import com.sonicsw.mf.common.ILogger;
import com.sonicsw.mf.common.runtime.IContainerIdentity;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.jmx.client.CommunicationException;
import com.sonicsw.mf.jmx.client.IConnectionListener;
import com.sonicsw.mf.jmx.client.IExceptionListener;
import com.sonicsw.mf.jmx.client.IOrphanedReplyListener;
import com.sonicsw.mf.jmx.client.IRemoteMBeanServer;
import com.sonicsw.mf.jmx.client.JMSConnectorAddress;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;

/**
 * Can't use regular JMSConnectorClient as we need the container identity
 */
final class LaunchConnector
implements IRemoteMBeanServer
{
    // the underlying JMS connector
    protected ConnectorClient m_connector;

    // this container's identity
    private IContainerIdentity m_containerIdentity;

    // don't process requests if the connector is in a transient disconnect state
    private boolean m_failWhenDisconnected = false;

    // the prior connector address and server identity (if any)
    private JMSConnectorAddress m_address;
    private String m_serverIdentity;

    // timeout value for connection setup attempt
    private long m_connectTimeout = IContainerConstants.CONNECT_TIMEOUT_DEFAULT * 1000;

    // timeout value for socket connection setup attempt (each URL in the connection URL list)
    private long m_socketConnectTimeout = IContainerConstants.SOCKET_CONNECT_TIMEOUT_DEFAULT * 1000;

    // timeout value for client requests
    private long m_requestTimeout = IContainerConstants.REQUEST_TIMEOUT_DEFAULT * 1000;
    
    // trace mask for ConnectorClient
    private int m_traceMask = 0;

    // listener to connection changes including permanent failures
    private com.sonicsw.mf.jmx.client.IConnectionListener m_connectionListener;

    private ILogger m_logger;
    
    LaunchConnector(IContainerIdentity containerIdentity, ILogger logger)
    {
        m_containerIdentity = containerIdentity;
        m_logger = logger;
    }
    
    LaunchConnector(ConnectorClient connectorClient)
    {
        m_connector = connectorClient;
    }

    //
    // As per Sun's JDMK RemoteMBeanServer interface
    //

    /**
     * Connects the connector to the underlying JMS transport.
     * <p>
     * NOTE: connect() will block until either the connection is established, a non-recoverable
     *       condition was determined (in which case a runtime exception will be thrown) or
     *       the thread invoking connect() is interrupted. Recoverable conditions include those
     *       consitions that might reasonably be expected, such as when the SonicMQ broker designated
     *       to carry management communications comes on line. Non-recoverable conditions include
     *       more permanent condition like the inability to resolve an IP address or a security
     *       failure such as an inauthentic client.
     *
     * @param address The attributes describing how to connect to the underlying JMS transport.
     * @return If the connector address dedicates the connector to a particular MBeanServer, the
     *         MBeanServer identity is returned, otherwise null is returned.
     *
     * @see #disconnect()
     */
    @Override
    public synchronized String connect(JMSConnectorAddress address)
    {
        return connect(address, 0, false);
    }

    /**
     * Connects the connector to the underlying JMS transport.
     * <p>
     * NOTE: connect() will block until either the connection is established, a non-recoverable
     *       condition was determined (in which case a runtime exception will be thrown) or
     *       the thread invoking connect() is interrupted. Recoverable conditions include those
     *       consitions that might reasonably be expected, such as when the SonicMQ broker designated
     *       to carry management communications comes on line. Non-recoverable conditions include
     *       more permanent condition like the inability to resolve an IP address or a security
     *       failure such as an inauthentic client.
     *
     * @param address The attributes describing how to connect to the underlying JMS transport.
     * @param timeout The length of time in which the initial connection must be established.
     * @param validateNodeName Indicates if a check should be made that the node name of an established
     *        connection should match that specified in the address (only applicable when creating a
     *        direct connection for the purpose of generating a cache).
     * @return If the connector address dedicates the connector to a particular MBeanServer, the
     *         MBeanServer identity is returned, otherwise null is returned.
     *
     * @see #disconnect()
     */
    public synchronized String connect(JMSConnectorAddress address, long timeout, boolean validateNodeName)
    {
        if (m_address != null && !address.equals(m_address))
        {
            throw new IllegalArgumentException("Cannot connect with a different address.");
        }

        if (m_connector != null)
        {
            throw new java.lang.IllegalStateException("Already connected");
        }

        m_connector = new ConnectorClient(m_containerIdentity);
        m_connector.setRequestTimeout(m_requestTimeout);
        m_connector.setConnectTimeout(m_connectTimeout);
        m_connector.setSocketConnectTimeout(m_socketConnectTimeout);
        m_connector.setTraceMask(m_traceMask);

        String managementNode = address.getManagementNode();
        if (managementNode != null && managementNode.length() > 0)
        {
            m_connector.setManagementNode(managementNode);
        }
        m_connector.setConnectionListener(new com.sonicsw.mf.comm.IConnectionListener()
        {
            public String lrn;
            @Override
            public void onFailure(Exception e)
            {
                if (LaunchConnector.this.m_connectionListener != null)
                {
                    LaunchConnector.this.m_connectionListener.onFailure(e);
                }
            }

            @Override
            public void onReconnect(String localRoutingNode)
            {
                // To avoid potential deadlock, commented out below section,
                // and instead added code to create new thread to invoke
                // connection listener's "onReconnect" method...
                /* synchronized(LaunchConnectorClient.this)
                {
                    LaunchConnectorClient.this.restartSubscriptionRenewalThreads();
                    if (LaunchConnectorClient.this.m_connectionListener != null)
                        LaunchConnectorClient.this.m_connectionListener.onReconnect(localRoutingNode);
                }
                */

                lrn = localRoutingNode;
                Thread reconnectHandler = new Thread("LaunchConnectorClient - Reconnect Handler")
                {
                    @Override
                    public void run()
                    {
                        synchronized(LaunchConnector.this)
                        {
                            if (LaunchConnector.this.m_connectionListener != null)
                            {
                                LaunchConnector.this.m_connectionListener.onReconnect(lrn);
                            }
                        }
                    }
                };
                reconnectHandler.setDaemon(true);
                reconnectHandler.start();
            }

            @Override
            public void onDisconnect()
            {
                synchronized(LaunchConnector.this)
                {
                    if (LaunchConnector.this.m_connectionListener != null)
                    {
                        LaunchConnector.this.m_connectionListener.onDisconnect();
                    }
                }
            }
        });

        try
        {
            m_connector.connect(address.getEnv(), timeout);
        }
        catch(Exception e)
        {
            if (m_connector != null)
            {
                m_connector.close();
                m_connector = null;
            }

            // set the cause through reflection in case this is not J2SE >= 1.4
            JMRuntimeException jmre = new JMRuntimeException(e.getMessage());
            try
            {
                Method method = jmre.getClass().getMethod("initCause", new Class[] { Throwable.class});
                method.invoke(jmre, new Object[] { e });
            }
            catch(Exception ex) { }
            throw jmre;
        }

        if (validateNodeName)
        {
            if (!m_connector.getLocalRoutingNodeName().equals(managementNode))
            {
                String message = "Configured management node [" + managementNode + "] does not match connected node [" + m_connector.getLocalRoutingNodeName() + "]";
                if (m_connector != null)
                {
                    m_connector.close();
                    m_connector = null;
                }

                throw new JMRuntimeException(message);
            }
        }

        m_address = address;
        m_serverIdentity = address.getServerIdentity();
        return m_serverIdentity;
    }

    /**
     * Gets the (singleton) exception listener for the connector.
     *
     * @deprecated IConnectionListener superseeds IExceptionListener
     *
     * @return Reference to an instance of a registered IExceptionListener implementation
     * @see #getConnectionListener(IConnectionListener)
     * @see #setConnectionListener(IConnectionListener)
     */
    public IExceptionListener getExceptionListener() { return null; }

    /**
     * Sets the (singleton) exception listener for the connector.
     *
     * @deprecated IConnectionListener supercedes IExceptionListener
     *
     * @param listener Reference to an IExceptionListener implementation
     * @see #getConnectionListener(IConnectionListener)
     * @see #setConnectionListener(IConnectionListener)
     */
    public synchronized void setExceptionListener(IExceptionListener listener) { }

    /**
     * Gets the (singleton) connection listener for the connector.
     *
     * @return Reference to a registered IConnectionListener implementation
     */
    public IConnectionListener getConnectionListener() { return m_connectionListener; }

    /**
     * Sets the (singleton) connection listener for the connector.
     *
     * @param listener Reference to an instance of an IConnectionListener implementation
     */
    public synchronized void setConnectionListener(IConnectionListener listener) { m_connectionListener = listener; }

    /**
     * Gets the (singleton) orphaned reply listener for the connector.
     *
     * @return Reference to a registered IOrphanedReplyListener instance
     */
    public IOrphanedReplyListener getOrphanedReplyListener()
    {
        throw new IllegalAccessError();
    }

    /**
     * Sets the (singleton) orphaned reply listener for the connector.
     *
     * @param listener Reference to an instance of an IOrphanedReplyListener implementation
     */
    public synchronized void setOrphanedReplyListener(IOrphanedReplyListener listener)
    {       
        throw new IllegalAccessError();
    }

    /**
     * Disconnect the connector from the underlying JMS transport.
     *
     * @see #connect(JMSConnectorAddress)
     */
    @Override
    public synchronized void disconnect()
    {
        if (m_connector == null)
        {
            return;
        }

        m_connector.closePending();

        m_connector.close();
        m_connector = null;
    }

    /**
     * Sets the default timeout the connector waits for responses from remote requests. The minimum is
     * 10000 milliseconds (silently enforced).
     *
     * @param timeout Timeout value in milliseconds
     */
    public void setRequestTimeout(long timeout)
    {
        m_requestTimeout = (timeout >= ConnectorClient.REQUEST_TIMEOUT_MINIMUM) ? timeout : ConnectorClient.REQUEST_TIMEOUT_MINIMUM;

        if (m_connector != null)
        {
            m_connector.setRequestTimeout(m_requestTimeout);
        }
    }

    /**
     * Gets the default timeout the connector waits for responses from remote requests.
     *
     * @return Current request timeout value in milliseconds
     */
    public long getRequestTimeout()
    {
        return m_requestTimeout;
    }

    public void setFailWhenDisconnected(boolean failWhenDisconnected) { m_failWhenDisconnected = failWhenDisconnected; }

    public boolean getFailWhenDisconnected() {  return m_failWhenDisconnected; }

    @Override
    public ObjectInstance createMBean(String className, ObjectName objectName)
    throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException,
           MBeanException, NotCompliantMBeanException
    {
        throw new IllegalAccessError();
    }

    @Override
    public ObjectInstance createMBean(String className, ObjectName objectName, Object[] params, String[] signature)
    throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException,
           MBeanException, NotCompliantMBeanException
    {
        throw new IllegalAccessError();
    }

    @Override
    public ObjectInstance createMBean(String className, ObjectName objectName, ObjectName loaderName)
    throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException,
           MBeanException, NotCompliantMBeanException, InstanceNotFoundException
    {
        throw new IllegalAccessError();
    }

    @Override
    public ObjectInstance createMBean(String className, ObjectName objectName, ObjectName loaderName, Object[] params, String[] signature)
    throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException,
           MBeanException, NotCompliantMBeanException, InstanceNotFoundException
    {
        throw new IllegalAccessError();
    }

    @Override
    public Object getAttribute(ObjectName objectName, String attribute)
    throws MBeanException, AttributeNotFoundException, InstanceNotFoundException, ReflectionException
    {
        throw new IllegalAccessError();
    }

    @Override
    public AttributeList getAttributes(ObjectName objectName, String[] attributes)
    throws InstanceNotFoundException, ReflectionException
    {
        throw new IllegalAccessError();
    }

    @Override
    public String getDefaultDomain()
    {
        throw new IllegalAccessError();
    }

    @Override
    public Integer getMBeanCount()
    {
        throw new IllegalAccessError();
    }

    @Override
    public MBeanInfo getMBeanInfo(ObjectName objectName)
    throws InstanceNotFoundException, IntrospectionException, ReflectionException
    {
        throw new IllegalAccessError();
    }

    @Override
    public ObjectInstance getObjectInstance(ObjectName objectName)
    throws InstanceNotFoundException
    {
        throw new IllegalAccessError();
    }

    @Override
    public Object invoke(ObjectName objectName, String operationName, Object[] params, String[] signature)
    throws InstanceNotFoundException, MBeanException, ReflectionException
    {
        return invoke(objectName, operationName, params, signature, null);
    }

    // non standard implementation to simulate internal comms
    public Object invoke(ObjectName objectName, String operationName, Object[] params, String[] signature, ClassLoader loader)
    throws InstanceNotFoundException, MBeanException, ReflectionException
    {
        checkNotNull("objectName", objectName);
        checkNotNull("operationName", operationName);
        checkNotNull("params", params);
        checkNotNull("signature", signature);

        testConnector();
        testFailWhenDisconnected();

        Object returnValue = null;
        ClassLoader existingContextLoader = null;

        try
        {
            if (loader != null)
            {
                existingContextLoader = Thread.currentThread().getContextClassLoader();
                Thread.currentThread().setContextClassLoader(loader);
            }
            returnValue = m_connector.invoke(IContainer.INTERNAL_COMMS_TYPE, objectName.getDomain(), objectName.getCanonicalName(), operationName, params, signature);
        }
        catch(MBeanException mbe) { throw mbe; }
        catch(InstanceNotFoundException infe) { throw infe; }
        catch(ReflectionException re) { throw re; }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed invoke");
        }
        finally
        {
            if (loader != null)
            {
                Thread.currentThread().setContextClassLoader(existingContextLoader);
            }
        }

        return returnValue;
    }

    /**
     * @return Returns true if the connector has a live connection to the underlying transport.
     *
     * Gets the connection state of the connector.
     */
    @Override
    public boolean isConnected()
    {
        if (m_connector == null)
        {
            return false;
        }

        try
        {
            return m_connector.isConnected();
        }
        catch(NullPointerException e)
        {
            return false;
        }
    }

    @Override
    public boolean isInstanceOf(ObjectName objectName, String className)
    throws InstanceNotFoundException
    {
        throw new IllegalAccessError();
    }

    @Override
    public boolean isRegistered(ObjectName objectName)
    {
        throw new IllegalAccessError();
    }

    @Override
    public Set queryMBeans(ObjectName objectName, QueryExp query)
    {
        throw new IllegalAccessError();
    }

    @Override
    public Set queryNames(ObjectName objectName, QueryExp query)
    {
        throw new IllegalAccessError();
    }

    @Override
    public void setAttribute(ObjectName objectName, Attribute attribute)
    throws InstanceNotFoundException, AttributeNotFoundException, InvalidAttributeValueException,
           MBeanException, ReflectionException
    {
        throw new IllegalAccessError();
    }

    @Override
    public AttributeList setAttributes(ObjectName objectName, AttributeList attributes)
    throws InstanceNotFoundException, ReflectionException
    {
        throw new IllegalAccessError();
    }

    @Override
    public void unregisterMBean(ObjectName objectName)
    throws InstanceNotFoundException, MBeanRegistrationException
    {
        throw new IllegalAccessError();
    }

    @Override
    public void addNotificationListener(ObjectName objectName, NotificationListener listener, NotificationFilter filter, Object handback)
    throws InstanceNotFoundException
    {
        throw new IllegalAccessError();
    }

    public void addNotificationListener(ObjectName objectName, NotificationListener listener, NotificationFilter filter, Object handback, Long timeout)
    throws InstanceNotFoundException
    {
        throw new IllegalAccessError();
    }

    @Override
    public void removeNotificationListener(ObjectName objectName, NotificationListener listener)
    throws InstanceNotFoundException, ListenerNotFoundException
    {
        throw new IllegalAccessError();
    }

    @Override
    public void removeNotificationListener(ObjectName objectName, NotificationListener listener, NotificationFilter filter, Object handback)
    throws InstanceNotFoundException, ListenerNotFoundException
    {
        throw new IllegalAccessError();
    }

    /**
     * Sets System.out runtime debug tracing mask. Values are as per JMS CONNECTOR
     * values.
     */
    public void setTraceMask(int maskValue) {
        m_traceMask = maskValue;
        if (m_connector != null)
        {
            m_connector.setTraceMask(maskValue);
        }
    }

    /**
     * Gets System.out runtime debug tracing mask. Values are as per JMS CONNECTOR
     * values.
     */
    public int getTraceMask() { return m_traceMask; }

    /**
     * Sets the connect timeout
     */
    public void setConnectTimeout(long connectTimeout)
    {
        m_connectTimeout = (connectTimeout >= ConnectorClient.CONNECT_TIMEOUT_MINIMUM) ? connectTimeout : ConnectorClient.CONNECT_TIMEOUT_MINIMUM;

        if (m_connector != null)
        {
            m_connector.setConnectTimeout(m_connectTimeout);
        }
    }

    /**
     * Gets the connect timeout
     */
    public long getConnectTimeout()
    {
        return m_connectTimeout;
    }

    /**
     * Sets the socket connect timeout
     */
    public void setSocketConnectTimeout(long socketConnectTimeout)
    {
        m_socketConnectTimeout = (socketConnectTimeout >= ConnectorClient.SOCKET_CONNECT_TIMEOUT_MINIMUM) ? socketConnectTimeout : ConnectorClient.SOCKET_CONNECT_TIMEOUT_MINIMUM;

        if (m_connector != null)
        {
            m_connector.setSocketConnectTimeout(m_socketConnectTimeout);
        }
    }

    /**
     * Gets the connect timeout
     */
    public long getSocketConnectTimeout()
    {
        return m_socketConnectTimeout;
    }
    
    public void logMessage(String message, int severityLevel)
    {
        if (m_logger != null)
        {
            m_logger.logMessage(message, severityLevel);
        }
    }

    public void logMessage(String message, Throwable exception, int severityLevel)
    {
        if (m_logger != null)
        {
            m_logger.logMessage(message, exception, severityLevel);
        }
    }

    private void checkNotNull(String paramName, Object object)
    {
        if (object == null)
        {
            IllegalArgumentException e = new IllegalArgumentException(paramName + " must not be null");
            throw new RuntimeOperationsException(e);
        }
    }

    private void testConnector()
    {
        if (m_connector == null)
        {
            throw new java.lang.IllegalStateException("Not connected");
        }
    }

    private void testFailWhenDisconnected()
    {
        if (m_failWhenDisconnected && !isConnected())
        {
            throw CommunicationException.create("Not currently connected");
        }
    }

    @Override
    public String[] getDomains()
    throws IOException
    {
        throw new IllegalAccessError();
    }

    @Override
    public void addNotificationListener(ObjectName objectName, ObjectName listener, NotificationFilter filter, Object handback)
    throws InstanceNotFoundException
    {
        throw new IllegalAccessError();
    }

    @Override
    public void removeNotificationListener(ObjectName objectName, ObjectName listener)
    throws InstanceNotFoundException, ListenerNotFoundException
    {
        throw new IllegalAccessError();
    }

    @Override
    public void removeNotificationListener(ObjectName objectName, ObjectName listener, NotificationFilter filter, Object handback)
    throws InstanceNotFoundException, ListenerNotFoundException
    {
        throw new IllegalAccessError();
    }
}
