package com.sonicsw.mf.framework.agent.ci;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.StringTokenizer;

import javax.management.JMRuntimeException;

import com.sonicsw.mf.comm.ConnectTimeoutException;
import com.sonicsw.mf.comm.IConnectorClient;
import com.sonicsw.mf.comm.InvokeTimeoutCommsException;
import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.comm.jms.ConnectorClient;
import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.ILogger;
import com.sonicsw.mf.common.runtime.IContainerExitCodes;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.agent.ContainerUtil;
import com.sonicsw.mf.jmx.client.IRemoteMBeanServer;
import com.sonicsw.mf.jmx.client.JMSConnectorClient;
import com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException;

public final class LaunchContainer
{
    static final int CONTAINER_LAUNCH_TRACE_MASK = 4096;

    private static final String DEFAULT_JVM_HOME = System.getProperty("java.home");

    // By default, generate a BAT script on windows and a SH script on UNIX. But if GENERATE_SH_ON_WINDOWS is true
    // (for because was called from startcontainer.sh), generate SH even on Windows.
    private static boolean ON_WINDOWS = File.separatorChar == '\\';

    private static final String SET_VERSION_SCRIPT_NAME = "set_launcher_version";

    private static boolean BAT_LAUNCH_SCRIPT;

    private static final String SUN_PATH = "LD_LIBRARY_PATH";

    private static final String LINUX_PATH = "LD_LIBRARY_PATH";

    private static final String HP_PATH = "SHLIB_PATH";

    private static final String AIX_PATH = "LIBPATH";

    private static final String WINDOWS_PATH = "PATH";

    private static final String DEFAULT_PATH = "LD_LIBRARY_PATH";

    private static final String PATH_VAR_NAME = getPathVarName();

    private Object m_shutdownThreadLockObj = new Object();
    private boolean m_shutdownRequested;

    static
    {
        BAT_LAUNCH_SCRIPT = ON_WINDOWS;
        String shOnWindows = System.getProperty(IContainer.GENERATE_SH_ON_WINDOWS);
        if (shOnWindows != null)
        {
            BAT_LAUNCH_SCRIPT = !shOnWindows.equalsIgnoreCase("true");
        }
    }

    ILogger m_logger;

    String m_launchScriptPath;

    public static void main(String[] args)
    {
        if (args.length < 1 || args.length > 2)
        {
            printUsage();
            System.exit(IContainerExitCodes.INVALID_CMD_LINE_EXIT_CODE);
        }

        String bootFileNames = args[0];
        String dsBootFileName = null;
        if (args.length > 1)
        {
            dsBootFileName = args[1];
        }
        new LaunchContainer().deployAndStart(bootFileNames, dsBootFileName, true);
        if (!System.getProperty("sonicsw.mf.devSingleJVM", "false").equals("true"))
        {
            Runtime.getRuntime().halt(0);
        }
    }

    private static String getPathVarName()
    {
        String osName = System.getProperty("os.name").toLowerCase();

        if (ON_WINDOWS)
        {
            return WINDOWS_PATH;
        }
        else if (osName.indexOf("sunos") != -1)
        {
            return SUN_PATH;
        }
        else if (osName.indexOf("linux") != -1)
        {
            return LINUX_PATH;
        }
        else if (osName.indexOf("hp") != -1)
        {
            return HP_PATH;
        }
        else if (osName.indexOf("aix") != -1)
        {
            return AIX_PATH;
        }
        else
        {
            return DEFAULT_PATH;
        }
    }

    private static String appendPathLine(String path)
    {
        if (BAT_LAUNCH_SCRIPT)
        {
            return "set PATH=" + path + File.pathSeparatorChar + "%PATH%";
        }
        else
        {
            return PATH_VAR_NAME + "=\"$" + PATH_VAR_NAME + File.pathSeparatorChar + path + "\"; export " + PATH_VAR_NAME;
        }
    }

    void deployAndStart(String bootFileNames, String dsBootFileName, boolean generateScriptOnly)
    {
        if (generateScriptOnly)
        {
            String scriptExtension = BAT_LAUNCH_SCRIPT ? ".bat" : ".sh";
            if (bootFileNames.indexOf(',') > -1 || bootFileNames.equals("*.xml"))
            {
                m_launchScriptPath = "default" + scriptExtension;
            }
            else
            {
                m_launchScriptPath = bootFileNames + scriptExtension;
            }
            new File(m_launchScriptPath).delete();
        }

        ArrayList bootfiles = new ArrayList();

        if (bootFileNames.equals("*.xml")) // can specify using wildcard when running from working directory containing multiple bootfiles
        {
            File workingDir = new File(".");
            String[] bootfileList = workingDir.list(new FilenameFilter()
            {
                @Override
                public boolean accept(File dir, String name)
                {
                    return name.endsWith(".xml");
                }
            });
            for (int i = 0; i < bootfileList.length; i++)
            {
                bootfiles.add(bootfileList[i]);
            }
        }
        else // or by a straight list
        {
            StringTokenizer bootFileTokens = new StringTokenizer(bootFileNames, ",");
            while (bootFileTokens.hasMoreTokens())
            {
                bootfiles.add(bootFileTokens.nextToken());
            }
        }

        if (bootfiles.size() > 1)
        {
            System.setProperty(IContainer.MF_LSD_COLOCATE_PROPERTY, "true");
        }

        DeployContainerResources[] deployments = new DeployContainerResources[bootfiles.size()];
        Iterator bootfileIterator = bootfiles.iterator();

        String originalThreadName = Thread.currentThread().getName();
        try
        {
            for (int i = 0; i < deployments.length; i++)
            {
                String bootFileName = (String)bootfileIterator.next();
                if (bootfiles.size() > 1)
                {
                    Thread.currentThread().setName(originalThreadName + " [" + bootFileName + "]");
                }
                try
                {
                    deployments[i] = new DeployContainerResources(dsBootFileName);
                    m_logger = deployments[i].getLogger();
                    deployments[i].init(bootFileName);
                    try
                    {
                        deployments[i].connectAndGetConfiguration();
                        deployments[i].deployResources();
                    }
                    catch (Throwable t)
                    {
                        // We don't throw this throwable - we'll try to start the container with resources refreshment
                        if (t instanceof JMRuntimeException && ((JMRuntimeException)t).getCause() instanceof ConnectTimeoutException)
                        {
                            logMessage("Failed to refresh resources due to a connection timeout", Level.WARNING);
                        }
                        else
                        {
                            if (t instanceof ProxyRuntimeException && t.getCause() != null)
                            {
                                t = t.getCause();
                            }
                            if (t instanceof InvokeTimeoutCommsException)
                            {
                                logMessage("Failed to refresh resources due to a connection timeout", Level.WARNING);
                            }
                            else if (t instanceof InvokeTimeoutException)
                            {
                                logMessage("Failed to refresh resources due to a request timeout", Level.WARNING);
                            }
                            else if (!(t instanceof ConfigureFromCacheException))
                            {
                                logMessage("Failed to refresh resources, trace follows...", t, Level.WARNING);
                            }
                        }
                    }

                    if (deployments[i].m_extractedConfig == null || deployments[i].m_logicalArchivePath == null)
                    {
                        logMessage("Failed to start the container", Level.SEVERE);
                        releaseResources(deployments[i]);
                        System.exit(IContainerExitCodes.CONFIGURATION_FAILURE_EXIT_CODE);
                    }

                    if (deployments[i].m_useCachedConfig)
                    {
                        logMessage("Using cached container configuration", Boolean.getBoolean(IContainer.CONFIGURE_FROM_CACHE_PROPERTY) ? Level.INFO : Level.WARNING);
                    }

                    if (System.getProperty("sonicsw.mf.devSingleJVM", "false").equals("true"))
                    {
                        // close the cache so the container can re-open it
                        releaseResources(deployments[i]);
                        ContainerUtil.storeCachePassword(new File(deployments[i].m_cacheHostDirectoryName), deployments[i].m_extractedConfig.m_cachePassword);

                        logMessage("System property \"sonicsw.mf.devSingleJVM=true\" - container will be started using launcher's JVM", Level.CONFIG);

                        // TODO: for consistency with normal (devSingleJVM) startup we should pick up
                        // the EC2 and internal/external hostname override properties from the container
                        // configuration, not just the current environment.  See StartContainerCommand.
                        boolean skipEC2 = Boolean.getBoolean(IContainer.CONTAINER_SKIP_EC2_CHECKS_PROPERTY);
                        boolean traceEC2 = Boolean.getBoolean(IContainer.CONTAINER_TRACE_EC2_CHECKS_PROPERTY);
                        int timeoutEC2 = Integer.getInteger(IContainer.CONTAINER_TIMEOUT_EC2_CHECKS_PROPERTY, IContainer.CONTAINER_TIMEOUT_EC2_CHECKS_DEFAULT);
                        
                        HostHelper hostHelper = new HostHelper(m_logger, skipEC2, traceEC2, timeoutEC2);
                        
                        if (System.getProperty(IContainer.CONTAINER_PRIVATE_HOST_PROPERTY) == null)
                        {
                            String privateHost = hostHelper.getPrivateHost();
                            if (privateHost != null)
                            {
                                System.setProperty(IContainer.CONTAINER_PRIVATE_HOST_PROPERTY, privateHost);
                            }
                        }
                        
                        if (System.getProperty(IContainer.CONTAINER_PUBLIC_HOST_PROPERTY) == null)
                        {
                            String publicHost = hostHelper.getPublicHost();
                            if (publicHost != null)
                            {
                                System.setProperty(IContainer.CONTAINER_PUBLIC_HOST_PROPERTY, publicHost);
                            }
                        }

                        new Agent(deployments[i].m_cacheHostDirectoryName, deployments[i].m_containerName, deployments[i].m_containerID);
                        return;
                    }

                    // if this is the last bootfile in the list (there is only one in the list for normal startup!)
                    if (i == (deployments.length - 1))
                    {
                        String upgradeVersion = null;

                        boolean dsFilesMissing = false;
                        //Only launcher install single container launches support the automatic update of launcher files
                        if (deployments.length == 1 && generateScriptOnly)
                        {
                            DeployContainerResources dcr = deployments[0];
                            dsFilesMissing = dcr.mfDSLaunchFilesMissing();

                            if (dcr.mfDirectoryJARMissing())
                            {
                                LauncherJARBuilder.createMFdirectoryJAR(dcr.getArchiveRoot(), dcr.getMFDirectory());
                                logMessage("...Missing \" " + dcr.getMFDirectory().getAbsolutePath() + "\" retrieved", Level.INFO);
                            }

                            LauncherFilesInstaller launcherInstaller = new LauncherFilesInstaller();
                            upgradeVersion = launcherInstaller.installLauncherFiles(dcr.getArchiveRoot());
                        }

                        String workingDir = deployments[deployments.length - 1].getWorkingDir(); // if there are multiple containers, just use the working dir of the last one
                        StartContainerCommand startCommand = createStartCommand(deployments, bootFileName + ".plt", m_logger, generateScriptOnly, null, false, upgradeVersion != null || dsFilesMissing);

                        if (upgradeVersion != null)
                        {
                            updateSetVersionScript(upgradeVersion);
                            logMessage("Upgrading the launcher to version " + upgradeVersion + " and restarting...", Level.INFO);
                        }
                        else if (dsFilesMissing)
                        {
                            logMessage("Restarting after retrieving missing Directory Service file(s)", Level.INFO);
                        }
                        else
                        {
                            logMessage("The container working directory is \"" + workingDir + '"', Level.INFO);
                        }

                        if (generateScriptOnly)
                        {
                            generateStartContainerScript(workingDir, startCommand.getCommandString(), startCommand.getNativePath());
                            // If the appropriate system property is set then we send a bell character to indicate
                            // startup has completed; we use the bell because its not printable and we write directly
                            // to stdout rather than using the logging mechanism (thus it does not get sent to the
                            // log file if any).
                            // This facility is used only by our NT service and our QA classes to avoid depenedency
                            // on looking for a particular startup string like that below !
                            if (IContainer.SIGNAL_MODE)
                            {
                                System.out.print("" + IContainer.SHUTDOWN_SIGNAL_CHAR + 0 + IContainer.SHUTDOWN_SIGNAL_CHAR);
                                System.out.flush();
                            }
                        }
                        else
                        {
                            startContainer(deployments[i].getContainerName(), workingDir, startCommand.getCommandArray());
                        }
                    }
                }
                catch (Throwable t)
                {
                    int exitCode = (t instanceof com.sonicsw.mf.framework.agent.cache.CacheIsLocked) ? IContainerExitCodes.CONTAINER_ALREADY_RUNNING_EXIT_CODE : IContainerExitCodes.FAILED_TO_START_CONTAINER_FROM_CI_DEPLOYANDSTART;
                    logMessage("", t, Level.SEVERE);
                    releaseResources(deployments);
                    // for NT service
                    if (IContainer.SIGNAL_MODE)
                    {
                        System.out.print("" + IContainer.SHUTDOWN_SIGNAL_CHAR + exitCode + IContainer.SHUTDOWN_SIGNAL_CHAR);
                        System.out.flush();
                        try
                        {
                            Thread.sleep(2000);
                        }
                        catch (Exception e)
                        {
                        }
                    }
                    System.exit(exitCode);
                }
                finally
                {
                    releaseResources(deployments[i]);
                }
            }
        }
        finally
        {
            Thread.currentThread().setName(originalThreadName);
            if (m_logger != null && m_logger instanceof CILogger)
            {
                ((CILogger)m_logger).close();
            }
        }
    }

    private void updateSetVersionScript(String carVersion) throws IOException
    {
        if (ON_WINDOWS)
        {
            StringBuffer sb = new StringBuffer();
            sb.append("set SONIC_LAUNCHER_VERSION=").append(carVersion);
            createScript(SET_VERSION_SCRIPT_NAME + ".bat", sb.toString());
        }
        else
        {

            StringBuffer sb = new StringBuffer();
            sb.append("#!/bin/sh").append("\n");
            sb.append("SONIC_LAUNCHER_VERSION=").append(carVersion);
            createScript(SET_VERSION_SCRIPT_NAME + ".sh", sb.toString());
        }
    }


    private void createScript(String scriptPath, String text) throws IOException
    {
        PrintWriter writer = new PrintWriter(new BufferedWriter(new FileWriter(scriptPath, false)), true);
        writer.println(text);
        writer.close();
    }

    private static StartContainerCommand createStartCommand(DeployContainerResources[] deployments, String platformArgsFile, ILogger logger, boolean generateScriptOnly, HashMap additionalProps, boolean specifyLibPath, boolean doRestart)
    throws Exception
    {
        int lastIndex = deployments.length - 1;

        String jvmHome = deployments[lastIndex].getJVMHome();
        if (jvmHome == null)
        {
            jvmHome = DEFAULT_JVM_HOME;
        }

        StartContainerCommand startCommand = new StartContainerCommand(deployments, jvmHome, logger, doRestart);
        deployments[lastIndex].getLocalFileReferences(); // Substitute sonicfs references by local cache references
        GeneratePlatformArguments generatePlatformArgs = new GeneratePlatformArguments(jvmHome, deployments[lastIndex].m_classpath.getLaunchClasspath(), platformArgsFile, logger);
        deployments[lastIndex].mergePlatformDependentArgs(generatePlatformArgs.getJVMArguments());
        deployments[lastIndex].mergePlatformProps(generatePlatformArgs.getSystemProprties(), !deployments[lastIndex].m_useCachedConfig);

        deployments[lastIndex].m_systemProps.addProps(additionalProps);

        startCommand.createExecCommand(generateScriptOnly, specifyLibPath);

        if ((deployments[lastIndex].m_traceMask & CONTAINER_LAUNCH_TRACE_MASK) != 0)
        {
            logger.logMessage("Container startup command:" + IContainer.NEWLINE + startCommand, Level.TRACE);
        }

        return startCommand;
    }

    // called from ContainerImpl
    public static ArrayList prepareCacheForActivatedContainer(IConnectorClient connector, IDirectoryAdminService ds, ILogger logger, String domainName, String containerName, String containerID, String containerWorkDirDirectoryName,
                                                              String cacheHostDirectoryName, String cachePassword, HashMap activationProps)
    throws Exception
    {
        LaunchConnector preconnectedConnector = new LaunchConnector((ConnectorClient)connector);
        return internalPrepareCacheForActivatedContainer(preconnectedConnector, ds, logger, domainName, containerName, containerID, containerWorkDirDirectoryName,
                                                         cacheHostDirectoryName, cachePassword, activationProps);
    }

    // called from Eclipse
    public static ArrayList prepareCacheForActivatedContainer(JMSConnectorClient connector, IDirectoryAdminService ds, ILogger logger, String domainName, String containerName, String containerID, String containerWorkDirDirectoryName,
                                                              String cacheHostDirectoryName, String cachePassword, HashMap activationProps)
    throws Exception
    {
        return internalPrepareCacheForActivatedContainer(connector, ds, logger, domainName, containerName, containerID, containerWorkDirDirectoryName,
                                                         cacheHostDirectoryName, cachePassword, activationProps);
    }

    private static ArrayList internalPrepareCacheForActivatedContainer(IRemoteMBeanServer connector, IDirectoryAdminService ds, ILogger logger, String domainName, String containerName, String containerID, String containerWorkDirDirectoryName,
                                                              String cacheHostDirectoryName, String cachePassword, HashMap activationProps)
    throws Exception
    {
        DeployContainerResources deployment = new DeployContainerResources(null);

        try
        {
            deployment.init(logger, domainName, containerName, containerID, containerWorkDirDirectoryName, cacheHostDirectoryName, cachePassword, true);
            try
            {
                deployment.getConfiguration(connector, ds);
                deployment.deployResources();
            }
            catch (Throwable t)
            {
                // We don't throw this throwable - we'll try to start the container with resources refreshment
                logger.logMessage("Resource refreshment for container \"" + containerName + "\" failed, trace follows...", t, Level.WARNING);
            }

            if (deployment.m_extractedConfig == null || deployment.m_logicalArchivePath == null)
            {
                throw new Exception("Cannot start container \"" + containerName + "\"");
            }

            if (deployment.m_useCachedConfig)
            {
                logger.logMessage("Using cached container configuration for container \"" + containerName + "\"", Level.WARNING);
            }

            StartContainerCommand startCommand = createStartCommand(new DeployContainerResources[] { deployment }, containerName + ".plt", logger, false, activationProps, true, false);

            return startCommand.getCommandArray();

        }
        finally
        {
            releaseResources(deployment); // Must close the cache before starting the container (and also can disconnect the connector)
        }
    }

    private void startContainer(String containerName, String workingDir, ArrayList startContainerCommand)
    throws Exception
    {
        //For the case this JVM is interrupted with CTRL-C: We don't want to exit before ContainerProcess is exiting
        Thread shutdownThread = setShutdownThread();

        ContainerProcess containerProc = new ContainerProcess(startContainerCommand, containerName, workingDir, m_logger);

        containerProc.exec();

        int containerExitCode = containerProc.waitFor();
        synchronized (m_shutdownThreadLockObj)
        {
            m_shutdownRequested = true;
            m_shutdownThreadLockObj.notifyAll(); // Tell the shutdown thread that it's ok to terminate
        }

        Runtime.getRuntime().halt(containerExitCode); // Now halt so that the shutdownThread will not be called again
    }

    private void generateStartContainerScript(String workingDir, String startContainerCommand, String nativePath)
    throws Exception
    {
        PrintWriter writer = new PrintWriter(new BufferedWriter(new FileWriter(m_launchScriptPath, false)), true);
        String cdCommand = null;
        String commentPrefix = null;
        if (BAT_LAUNCH_SCRIPT)
        {
            cdCommand = "cd /d " + "\"" + workingDir + "\"";
            commentPrefix = "rem ";
        }
        else
        {
            cdCommand = "cd " + "\"" + workingDir + "\"";
            commentPrefix = "# ";
        }

        writer.println(commentPrefix + "Warning: This script is generated and invoked by LaunchContainer." + " It may not be modified or invoked directly.");

        if (ON_WINDOWS && !BAT_LAUNCH_SCRIPT)
        {
            cdCommand = cdCommand.replace('\\', '/');
            startContainerCommand = startContainerCommand.replace('\\', '/');
        }

        writer.println(cdCommand);
        writer.println(appendPathLine(nativePath));
        writer.println(startContainerCommand);
    }

    private static void releaseResources(DeployContainerResources deployment)
    {
        deployment.releaseResources();
    }

    private static void releaseResources(DeployContainerResources[] deployments)
    {
        for (int i = 0; i < deployments.length; i++)
        {
            deployments[i].releaseResources();
        }
    }

    private Thread setShutdownThread()
    {
        Thread shutdownThread = new Thread()
        {
            @Override
            public void run()
            {
                synchronized (m_shutdownThreadLockObj)
                {
                    // Wait until notified - it's ok to terminate
                    try
                    {
                        while (!m_shutdownRequested)
                        {
                            m_shutdownThreadLockObj.wait();
                        }
                    }
                    catch (InterruptedException ie)
                    {
                    } // ignore
                }
            }
        };
        Runtime.getRuntime().addShutdownHook(shutdownThread);
        return shutdownThread;
    }

    void logMessage(String message, int severityLevel)
    {
        m_logger.logMessage(message, severityLevel);
    }

    void logMessage(String message, Throwable t, int severityLevel)
    {
        m_logger.logMessage(message, t, severityLevel);
    }

    private static void printUsage()
    {
        System.out.println();
        System.out.println("Starts a container.");
        System.out.println();
        System.out.println("Usage: <java> [-Dsonic.home=<home>] [-Dsonicsw.mf.password=<password>] <boot file> [<ds xml boot file>");   //NOSONAR Password is not defined.
        System.out.println();
        System.out.println("  <boot file>  Container XML or INI configuration file.");
        System.out.println();
        System.out.println("  <ds xml boot file>  If the container hosts the DS (optional).");
        System.out.println();
        System.out.println("  <home> The directory above the sonic lib directory (not required in a Centralized Install setup).");
        System.out.println();
        System.out.println("  <password> Password to decrypt disk based configuration information (optional).");

        System.out.flush();
    }

}
