package com.sonicsw.mf.framework.agent.ci;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Properties;

import javax.management.ObjectName;

import com.sonicsw.mf.common.util.LockFile;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.agent.ContainerSetup;
import com.sonicsw.mf.framework.agent.ContainerUtil;
import com.sonicsw.mf.jmx.client.JMSConnectorClient;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;

final public class LauncherContainerDriver implements ILauncherContainerDriver
{
    private static String INSTALL_AND_START_LOG = "winservice_install_and_start.log";
    private static String INSTALL_AND_START_ERROR_LOG = "winservice_install_and_start_error.log";
    private static String CONTAINER_LAUNCH_ERROR_LOG = "container_launch_error.log";
    private static final String INSTALL_AND_START_SCRIPT_NAME = "winservice_install_and_start.bat";
    private static final String LAUNCH_CONTAINER_BAT_SCRIPT_NAME = "launchcontainer.bat";
    private static final String LAUNCH_CONTAINER_SH_SCRIPT_NAME = "launchcontainer.sh";
    private static final String FAILED_TO_INSTALL_WS_ERROR_TEXT = "Failed to start the Windows Service install process: Exit error ";
    private static final String FAILED_TO_LAUNCH_BAT_ERROR_TEXT = "Failed to start windows cmd to launch the container: Exit error ";


    private static final String FILELOG_SUFFIX = ".log";
    private static final String RESTART_LOG_ENTRY = "Restart initiated";
    private static final String STARTUP_COMPLETE_LOG_ENTRY = "startup complete";
    private static final String SHUTDOWN_EXIT_LOG_ENTRY = "(info) Exiting...";

    private static String INSTALL_AND_START_INSTRUCTIONS;

    private static String START_WS_LOG = "start_ws.log";
    private static String START_WS_ERROR_LOG = "winservice_start_error.log";
    private static final String FAILED_TO_START_WS_ERROR_TEXT = "Failed to start the Windows Service start process: Exit error ";
    private static final String START_WS_SCRIPT_NAME = "winservice_start.bat";
    private static String START_WS_INSTRUCTIONS;

    static
    {
        StringBuffer sb = new StringBuffer();
        sb.append("call .\\winservice /install > ").append(INSTALL_AND_START_LOG).append(IContainer.NEWLINE);
        sb.append("sleep 5").append(IContainer.NEWLINE);
        sb.append("call .\\winservice /start >> ").append(INSTALL_AND_START_LOG);
        INSTALL_AND_START_INSTRUCTIONS = sb.toString();

        StringBuffer wsSB = new StringBuffer();
        wsSB.append("call .\\winservice /start >> ").append(START_WS_LOG);
        START_WS_INSTRUCTIONS = wsSB.toString();
    }

    File m_containerDir;
    File m_logFile;
    String m_containerDSPath;
    String m_containerName;
    String m_fullContainerName;
    String m_domainName;

    //Local container shutdown program
    public static void main(final String[] args) throws Exception
    {
        if (args.length != 3)
        {
            System.err.println("The com.sonicsw.mf.framework.agent.ci.LauncherContainerDriver container shutdown programs requires 3 arguments: ");
            System.err.println("Working directory path, synchronous (true | false) and forced shutdown (true | false)");
            throw new Exception("Bad Arguments");
        }
        File workingDir = new File(args[0]);
        boolean synchronous = new Boolean(args[1]).booleanValue();
        boolean force = new Boolean(args[2]).booleanValue();
        shutdownLocalContainer(workingDir, synchronous, force);
    }


    LauncherContainerDriver(File containerDir) throws IOException
    {
        File containerINI = new File(containerDir, ContainerSetup.CONTAINER_INI_FILE);
        Properties props = ContainerUtil.readProperties(new FileInputStream(containerINI));
        init(containerDir, props.getProperty(IContainerConstants.DOMAIN_NAME_ATTR), props.getProperty(IContainer.CONTAINER_PATH_ATTR));
    }

    LauncherContainerDriver(File containerDir, String domainName, String containerDSPath)
    {
        init(containerDir, domainName, containerDSPath);
    }

    private void init(File containerDir, String domainName, String containerDSPath)
    {
        m_containerDir = containerDir;
        m_domainName = domainName;
        m_containerDSPath = containerDSPath;
        m_containerName = new File(m_containerDSPath).getName();
        m_fullContainerName = m_domainName + "." + m_containerName;

        String fullDefaultLogName = m_fullContainerName + FILELOG_SUFFIX;
        // if the file log_location exists, that's where the log is, use the contents of
        // that file to figure out where the log is. Cannot assume the log is in the 
        // default location; SDM places the log in a different area, for instance
        String logLocation = null;
        File logLocationFile = new File(m_containerDir, IContainer.LOG_LOCATION_FILE);
        if (logLocationFile.exists())
        {
            try
            {
                BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(logLocationFile)));
                try
                {
                    String line = reader.readLine();
                    if (line != null)
                    {
                        line = line.trim();
                        if (line.length() > 0)
                        {
                            logLocation = line;
                        }
                    }
                }
                finally
                {
                    reader.close();
                }
            }
            catch (IOException e)
            {
              // If cannot get the log directory from LOG_LOCATION_FILE assume the default location
            }
        }
        File logDir = null;
        if (logLocation != null)
        {
            File tempFile = new File(logLocation);
            if (tempFile.isAbsolute())
            {
                logDir = tempFile;
            }
            else
            {
                logDir = new File(m_containerDir, logLocation);
            }               
        }
        else
        {
            logDir = new File(m_containerDir, fullDefaultLogName);
        }
        m_logFile = new File(logDir, fullDefaultLogName);
    }

    @Override
    public void installAsWindowsServiceAndStart() throws Exception
    {
        String scriptPath = new File(m_containerDir,INSTALL_AND_START_SCRIPT_NAME).getAbsolutePath();
        writeScript(INSTALL_AND_START_INSTRUCTIONS, scriptPath);
        int logInitialLength = countFileLines(m_logFile);
        startWindowsCMDProcess(INSTALL_AND_START_SCRIPT_NAME, INSTALL_AND_START_ERROR_LOG, FAILED_TO_INSTALL_WS_ERROR_TEXT);
        new LogWatcher(m_logFile, STARTUP_COMPLETE_LOG_ENTRY, logInitialLength).waitForTextFound();
    }

    @Override
    public void launchContainerAsWindowsService() throws Exception
    {
    	String scriptPath = new File(m_containerDir,START_WS_SCRIPT_NAME).getAbsolutePath();
        writeScript(START_WS_INSTRUCTIONS, scriptPath);
    	int logInitialLength = countFileLines(m_logFile);
        startWindowsCMDProcess(START_WS_SCRIPT_NAME, START_WS_ERROR_LOG, FAILED_TO_START_WS_ERROR_TEXT);
        new LogWatcher(m_logFile, STARTUP_COMPLETE_LOG_ENTRY, logInitialLength).waitForTextFound();
    }

    @Override
    public void launchContainerWithBAT() throws Exception
    {
        int logInitialLength = countFileLines(m_logFile);
        startWindowsCMDProcess(LAUNCH_CONTAINER_BAT_SCRIPT_NAME, CONTAINER_LAUNCH_ERROR_LOG, FAILED_TO_LAUNCH_BAT_ERROR_TEXT);
        new LogWatcher(m_logFile, STARTUP_COMPLETE_LOG_ENTRY, logInitialLength).waitForTextFound();
    }

    @Override
    public void launchContainerWithSH() throws Exception
    {
        ContainerProcess tempContainerProc = null;
        int logInitialLength = countFileLines(m_logFile);
        String scriptPath = new File(m_containerDir,LAUNCH_CONTAINER_SH_SCRIPT_NAME).getAbsolutePath();
        try
        {
            ArrayList<String> execCommand = new ArrayList<String>();
            execCommand.add(scriptPath);
            tempContainerProc = new ContainerProcess(execCommand, m_containerDSPath, m_containerDir.getAbsolutePath(), null);
            tempContainerProc.exec();
        }
        catch (Throwable t)
        {
            log(printStackTrace(t, t.getMessage()), new File(m_containerDir, CONTAINER_LAUNCH_ERROR_LOG).getAbsolutePath());
        }

        final LogWatcher logWatcher = new LogWatcher(m_logFile, STARTUP_COMPLETE_LOG_ENTRY, logInitialLength);

        final ContainerProcess containerProc = tempContainerProc;
        Thread procThread = new Thread("com.sonicsw.mf.framework.agent.ci.LauncherContainerDriver.launchContainerWithSH process threasd")
        {
            @Override
            public void run()
            {
                try
                {
                    int exitCode = containerProc.waitFor();
                    if (exitCode != 0)
                    {
                        StringBuffer sb = new StringBuffer();
                        sb.append("Failed to launch container: Exit error ").append(new Integer(exitCode).toString()).append(IContainer.NEWLINE);
                        String outMsg = containerProc.getOutput();
                        if (outMsg != null && outMsg.length() > 0)
                        {
                            sb.append(outMsg).append(IContainer.NEWLINE);
                        }
                        outMsg = containerProc.getError();
                        if (outMsg != null && outMsg.length() > 0)
                        {
                            sb.append(outMsg);
                        }
                        logWatcher.setException(new Exception(sb.toString()));
                    }
                }
                catch (Exception e)
                {
                    logWatcher.setException(e);
                    return;
                }

            }
        };
        procThread.setDaemon(true);
        procThread.start();

        logWatcher.waitForTextFound();
    }

    @Override
    public void shutdown(JMSConnectorClient connector) throws Exception
    {
        shutdownContainer(connector, m_domainName, m_containerName, m_logFile);
    }

    public static void shutdownContainer(JMSConnectorClient connector, String domainName, String containerName, File containerLog) throws Exception
    {
        int logInitialLength = countFileLines(containerLog);
        ObjectName agentComponent = new ObjectName(domainName + '.' + containerName + ':' + "ID=AGENT");
        connector.invoke(agentComponent, "shutdown", new Object[0], new String[0]);
        new LogWatcher(containerLog, SHUTDOWN_EXIT_LOG_ENTRY, logInitialLength).waitForTextFound();
    }

    public static void restartContainer(JMSConnectorClient connector, String domainName, String containerName, File containerLog) throws Exception
    {
        restartContainer(connector, domainName, containerName, containerLog, false);
    }

    public static void restartContainer(JMSConnectorClient connector, String domainName, String containerName, File containerLog, boolean cleanRestart) throws Exception
    {
        int logInitialLength = countFileLines(containerLog);
        ObjectName agentComponent = new ObjectName(domainName + '.' + containerName + ':' + "ID=AGENT");

        connector.invoke(agentComponent, cleanRestart ? "cleanRestart" : "restart", new Object[0], new String[0]);

        new LogWatcher(containerLog, RESTART_LOG_ENTRY, logInitialLength).waitForTextFound();
        new LogWatcher(containerLog, STARTUP_COMPLETE_LOG_ENTRY, logInitialLength).waitForTextFound();
    }

    private void writeScript(String scriptText, String scriptPath) throws IOException
    {
        PrintWriter writer = new PrintWriter(new BufferedWriter(new FileWriter(scriptPath, false)), true);
        writer.println(scriptText);
        writer.close();
    }

    void startWindowsCMDProcess(String scriptName, String launchErrorLog, String errorText)
    {
        try
        {
            ArrayList<String> execCommand = new ArrayList<String>();
            execCommand.add("cmd.exe");
            execCommand.add("/C");
            execCommand.add("start");
            // add another cmd /C here because just using start will implicitly do cmd /K (e.g. not close the prompt).
            execCommand.add("cmd.exe");
            execCommand.add("/C");
            execCommand.add(scriptName);

            ContainerProcess cmdProcess = new ContainerProcess(execCommand, m_containerDSPath, m_containerDir.getAbsolutePath(), null);
            cmdProcess.exec();
            int exitCode = cmdProcess.waitFor();
            if (exitCode != 0)
            {
                StringBuffer sb = new StringBuffer();
                sb.append(errorText).append(new Integer(exitCode).toString()).append(IContainer.NEWLINE);
                String outMsg = cmdProcess.getOutput();
                if (outMsg != null && outMsg.length() > 0)
                {
                    sb.append(outMsg).append(IContainer.NEWLINE);
                }
                outMsg = cmdProcess.getError();
                if (outMsg != null && outMsg.length() > 0)
                {
                    sb.append(outMsg);
                }
                log(sb.toString(), launchErrorLog);
            }
        }
        catch (Throwable t)
        {
            try
            {
                log(printStackTrace(t, t.getMessage()), new File(m_containerDir, launchErrorLog).getAbsolutePath());
            }
            catch (Exception e1)
            {
            }
        }
    }



    private static void log(String msg, String fileName) throws Exception
    {

        PrintWriter writer = new PrintWriter(new BufferedWriter(new FileWriter(fileName, false)), true);
        writer.println(msg);
        writer.close();
    }

    private static String printStackTrace(Throwable exception, String message)
    {
        StringBuffer buffer = new StringBuffer(message);
        buffer.append(IContainer.NEWLINE);
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        PrintWriter writer = new PrintWriter(out, true);
        exception.printStackTrace(writer);
        byte[] data = out.toByteArray();

        try
        {
            out.close();
        }
        catch (IOException io) { }

        buffer.append(new String(data));
        return buffer.toString();
   }

   public static int countFileLines(File logFile) throws IOException
   {
        if (!(logFile.exists()))
        {
            return 0;
        }

        BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(logFile)));
        int count = 0;
        while (reader.readLine() != null)
        {
            count++;
        }

        reader.close();
        return count;
   }
   public static boolean lookForText(String text, File logFile, int afterLine) throws IOException
   {
       return lookForText(text, logFile, afterLine, null);
   }

   private  static boolean lookForText(String text, File logFile, int afterLine, Boolean[] rolledOverPoniter) throws IOException
   {
        if (!(logFile.exists()))
        {
            return false;
        }

        BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(logFile)));
        int lineNum = 0;
        int lineFound = 0;
        while (true)
        {
            String line = reader.readLine();
            if (line == null)
            {
                break;
            }

            lineNum++;

            if (line.indexOf(text) != -1)
            {
                lineFound = lineNum;
            }
        }

        reader.close();

        boolean logRolledOver = lineNum < afterLine;

        if (rolledOverPoniter != null && logRolledOver)
        {
            rolledOverPoniter[0] = Boolean.TRUE;
        }

        return (lineFound > afterLine) || (logRolledOver && lineFound > 0);
   }

   public static String getLineText(File logFile, int targetLineNum) throws IOException
   {
        if (!(logFile.exists()) || targetLineNum <= 0)
        {
            return null;
        }

        BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(logFile)));
        int lineNum = 0;
        while (true)
        {
            String line = reader.readLine();
            if (line == null)
            {
                break;
            }

            if (++lineNum == targetLineNum)
            {
                reader.close();
                return line;
            }
        }

        reader.close();
        return null;

   }

   //Create an IContainer.SHUTDOWN_CONTAINER_FILE file.  Returns 'false' of the container is not running
   public static boolean shutdownLocalContainer(File containerWD, boolean waitForShutdown, boolean force) throws Exception
   {
       File timestampFile = new File(containerWD, IContainer.START_TIMESTAMP_FILE);
       File shutdownFile = new File(containerWD, IContainer.SHUTDOWN_CONTAINER_FILE);

       String startupTimestamp = null;
       BufferedReader reader = null;
       try
       {
           reader = new BufferedReader(new InputStreamReader(new FileInputStream(timestampFile)));
           startupTimestamp = reader.readLine().trim();
       }
       catch (Exception e)
       {

       }
       finally
       {
           if (reader != null)
        {
            try
               {
                   reader.close();
               }
               catch (Exception e){}
        }
       }


       if (startupTimestamp == null)
    {
        return false;
    }

       String doForce = force ? "true" : "false";

       shutdownFile.delete();
       PrintWriter writer = new PrintWriter(new BufferedWriter(new FileWriter(shutdownFile, false)), true);
       writer.println(startupTimestamp);
       writer.println(doForce);
       writer.close();

       if (!waitForShutdown)
    {
        return true;
    }

       waitForContainerToShutdown(containerWD);

       return true;
   }

   public static void waitForContainerToShutdown(File containerWD) throws Exception
   {
       LockFile containerRunningLock = new LockFile(new File(containerWD, IContainer.CONTAINER_RUNNING_LOCK_FILE).getAbsolutePath());

       while (!containerRunningLock.lock())
    {
        Thread.sleep(2000);
    }

       containerRunningLock.unlock();
   }


   //Watches for the appearance of specific test in a log
   public static class LogWatcher
   {
       private File m_logFile;
       private String m_text;
       private boolean m_found;
       private boolean m_stop;
       private Exception m_exception;
       private int m_initialLength;
       private boolean m_logRolledOver;

       public LogWatcher(File logFile, String text, int initialLength)
       {
            m_logFile = logFile;
            m_text = text;
            m_found = false;
            m_stop = false;
            m_exception = null;
            m_initialLength = initialLength;
            m_logRolledOver = false;

            Thread watchThread = new Thread("com.sonicsw.mf.framework.agent.ci.LauncherContainerDriver log watcher")
            {
                @Override
                public void run()
                {
                    try
                    {
                        while (!m_stop)
                        {
                            Boolean[] rolledOverPointer = !m_logRolledOver ? new Boolean[1] : null;
                            boolean textFound = lookForText(m_text, m_logFile, m_initialLength, rolledOverPointer);
                            if (!textFound && !m_logRolledOver && rolledOverPointer[0] != null)
                            {
                                m_initialLength = 0;
                                m_logRolledOver = true;
                            }

                            if (textFound)
                            {
                                setFound();
                                return;
                            }
                            Thread.sleep(5000);
                        }
                    }
                    catch (Exception e)
                    {
                        setException(e);
                        return;
                    }

                }
            };
            watchThread.setDaemon(true);
            watchThread.start();
       }
       public void waitForTextFound() throws Exception
       {
           waitForTextFound(-1);
       }

       public synchronized void waitForTextFound(long waitPeriod) throws Exception
       {
           long waitCounter = 0;
           while (!m_found && m_exception == null)
           {
               try
               {
                   wait(1000);
                   waitCounter += 1000;
               }
               catch (Exception e)
               {
               }
               if (waitPeriod > 0 && waitCounter >= waitPeriod)
               {
                   m_stop = true;
                   throw new Exception("The \"" + m_text + "\" text was not found in log \"" + m_logFile.getAbsolutePath() + "\" after " + waitPeriod + " millis");
               }

               if (m_exception != null)
            {
                throw m_exception;
            }
           }
       }


       private synchronized void setFound()
       {
           m_found = true;
           notifyAll();
       }

       synchronized final void setException(Exception e)
       {
           m_exception = e;
           notifyAll();
       }


   }
}