package com.sonicsw.mf.framework.agent.ci;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Properties;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.jar.Pack200;
import java.util.jar.Pack200.Unpacker;

import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.agent.ContainerSetup;
import com.sonicsw.mf.framework.agent.ContainerUtil;


public final class LauncherFilesInstaller
{
    private static final String EXT_PACK = ".pack";
    private static final String JRE_PROP = "JRE";
    private static final boolean ON_WINDOWS = File.separatorChar == '\\';
    private static final String SONIC_HOME = System.getProperty(IContainer.SONICSW_HOME_PROPERTY);
    private static final String WINSERVICE_DIR = "winservice";
    private static final String BAT_FILE = ".bat";
    private static final String SH_FILE = ".sh";
    private static final String CAR_MANIFEST_FILE = "META-INF/MANIFEST.MF";
    private static final String MANIFEST_VERSION_LABEL = "Implementation-Version: ";
    private static final String CAR_LAUNCHER_FILE_LIST = "launch/maps/car_to_launcher_map";
    private static final String SH_SETUP_SCRIPT = "container_setup/setup.sh";
    private static final String BAT_SETUP_SCRIPT = "container_setup/setup.bat";
    private static final String TEMP_SETUP_SCRIPT = "container_setup/setup.temp";
    private static final String BAT_COMMENT = "REM";
    private static final String SH_COMMENT = "#";
    private static final String TAILOR_MARKER = "$";

    private byte[] m_buffer = new byte[8192];


    //Called by the build to generate the install bundle
    public static void main(String[] args) throws Exception
    {
        String archiveRootPath = args[0];
        if (archiveRootPath == null)
        {
            throw new Exception("Usage: com.sonicsw.mf.framework.agent.ci.LauncherFilesInstaller <expanded-MFcontainer.car-root-path>");
        }
        File archiveRootDir = new File(archiveRootPath);
        new LauncherFilesInstaller().installLauncherFiles(archiveRootDir, archiveRootDir.getParentFile().getAbsolutePath(), true, false, false);
    }

    //Install from an archive file without MFdirectory.jar
    public String installLauncherFiles(String archivePath, String sonicHome) throws IOException
    {
        return installLauncherFiles(archivePath, sonicHome, false);
    }

     //Install from an archive file with or without MFdirectory.jar
    String installLauncherFiles(String archivePath, String sonicHome, boolean instalMFdirectoryJAR) throws IOException
    {
        File archiveFile = new File(archivePath);
        String tempDir = System.getProperty("java.io.tmpdir");
        File tempRoot = new File(tempDir, archiveFile.getName());
        if (tempRoot.exists())
        {
            clearDestination(tempRoot);
        }
        expandArchive(archiveFile, tempRoot);
        String version = installLauncherFiles(tempRoot, sonicHome, true, true, instalMFdirectoryJAR);
        clearDestination(tempRoot);
        return version;
    }

    //Install into a given sonic home directory
    public String installLauncherFiles(File archiveRoot, File homeDir) throws IOException
    {
        return installLauncherFiles(archiveRoot, homeDir.getAbsolutePath(), true, true, false);
    }

    //Install if the current version doesn't match the archive version
    String installLauncherFiles(File archiveRoot) throws IOException
    {
        return installLauncherFiles(archiveRoot, SONIC_HOME, false, true, false);
    }

    private String installLauncherFiles(File archiveRoot, String sonicHome, boolean ignoreCurrentVersion, boolean tailorSetupScripts, boolean installMFdirectoryJAR) throws IOException
    {
        if (IContainer.CURRENT_LAUNCHER_VERSION == null && !ignoreCurrentVersion)
        {
            return null;
        }

        String carVersion = getCarManifestVersion(archiveRoot);
        if (null == carVersion)
        {
            return null;
        }

        try
        {
            upgradeWDScriptsIfNeeded(sonicHome, IContainer.CURRENT_LAUNCHER_VERSION);
        }
        catch (IOException e)
        {
            System.err.println("WARNING: Might have failed to upgrade a working directory script");
            e.printStackTrace();
        }

        if (!ignoreCurrentVersion && carVersion.equals(IContainer.CURRENT_LAUNCHER_VERSION))
        {
            return null;
        }

        File launcherListFile = new File(archiveRoot, CAR_LAUNCHER_FILE_LIST);
        String launcherDirName = IContainer.LAUNCHERS_ROOT_DIR+carVersion;
        File updatedLauncherPath = new File(sonicHome, launcherDirName);
        File updatesLauncherMFDirJAR = new File(updatedLauncherPath, LauncherJARBuilder.MF_DIR_JAR_PATH);

        File currentLauncherPath = new File(sonicHome, IContainer.LAUNCHER_DIR);
        File currentLauncherMFDirJAR = new File(currentLauncherPath, LauncherJARBuilder.MF_DIR_JAR_PATH);


        LauncherInstallManager launchManager = new LauncherInstallManager(new File(sonicHome), launcherDirName);
        try
        {
            if (launchManager.startInstall())
            {
                copyFiles(launcherListFile, archiveRoot, updatedLauncherPath, ON_WINDOWS);

                //If the current launcher contain MFdirectory.jar then we need one in the upgraded launcher as well
                if (currentLauncherMFDirJAR.exists() || installMFdirectoryJAR)
                {
                    LauncherJARBuilder.createMFdirectoryJAR(archiveRoot, updatesLauncherMFDirJAR);
                }

                if (tailorSetupScripts)
                {
                    tailorSetupScripts(updatedLauncherPath, ON_WINDOWS);
                }

                if (!ON_WINDOWS)
                {
                    ContainerSetup.unixChmodSHFilesToExecutables(updatedLauncherPath);
                }
            }

            return carVersion;
        }
        finally
        {
            //We delete the SH files on windows and the BAT files on non Windows Platforms. But we do that only if tailorSetupScripts
            // since that means that the generated launcher install target is this current machine. When we create a launcher bundle
            // during the build we want SH and BAT files to be there since the bundle could be used on different platforms.
            if (tailorSetupScripts)
            {
                deleteFilesOfType(updatedLauncherPath, ON_WINDOWS ? SH_FILE : BAT_FILE);
            }

            launchManager.installDone();
        }
    }

    //The need to upgrade the container's WD scripts is rare since those scripts are mostly immutable - updates are typically restricted
    // to the launcher scripts. But whenever WD scripts upgrade is required, it happens in upgradeWDScriptsIfNeeded. For example, shutdowncontainer.bat(sh)
    // has changed between 8.0 and 8.1. Note that upgradeWDScriptsIfNeeded will always be called twice during the launch sequence in cases where a launcher upgrade
    // is required - before and after the upgrade (but not when we upgrade from 8.0 to 8.1 since upgradeWDScriptsIfNeeded didn't exist in 8.0). So any code added
    // to upgradeWDScriptsIfNeeded must check whether the script in question was already upgraded.
    private void upgradeWDScriptsIfNeeded(String sonicHome, String currentLauncherVersion) throws IOException
    {
        String scriptName = "shutdowncontainer" + (ON_WINDOWS ? ".bat" : ".sh");
        File launcherDir = new File(sonicHome, IContainer.LAUNCHERS_ROOT_DIR+currentLauncherVersion);
        File templatesDir = new File(launcherDir, ContainerSetup.CONTAINER_TEMPLATES_DIR);
        File targetScript = new File(scriptName);
        File srcScript = new File(templatesDir, scriptName);

        if (!targetScript.exists())
        {
            return;
        }
        //Check whether shutdowncontainer requires upgrade.
        BufferedReader reader = null;
        boolean oldScript = true;
        try
        {
            reader = new BufferedReader (new InputStreamReader(new FileInputStream(targetScript)));
            while (true)
            {
                String line = reader.readLine();
                if (line == null)
                {
                    break;
                }
                if (line.indexOf("Tailored variables") != -1)
                {
                    oldScript = false;
                    break;
                }
            }
        }
        finally
        {
            if (reader != null)
            {
                reader.close();
            }
        }

        if (!oldScript)
        {
            return;
        }

        targetScript.delete();

        File currentWD = new File(System.getProperty("user.dir"));
        Properties tailorProps = new Properties();
        Properties pathProps = new Properties(); //Can be empty since path adjustments are not needed

        tailorProps.setProperty(ContainerSetup.SONIC_HOME_SCRIPT_VAR, sonicHome);
        tailorProps.setProperty(IContainer.WORKING_DIRECTORY_ATTR, currentWD.getAbsolutePath());

        ContainerSetup.tailorFile(srcScript, targetScript, tailorProps, pathProps, ON_WINDOWS);

        if (!ON_WINDOWS)
        {
            ArrayList chmodList = new ArrayList();
            chmodList.add(targetScript);
            ContainerSetup.unixChmodSHFilesToExecutables(chmodList);
        }
    }


    /**
     * Reads the {@link #CAR_MANIFEST_FILE} to get the version number.
     *
     * @param archiveRoot
     *            file name of the CAR
     * @return version/build string
     * @throws FileNotFoundException
     *             if the version file does not exist
     * @throws IOException
     *             if we cannot read the file
     */
    private String getCarManifestVersion(File archiveRoot)
            throws FileNotFoundException, IOException
    {
        File manifetFile = new File(archiveRoot, CAR_MANIFEST_FILE);
        if (!manifetFile.exists())
        {
            throw new IOException("The container Archive's " + manifetFile.getAbsolutePath() + " does not exist");
        }

        BufferedReader reader = new BufferedReader(new InputStreamReader(
                new FileInputStream(manifetFile)));

        String version = null;
        try
        {
            while (true)
            {
                String line = reader.readLine();
                if (line == null)
                {
                    break;
                }

                line = line.trim();

                int labelIndex = line.indexOf(MANIFEST_VERSION_LABEL);

                if (labelIndex == -1)
                {
                    continue;
                }

                return line.substring(MANIFEST_VERSION_LABEL.length());
            }
            throw new IOException("Could not find the \"" + MANIFEST_VERSION_LABEL + "\" label in \"" + manifetFile.getAbsolutePath() + "\"");
        }
        finally
        {
            // close all streams/readers including the FileInputStream
            reader.close();
        }
    }

    //This is used only by tests to simulate container archive upgrades.
    public static void setManifestVersionInContainerExpandedArchive(File archiveRoot, String testVersion) throws IOException
    {
        File manifetFile = new File(archiveRoot, CAR_MANIFEST_FILE);
        BufferedReader reader = new BufferedReader (new InputStreamReader(new FileInputStream(manifetFile)));
        File tmpFile = new File(manifetFile.getAbsolutePath() + ".tmp");
        PrintWriter writer = new PrintWriter(new BufferedWriter(new FileWriter(tmpFile, false)), true);

        while (true)
        {
            String srcLine = reader.readLine();
            if (srcLine == null)
            {
                break;
            }

            srcLine = srcLine.trim();

            String newLine = srcLine.indexOf(MANIFEST_VERSION_LABEL) != -1 ?
                             MANIFEST_VERSION_LABEL + testVersion        :
                             srcLine;

            writer.println(newLine);

        }
        reader.close();
        writer.close();

        manifetFile.delete();
        tmpFile.renameTo(manifetFile);


    }


    private void expandArchive(File archiveFile, File tempRoot) throws IOException
    {
        JarFile jarFile = new JarFile(archiveFile);
        clearDestination(tempRoot);

        Enumeration<JarEntry> entries = jarFile.entries();
        while (entries.hasMoreElements())
        {
            JarEntry entry = entries.nextElement();
            File copyFile = new File(tempRoot, entry.getName());

            if (entry.isDirectory())
            {
                if (!copyFile.exists() && !copyFile.mkdirs())
                {
                    throw new IOException("Could not create " + copyFile.getAbsolutePath());
                }
            }
            else
            {
                copyFileFromJAR(jarFile.getInputStream(entry), copyFile);
            }
        }
        jarFile.close();
    }

    private void copyFileFromJAR(InputStream stream0, File jarFile) throws IOException
    {
        BufferedInputStream is = new BufferedInputStream(stream0, m_buffer.length);
        BufferedOutputStream os = new BufferedOutputStream(new FileOutputStream(jarFile), m_buffer.length);

        while (true)
        {
            int numRead = is.read(m_buffer);

            if (numRead <= 0)
            {
                break;
            }

            os.write(m_buffer, 0, numRead);
        }

        is.close();
        os.close();
    }



    private void tailorSetupScripts(File updatedLauncherPath, boolean windows) throws IOException
    {
        Properties tailorProps = new Properties();
        tailorProps.put(JRE_PROP, ContainerSetup.JAVA_EXEC);

        tailorScript(new File (updatedLauncherPath, SH_SETUP_SCRIPT), new File (updatedLauncherPath, TEMP_SETUP_SCRIPT), tailorProps, false);
        if (ON_WINDOWS)
        {
            tailorScript(new File (updatedLauncherPath, BAT_SETUP_SCRIPT), new File (updatedLauncherPath, TEMP_SETUP_SCRIPT), tailorProps, true);
        }
    }

    private void tailorScript(File srcFile, File tempFile, Properties tailorProps, boolean batFile) throws IOException
    {
        tailorFile(srcFile, tempFile, tailorProps, batFile);
        boolean delOk = srcFile.delete();
        boolean renameOk = tempFile.renameTo(srcFile);
        if (!delOk || !renameOk)
        {
            throw new IOException("Failed to rename " + tempFile.getAbsolutePath() + " to " + srcFile.getAbsolutePath());
        }
    }

    private void tailorFile(File srcFile, File trgtFile, Properties tailorProps,  boolean batFile) throws IOException
    {
        BufferedReader reader = new BufferedReader (new InputStreamReader(new FileInputStream(srcFile)));
        PrintWriter writer = new PrintWriter(new BufferedWriter(new FileWriter(trgtFile, false)), true);

        while (true)
        {
            String srcLine = reader.readLine();
            if (srcLine == null)
            {
                break;
            }

            String commentPrefix = batFile ? BAT_COMMENT : SH_COMMENT;
            if (srcLine.toUpperCase().startsWith(commentPrefix))
            {
                writer.println(srcLine);
            }
            else
            {
                writer.println(tailorLine(srcLine, tailorProps));
            }
        }
        reader.close();
        writer.close();
    }

    private String tailorLine(String srcLine, Properties tailorProps)
    {
        Enumeration tailorList = tailorProps.keys();
        String resultLine = srcLine;

        while (tailorList.hasMoreElements())
        {
            String prop = (String)tailorList.nextElement();
            if (srcLine.indexOf(TAILOR_MARKER + prop + TAILOR_MARKER) == -1)
            {
                continue;
            }
            String val = (String)tailorProps.get(prop);

            String oldString = TAILOR_MARKER + prop + TAILOR_MARKER;
            int start = srcLine.indexOf(oldString);
            resultLine = srcLine.substring(0, start);
            resultLine += val + srcLine.substring(start + oldString.length());
        }

        return resultLine;

    }

    static File findCacheFile(File rootDir, String pathInCache)
    {
        File cachedFile = new File(rootDir, pathInCache);

        if (!cachedFile.exists())
        {
            //If the file is not there, perhaps the file was already unpacked so its unpacked
            // version is there. That happens when the archive file is cached and expanded in the container
            // cache, the JARs are unpacked - SNC00076219.
            if (pathInCache.endsWith(EXT_PACK))
            {
                int trimmedLength = pathInCache.length() - EXT_PACK.length();
                pathInCache = pathInCache.substring(0, trimmedLength);
                cachedFile = new File(rootDir, pathInCache);
                if (!cachedFile.exists())
                {
                    return null;
                }
                else
                {
                    return cachedFile;
                }
            }
            else
            {
                return null;
            }
        }
        else
        {
            return cachedFile;
        }
    }

    private void copyFiles(File launcherListFile, File srcDir, File destDir, boolean windows) throws IOException
    {

        Properties fileMap = ContainerUtil.readProperties(new FileInputStream(launcherListFile));
        Iterator mapIt = fileMap.keySet().iterator();

        while (mapIt.hasNext())
        {
            String targetPath = (String)mapIt.next();
            String srcPath = fileMap.getProperty(targetPath);


            if (!windows && windowsFile(targetPath))
            {
                continue;
            }

            File srcFile = findCacheFile(srcDir, srcPath);
            if (srcFile == null)
            {
                continue;
            }

            boolean unpack = srcFile.getName().endsWith(EXT_PACK);
            boolean dirCopy = srcFile.isDirectory();

            File destFile = new File(destDir, targetPath);

            File destParent = dirCopy ? destFile : destFile.getParentFile();

            if (!destParent.exists())
            {
                if (!destParent.mkdirs())
                {
                    throw new IOException("Failed to create destination directory " + destDir.getAbsolutePath());
                }
            }


            if (dirCopy)
            {
                copyDirectory(srcFile, destFile);
            }
            else if (unpack)
            {
                unpack(srcFile, destFile);
            }
            else
            {
                copyFile(srcFile, destFile);
            }

        }

    }

    private void unpack(File srcFile, File destFile) throws IOException
    {
        Unpacker unpacker = Pack200.newUnpacker();
        FileOutputStream fos = new FileOutputStream(destFile);
        JarOutputStream jout = new JarOutputStream(
                new BufferedOutputStream(fos));
        unpacker.unpack(srcFile, jout);
        jout.close();
    }

    private void clearDestination(File fileToDelete)
    {
      if (fileToDelete.isDirectory())
      {
          String[] files = fileToDelete.list();
          for (int i=0; i < files.length; i++)
        {
            clearDestination(new File(fileToDelete, files[i]));
        }
          fileToDelete.delete();
      }
    else
    {
        fileToDelete.delete();
    }
    }

    private void deleteFilesOfType(File directory, String type)
    {
        String[] files = directory.list();
        for (int i=0; i < files.length; i++)
        {
            File child = new File(directory, files[i]);
            if (files[i].endsWith(type))
            {
                child.delete();
            }
            else if (child.isDirectory())
            {
                deleteFilesOfType(child, type);
            }
        }
    }

    private boolean windowsFile(String path)
    {
        return path.startsWith(WINSERVICE_DIR) || path.endsWith(BAT_FILE);
    }

    private void copyDirectory(File srcDir, File destDir) throws IOException
    {
        String[] files = srcDir.list();
        for (int i = 0; i < files.length; i++)
        {
            copyFile(new File(srcDir, files[i]), new File(destDir, files[i]));
        }
    }

    private void copyFile(File srcFile, File destFile) throws IOException
    {
        copyFile(srcFile, destFile, m_buffer);
    }

    static void copyFile(File srcFile, File destFile, byte[] buffer) throws IOException
    {
        RandomAccessFile src = new RandomAccessFile(srcFile, "r");
        RandomAccessFile dest = new RandomAccessFile(destFile, "rw");

        while (true)
        {
            int numRead = src.read(buffer);

            if (numRead > 0)
            {
                dest.write(buffer, 0, numRead);
            }

            if ( numRead < buffer.length)
            {
                break;
            }
        }

        src.close();
        dest.close();
    }

}
