/* Copyright (c) 2009 Progress Software Corporation.  All Rights Reserved. */

package com.sonicsw.mf.framework.agent.ci;

import java.io.File;
import java.io.IOException;

import com.sonicsw.mf.common.util.LockFile;


//Coordinates the upgrade install of the launcher when multiple root containers use a single launcher install; and guarantees install integrity and recovery
final class LauncherInstallManager
{
    private static final String NOT_COMPLETE_FILE_NAME = "install_is_not_complete";
    File m_homeDir;
    String m_launcherDir;
    File m_installRoot;
    LockFile m_lock;
    File m_incompleteMarker;


    //For unit tests
    public static void main(String[] args) throws Exception
    {
        LauncherInstallManager mngr = new LauncherInstallManager(new File("C:/test/inst"), "test8");
        if (mngr.startInstall())
        {
            mngr.testInstall(args[0]);
        }
        mngr.installDone();
    }

    //For unit test
    void testInstall(String pName) throws Exception
    {
        new File(m_installRoot, "f1" + pName).mkdir();
        System.out.println("Created f1");
        Thread.sleep(2000);
        new File(m_installRoot, "f2" + pName).mkdir();
        Thread.sleep(2000);
        new File(m_installRoot, "f3" + pName).mkdir();
        Thread.sleep(2000);
        new File(m_installRoot, "f4" + pName).mkdir();
        System.out.println("INSTALL DONE");
    }

    LauncherInstallManager(File homeDir, String launcherDir) throws IOException
    {
        if (!homeDir.exists())
        {
            if (!homeDir.mkdirs())
            {
                throw new IOException("Sonic home directory \"" + homeDir.getAbsolutePath() + "\" does not exist and cannot be created");
            }
        }
        else if (!homeDir.isDirectory())
        {
            throw new IOException("Sonic home \"" + homeDir.getAbsolutePath() + "\" is not a directory");
        }

        File launcherParent = new  File(homeDir, com.sonicsw.mf.framework.IContainer.LAUNCHERS_ROOT_DIR);
        if (!launcherParent.exists() && !launcherParent.mkdir())
        {
            throw new IOException("Cannot create " + launcherParent.getAbsolutePath());
        }

        m_homeDir = homeDir;
        m_launcherDir = launcherDir;
        m_installRoot = new File(m_homeDir, m_launcherDir);
        m_lock = new LockFile(new File(m_homeDir, m_launcherDir + ".lock").getAbsolutePath());
        m_incompleteMarker = new File( m_installRoot, NOT_COMPLETE_FILE_NAME);
    }

    //Return 'true' if ok to perform the install under new File(homeDir, launcherDir). The caller must call installDone() when done.
    //Return 'false' if the install was already performed by another container.
    //If an install process is on going by another container, wait until that container is done (or abnormally terminated)
    boolean startInstall() throws IOException
    {
        while (!m_lock.lock())
        {
            try
            {
                Thread.sleep(1000);
            }
            catch (InterruptedException e)
            {
                throw new IOException ("LauncherInstallManager.startInstall() was interrupted while waiting on a lock");
            }
        }

        if (!m_installRoot.exists() && !m_installRoot.mkdir())
        {
            throw new IOException("Failed to create launcher directory \"" + m_installRoot.getAbsolutePath() + "\"");
        }

        String[] installFiles = m_installRoot.list();

        if (installFiles.length == 0)
        {
            m_incompleteMarker.mkdir();
            return true;
        }
        else if (m_incompleteMarker.exists())
        {
            deleteObsolete(m_installRoot);

            if (m_installRoot.exists())
            {
                throw new IOException("Failed to delete obsolete launcher directory \"" + m_installRoot.getAbsolutePath() + "\"");
            }

            if (!m_installRoot.mkdir())
            {
                throw new IOException("Failed to create launcher directory \"" + m_installRoot.getAbsolutePath() + "\"");
            }

            m_incompleteMarker.mkdir();
            return true;
        }
        else
        {
            return false;
        }
    }

    void installDone() throws IOException
    {
        if (m_incompleteMarker.exists() && !m_incompleteMarker.delete())
        {
            throw new IOException("Failed to delete marker directory \"" + m_incompleteMarker.getAbsolutePath() + "\"");
        }
        m_lock.unlock();
    }

    static void deleteObsolete(File fileToDelete)
    {
      if (fileToDelete.isDirectory())
      {
          String[] files = fileToDelete.list();
          for (int i=0; i < files.length; i++)
        {
            deleteObsolete(new File(fileToDelete, files[i]));
        }
          fileToDelete.delete();
      }
    else
    {
        fileToDelete.delete();
    }
    }



}