package com.sonicsw.mf.framework.agent.ci;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.jar.Pack200;
import java.util.jar.Pack200.Unpacker;

import com.sonicsw.mf.common.util.ZipSerializer;
import com.sonicsw.mf.framework.IContainer;


/*
#Map classes from MFcontainer.car to MFdirectory.jar

private/lib/MFagent.jar: com/sonicsw/mf/framework/*.class
private/lib/MFagent.jar: com/sonicsw/mf/framework/agent/*.class
private/lib/MFagent.jar: com/sonicsw/mf/framework/directory/*
shared/lib/MFcore.jar: com/sonicsw/mf/framework/elementversion/*.class
global/lib/MFcommon.jar: com/sonicsw/security/pass/mf/*.class
private/lib/pse.jar: *
private/lib/blackbird.jar: *
private/lib/colt.jar: *
exclude: com/sonicsw/mf/common/xml/

*/
//Build a JAR file needed by the launcher from files in the CAR archive. Used to build MFdirectory.jar from MFcontainer.car
public final class LauncherJARBuilder
{
    private static final String EXT_PACK = ".pack";

    static final String MF_DIR_JAR_PATH = "lib/MFdirectory.jar";

    private static final String SONICSW_HOME = System.getProperty(IContainer.SONICSW_HOME_PROPERTY);
    private static final String CONTAINER_ARCHIVE_TO_DIRECTORY_JAR_MAP = "launch/maps/car_to_mfdirectory_map";


    private static final String LINE_DELIM = ": ";
    private static final int LINE_DELIM_LENGTH = LINE_DELIM.length();
    private static final String EXCLUDE_PHRASE = "exclude";

    private HashMap<String, ArrayList<String>> m_jarMaps;
    private File m_archiveSourceRoot;
    private File m_jarFileToBuild;
    private File m_mapFile;
    private ArrayList<String> m_exclusions;
    private byte[] m_buffer;

    //Unit test
    public static void main(String[] args) throws IOException
    {
        createMFdirectoryJAR();
    }

    public static boolean createMFdirectoryJAR() throws IOException
    {
        if (IContainer.CURRENT_LAUNCHER_VERSION == null)
        {
            return false;
        }

        File launcherHomeDir = new File(new File(SONICSW_HOME), IContainer.LAUNCHER_DIR);
        File MFdirectoryJAR = new File(launcherHomeDir, MF_DIR_JAR_PATH);
        if (MFdirectoryJAR.exists())
        {
            return false;
        }


        String rootMFContainerPath = System.getProperty(IContainer.MF_CONTAINER_CAR_ROOT_PROPERTY);
        if (rootMFContainerPath == null)
        {
            throw new IOException("The \"" + IContainer.MF_CONTAINER_CAR_ROOT_PROPERTY + "\" system property was not defined");
        }
        File rootMFContainerArchive = new File(rootMFContainerPath);
        File carToMFdirectoryMap = new File(rootMFContainerArchive, CONTAINER_ARCHIVE_TO_DIRECTORY_JAR_MAP);


        new LauncherJARBuilder(carToMFdirectoryMap, rootMFContainerArchive, MFdirectoryJAR).buildJAR();
        return true;
    }

    static boolean mfDirectoryJARIsMissing() throws IOException
    {
        //This is relevant only with a launcher install - so return false if not a launcher install
        if (IContainer.CURRENT_LAUNCHER_VERSION == null)
        {
            return false;
        }

        File launcherHomeDir = new File(new File(SONICSW_HOME), IContainer.LAUNCHER_DIR);
        File MFdirectoryJAR = new File(launcherHomeDir, MF_DIR_JAR_PATH);
        return !MFdirectoryJAR.exists();
    }

    static void createMFdirectoryJAR(File rootMFContainerArchive, File MFdirectoryJAR) throws IOException
    {
        File carToMFdirectoryMap = new File(rootMFContainerArchive, CONTAINER_ARCHIVE_TO_DIRECTORY_JAR_MAP);
        new LauncherJARBuilder(carToMFdirectoryMap, rootMFContainerArchive, MFdirectoryJAR).buildJAR();
    }

    LauncherJARBuilder(File mapFile, File archiveSourceRoot, File jarFileToBuild)
    {
        m_jarMaps = new HashMap<String, ArrayList<String>>();
        m_exclusions = new ArrayList<String>();
        m_exclusions.add("META-INF/");
        m_buffer = new byte[1024];
        m_jarFileToBuild = jarFileToBuild;
        m_mapFile = mapFile;
        m_archiveSourceRoot = archiveSourceRoot;
    }

    public void buildJAR() throws IOException
    {
        readMap(m_mapFile);
        createJAR();
    }

    private void readMap(File mapFile) throws IOException
    {
        BufferedReader reader = null;

        try
        {
            reader = new BufferedReader(new InputStreamReader(new FileInputStream(mapFile)));
            while (true)
            {
                String line = reader.readLine();
                if (line == null)
                {
                    break;
                }

                int delimIndex = line.indexOf(LINE_DELIM);
                if (delimIndex == -1)
                {
                    continue;
                }

                String src = line.substring(0, delimIndex).trim();
                String target = line.substring(delimIndex + LINE_DELIM_LENGTH).trim();

                if (src.equalsIgnoreCase(EXCLUDE_PHRASE))
                {
                    m_exclusions.add(target);
                    continue;
                }

                ArrayList<String> targets = m_jarMaps.get(src);
                if (targets == null)
                {
                    targets = new ArrayList<String>();
                }
                targets.add(target);

                m_jarMaps.put(src, targets);
            }
        }
        finally
        {
            if (reader != null)
            {
                reader.close();
            }
        }
    }

    private void createJAR() throws IOException
    {
        if (m_jarFileToBuild.exists() && !m_jarFileToBuild.delete())
        {
            throw new IOException("\"" + m_jarFileToBuild.getAbsolutePath() + "\" already exists and cannot be deleted");
        }

        String jarName = m_jarFileToBuild.getName();
        int dotIndex = jarName.lastIndexOf(".");
        String tmpName = ((dotIndex == -1) ? jarName : jarName.substring(0, dotIndex)) + "_tmp";
        File tempDestination = new File(m_jarFileToBuild.getParentFile(), tmpName);
        clearDestination(tempDestination);
        if (tempDestination.exists())
        {
            throw new IOException("Could not delete \"" + tempDestination.getAbsolutePath() + "\" ");
        }

        try
        {
            Iterator<String> srcIter = m_jarMaps.keySet().iterator();
            while (srcIter.hasNext())
            {
                String source = srcIter.next();
                ArrayList<String> targets = m_jarMaps.get(source);
                processSource(source, targets, tempDestination);
            }
            new ZipSerializer(tempDestination).zipDir(m_jarFileToBuild);
        }
        finally
        {
            clearDestination(tempDestination);
        }
    }

    private void processSource(String source, ArrayList<String> targets, File tempDestination) throws IOException
    {
        File unpackedFile = getSourceFile(source);
        JarFile jarFile = new JarFile(unpackedFile);

        Enumeration<JarEntry> entries = jarFile.entries();
        while (entries.hasMoreElements())
        {
            JarEntry entry = entries.nextElement();
            if (!entryIsInTargetList(entry, targets))
            {
                continue;
            }

            File copyFile = new File(tempDestination, entry.getName());
            File parentToCreate = entry.isDirectory() ? copyFile : copyFile.getParentFile();

            if (!parentToCreate.exists() && !parentToCreate.mkdirs())
            {
                throw new IOException("Could not create " + parentToCreate.getAbsolutePath());
            }

            if (!entry.isDirectory())
            {
                createFileFromJAR(jarFile.getInputStream(entry), copyFile);
            }
        }
        jarFile.close();
    }

    private File getSourceFile(String source) throws FileNotFoundException,
            IOException
    {
        File srcFile = LauncherFilesInstaller.findCacheFile(m_archiveSourceRoot, source);
        if (srcFile == null)
        {
            throw new FileNotFoundException(source);
        }
        else if (!srcFile.getName().endsWith(EXT_PACK))
        {
            return srcFile;
        }

        Unpacker unpacker = Pack200.newUnpacker();
        File unpackedFile = new File(m_archiveSourceRoot, source.toString().concat(".jar"));
        FileOutputStream fos = new FileOutputStream(unpackedFile);
        JarOutputStream jout = new JarOutputStream(
                new BufferedOutputStream(fos));
        unpacker.unpack(srcFile, jout);
        jout.close();

        return unpackedFile;
    }

    private boolean entryIsInTargetList(JarEntry entry, ArrayList<String> targets)
    {
        String entryName = entry.getName();
        if (entryName.toUpperCase().endsWith(".TXT"))
        {
            return false;
        }

        for (int i = 0; i < m_exclusions.size(); i++)
        {
            if (entryName.startsWith((String)m_exclusions.get(i)))
            {
                return false;
            }
        }

        for (int i = 0; i < targets.size(); i++)
        {
            File entryFile = new File(entryName);
            File targetFile = new File(targets.get(i));

            String targetName = targetFile.getName();

            String targetParent = targetFile.getParent();
            if (targetParent == null)
            {
                targetParent = "";
            }

            String entryParent = entryFile.getParent();
            if (entryParent == null)
            {
                entryParent = "";
            }


            boolean directoryTarget = targetName.equals("*");
            if (directoryTarget)
            {

                if (entryFile.getPath().startsWith(targetParent))
                {
                    return true;
                }
            }
            else if (entryParent.equals(targetParent) && entryFile.getName().endsWith(targetName.substring(1)))
            {
                return true;
            }

        }
        return false;
    }

    private void createFileFromJAR(InputStream stream0, File createdFile) throws IOException
    {
        BufferedInputStream is = new BufferedInputStream(stream0, m_buffer.length);
        BufferedOutputStream os = new BufferedOutputStream(new FileOutputStream(createdFile), m_buffer.length);

        while (true)
        {
            int numRead = is.read(m_buffer);

            if (numRead <= 0)
            {
                break;
            }

            os.write(m_buffer, 0, numRead);
        }

        is.close();
        os.close();
    }

    private void clearDestination(File fileToDelete)
    {
      if (fileToDelete.isDirectory())
      {
          String[] files = fileToDelete.list();
          for (int i=0; i < files.length; i++)
        {
            clearDestination(new File(fileToDelete, files[i]));
        }
          fileToDelete.delete();
      }
    else
    {
        fileToDelete.delete();
    }
    }
}