package com.sonicsw.mf.framework.agent.ci;

import java.io.File;
import java.util.HashMap;
import java.util.Hashtable;

import com.odi.ReplicationController;
import com.odi.ReplicationStateHandler;

import com.sonicsw.mf.common.IComponent;
import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.ILogger;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.runtime.IContainerExitCodes;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.directory.DirectoryServiceFactory;
import com.sonicsw.mf.framework.directory.IDirectoryMFService;
import com.sonicsw.mf.framework.directory.IDirectoryService;
import com.sonicsw.mf.framework.directory.IFSStorage;
import com.sonicsw.mf.mgmtapi.config.constants.IDirectoryServiceConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IReplicationConnectionConstants;


// Provides DS access when launching a DS containing container
public final class LocalDirectoryService
{
    private IDirectoryService m_ds = null;
    private com.odi.Storage m_dsStorage = null;
    private boolean m_closing = false;

    private ILogger m_logger = null;
    private String m_domainName = null;
    private Hashtable m_dsEnv = null;
    private int m_traceMask = 0;

    private DSStateHandler m_stateHandler = null;

    // Unit testing
    public static void main(String[] args) throws Exception
    {
        String hostDir = args[0];
        String op = args[1];
        boolean primary = false;
        boolean create = false;

        if (op.equals("primary"))
        {
            primary = true;
        }
        else if (op.equals("create"))
        {
            create = true;
        }


        if (primary)
        {
            System.out.println("Start as PRIMARY");
        }
        else if (create)
        {
            System.out.println("Start as CREATE");
        }
        else
        {
            System.out.println("Start as BACKUP");
        }

        HashMap commConfig = new HashMap();

        commConfig.put("BACKUP_PORT", "2507");
        commConfig.put("PRIMARY_PORT", "2506");
        HashMap[] commConfigs = new HashMap[1];
        commConfigs[0] = commConfig;

        Hashtable directoryEnv = new Hashtable();
        directoryEnv.put(IFSStorage.HOST_DIRECTORY_ATTRIBUTE, hostDir);
        directoryEnv.put(IReplicationConnectionConstants.REPLICATION_CONNECTIONS_ATTR, commConfigs);
        directoryEnv.put(IDirectoryService.STORAGE_TYPE_ATTRIBUTE, IDirectoryService.PSE_STORAGE);
        directoryEnv.put(IDirectoryServiceConstants.DUAL_ACTIVE_RESOLUTION_ATTR, Boolean.TRUE);

        LocalDirectoryService ld = new LocalDirectoryService();
        IDirectoryAdminService ds;
        if (create)
        {
            ds = ld.getDS(new TestLogger(), "Domain1", directoryEnv);
            return;
        }

        ld.openReplicatedStorage(new TestLogger(), "Domain1", directoryEnv, new Boolean(primary), Boolean.FALSE);
        ds = ld.getReplicatedDS();
        System.out.println("Got ds " + ds);
    }

    static class TestLogger implements ILogger
    {
        @Override
        public void logMessage(String message, Throwable exception, int severityLevel)
        {
            System.out.println(message + " Cause: ");
            exception.printStackTrace();
        }
        @Override
        public void logMessage(String message, int severityLevel)
        {
            System.out.println(message);
        }
    }

    public IDirectoryAdminService getDS(ILogger logger,
                                 String domainName,
                                 String hostDir,
                                 String encryptionPassword,
                                 Boolean isSharedHostDirectory) throws Exception
    {
        m_logger = logger;
        m_domainName = domainName;

        Hashtable directoryEnv = new Hashtable();
        directoryEnv.put(IDirectoryService.STORAGE_TYPE_ATTRIBUTE, IDirectoryService.PSE_STORAGE);

        if (hostDir != null)
        {
            directoryEnv.put(IFSStorage.HOST_DIRECTORY_ATTRIBUTE, hostDir);
        }

        if ((encryptionPassword != null) && (encryptionPassword.length() != 0))
        {
            directoryEnv.put(IFSStorage.PASSWORD_ATTRIBUTE, encryptionPassword);
        }

        if (isSharedHostDirectory.booleanValue())
        {
            directoryEnv.put(IDirectoryMFService.IS_SHARED_STORAGE, Boolean.TRUE);
        }

        directoryEnv.put(IDirectoryMFService.CI_FIRST_PHASE_ATTRIBUTE, Boolean.TRUE);

        DirectoryServiceFactory factory = new DirectoryServiceFactory(directoryEnv);
        m_ds = factory.createDirectoryService(domainName, logger);
        return m_ds;
    }

    public IDirectoryAdminService getDS(ILogger logger, String domainName, Hashtable dsEnv) throws Exception
    {
        m_logger = logger;
        m_domainName = domainName;
        m_dsEnv = dsEnv;
        Integer tmp = (Integer)dsEnv.get("_TRACE_MASK");
        if (tmp != null)
        {
            m_traceMask = tmp.intValue();
        }

        return getDS();
    }

    private IDirectoryAdminService getDS() throws Exception
    {
        m_dsEnv.put(IDirectoryMFService.CI_FIRST_PHASE_ATTRIBUTE, Boolean.TRUE);
        m_dsEnv.put(IDirectoryService.STORAGE_TYPE_ATTRIBUTE, IDirectoryService.PSE_STORAGE);

        DirectoryServiceFactory factory = new DirectoryServiceFactory(m_dsEnv);
        m_ds = factory.createDirectoryService(m_domainName, m_logger);
        return m_ds;
    }

    // Get the local DS in case of DS replication. In that case, we don't nitially know whether the local DS is the one that will be active.
    // So the process of activating the DS happens asynchronously: First waiting tosee what is the initial state of the local DS. If the state
    // becomes active then we open the DS asynchronously (so we don't block the PSE state reporting mechanism) and wait until activation
    // is done.
    public void openReplicatedStorage(ILogger logger,
                                      String domainName,
                                      Hashtable dsEnv,
                                      Boolean isPrimary,
                                      Boolean startActive) throws Exception
    {
        m_logger = logger;
        m_logger.logMessage("Starting Directory Service Replication with the Start Active option " +
                            (startActive.booleanValue() ? "enabled" : "disabled"),  Level.INFO);
        m_domainName = domainName;
        m_dsEnv = dsEnv;
        Integer tmp = (Integer)dsEnv.get("_TRACE_MASK");
        if (tmp != null)
        {
            m_traceMask = tmp.intValue();
        }

        File hostDir = new File((String)dsEnv.get(IFSStorage.HOST_DIRECTORY_ATTRIBUTE));
        if (!hostDir.exists())
        {
            throw new DirectoryServiceException("Directory Service  host directory \"" + hostDir.getPath() + "\" was not found.");
        }

        File domainDir = new File(hostDir, domainName);

        // If this DS instance is going to be standby and was not yet created then we have to create it so we have a place to create the new storage
        if (!domainDir.exists() && !domainDir.mkdir())
        {
            throw new DirectoryServiceException("Domain directory \"" + domainDir.getPath() + "\" was not found and could not be created.");
        }

        File storageDir = new File(domainDir, IDirectoryMFService.DS_STORAGE_NAME);

        HashMap storageParameters = new HashMap();
        storageParameters.put("NO_READ_LOCK", Boolean.TRUE);

        storageParameters.put(com.odi.Storage.REPLICATION_RETRY_INTERVAL_ATTR, dsEnv.get(IDirectoryServiceConstants.RETRY_INTERVAL_ATTR));
        storageParameters.put(com.odi.Storage.REPLICATION_PING_INTERVAL_ATTR, dsEnv.get(IDirectoryServiceConstants.PING_INTERVAL_ATTR));
        storageParameters.put(com.odi.Storage.REPLICATION_FAILURE_DETECTION_TIMEOUT_ATTR,
                              dsEnv.get(IDirectoryServiceConstants.FAILURE_DETECTION_TIMEOUT_ATTR));
        storageParameters.put(com.odi.Storage.MAX_REPLICATION_LOG_SIZE_ATTR, dsEnv.get(IDirectoryServiceConstants.MAX_REPLICATION_LOG_SIZE_ATTR));
        storageParameters.put(com.odi.Storage.REPLICATION_TIMEOUT_ATTR, dsEnv.get(IDirectoryServiceConstants.REPLICATION_TIMEOUT_ATTR));
        storageParameters.put(com.odi.Storage.REPLICATION_SSL_ATTR, dsEnv.get(IDirectoryServiceConstants.SSL_PARAMETERS_ATTR));
        boolean trace = (m_traceMask & LaunchContainer.CONTAINER_LAUNCH_TRACE_MASK) > 0;
        boolean trace_verbose = (m_traceMask & IComponent.TRACE_VERBOSE) > 0;
        com.odi.Storage.setReplicationTracing(trace, trace_verbose);

        //This connection is temporary in the the sense that it's established only to determine
        // which storage instance should be active and then closed. The hint that this connection
        // is temporary is used to avoid logging the the loss of the connection (on the peer side).
        storageParameters.put("_TEMPORARY_CONNECTION", Boolean.TRUE);

        m_stateHandler =  new DSStateHandler();

        try
        {
            m_dsStorage = com.odi.Storage.openStorage(new DSReplicationConroller(),
                                                      m_stateHandler,
                                                      storageDir.getAbsolutePath(),
                                                      storageParameters,
                                                      isPrimary.booleanValue(),
                                                      startActive.booleanValue(),
                                                      ((Boolean)dsEnv.get(IDirectoryServiceConstants.DUAL_ACTIVE_RESOLUTION_ATTR)).booleanValue(),
                                                      (HashMap[])dsEnv.get(IReplicationConnectionConstants.REPLICATION_CONNECTIONS_ATTR));
        }
        catch (Exception e)
        {
            m_logger.logMessage("Failed to open the Directory Service store: " + e.toString(), Level.SEVERE);
            System.exit(IContainerExitCodes.DS_FAILURE_EXIT_CODE);
        }

    }

    public IDirectoryAdminService getReplicatedDS() throws Exception
    {
        //Sanity check
        if (m_dsStorage == null || m_stateHandler == null)
        {
            throw new Exception("Storage is not open.");
        }

        return getDSWhenReady(m_stateHandler);
    }

    private IDirectoryAdminService getDSWhenReady(DSStateHandler stateHandler) throws Exception
    {

        stateHandler.waitForInitialState();
        if (!stateHandler.m_activationStarted)
        {
            return null;
        }

        //Activation was started, wait for it to be done
        if (stateHandler.waitForActivation())
        {
            return m_ds;
        }
        else if (stateHandler.m_activationFailure != null)
        {
            if (stateHandler.m_activationFailure instanceof Exception)
            {
                throw (Exception)stateHandler.m_activationFailure;
            }
            else
            {
                throw new Exception(stateHandler.m_activationFailure.toString());
            }
        }
        else
        {
            throw new Exception("Failed to open the Directory Service");
        }
   }



    private class DSStateHandler extends ReplicationStateHandler
    {
       private boolean m_gotInitialState = false;
       private boolean m_activationSucceeded = false;
       private boolean m_activationFailed = false;
       private boolean m_activationStarted = false;

       Throwable m_activationFailure = null;

       @Override
    public void newState(short state)
       {
           if (state == DEEP_SYNCHRONIZING_STANDBY)
        {
            m_logger.logMessage("Using Deep Synchronization to replicate the Directory Service",  Level.INFO);
        }
           if (state == SYNCHRONIZING_STANDBY)
        {
            m_logger.logMessage("The Directory Service is performing replication synchronization",  Level.INFO);
        }
           if ((m_traceMask & LaunchContainer.CONTAINER_LAUNCH_TRACE_MASK) != 0)
        {
            m_logger.logMessage("Storage replication state changed to " + stateToString(state),  Level.TRACE);
        }
           setState(state);
       }

       @Override
    public void shutdownState(String shutdownReason, Exception e)
       {
           m_logger.logMessage(shutdownReason, e, Level.SEVERE);

           int exitCode = IContainerExitCodes.DS_FAILURE_EXIT_CODE;
           if (shutdownReason != null && shutdownReason.indexOf("Both") != -1 && shutdownReason.indexOf("unreplicated") != -1)
        {
            exitCode = IContainerExitCodes.DS_DUAL_ACTIVE_EXIT_CODE;
        }
           System.exit(exitCode);
       }

       @Override
    public void newMessage(String message, boolean warning)
       {
           m_logger.logMessage(message, warning ? Level.WARNING : Level.INFO);
       }

       private synchronized void setActivationResult(boolean ok, Throwable failureCause)
       {
           if (ok)
        {
            m_activationSucceeded = true;
        }
        else
           {
               m_activationFailed = true;
               m_activationFailure = failureCause;
           }

           notifyAll();
       }

       private synchronized void close()
       {
           m_closing = true;
           notifyAll();
       }

       private synchronized void setState(short state)
       {

           m_gotInitialState = true;
           notifyAll();
           if (state == SYNCHRONIZING_ACTIVE || state == ACTIVE)
        {
            activateInThread();
        }
       }

       private synchronized void waitForInitialState() throws InterruptedException
       {
           while (!m_gotInitialState && !m_closing)
        {
            wait();
        }
       }

       private synchronized boolean waitForActivation() throws InterruptedException
       {
           while (!m_activationSucceeded && !m_activationFailed && !m_closing)
        {
            wait();
        }

           return m_activationSucceeded ? true : false;
       }

       private void activateInThread()
        {
            if (m_activationStarted)
            {
                return;
            }

            m_activationStarted = true;
            if (m_closing)
            {
                return;
            }

            Thread activation = new Thread("com.sonicsw.mf.framework.agent.ci.LocalDirectoryService launcher Fault Tolerant Activation Thread")
            {
                @Override
                public void run()
                {
                    try
                    {
                        if (m_closing)
                        {
                            return;
                        }
                        getDS();
                        setActivationResult(true, null);
                    }
                    catch (Throwable t)
                    {
                        setActivationResult(false, t);
                    }
                }
            };
            activation.setDaemon(true);
            activation.start();
        }
    }

    private class DSReplicationConroller extends ReplicationController
    {
        @Override
        public boolean allowedToGetActive()
        {
              return true;
        }
    }

    public void close()throws Exception
    {
        if (m_ds != null)
        {
            m_ds.close();
        }
        m_ds = null;
        if (m_dsStorage != null)
        {
            m_dsStorage.close();
        }
        m_dsStorage = null;
        if (m_stateHandler != null)
        {
            m_stateHandler.close();
        }
    }

}

