package com.sonicsw.mf.framework.agent.ci;

import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;

import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.ILogger;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.agent.cache.IConfigCache;
import com.sonicsw.mf.framework.agent.cache.IConfigCacheView;
import com.sonicsw.mf.framework.agent.cache.LatestVersionMissingException;


// Caching missing or obsolete resources is done as following:
// 1) Put in cache all the envelope elements that are missing or obsolete.
// 2) Fetch cacheand the files cache the files for envelope elements that don't have the recent version
// This alogorithm insures that if we fail in the middle we don't have obsolete files but rather only
// missing files.
final class PopulateCacheResources
{
    private IDirectoryAdminService m_ds;
    private IConfigCache m_configCache;
    private IConfigCacheView m_configCacheView;
    private String[] m_resourceList;
    private ILogger m_logger;

    PopulateCacheResources(IDirectoryAdminService ds, IConfigCache configCache, String[] resourceList, ILogger logger)
    {
        m_ds = ds;
        m_configCache = configCache;
        m_resourceList = resourceList;
        m_logger = logger;
    }

    void updateCache() throws Exception
    {
        m_configCacheView = m_configCache.getCacheView();
        ArrayList updateRequiredList = getUpdateList();

        if (!updateRequiredList.isEmpty())
        {
            logMessage("Updating the container cache...", Level.INFO);
        }

        updateResources(updateRequiredList);

        if (!updateRequiredList.isEmpty())
        {
            logMessage("Cache update is done.", Level.INFO);
        }
    }

    private ArrayList getUpdateList() throws Exception
    {
        ArrayList updateList = new ArrayList();
        if (m_resourceList.length == 0)
        {
            return updateList;
        }

        IElement[] envelopeElements = m_ds.getElementsByLogicalNames(m_resourceList);
        for (int i = 0; i < envelopeElements.length; i++)
        {
            if (envelopeElements[i] == null) // Not in the DS
            {
                logMessage("Could not find file \"" + DeployContainerResources.sonicfsPrefix +
                            m_resourceList[i] + "\" in the Directory Service", Level.WARNING);
                continue;
            }

            IElementIdentity dsID =  envelopeElements[i].getIdentity();
            IElement cacheElement = m_configCacheView.getElement(dsID.getName());

            // Not in the cache
            boolean cacheObsolete = false;
            if (cacheElement == null || cacheElement.getIdentity().getVersion() < dsID.getVersion())
            {
                cacheObsolete = true;
            }
            else // Check the case the envelope element in the cache is up-to-date, but not the actual file
            {
                File file = null;
                try
                {
                    file = m_configCacheView.getFileByLogicalName(m_resourceList[i], false);
                }
                catch (LatestVersionMissingException e){}

                if (file == null)
                {
                    cacheObsolete = true;
                }
            }

            if (cacheObsolete)
            {
                updateList.add(m_resourceList[i]);
                m_configCache.setElement(envelopeElements[i]);
            }
        }

        return updateList;
    }


    private void logMessage(String message, int sevirityLevel)
    {
        m_logger.logMessage(message, sevirityLevel);
    }

    private void updateResources(ArrayList updateRequiredList) throws Exception
    {
        // Some of the resources might be contained in  the same EXPAND_IN_CACHE
        // archive, so we have to make sure we don't store twice
        HashSet blobsSeen = new HashSet();

        for (int i = 0; i < updateRequiredList.size(); i++)
        {
            String logicalName = (String)updateRequiredList.get(i);
            IBlob blob = m_ds.getBlobByLogicalName(logicalName);
            if (blob == null)
            {
                continue;
            }
            if (blob.getBlobStream() == null)
            {
                throw new Exception("The data of file \"" + logicalName + "\" is missing from the Directory Service.");
            }
            IElement blobElement = blob.getElement();
            String storageName = blobElement.getIdentity().getName();
            if (!blobsSeen.contains(storageName))
            {
                String archiveName = blobElement.getArchiveName();
                if (archiveName == null)
                {
                    archiveName = logicalName;
                }
                m_configCache.setBlobByLogicalName(archiveName, blob, false, true);
                blobsSeen.add(storageName);
                logMessage("Stored \"" + archiveName + "\" in the cache.", Level.INFO);
            }
        }
    }
}
