package com.sonicsw.mf.framework.agent.ci;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;

import javax.management.ObjectName;

import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.ILogger;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.agent.ContainerUtil;
import com.sonicsw.mf.jmx.client.DirectoryServiceProxy;
import com.sonicsw.mf.jmx.client.JMSConnectorAddress;
import com.sonicsw.mf.jmx.client.JMSConnectorClient;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;

final class SetCIConfigUtility
{
    private static final String[] EMPTY_STRING_ARRAY = new String[0];

    private String m_containerBootFileName;
    private String m_containerName;
    private JMSConnectorClient m_connector;
    private IDirectoryAdminService m_ds;
    private String m_containerID;
    private ExtractedContainerConfig m_bootExtractedConfig;
    private static ILogger m_logger = new CILogger("Centralized Install Configuration Utility");

    // Configuration
    ExtractedContainerConfig m_extractedConfig;   // container configuration we use



    public static void main (String[] args)
    {
        if (args.length != 2)
        {
            System.out.println("Usage: java com.sonicsw.mf.framework.agent.ci.SetCIConfigUtility <container boot file> <attributes file>");
            System.exit(1);
        }
        try
        {
            new SetCIConfigUtility(args[0], args[1]);
        }
        catch (Throwable t)
        {
            logMessage("", t,  Level.SEVERE);
            System.exit(1);
        }
    }

    SetCIConfigUtility(String containerBootFileName, String attributesFile) throws Exception
    {
        File containerBootFile = new File(containerBootFileName);
        m_containerBootFileName = containerBootFile.getAbsolutePath();
        if (!containerBootFile.exists() || !containerBootFile.isFile() || !containerBootFile.canRead())
        {
            String problem = !containerBootFile.exists() ?  " not found." : " cannot be read";
            String errMessage = "Bootfile " + containerBootFileName + problem;
            logMessage(errMessage, Level.SEVERE);
            throw new Exception(errMessage);
        }
        logMessage("Open container boot file \"" + containerBootFile.getAbsolutePath() + '"', Level.INFO);


        IElement containerConfiguration = ContainerUtil.importConfiguration(containerBootFile, IContainer.PASSWORD);
        m_containerID = containerConfiguration.getIdentity().getName();
        IAttributeSet configAttributes = containerConfiguration.getAttributes();
        m_bootExtractedConfig = new ExtractedContainerConfig();
        extractConfigInfo(configAttributes, m_bootExtractedConfig);
        m_containerName = createContainerFullName(m_bootExtractedConfig);

        logMessage("Updating container configuration \"" + m_containerName + "\"...", Level.INFO);

        createDSProxy(createConnector(configAttributes), m_bootExtractedConfig.m_domainName);
        ExtractedContainerConfig dsExtractedConfig = new ExtractedContainerConfig();
        populateProperties(attributesFile);
        logMessage("...configuration update complete", Level.INFO);
        System.exit(0);
    }

    private void populateProperties(String attFileName) throws Exception
    {
        logMessage("Update container configuration ID \"" + m_containerID +
                   "\" in the Directory Service from file \"" + attFileName +"\"", Level.INFO);

        IDirElement configElement = m_ds.getElement(m_containerID, true);
        IAttributeSet configAtts = configElement.getAttributes();
        IAttributeSet jvmArgs = (IAttributeSet)configAtts.getAttribute(IContainerConstants.JVM_ARGUMENTS_ATTR);

        BufferedReader reader = new BufferedReader
            (new InputStreamReader(new FileInputStream(attFileName)));

        boolean updated = false;
        while (true)
        {
            String line = reader.readLine();
            if (line == null)
            {
                break;
            }

            updated = true;

            int eqIndex = line.indexOf('=');
            String key = line.substring(0, eqIndex);
            String value = line.substring(eqIndex+1);

            logMessage("Set " + key + " to " + value,  Level.INFO);

            if (key.startsWith(IContainerConstants.JVM_ARGUMENTS_ATTR))
            {
                if (jvmArgs == null)
                {
                    jvmArgs = configAtts.createAttributeSet(IContainerConstants.JVM_ARGUMENTS_ATTR);
                }
                jvmArgs.setStringAttribute(
                    key.substring(IContainerConstants.JVM_ARGUMENTS_ATTR.length() + 1), value);
            }
            else
            {
                configAtts.setStringAttribute(key, value);
            }
        }
        if (updated)
        {
            m_ds.setElement(configElement.doneUpdate(), null);
        }
    }

/*
    private void populateProperties(String attributesFile) throws Exception
    {
        logMessage("Update container configuration configuration ID \"" + m_containerID +
                   "\" in the Directory Service.", Level.INFO);

        try
        {
            Properties props = new Properties();
            props.load(new FileInputStream(attributesFile));
            Enumeration keyEnum = props.keys();

            IDirElement configElement = m_ds.getElement(m_containerID, true);
            IAttributeSet configAtts = configElement.getAttributes();
            IAttributeSet jvmArgs = (IAttributeSet)configAtts.getAttribute(IContainerConstants.JVM_ARGUMENTS_ATTR);

            boolean updated = false;
            while(keyEnum.hasMoreElements())
            {
                updated = true;
                String key = (String)keyEnum.nextElement();
                String value = (String)props.get(key);

                // Some clumsy code to work around the Properties limitation of having ":" in keys
                if (value.length() > 0 && value.charAt(0) == '=')
                {
                    value = value.substring(1);
                    key += ":";
                }

                logMessage("Set " + key + " TO " + value,  Level.INFO);

                if (key.startsWith(IContainerConstants.JVM_ARGUMENTS_ATTR))
                {
                    if (jvmArgs == null)
                        jvmArgs = configAtts.createAttributeSet(IContainerConstants.JVM_ARGUMENTS_ATTR);
                    jvmArgs.setStringAttribute(key.substring(IContainerConstants.JVM_ARGUMENTS_ATTR.length() + 1), value);
                }
                else
                    configAtts.setStringAttribute(key, value);
            }
            if (updated)
                m_ds.setElement(configElement.doneUpdate(), null);

        }
        catch (Exception e)
        {
            throw e;
        }
    }
*/

    private void createDSProxy(JMSConnectorClient connector, String domainName) throws Exception
    {
        m_ds = new DirectoryServiceProxy(connector, new ObjectName(domainName + "." + IContainer.DS_ADDRESS));
        m_ds.getDirectoryServiceVersion(); // Check the accessibility of the DS
    }


    private void extractConfigInfo(IAttributeSet containerAttrs, ExtractedContainerConfig containerConfig) throws Exception
    {
        containerConfig.m_domainName = (String)containerAttrs.getAttribute(IContainerConstants.DOMAIN_NAME_ATTR);
        if (containerConfig.m_domainName == null)
        {
            containerConfig.m_domainName = IContainerConstants.DOMAIN_NAME_DEFAULT;
        }

        containerConfig.m_containerName = (String)containerAttrs.getAttribute(IContainerConstants.CONTAINER_NAME_ATTR);

        containerConfig.m_classpath = (String)containerAttrs.getAttribute(IContainerConstants.CLASSPATH_ATTR);

        containerConfig.m_deployTraceMask = (Integer)containerAttrs.getAttribute(IContainerConstants.TRACE_MASK_ATTR);
        if (containerConfig.m_deployTraceMask == null)
        {
            containerConfig.m_deployTraceMask = new Integer(IContainerConstants.TRACE_MASK_DEFAULT);
        }

        IAttributeSet cacheAttrs = (IAttributeSet)containerAttrs.getAttribute(IContainerConstants.CACHE_ATTR);

        containerConfig.m_cacheHostDirectoryName = (String)cacheAttrs.getAttribute(IContainerConstants.CACHE_DIRECTORY_ATTR);
        if (containerConfig.m_cacheHostDirectoryName == null)
        {
            containerConfig.m_cacheHostDirectoryName = IContainerConstants.CACHE_DIRECTORY_DEFAULT;
        }
        containerConfig.m_cachePassword = (String)cacheAttrs.getAttribute(IContainerConstants.PASSWORD_ATTR);

        IAttributeSet jvmArgs = (IAttributeSet)containerAttrs.getAttribute(IContainerConstants.JVM_ARGUMENTS_ATTR);
        if (jvmArgs != null)
        {
            containerConfig.m_jvmArgs = jvmArgs.getAttributes();
        }
        else
        {
            containerConfig.m_jvmArgs = new HashMap();
        }
    }

    private JMSConnectorClient createConnector(IAttributeSet configAttributes) throws Exception
    {

        IAttributeSet connectionAttrs = (IAttributeSet)configAttributes.getAttribute(IContainerConstants.CONNECTION_ATTR);
        Map factoryAttrs = (Map)connectionAttrs.getAttributes().clone();
        factoryAttrs.remove(IContainerConstants.MANAGEMENT_NODE_ATTR);
        factoryAttrs.remove(IContainerConstants.REQUEST_TIMEOUT_ATTR);
        factoryAttrs.remove(IContainerConstants.CONNECT_TIMEOUT_ATTR);
        factoryAttrs.remove(IContainerConstants.SOCKET_CONNECT_TIMEOUT_ATTR);

        Hashtable env = new Hashtable(factoryAttrs);
        if (env.get(IContainerConstants.CONNECTIONURLS_ATTR) == null)
        {
            env.put(IContainerConstants.CONNECTIONURLS_ATTR, IContainer.CONNECTIONURLS_DEFAULT);
        }
        JMSConnectorAddress address = new JMSConnectorAddress(env);

        String managementNode = (String)connectionAttrs.getAttribute(IContainerConstants.MANAGEMENT_NODE_ATTR);
        if (managementNode != null && managementNode.length() > 0)
        {
            address.setManagementNode(managementNode);
        }

        JMSConnectorClient connector = new JMSConnectorClient();

        // REQUEST_TIMEOUT_ATTR is in seconds, setRequestTimeout is passed millis
        Integer requestTimeout = (Integer)connectionAttrs.getAttribute(IContainerConstants.REQUEST_TIMEOUT_ATTR);
        if (requestTimeout != null)
        {
            connector.setRequestTimeout(requestTimeout.intValue() * 1000);
        }

        // CONNECT_TIMEOUT_ATTR is in seconds, setConnectTimeout is passed millis
        Integer connectTimeout = (Integer)connectionAttrs.getAttribute(IContainerConstants.CONNECT_TIMEOUT_ATTR);
        if (connectTimeout != null)
        {
            connector.setConnectTimeout(connectTimeout.intValue() * 1000);
        }

        // SOCKET_CONNECT_TIMEOUT_ATTR is in seconds, passed to connect() in millis
        Integer socketConnectTimeout = (Integer)connectionAttrs.getAttribute(IContainerConstants.SOCKET_CONNECT_TIMEOUT_ATTR);
        if (socketConnectTimeout != null)
        {
            connector.setSocketConnectTimeout(socketConnectTimeout.intValue() * 1000);
        }

        logMessage("Connecting to the Directory Service to refresh the container's resources",  Level.INFO);
        connector.connect(address);

        m_connector = connector;
        return connector;

    }

    static void logMessage(String message, int severityLevel)
    {
        m_logger.logMessage(message, severityLevel);
    }

    static void logMessage(String message, Throwable t, int severityLevel)
    {
        m_logger.logMessage(message, t, severityLevel);
    }

    private static String createContainerFullName(ExtractedContainerConfig extractedConfig)
    {
        return createContainerFullName(extractedConfig.m_domainName,  extractedConfig.m_containerName);
    }

    private static String createContainerFullName(String domainName, String containerName)
    {
        return domainName + "." + containerName;
    }

    // This configuration is extracted from the container.xml boot file or from the
    // actual DS configuration. By incapsulating it in class, we can use the same code
    // in both cases and reconcile differences according to some rules.
    private static class ExtractedContainerConfig implements Serializable
    {
        String m_domainName = null;
        String m_containerName = null;
        String m_classpath = null;
        Integer m_deployTraceMask = null;
        String  m_cacheHostDirectoryName = null;
        String m_cachePassword = null;
        HashMap m_jvmArgs;

        ExtractedContainerConfig deepClone()
        {
          try
          {
               ByteArrayOutputStream out = new ByteArrayOutputStream();
               ObjectOutputStream objectOut = new ObjectOutputStream(out);
               objectOut.writeObject(this);
               byte[] bytes = out.toByteArray();
               objectOut.close();

               ByteArrayInputStream in = new ByteArrayInputStream(bytes);
               ObjectInputStream objectIn = new ObjectInputStream(in);
               return (ExtractedContainerConfig) objectIn.readObject();
          }
          catch (Exception e)
          {
                throw new Error("deepClone failed.");
          }
        }


    }

}
