package com.sonicsw.mf.framework.agent.ci;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.StringTokenizer;

import com.sonicsw.mf.common.ILogger;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.agent.ContainerUtil;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;

public final class StartContainerCommand
{
    private static boolean ON_WINDOWS = File.separatorChar == '\\';
    private static final String WHITE_SPACE_CHARS = " \t\r\n\f";
    private static final String PATH_SEPARATOR = System.getProperty("path.separator");
    private static final String DEV_JVM_ARGS_PROPERTY = "sonicsw.mf.devJVMArgs";
    private static final String CONTAINER_CLASS = "com.sonicsw.mf.framework.agent.ci.Agent";
    private static final String QUOTE_STRING = "\"";
    private static final String[] EMPTY_STRING_ARRAY = new String[0];

    private ContainerResources[] m_deployments;
    private int lastDeploymentIndex;
    private String m_jvmHome;
    private String m_devArgsFileName;
    private String m_nativePath;
    private QuotedItemsList m_execCommand;
    private ILogger m_logger;
    private boolean m_doRestart;

    public StartContainerCommand(ContainerResources[] deployments, String jvmHome, ILogger logger) throws Exception
    {
        m_logger = logger;
        m_doRestart = false;
        m_deployments = deployments;
        m_jvmHome = jvmHome;
        lastDeploymentIndex = m_deployments.length - 1;
    }

    StartContainerCommand(ContainerResources[] deployments, String jvmHome, ILogger logger, boolean doRestart) throws Exception
    {
        m_logger = logger;
        m_doRestart = doRestart;
        m_deployments = deployments;
        m_jvmHome = jvmHome;
        lastDeploymentIndex = m_deployments.length - 1;
    }

    public void createExecCommand(boolean doQuote, boolean specifyLibraryPath) throws Exception
    {
        m_execCommand = new QuotedItemsList(doQuote);

        m_execCommand.add(m_jvmHome + File.separatorChar + "bin" +  File.separatorChar + "java" + (ON_WINDOWS ? ".exe" : ""));

        if (m_doRestart)
        {
            m_execCommand.add("-Dsonicsw.mf.launcher_upgrade_restart=true");
        }

        if (m_deployments.length == 1)
        {
            m_execCommand.add("-D" + IContainer.CONTAINER_ID_PROPERTY + '=' + m_deployments[lastDeploymentIndex].m_containerName);
        }
        
        String d64 = System.getProperty("sonicsw.mf.dev.d64");
        if (d64 != null && d64.equalsIgnoreCase("true"))
        {
            m_execCommand.add("-d64");
        }

        m_execCommand.addAll(extractJVMArgs());

        String[] bootList = m_deployments[lastDeploymentIndex].m_classpath.getBootClasspath();
        if (bootList.length > 0)
        {
            m_execCommand.add(DeployContainerResources.XARG_BOOT + createClasspathString(bootList));
        }

        bootList = m_deployments[lastDeploymentIndex].m_classpath.getBootPrependClasspath();
        if (bootList.length > 0)
        {
            m_execCommand.add(DeployContainerResources.XARG_BOOT_P + createClasspathString(bootList));
        }

        bootList = m_deployments[lastDeploymentIndex].m_classpath.getBootAppendClasspath();
        if (bootList.length > 0)
        {
            m_execCommand.add(DeployContainerResources.XARG_BOOT_A + createClasspathString(bootList));
        }

        m_nativePath = generateNativePath(specifyLibraryPath);
        if (specifyLibraryPath)
        {
            m_execCommand.add("-D" +  "java.library.path" + "=" + m_nativePath);
        }

        if (m_deployments[lastDeploymentIndex].m_localDSParam != null && m_deployments[lastDeploymentIndex].m_localDSParam.m_isFaultTolerant.booleanValue())
        {
            String fsSanityProperty = m_deployments[lastDeploymentIndex].m_localDSParam.m_isBackup ? ("-D" + IContainer.SANITY_DS_FT_ROLE_BACKUP + "=true") :
                                                                           ("-D" + IContainer.SANITY_DS_FT_ROLE_PRIMARY + "=true");
            m_execCommand.add(fsSanityProperty);
        }

        // Add to the command the configured system properties
        HashMap systemProperties = m_deployments[lastDeploymentIndex].m_systemProps.getProperties();
        
        // test if there are any AOP based Actional interceptors configured by looking for the
        // system property "com.actional.aops"; if so then add the -javaagent argument
        String actionalAOPproperty = (String)systemProperties.get("com.actional.aops");
        if (actionalAOPproperty != null && actionalAOPproperty.length() > 0)
        {
            if (m_deployments[lastDeploymentIndex].m_actionalPlugMakerArchive == null)
            {
                throw new Exception("The " + IContainerConstants.ACTIONAL_PLUGMAKER_ARCHIVE_NAME_ATTR +
                                    " attribute is missing from the container's configuration or the archive could not be found.");
            }
            else
            {
                m_execCommand.add("-javaagent:" + m_deployments[lastDeploymentIndex].m_actionalPlugMakerArchive.getPath());
            }
        }
        
        // SNC00076492 remove redundant/misleading properties carried forward
        // from parent container if this container is launched via an ActivationDaemon
        systemProperties.remove(IContainer.CONTAINER_ID_PROPERTY);
        systemProperties.remove(IContainer.MF_CONTAINER_CAR_ROOT_PROPERTY); 
        
        Iterator propKeys = systemProperties.keySet().iterator();
        while (propKeys.hasNext())
        {
            String propName = (String)propKeys.next();
            String propValue = (String)systemProperties.get(propName);
            if (propValue == null)
            {
                propValue = "MISSING_FROM_CACHE";
            }
            m_execCommand.add("-D" + propName + "=" + propValue);
        }
        
        // All the user-configured system properties have been added.  Now add
        // internal/external host properties if the user hasn't already set them.
        // Reading these properties back from what we've built of the command line
        // so far means we should pick them up whether they've been set in the
        // container configuration (in the DS) or in the current Java environment.
        String skipEC2 = m_execCommand.getProperty("-D" + IContainer.CONTAINER_SKIP_EC2_CHECKS_PROPERTY);
        String traceEC2 = m_execCommand.getProperty("-D" + IContainer.CONTAINER_TRACE_EC2_CHECKS_PROPERTY);
        String timeoutEC2 = m_execCommand.getProperty("-D" + IContainer.CONTAINER_TIMEOUT_EC2_CHECKS_PROPERTY);

        int timeout = IContainer.CONTAINER_TIMEOUT_EC2_CHECKS_DEFAULT;
        try
        {
            timeout = Integer.parseInt(timeoutEC2);
        }
        catch (NumberFormatException nfe)
        {
            // expected
        }

        HostHelper hostHelper = new HostHelper(m_logger,
                                               skipEC2 != null && skipEC2.equalsIgnoreCase("true"),
                                               traceEC2 != null && traceEC2.equalsIgnoreCase("true"),
                                               timeout);
        
        if (m_execCommand.getProperty("-D" + IContainer.CONTAINER_PRIVATE_HOST_PROPERTY) == null)
        {
            String privateHost = hostHelper.getPrivateHost();
            if (privateHost != null)
            {
                m_execCommand.add("-D" + IContainer.CONTAINER_PRIVATE_HOST_PROPERTY + '=' + privateHost);
            }
        }
        
        if (m_execCommand.getProperty("-D" + IContainer.CONTAINER_PUBLIC_HOST_PROPERTY) == null)
        {
            String publicHost = hostHelper.getPublicHost();
            if (publicHost != null)
            {
                m_execCommand.add("-D" + IContainer.CONTAINER_PUBLIC_HOST_PROPERTY + '=' + publicHost);
            }
        }
        

        ContainerUtil.storeCachePassword(new File(m_deployments[lastDeploymentIndex].m_cacheHostDirectoryName),
                                                  m_deployments[lastDeploymentIndex].m_extractedConfig.m_cachePassword);

        String[] containerClasspathList = m_deployments[lastDeploymentIndex].m_configClasspath.getSubstitutedResources();
        
        // append the Actional SDK so that we get a classpath for the global loader that eventually evaluate to:
        //  <customner defined prepend classpath> + <actional SDK> + <container car file define global jars>
        
        if (m_deployments[lastDeploymentIndex].m_actionalSDKArchive.exists())
        {
            String[] newList = new String[containerClasspathList.length + 1];
            System.arraycopy(containerClasspathList, 0, newList, 0, containerClasspathList.length);
            newList[newList.length - 1] = m_deployments[lastDeploymentIndex].m_actionalSDKArchive.getPath();
            containerClasspathList = newList;
        }

        if (containerClasspathList.length > 0)
        {
            m_execCommand.add("-D" + IContainer.CONTAINER_CLASSPATH_PROPERTY + "=" + createClasspathString(containerClasspathList));
        }

        String[] archiveClasspathList = m_deployments[lastDeploymentIndex].m_classpath.getLaunchClasspath();
        String[] classpathList = mergeLists(System.getProperty(IContainer.MF_DEV_PRIVATE_CLASSPATH_PROPERTY),
                                            containerClasspathList, archiveClasspathList);

        if (classpathList.length > 0)
        {
            m_execCommand.add("-cp");
            m_execCommand.add(createClasspathString(classpathList));
        }

        m_execCommand.add(CONTAINER_CLASS);

        m_execCommand.add(m_deployments[lastDeploymentIndex].m_cacheHostDirectoryName);

        String containerNames = m_deployments[0].m_containerName;
        String containerIDs = m_deployments[0].m_containerID;

        if (m_deployments.length > 1)
        {
            for (int i = 1; i < m_deployments.length; i++)
            {
                containerNames += ',' + m_deployments[i].m_containerName;
                containerIDs += ',' + m_deployments[i].m_containerID;
            }
        }

        m_execCommand.add(containerNames);
        m_execCommand.add(containerIDs);
    }

    String getNativePath()
    {
        return m_nativePath;
    }

    private String generateNativePath(boolean appendExistingPath) throws Exception
    {
        // The native library path is comprised of the native libraries in the cache, the configured
        // native path and the java.library.path system path
        String sonicNativePath = m_deployments[lastDeploymentIndex].getNativeLibDirectory();
        String configNativePath = m_deployments[lastDeploymentIndex].m_extractedConfig.m_configNativePath;
        if (configNativePath != null && configNativePath.length() > 0)
        {
            sonicNativePath += java.io.File.pathSeparatorChar + configNativePath;
        }
        String systemNativePath = null;
        if (appendExistingPath)
        {
            systemNativePath = System.getProperty("java.library.path");
        }
        if (systemNativePath != null && systemNativePath.length() > 0)
        {
            sonicNativePath += java.io.File.pathSeparatorChar + systemNativePath;
        }

        //Fix Sonic00035121 & SNC00069893
        if (ON_WINDOWS && sonicNativePath != null)
        {
            sonicNativePath = sonicNativePath.replaceAll("\"", "");
            sonicNativePath = sonicNativePath.endsWith("\\") ? sonicNativePath.substring(0, sonicNativePath.length() - 1) : sonicNativePath;
        }

        return sonicNativePath;
    }

    public ArrayList getCommandArray()
    {
        return m_execCommand;
    }

    public String getCommandString()
    {
        return toString(false);
    }

    @Override
    public String toString()
    {
        return toString(true);
    }

    private  String toString(boolean withNewline)
    {
        StringBuffer stBuffer = new StringBuffer();
        for (int i = 0; i < m_execCommand.size(); i++)
        {
            stBuffer.append((String)m_execCommand.get(i));
            if (withNewline)
            {
                stBuffer.append(IContainer.NEWLINE);
            }
            else
            {
                stBuffer.append(' ');
            }
        }
        return stBuffer.toString();
    }

    private ArrayList extractJVMArgs() throws IOException
    {
        ArrayList argList = new ArrayList();
        HashMap jvmArgs = m_deployments[lastDeploymentIndex].m_extractedConfig.m_jvmArgs;
        String configuredArgs = null;

        // -server must come first on some platforms
        boolean serverOption = false;
        if (jvmArgs.containsKey("-server"))
        {
            jvmArgs.remove("-server");
            serverOption = true;
        }

        if (jvmArgs.containsKey(DeployContainerResources.CONFIGURED_JVM_OPTIONS))
        {
            configuredArgs = (String)jvmArgs.remove(DeployContainerResources.CONFIGURED_JVM_OPTIONS);
        }

        Iterator keys = jvmArgs.keySet().iterator();

        while (keys.hasNext())
        {
            String arg = (String)keys.next();
            String value = (String)jvmArgs.get(arg);
            if (value == null)
            {
                value = "";
            }
            argList.add(arg + value);
        }

        argList.addAll(getOptionList(configuredArgs));
        addDevelopmentArgs(argList);

        boolean clientOption = false;
        for (int i = 0; i < argList.size(); i++)
        {
            if (argList.get(i).equals("-client"))
            {
                clientOption = true;
            }
        }

        if (serverOption && !clientOption)
        {
            argList.add(0, "-server");
        }

        return argList;

    }

    private static ArrayList getOptionList(String src)
    {
        ArrayList list = new ArrayList();
        if (src == null)
        {
            return list;
        }

        while(true)
        {
            src = src.trim();
            int lastOptionIndex = getLastOptionIndex(src);
            if (lastOptionIndex == -1)
            {
                break;
            }
            list.add(src.substring(lastOptionIndex));
            src = src.substring(0, lastOptionIndex);
        }

        Collections.reverse(list);
        return list;
    }

    private static int getLastOptionIndex(String src)
    {
        for (int i = src.length() - 1; i >= 0; i--)
        {
            boolean dashFound = src.charAt(i) == '-';
            if (dashFound && (i == 0 || WHITE_SPACE_CHARS.indexOf(src.charAt(i-1)) != -1))
            {
                return i;
            }
        }
        return -1;
    }


    private void addDevelopmentArgs(ArrayList argList) throws IOException
    {
        m_devArgsFileName = System.getProperty(DEV_JVM_ARGS_PROPERTY);
        if (m_devArgsFileName == null)
        {
            return;
        }

        BufferedReader reader = new BufferedReader (new InputStreamReader(new FileInputStream(m_devArgsFileName)));
        String line = reader.readLine();
        reader.close();

        if (line == null || line.length() == 0)
        {
            return;
        }

        StringTokenizer tokens = new StringTokenizer(line);
        while (tokens.hasMoreElements())
        {
            argList.add(tokens.nextToken());
        }
    }

    private static String[] mergeLists(String developmentClasspath, String[] classpathList0, String[] classpathList1)
    {
        String[] merged = new String[classpathList0.length + classpathList1.length + (developmentClasspath != null ? 1 : 0)];

        int startIdx = 0;
        if (developmentClasspath != null)
        {
            startIdx = 1;
            merged[0] = developmentClasspath;
        }

        for (int i = 0; i < classpathList0.length; i++)
        {
            merged[i+startIdx] = classpathList0[i];
        }

        for (int i = classpathList0.length, count = 0; count < classpathList1.length; i++, count++)
        {
            merged[i+startIdx] = classpathList1[count];
        }

        return merged;
    }

    static String createClasspathString(String[] classpathList)
    {
        StringBuffer classpathString = new StringBuffer();

        for (int i = 0; i < classpathList.length; i++)
        {
            // This file was not found
            if (classpathList[i] == null)
            {
                continue;
            }

            classpathString.append(classpathList[i]);
            if (i+1 < classpathList.length)
            {
                classpathString.append(PATH_SEPARATOR);
            }
        }
        return classpathString.toString();
    }

    private class QuotedItemsList extends ArrayList
    {
        String m_quote;

        QuotedItemsList(boolean doQuote)
        {
            super();
            m_quote = doQuote ? QUOTE_STRING : "";
        }

        @Override
        public boolean add(Object item)
        {
            return super.add(m_quote + item.toString() + m_quote);
        }

        @Override
        public boolean addAll(Collection c)
        {
            Iterator iter = c.iterator();
            while (iter.hasNext())
            {
                add(iter.next());
            }

            return c.size() != 0;
        }
        
        public String getProperty(String propertyName)
        {
            String matchStr = m_quote + propertyName + '=';

            String propertyValue = null;
            for (Object obj : this)
            {
                String str = (String)obj;
                if (str.startsWith(matchStr))
                {
                    propertyValue = str.substring(matchStr.length(), str.length() - m_quote.length());
                    break;
                }
            }
            
            return propertyValue;
        }        
    }
}
