package com.sonicsw.mf.framework.agent.ci;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;

import javax.management.ObjectName;

import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.ILogger;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.agent.ContainerUtil;
import com.sonicsw.mf.jmx.client.DirectoryServiceProxy;
import com.sonicsw.mf.jmx.client.JMSConnectorAddress;
import com.sonicsw.mf.jmx.client.JMSConnectorClient;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;

final class StoreFilesUtility
{
    private static final String[] EMPTY_STRING_ARRAY = new String[0];
    private static final String MF_FILE = "MF_FILE";

    private String m_containerBootFileName;
    private JMSConnectorClient m_connector;
    private IDirectoryFileSystemService m_ds;

    private static ILogger m_logger = new CILogger("DS File Utility");

    // Configuration
    ExtractedContainerConfig m_extractedConfig;   // container configuration we use



    public static void main (String[] args)
    {
        if (args.length != 2)
        {
            System.out.println("Usage: java com.sonicsw.mf.framework.agent.ci.SetCIConfigUtility containerbootFile copyList");
            System.exit(1);
        }
        try
        {
            new StoreFilesUtility(args[0], args[1]);
        }
        catch (Throwable t)
        {
            logMessage("", t,  Level.SEVERE);
            System.exit(1);
        }
    }

    StoreFilesUtility(String containerBootFileName, String listFile) throws Exception
    {
        File containerBootFile = new File(containerBootFileName);
        m_containerBootFileName = containerBootFile.getAbsolutePath();
        if (!containerBootFile.exists() || !containerBootFile.isFile() || !containerBootFile.canRead())
        {
            String problem = !containerBootFile.exists() ?  " not found." : " cannot be read";
            String errMessage = "Bootfile " + containerBootFileName + problem;
            logMessage(errMessage, Level.SEVERE);
            throw new Exception(errMessage);
        }
        logMessage("Open container boot file \"" + containerBootFile.getAbsolutePath() + '"', Level.INFO);


        IElement containerConfiguration = ContainerUtil.importConfiguration(containerBootFile, IContainer.PASSWORD);
        IAttributeSet configAttributes = containerConfiguration.getAttributes();
        ExtractedContainerConfig bootExtractedConfig = new ExtractedContainerConfig();
        extractConfigInfo(configAttributes, bootExtractedConfig);


        logMessage("Starting to store files...", Level.INFO);
        createDSProxy(createConnector(configAttributes), bootExtractedConfig.m_domainName);
        storeFiles(listFile);
        logMessage("...store complete", Level.INFO);
        System.exit(0);
    }

   private void storeFile(String filePath, String dsFolder) throws Exception
   {
       logMessage("Store " + filePath + " in " + dsFolder,  Level.INFO);
       File file = new File(filePath);

       // Create the folder if doesn't exit
       EntityName eFolderName = new EntityName(dsFolder);
       String[] nameComponents = eFolderName.getNameComponents();
       String currentLevel = "";
       for (int i = 0; i < nameComponents.length; i++)
       {
           currentLevel += "/" + nameComponents[i];
           m_ds.createFolder(currentLevel, true);
       }

       EntityName eDsFilePath = eFolderName.createChild(file.getName());

       IDirElement envelopeElement = m_ds.getFSElement(eDsFilePath.getName(), true);


       if (envelopeElement == null)
    {
        envelopeElement =  (IDirElement)ElementFactory.createElement(eDsFilePath.getName(),
                                                                        MF_FILE,
                                                                        IContainerConstants.DS_C_VERSION);
    }

       BufferedInputStream bIn = new BufferedInputStream(new FileInputStream(filePath));
       m_ds.attachFSBlob(envelopeElement.doneUpdate(), bIn);

   }


    private void storeFiles(String listFileName) throws Exception
    {
        BufferedReader reader = new BufferedReader
            (new InputStreamReader(new FileInputStream(listFileName)));

        while (true)
        {
            String line = reader.readLine();
            if (line == null)
            {
                break;
            }

            int eqIndex = line.indexOf('=');
            String dsFolder = line.substring(0, eqIndex);
            String fsFile = line.substring(eqIndex+1);
            storeFile(fsFile, dsFolder);
        }
    }

    private void createDSProxy(JMSConnectorClient connector, String domainName) throws Exception
    {
        m_ds = new DirectoryServiceProxy(connector, new ObjectName(domainName + "." + IContainer.DS_ADDRESS));
    }


    private void extractConfigInfo(IAttributeSet containerAttrs, ExtractedContainerConfig containerConfig) throws Exception
    {
        containerConfig.m_domainName = (String)containerAttrs.getAttribute(IContainerConstants.DOMAIN_NAME_ATTR);
        if (containerConfig.m_domainName == null)
        {
            containerConfig.m_domainName = IContainerConstants.DOMAIN_NAME_DEFAULT;
        }

        containerConfig.m_containerName = (String)containerAttrs.getAttribute(IContainerConstants.CONTAINER_NAME_ATTR);

        containerConfig.m_classpath = (String)containerAttrs.getAttribute(IContainerConstants.CLASSPATH_ATTR);

        containerConfig.m_deployTraceMask = (Integer)containerAttrs.getAttribute(IContainerConstants.TRACE_MASK_ATTR);
        if (containerConfig.m_deployTraceMask == null)
        {
            containerConfig.m_deployTraceMask = new Integer(IContainerConstants.TRACE_MASK_DEFAULT);
        }

        IAttributeSet cacheAttrs = (IAttributeSet)containerAttrs.getAttribute(IContainerConstants.CACHE_ATTR);

        containerConfig.m_cacheHostDirectoryName = (String)cacheAttrs.getAttribute(IContainerConstants.CACHE_DIRECTORY_ATTR);
        if (containerConfig.m_cacheHostDirectoryName == null)
        {
            containerConfig.m_cacheHostDirectoryName = IContainerConstants.CACHE_DIRECTORY_DEFAULT;
        }
        containerConfig.m_cachePassword = (String)cacheAttrs.getAttribute(IContainerConstants.PASSWORD_ATTR);

        IAttributeSet jvmArgs = (IAttributeSet)containerAttrs.getAttribute(IContainerConstants.JVM_ARGUMENTS_ATTR);
        if (jvmArgs != null)
        {
            containerConfig.m_jvmArgs = jvmArgs.getAttributes();
        }
        else
        {
            containerConfig.m_jvmArgs = new HashMap();
        }
    }

    private JMSConnectorClient createConnector(IAttributeSet configAttributes) throws Exception
    {

        IAttributeSet connectionAttrs = (IAttributeSet)configAttributes.getAttribute(IContainerConstants.CONNECTION_ATTR);
        Map factoryAttrs = (Map)connectionAttrs.getAttributes().clone();
        factoryAttrs.remove(IContainerConstants.MANAGEMENT_NODE_ATTR);
        factoryAttrs.remove(IContainerConstants.REQUEST_TIMEOUT_ATTR);
        factoryAttrs.remove(IContainerConstants.CONNECT_TIMEOUT_ATTR);
        factoryAttrs.remove(IContainerConstants.SOCKET_CONNECT_TIMEOUT_ATTR);

        Hashtable env = new Hashtable(factoryAttrs);
        if (env.get(IContainerConstants.CONNECTIONURLS_ATTR) == null)
        {
            env.put(IContainerConstants.CONNECTIONURLS_ATTR, IContainer.CONNECTIONURLS_DEFAULT);
        }
        JMSConnectorAddress address = new JMSConnectorAddress(env);

        String managementNode = (String)connectionAttrs.getAttribute(IContainerConstants.MANAGEMENT_NODE_ATTR);
        if (managementNode != null && managementNode.length() > 0)
        {
            address.setManagementNode(managementNode);
        }

        JMSConnectorClient connector = new JMSConnectorClient();

        // REQUEST_TIMEOUT_ATTR is in seconds, setRequestTimeout is passed millis
        Integer requestTimeout = (Integer)connectionAttrs.getAttribute(IContainerConstants.REQUEST_TIMEOUT_ATTR);
        if (requestTimeout != null)
        {
            connector.setRequestTimeout(requestTimeout.intValue() * 1000);
        }

        // CONNECT_TIMEOUT_ATTR is in seconds, setConnectTimeout is passed millis
        Integer connectTimeout = (Integer)connectionAttrs.getAttribute(IContainerConstants.CONNECT_TIMEOUT_ATTR);
        if (connectTimeout != null)
        {
            connector.setConnectTimeout(connectTimeout.intValue() * 1000);
        }

        // SOCKET_CONNECT_TIMEOUT_ATTR is in seconds, passed to connect() in millis
        Integer socketConnectTimeout = (Integer)connectionAttrs.getAttribute(IContainerConstants.SOCKET_CONNECT_TIMEOUT_ATTR);
        if (socketConnectTimeout != null)
        {
            connector.setSocketConnectTimeout(socketConnectTimeout.intValue() * 1000);
        }

        logMessage("Connecting to the Directory Service",  Level.INFO);
        connector.connect(address);

        m_connector = connector;
        return connector;

    }

    static void logMessage(String message, int severityLevel)
    {
        m_logger.logMessage(message, severityLevel);
    }

    static void logMessage(String message, Throwable t, int severityLevel)
    {
        m_logger.logMessage(message, t, severityLevel);
    }

    // This configuration is extracted from the container.xml boot file or from the
    // actual DS configuration. By incapsulating it in class, we can use the same code
    // in both cases and reconcile differences according to some rules.
    private static class ExtractedContainerConfig implements Serializable
    {
        String m_domainName = null;
        String m_containerName = null;
        String m_classpath = null;
        Integer m_deployTraceMask = null;
        String  m_cacheHostDirectoryName = null;
        String m_cachePassword = null;
        HashMap m_jvmArgs;

        ExtractedContainerConfig deepClone()
        {
          try
          {
               ByteArrayOutputStream out = new ByteArrayOutputStream();
               ObjectOutputStream objectOut = new ObjectOutputStream(out);
               objectOut.writeObject(this);
               byte[] bytes = out.toByteArray();
               objectOut.close();

               ByteArrayInputStream in = new ByteArrayInputStream(bytes);
               ObjectInputStream objectIn = new ObjectInputStream(in);
               return (ExtractedContainerConfig) objectIn.readObject();
          }
          catch (Exception e)
          {
                throw new Error("deepClone failed.");
          }
        }


    }

}
