package com.sonicsw.mf.framework.agent.ci;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Properties;

import com.sonicsw.mf.common.ILogger;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.IContainer;


// Maintains a list of system properties - some of them point to sonicfs:/// resources
final public class SystemProperties
{
    private static final String[] EMPTY_STRING_ARRAY = new String[0];
    private static final String VENDOR = System.getProperty("java.vm.vendor");
    private static final String PASS_THROUGH_PREFIX = "_pass_to_container_jvm";
    private static final String JAVA_PROTOCOL = "java.protocol";
    private static final int JAVA_PROTOCOL_LENGTH = JAVA_PROTOCOL.length();
    private static final String SONIC = "sonic";
    private static final int SONIC_LENGTH = SONIC.length();
    private static String[] EXCLUDE_IF_STARTS_WITH = {"com.ibm","console.encoding"};

    private static final HashSet PROPERTY_EXCLUSION_SET = new HashSet();
    static
    {
        PROPERTY_EXCLUSION_SET.add("invokedviajava");
        PROPERTY_EXCLUSION_SET.add("awt");
        PROPERTY_EXCLUSION_SET.add("javax");
        PROPERTY_EXCLUSION_SET.add("java");
        PROPERTY_EXCLUSION_SET.add("sun");
        PROPERTY_EXCLUSION_SET.add("ibm");
        PROPERTY_EXCLUSION_SET.add("path");
        PROPERTY_EXCLUSION_SET.add("file");
        PROPERTY_EXCLUSION_SET.add("user");
        PROPERTY_EXCLUSION_SET.add("os");
        PROPERTY_EXCLUSION_SET.add("line");
    }

    private static final HashSet SONIC_FIRST_PHASE_ONLY = new HashSet();
    static
    {
        SONIC_FIRST_PHASE_ONLY.add(IContainer.GENERATE_SH_ON_WINDOWS);
    }

    private HashMap m_properties;
    private String[] m_sonicfsResources;
    private HashMap m_sonicfsMarkers;
    private ILogger m_logger;

    public SystemProperties (HashMap properties0, ILogger logger, boolean addSystemProperties) throws Exception
    {
        HashMap properties = addSystemProperties ? new HashMap(filterSystemProperties()) : new HashMap();
        if (properties0 != null)
        {
            properties.putAll(properties0);
        }

        m_logger = logger;
        m_properties = properties;
        ArrayList sonicfsResources = new ArrayList();
        m_sonicfsMarkers = new HashMap();

        if (m_properties == null)
        {
            m_properties = new HashMap();
            m_sonicfsResources = EMPTY_STRING_ARRAY;
            return;
        }

        Iterator propKeys = properties.keySet().iterator();
        int count = 0;
        while (propKeys.hasNext())
        {
            String key = (String)propKeys.next();
            String propValue = (String)properties.get(key);
            String path = getSonicfsPath(propValue);
            if (path != null)
            {
                sonicfsResources.add(path);
                m_sonicfsMarkers.put(new Integer(count++), key); //Map between the prop key and the sonicfs list
            }
        }

        m_sonicfsResources = (String[])sonicfsResources.toArray(EMPTY_STRING_ARRAY);
    }

    public void addProps(HashMap properties)
    {
        if (properties != null)
        {
            m_properties.putAll(properties);
        }
    }

    //Choose the JvmProperty.CentralConnection.xxx properties if 'centralConnection' is 'true' or the
    // JvmProperty.Connection.xxx properties if 'centralConnection' is 'false'
    static void connectionPropertiesFilter(HashMap table, boolean centralConnection)
    {
        ArrayList keepList = new ArrayList();
        ArrayList removeList = new ArrayList();

        Iterator propKeys = table.keySet().iterator();
        while (propKeys.hasNext())
        {
            String key = (String)propKeys.next();

            Boolean centralConnectionPrefix = null;

            if (key.regionMatches(true, 0, IContainer.CENTRAL_CONNECTION_PROP_PREFIX, 0, IContainer.CENTRAL_CONNECTION_PROP_PREFIX.length()))
            {
                centralConnectionPrefix = Boolean.TRUE;
            }
            else if (key.regionMatches(true, 0, IContainer.CONNECTION_PROP_PREFIX, 0, IContainer.CONNECTION_PROP_PREFIX.length()))
            {
                centralConnectionPrefix = Boolean.FALSE;
            }

            if (centralConnectionPrefix != null)
            {
                if (centralConnection && centralConnectionPrefix.booleanValue() ||
                    !centralConnection && !centralConnectionPrefix.booleanValue())
                {
                    keepList.add(key);
                }
                else
                {
                    removeList.add(key);
                }
            }

        }

        //Remove entries not required for this connection
        for (int i = 0; i < removeList.size(); i++)
        {
            table.remove(removeList.get(i));
        }

        //Remove the prefix from entries required by this connection
        for (int i = 0; i < keepList.size(); i++)
        {
            String fullKey = (String)keepList.get(i);
            int prefixLengthToRemove = centralConnection ? IContainer.CENTRAL_CONNECTION_PROP_PREFIX.length() : IContainer.CONNECTION_PROP_PREFIX.length();
            String key = fullKey.substring(prefixLengthToRemove);
            String value = (String)table.remove(fullKey);
            table.put(key, value);
        }


    }

    void mergePlatformDependentProps(HashMap dependentProperties)
    {
        dependentProperties.putAll(m_properties);
        m_properties = dependentProperties;
    }

    String[] getSonicfsResources()
    {
        return m_sonicfsResources;
    }

    HashMap getProperties()
    {
        return m_properties;
    }

    void substituteSonicfsResources(String[] substitutes)
    {
        for (int i = 0; i < substitutes.length; i++)
        {
            m_properties.put(m_sonicfsMarkers.get(new Integer(i)), substitutes[i]);
        }
    }

    private String getSonicfsPath(String sonicfsPath) throws Exception
    {
        // First see whether sonicfs:// is in the prefix. It's a common mistake to have sonicfs://xxx rather than
        // sonicfs:///xxx - and we would like to identify this case and log an error.
        String dsClasspathPrefix = IContainer.DS_CLASSPATH_PROTOCOL.substring(0, IContainer.DS_CLASSPATH_PROTOCOL.length() - 1);

        if (sonicfsPath.length() > dsClasspathPrefix.length() &&
            sonicfsPath.substring(0, dsClasspathPrefix.length()).equalsIgnoreCase(dsClasspathPrefix))
        {
            if (sonicfsPath.charAt(dsClasspathPrefix.length()) != '/')
            {
                logMessage("The container system properties contain an invalid URL: " + sonicfsPath, Level.SEVERE);
                return null;
            }

            return sonicfsPath.substring(IContainer.DS_CLASSPATH_PROTOCOL.length() - 1);
        }
        else
        {
            return null;
        }
    }

    void logMessage(String message, int severityLevel)
    {
        m_logger.logMessage(message,  severityLevel);
    }

    void logMessage(String message, Throwable t, int severityLevel)
    {
        m_logger.logMessage(message, t, severityLevel);
    }


    private static Properties filterSystemProperties()
    {
        Properties filtered = new Properties();
        Properties sysProperties = System.getProperties();

        Enumeration keys = sysProperties.keys();

        while (keys.hasMoreElements())
        {
            String key = (String)keys.nextElement();
            if (passThrough(key, sysProperties.getProperty(key), filtered))
            {
                continue;
            }

            if (inInclusionList(key) || !inExclusionList(key))
            {
                filtered.setProperty(key, sysProperties.getProperty(key));
            }
        }

        return filtered;
    }

    private static boolean passThrough(String key, String value, Properties filtered)
    {
        int dotIndx = key.indexOf(".");
        if (dotIndx < 0)
        {
            return false;
        }

        String firstComponent = key.substring(0, dotIndx);
        if (firstComponent.equalsIgnoreCase(PASS_THROUGH_PREFIX))
        {
             String newKey = key.substring(dotIndx+1); // Remove the PASS_THROUGH_PREFIX prefix
             filtered.setProperty(newKey, value);
             return true;
        }
        else
        {
            return false;
        }
    }

    private static boolean inExclusionList(String propKey)
    {
        if (SONIC_FIRST_PHASE_ONLY.contains(propKey))
        {
            return true;
        }

        for (int i = 0; i < EXCLUDE_IF_STARTS_WITH.length; i++)
        {
            if (propKey.startsWith(EXCLUDE_IF_STARTS_WITH[i]))
            {
                return true;
            }
        }

        int dotIndx = propKey.indexOf(".");

        String firstComponent = null;
        if (dotIndx < 0)
        {
            firstComponent = propKey;
        }
        else
        {
            firstComponent = propKey.substring(0, dotIndx);
        }

        if(VENDOR.regionMatches(true, 0, firstComponent, 0, 3))
        {
            return true;
        }

        if (PROPERTY_EXCLUSION_SET.contains(firstComponent))
        {
            return true;
        }

        return false;
    }

    private static boolean inInclusionList(String propKey)
    {
        if (SONIC_FIRST_PHASE_ONLY.contains(propKey))
        {
            return false;
        }

        if(propKey.regionMatches(true, 0, SONIC, 0, SONIC_LENGTH))
        {
            return true;
        }

        if(propKey.regionMatches(false, 0, JAVA_PROTOCOL, 0, JAVA_PROTOCOL_LENGTH))
        {
            return true;
        }

        return false;
    }

}

