package com.sonicsw.mf.framework.daemon;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

import javax.management.MBeanAttributeInfo;
import javax.management.MBeanNotificationInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanParameterInfo;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.MFException;
import com.sonicsw.mf.common.config.IAttributeChangeHandler;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.NotModifiedAttException;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.runtime.IChildContainerState;
import com.sonicsw.mf.common.runtime.IComponentState;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.AbstractFrameworkComponent;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.mgmtapi.config.constants.IActivationDaemonConstants;

public final class ActivationDaemon
extends AbstractFrameworkComponent
{
    private SdfMFTracingIntegration m_SdfMFTracingIntegration;

    static volatile String m_mfXSD;
    
    private static final String DAEMON_TRACE_MASK_VALUES = "16=child container exec,32=child container stdout,64=child container stderr";
    public static final int TRACE_CHILDPROCESS_EXEC = 16;
    public static final int TRACE_CHILDPROCESS_STDOUT = 32;
    public static final int TRACE_CHILDPROCESS_STDERR = 64;
    private static final String ENV_FILE_SUFFIX = "_ADENV";
    private static final String ENV_ESCAPED_FILE_SUFFIX = "_ESCAPED";
    private static final long MAX_ENV_WAIT_TIME = 60000; // 1 minute wait for the environment file to close

    private HashMap m_childContainers = new HashMap(0);
    private Scheduler m_adScheduler;
    private ArrayList m_ADNativeEnv = null;

    private IAttributeSet m_containersAttributSet;
    
    // AcivationDaemon notifications
    public static final String FAILURE_NOTIFICATION_TYPE = "Failure";
    public static final String ACTIVATE_NOTIFICATION_TYPE = "Activate";
    public static final String DEACTIVATE_NOTIFICATION_TYPE = "Deactivate";

    public static final IChildContainerState[] EMPTY_CHILD_CONTAINER_STATE_ARRAY = new IChildContainerState[0];

    private static final ArrayList ATTRIBUTE_INFOS = new ArrayList();
    private static final ArrayList OPERATION_INFOS = new ArrayList();
    private static final ArrayList NOTIFICATION_INFOS = new ArrayList();

    static
    {
        //
        // Attributes
        //

        //
        // Operations
        //
        MBeanParameterInfo[] mbParamInfos = null;

        // activate
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("id", String.class.getName(), "Runtime identity of the container to activate (e.g. \"Domain1.Container1\")")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("activate", "(Re)Activates a child container according to the schedule and activation rules defined for the container.",
                                                   mbParamInfos, Void.class.getName(), MBeanOperationInfo.ACTION));

        // deactivate
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("id", String.class.getName(), "Runtime identity of the container to deactivate (e.g. \"Domain1.Container1\")"),
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("deactivate", "Shuts down an active child container and disables the child from being re-launched due to the activation schedule and rules defined for the container.",
                                                   mbParamInfos, Void.class.getName(), MBeanOperationInfo.ACTION));

        // launch
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("configID", String.class.getName(), "The configuration identity of the container to launch."),
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("launch", "Dynamically launches the specified child container.",
                                                   mbParamInfos, Void.class.getName(), MBeanOperationInfo.ACTION));

        // get child states
        OPERATION_INFOS.add(new MBeanOperationInfo("getChildStates", "Gets the runtime state for all of the child containers managemed by this daemon.",
                                                   IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, IChildContainerState[].class.getName(), MBeanOperationInfo.INFO));

        //
        // Notifications
        //
        String[] notifTypes = null;

        // system.state.activate
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
            ACTIVATE_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, INotification.CLASSNAME, "Child container (re)launch complete."));
        // system.state.deactivate
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
            DEACTIVATE_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, INotification.CLASSNAME, "Child container shutdown complete."));
        // system.state.failure
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
            FAILURE_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, INotification.CLASSNAME, "Child container process has died."));
    }


    @Override
    public MBeanAttributeInfo[] getAttributeInfos() { return (MBeanAttributeInfo[])ATTRIBUTE_INFOS.toArray(IEmptyArray.EMPTY_ATTRIBUTE_INFO_ARRAY); }

    @Override
    public MBeanOperationInfo[] getOperationInfos() { return (MBeanOperationInfo[])OPERATION_INFOS.toArray(IEmptyArray.EMPTY_OPERATION_INFO_ARRAY); }

    @Override
    public MBeanNotificationInfo[] getNotificationInfos() { return (MBeanNotificationInfo[])NOTIFICATION_INFOS.toArray(IEmptyArray.EMPTY_NOTIFICATION_INFO_ARRAY); }

    @Override
    public void init(IComponentContext context)
    {
        super.init(context);

        // get the daemon's configuration
        IElement daemonConfig = context.getConfiguration(true);

        // as soon as one container is launched/activation we will need the MF schema
        // so get it now
        IElement xsdConfig = super.m_context.getConfiguration("/_MFSchemaDefinition", false);
        m_mfXSD = (String)xsdConfig.getAttributes().getAttribute("SCHEMA_DEFINITION");
        m_mfXSD = m_mfXSD.substring(m_mfXSD.indexOf("<"));

        // get the containers the daemon is configured to manage and create
        // an object to manage the child
        IAttributeSet daemonAttrs = daemonConfig.getAttributes();
        m_containersAttributSet = (IAttributeSet)daemonAttrs.getAttribute(IActivationDaemonConstants.CONTAINERS_ATTR);
        HashMap containers = m_containersAttributSet.getAttributes();
        if (containers != null)
        {
            Iterator iterator = containers.entrySet().iterator();
            while (iterator.hasNext())
            {
                Map.Entry entry = (Map.Entry)iterator.next();
                addChildContainer((String)entry.getKey(), (IAttributeSet)entry.getValue());
            }
        }

        // add a change listener to the containers set
        IAttributeChangeHandler changeHandler = new IAttributeChangeHandler()
        {
            @Override
            public void itemDeleted() { }
            @Override
            public void itemModified(Object value)
            {
                ActivationDaemon.this.handleContainersAttributeSetChange((IDeltaAttributeSet)value);
            }
        };
        m_containersAttributSet.registerAttributeChangeHandler(super.m_context, changeHandler);
    }

    @Override
    public synchronized void start()
    {

        m_SdfMFTracingIntegration = new SdfMFTracingIntegration();
        m_SdfMFTracingIntegration.register();

        if (super.m_state == IComponentState.STATE_ONLINE)
        {
            return;
        }
        
        // find the native environment of the AD
        try
        {
            m_ADNativeEnv = getADNativeEnvironment();
        }
        catch (Exception e)
        {
        	m_context.logMessage("Unable to start activation daemon " + ActivationDaemon.this.m_context.getComponentName().getComponentName(), e, Level.SEVERE);
        	return;
        }

        // iterate over child containers and start each child that is auto start
        Object[] childContainers = m_childContainers.values().toArray();
        for (int i = 0; i < childContainers.length; i++)
        {
            final ChildContainer child = (ChildContainer)childContainers[i];
            if (child.isAutoStart())
            {
                Runnable activator = new Runnable()
                {
                    @Override
                    public void run() { child.activate(true); }
                };
                super.m_context.scheduleTask(activator, new Date(System.currentTimeMillis()));
            }
        }

        // start the scheduler (which will activate all children that are not auto start)
        m_adScheduler = new Scheduler(super.m_frameworkContext, m_childContainers);
        m_adScheduler.start();

        super.start();
    }
    
    private String[] getEnvCommand()
    {
    	String OS = System.getProperty("os.name").toLowerCase();
        if (OS.indexOf("windows") > -1)
        {
            return new String[]{"cmd.exe", "/c set > \"" + getTempEnvFile() + '\"'};
        }
        else
        {
            return new String[] {"sh", "-c","env > \"" + getTempEnvFile() + '\"'};
        }
    }
    
    private File getTempEnvFile()
    {
    	return new File(m_container.getCacheDirectory(), m_container.getContainerIdentity().getContainerName() + ENV_FILE_SUFFIX);
    }
    
    private File escapeBackslashInFile() throws FileNotFoundException, IOException
    {
    	File envFile = getTempEnvFile();
    	if (!envFile.exists())
        {
            throw new FileNotFoundException("The environment file was not found! (" + envFile + ")");
        }
    	FileInputStream inputStream = new FileInputStream(envFile);
    	File modifiedEnvFile = new File(envFile.getCanonicalPath() + ENV_ESCAPED_FILE_SUFFIX);
    	FileOutputStream outStream = new FileOutputStream(modifiedEnvFile.getCanonicalPath());
    	
    	while (inputStream.available() > 0)
    	{
    		int byteRead = inputStream.read();
    		if (byteRead == '\\')
             {
                outStream.write('\\'); // need to escape the backslash
            }
    		outStream.write(byteRead);
    			
    	}
    	inputStream.close();
    	outStream.close();
    	return modifiedEnvFile;
    }
    
    private ArrayList getEnvFromFile() throws IOException
    {
    	File envFile = getTempEnvFile();
    	// on a slow machine, the process which creates the file
    	// with the native environment might be done but the file
    	// might not be closed yet. Give it some time to finish closing
    	// if this is the case
    	long startTime = System.currentTimeMillis();
    	while (System.currentTimeMillis() - startTime < MAX_ENV_WAIT_TIME )
        {
            try
            {
            	FileInputStream testStream = new FileInputStream(envFile);
            	testStream.close();
            	break;  // if we get here, we were able to open the file
            }
            catch (Exception streamEx) 
            {
            	try
            	{
            	    Thread.sleep(500);
            	}
            	catch (InterruptedException ex) {} // not expecting this, so continue
            }
        }
    	
    	File envQuotedFile = escapeBackslashInFile();
    	InputStream fileStream = new FileInputStream(envQuotedFile);
    	Properties props = new Properties();
    	props.load(fileStream);
    	Enumeration propNames = props.keys();
    	ArrayList env = new ArrayList();
    	while (propNames.hasMoreElements())
    	{
    		String key = (String)propNames.nextElement();
    		String value = (String)props.get(key);
    		env.add(key+ "=" + value);  	
    	}
    	fileStream.close();
    	boolean deleted = envFile.delete();
    	if (!deleted)
        {
            m_context.logMessage("Unable to delete temporary environment file " + envFile.getCanonicalPath(), Level.INFO);
        }
    	deleted = envQuotedFile.delete();
    	if (!deleted)
        {
            m_context.logMessage("Unable to delete temporary environment file " + envQuotedFile.getCanonicalPath(), Level.INFO);
        }
    	return env;
    }
    
    private ArrayList getADNativeEnvironment() throws Exception
    {
    	String[] command = getEnvCommand();
    	Process envP = Runtime.getRuntime().exec(command);
    	InputStream childOutput = envP.getInputStream();
    	InputStream childErr = envP.getErrorStream();
    	OutputStream childInput = envP.getOutputStream();

    	// consume the streams
    	GetEnvReader envOutput = new GetEnvReader(childOutput, true);
        GetEnvReader envErr = new GetEnvReader(childErr, false);

        Thread outputThread = new Thread(envOutput, ActivationDaemon.this.m_context.getComponentName().getComponentName() + "getEnv output Reader");
        outputThread.setDaemon(true);
        outputThread.start();

        Thread errThread = new Thread(envErr, ActivationDaemon.this.m_context.getComponentName().getComponentName() + "getEnv stderr Reader");
        errThread.setDaemon(true);
        errThread.start();
        
        GetEnvInputFlusher envInput= new GetEnvInputFlusher(childInput, outputThread);
        Thread inputThread = new Thread(envInput, ActivationDaemon.this.m_context.getComponentName().getComponentName() + "getEnv input flusher");
        inputThread.setDaemon(true);
        inputThread.start();

        while (outputThread.isAlive() || inputThread.isAlive() || errThread.isAlive())
        {
            Thread.sleep(1000);
        }
        return getEnvFromFile();
    }
    
    // used by the child containers to get a copy of the AD's native environment
    
    ArrayList getNativeEnvironment()
    {
    	// deep clone of the environment list, as the contents are passed
    	// to the child container's process and we don't own what happens to
    	// the environment strings there.
    	ArrayList copiedList = new ArrayList();
    	Iterator envIterator = m_ADNativeEnv.iterator();
    	while (envIterator.hasNext())
    	{
    		String envSetting = (String)envIterator.next();
    		copiedList.add(new String(envSetting));
    	}
    	return copiedList;
    }

    @Override
    public synchronized void stop()
    {
        if (super.m_state == IComponentState.STATE_OFFLINE)
        {
            return;
        }
        super.m_state = IComponentState.STATE_STOPPING;

        // stop schedular
        m_adScheduler.interrupt();
        m_adScheduler = null;

        // iterate over the child containers and deactivate them
        Object[] childContainers = m_childContainers.values().toArray();
        for (int i = 0; i < childContainers.length; i++)
        {
            ((ChildContainer)childContainers[i]).deactivate(IChildContainerState.STATE_INACTIVE);
        }

        super.stop();
    }

    @Override
    public void destroy()
    {
        m_containersAttributSet.unregisterAttributeChangeHandler(super.m_context);
        Object[] childContainers = m_childContainers.values().toArray();
        for (int i = 0; i < childContainers.length; i++)
        {
            ((ChildContainer)childContainers[i]).close();
        }
        m_childContainers.clear();
        super.destroy();
    }

    /**
     * This method (re)activates a child container according to the schedule
     * and activation rules defined for the child container.
     *
     * @param value The String value is a child container runtime name (such as Sonic.Container2)
     */
    public void activate(String containerName)
    throws MFException
    {
        if (super.m_state != IComponentState.STATE_ONLINE)
        {
            throw new IllegalStateException("Activation daemon must be online to activate a container");
        }

        final ChildContainer child = getChildContainer(containerName);
        if (child == null)
        {
            throw new MFException("Unknown container: " + containerName);
        }
        Runnable activator = new Runnable()
        {
            @Override
            public void run()
            {
                child.activate(false);
            }
        };
        super.m_context.scheduleTask(activator, new Date());
    }

    /**
     * This method shuts down an active child container
     * and disables the child from being re-launched due to the activation schedule
     * and rules defined for the container.
     * @param value The String value is a child container runtime name (such as "Sonic.Container2")
     */
    public void deactivate(String containerName)
    throws MFException
    {
        if (super.m_state != IComponentState.STATE_ONLINE)
        {
            throw new IllegalStateException("Activation daemon must be online to deactivate a container");
        }

        final ChildContainer child = getChildContainer(containerName);
        if (child == null)
        {
            throw new MFException("Unknown container: " + containerName);
        }
        Runnable deactivator = new Runnable()
        {
            @Override
            public void run()
            {
                child.deactivate(IChildContainerState.STATE_INACTIVE);
            }
        };
        super.m_context.scheduleTask(deactivator, new Date());
    }

    /**
     * This method dynamically launches the specified child container with no schedule or rules.
     *
     * @param value The String value is child container configuration ID
     */
    public void launch(final String childConfigID)
    throws MFException
    {
        if (super.m_state != IComponentState.STATE_ONLINE)
        {
            throw new IllegalStateException("Activation daemon must be online to launch a container");
        }

        final ChildContainer child = addChildContainer(childConfigID);
        Runnable activator = new Runnable()
        {
            @Override
            public void run()
            {
                child.activate(true);
            }
        };
        super.m_context.scheduleTask(activator, new Date(System.currentTimeMillis()));
    }

    public IChildContainerState[] getChildStates()
    {
        ArrayList childStates = new ArrayList();

        Object[] children = m_childContainers.values().toArray();
        for (int i = 0; i < children.length; i++)
        {
            childStates.add(((ChildContainer)children[i]).getContainerState());
        }

        return (IChildContainerState[])childStates.toArray(EMPTY_CHILD_CONTAINER_STATE_ARRAY);
    }

    @Override
    public synchronized void handleElementChange(IElementChange elementChange)
    {
        super.m_context.fireAttributeChangeHandlers();
    }

    @Override
    public void setTraceMask(Integer traceMask)
    {
        setTraceMask(traceMask, false);
    }

    private void setTraceMask(Integer traceMask, boolean fromSDF)
    {
        // If SDF is used then ignore trace setup from other sources
        // If SDF is not used - update it from other sources
        if (!fromSDF && m_SdfMFTracingIntegration != null)
        {
            if (m_SdfMFTracingIntegration.wasUpdated())
            {
                return;
            }
            else
            {
                m_SdfMFTracingIntegration.setTraceMask(traceMask);
            }
        }

        super.setTraceMask(traceMask);
    }

    @Override
    public String getTraceMaskValues() { return (super.getTraceMaskValues() + "," + DAEMON_TRACE_MASK_VALUES); }

    @Override
    public Integer getTraceMask() { return (super.getTraceMask()); }

    void notifyScheduler()
    {
        if (m_adScheduler != null)
        {
            m_adScheduler.changed();
        }
    }

    private ChildContainer getChildContainer(String runtimeName)
    {
        Object[] childContainers = m_childContainers.values().toArray();
        for (int i = 0; i < childContainers.length; i++)
        {
            ChildContainer child = (ChildContainer)childContainers[i];
            if (child.getContainerState().getRuntimeIdentity().getCanonicalName().compareTo(runtimeName) == 0)
            {
                return child;
            }
        }
        return null;
    }

    private synchronized ChildContainer addChildContainer(String childConfigID)
    {
        ChildContainer child = null;

        // look to see if its already there under the config ID
        child = (ChildContainer)m_childContainers.get(childConfigID);
        if (child != null)
        {
            return child;
        }

        // look to see if its there under the storage name
        Iterator iterator = m_childContainers.values().iterator();
        while (iterator.hasNext())
        {
            child = (ChildContainer)iterator.next();
            if (child.getConfigID().equals(childConfigID))
            {
                return child;
            }
        }

        child = new ChildContainer(this, super.m_frameworkContext, childConfigID, null);

        m_childContainers.put(childConfigID, child);
        return child;
    }

    private synchronized void addChildContainer(String childName, IAttributeSet activationAttrs)
    {
        if (m_childContainers.containsKey(childName))
         {
            return; // it's already there !
        }

        String childConfigID = ((Reference)activationAttrs.getAttribute(IActivationDaemonConstants.CONTAINER_REF_ATTR)).getElementName();

        Object child = m_childContainers.remove(childConfigID); // in case its already there under the configID
        if (child == null)
        {
            child = new ChildContainer(this, super.m_frameworkContext, childConfigID, activationAttrs);
        }

        m_childContainers.put(childName, child);
    }

    private synchronized void handleContainersAttributeSetChange(IDeltaAttributeSet value)
    {
        // individual child container changes are handled by the child container themselves

        String[] names = null;

        // handled deleted child containers
        names = value.getDeletedAttributesNames();
        for (int i = 0; i < names.length; i++)
        {
            ChildContainer child = (ChildContainer)m_childContainers.get(names[i]);
            if (child != null)
            {
                child.close();
                m_childContainers.remove(names[i]);
            }
        }

        // handled added child containers
        names = value.getNewAttributesNames();
        boolean notifyScheduler = false;
        for (int i = 0; i < names.length; i++)
        {
            try
            {
                IAttributeSet activationAttrs = (IAttributeSet)value.getNewValue(names[i]);
                ChildContainer child = (ChildContainer)m_childContainers.get(names[i]);
                if (child == null)
                {
                    // now check to see if there was a child container launched via the runtime API
                    // .. in which case the child would have been recorded under its configID
                    String childConfigID = ((Reference)activationAttrs.getAttribute(IActivationDaemonConstants.CONTAINER_REF_ATTR)).getElementName();
                    child = (ChildContainer)m_childContainers.get(childConfigID);
                    if (child == null)
                    {
                        addChildContainer(names[i], activationAttrs);
                        if (super.m_state == IComponentState.STATE_ONLINE)
                        {
                            final ChildContainer newChild = (ChildContainer)m_childContainers.get(names[i]);
                            if (newChild.isAutoStart())
                            {
                                Runnable activator = new Runnable()
                                {
                                    @Override
                                    public void run()
                                    {
                                        newChild.activate(true);
                                    }
                                };
                                super.m_context.scheduleTask(activator, new Date(System.currentTimeMillis()));
                            }
                            else
                            {
                                notifyScheduler = true;
                            }
                        }
                    }
                    else
                    {
                        synchronized(m_childContainers)
                        {
                            m_childContainers.remove(childConfigID);
                            m_childContainers.put(names[i], child);
                        }
                    }
                }

                // this covers the case where the container was launched manually under the config ID or another
                // thread got in and added the container
                if (child != null)
                {
                    child.setActivationSettings(activationAttrs);
                }

            } catch(NotModifiedAttException e) { }
        }

        // do we need to notify the scheduler ?
        if (notifyScheduler)
        {
            notifyScheduler();
        }
    }

    IContainer getContainer()
    {
        return m_container;
    }

    private class SdfMFTracingIntegration extends com.sonicsw.sdf.AbstractMFComponentTracing
    {
         private boolean m_updateTraceLevelWasCalled;

         SdfMFTracingIntegration()
         {
             super("sonic.mf.ad." + m_context.getComponentName().getComponentName().replace(' ', '_'), getTraceMaskValues());
             m_updateTraceLevelWasCalled = false;

             setTraceMask();
         }

         private void setTraceMask() {
             setTraceMask(new Integer(ActivationDaemon.this.m_traceMask));
         }
         
         boolean wasUpdated()
         {
             return m_updateTraceLevelWasCalled;
         }

         @Override
        public void updateTraceLevel(String doiIDNotUsed, HashMap parameters, StringBuffer buffer)
         {
             super.updateTraceLevel(doiIDNotUsed, parameters, buffer);
             m_updateTraceLevelWasCalled = true;
             ActivationDaemon.this.setTraceMask(getCurrentMask(), true);
         }
    }
    
    private final class GetEnvInputFlusher
    implements Runnable
    {
    	OutputStream m_input;    	
    	Thread m_readerThread;
    	
    	private GetEnvInputFlusher(OutputStream stream, Thread readerThread)
    	{
    		m_input = stream;    		
    		m_readerThread = readerThread;
    	}
    	
    
    	
    	@Override
        public void run()
    	{    		
    		while (m_readerThread.isAlive())
    		{
    			try
    			{
    				m_input.write((System.getProperty("line.separator")).getBytes());
    			    m_input.flush();
    			}
    			catch (IOException ioE) // When the container is shutting down, the pipes
    			                       // will close and so we'll stop flushing the input
    			{
    			    break;
    			} 
    			
    			try
    			{
    			    Thread.sleep(2000);
    			}
    			catch (InterruptedException interrupted)
    			{
    				break;
    			}
    		}
    		
    	}
    }
    
    private final class GetEnvReader
    implements Runnable
    {
        InputStream stream;
        boolean stdout = false;  // used for error message
        private GetEnvReader(InputStream stream, boolean stdout)
        {
            this.stream = stream;
            this.stdout = stdout;
        }
        
        @Override
        public void run()
        {
            BufferedReader reader = new BufferedReader(new InputStreamReader(this.stream));
            
                try
                {
                	while (stream.available() > 0)
                    {
                        reader.readLine();
                    }                    
                }
                catch(IOException e)
                {
                	m_context.logMessage("Error reading the getEnv " + (stdout ? "output " : "error " + "stream") + e, Level.WARNING);          		
                } 
        }
    }

}
