package com.sonicsw.mf.framework.daemon;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.StringTokenizer;

import com.sonicsw.mx.util.CLI;
import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.ILogger;
import com.sonicsw.mf.common.config.IAttributeChangeHandler;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.runtime.IChildContainerState;
import com.sonicsw.mf.common.runtime.IComponentState;
import com.sonicsw.mf.common.runtime.IContainerExitCodes;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.impl.CanonicalName;
import com.sonicsw.mf.common.runtime.impl.ChildContainerState;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.framework.agent.cache.impl.ConfigCache;
import com.sonicsw.mf.mgmtapi.config.constants.IActivationDaemonConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;

class ChildContainer
implements Runnable
{
    private IFrameworkComponentContext m_context;
    private String m_configID;
    private IElement m_config;

    private boolean m_isRunning = false;
    private boolean m_isClosing = false;
    private int m_activationRetries = 0;
    private ChildContainerState m_containerState;

    private CanonicalName m_containerName;
    private String m_cacheHostDir;
    private String m_cachePassword;
    private File m_workingDir;
    private File m_nativeLibrariesDir;
    private String m_nativeLibPath;
    private File m_cacheLockFile;

    private IAttributeSet m_containerAttrs;
    private IAttributeSet m_activationAttrs;
    private long m_retryInterval = 0;
    private int m_retryTimes = -1;
    private ArrayList m_schedules = new ArrayList();

    private Process m_containerProcess = null;
    private OutputStream m_containerStdin = null;
    private OutputReader m_containerStdout = null;
    private InputFlusher m_inputFlusher = null;
    private OutputReader m_containerStderr = null;
    private ActivationDaemon m_daemon = null;

    private static String m_rootWorkingDir;
    private static ArrayList m_defaultCmdArgs;
    private static HashMap m_activationProps;

    private static final String LINE_SEPARATOR = System.getProperty("line.separator");
    private static final String QUOTE = System.getProperty("path.separator").equals(";") ? "\"" : "";

    private static final long MAX_DEACTIVATION_TIME = 300000; // 5 minutes
    private boolean m_ldLibraryPathSet = false;
    private boolean m_pathSet = false;
    private boolean m_shLibPathSet = false;
    private boolean m_libPathSet = false;

    static
    {
        m_activationProps = new HashMap();
        m_activationProps.put(IContainer.MF_CLI_PROPERTY,  "true"); // force the local CLI
        m_activationProps.put(CLI.SHOW_PROMPT_PROP,  "false"); // force cmd line to not show prompts
        m_activationProps.put(CLI.ECHO_CMD_PROP,  "false"); // force cmd line echo off
        m_activationProps.put(IContainer.MF_SIGNAL_PROPERTY, "true"); // signals on startup/shutdown
        m_activationProps.put(IContainer.MF_ALLOW_RESTART_PROPERTY, "true"); // support for container triggered restart
    }

    static
    {
        m_rootWorkingDir = System.getProperty("user.dir") + File.separatorChar;

        m_defaultCmdArgs = new ArrayList();

        String bootClasspath = System.getProperty("sun.boot.class.path");
        if (bootClasspath != null)
        {
            m_defaultCmdArgs.add(QUOTE + "-Xbootclasspath:" + bootClasspath + QUOTE);
        }

        m_defaultCmdArgs.add("-D" + IContainer.MF_CLI_PROPERTY + "=true"); // force the local CLI
        m_defaultCmdArgs.add("-D" + CLI.SHOW_PROMPT_PROP + "=false"); // force cmd line to not show prompts
        m_defaultCmdArgs.add("-D" + CLI.ECHO_CMD_PROP + "=false"); // force cmd line echo off
        m_defaultCmdArgs.add("-D" + IContainer.MF_SIGNAL_PROPERTY + "=true"); // signals on startup/shutdown
        m_defaultCmdArgs.add("-D" + IContainer.MF_ALLOW_RESTART_PROPERTY + "=true"); // support for container triggered restart

        if (IContainer.PASSWORD != null)
        {
            // the bootfiles will be encrypted
            m_defaultCmdArgs.add("-D" + IContainer.MF_PASSWORD_PROPERTY + '=' + IContainer.PASSWORD);
        }

        m_defaultCmdArgs.add("-cp");
        m_defaultCmdArgs.add(QUOTE + System.getProperty("java.class.path") + QUOTE);
        m_defaultCmdArgs.add("com.sonicsw.mf.framework.agent.ci.Agent");
    }

    ChildContainer(ActivationDaemon daemon, IFrameworkComponentContext context, String configID, IAttributeSet activationAttrs)
    {
        m_daemon = daemon;
        m_context = context;
        m_configID = configID;
        setActivationSettings(activationAttrs);

        // get the configuration on another thread
        Thread configReader = new Thread(context.getComponentName().getComponentName() + " - Child Configuration Reader")
        {
            @Override
            public void run() { ChildContainer.this.getConfiguration(); }
        };
        configReader.setDaemon(true);
        configReader.start();
    }

    void setContainerSettings(IAttributeSet containerAttrs)
    throws Exception
    {
        m_containerAttrs = containerAttrs;

        parseContainerSettings();
        if (m_containerAttrs != null)
        {
            IAttributeChangeHandler changeHandler = new IAttributeChangeHandler()
            {
                @Override
                public void itemDeleted() { }
                @Override
                public void itemModified(Object value)
                {
                    ChildContainer.this.m_config = m_context.getConfiguration(m_configID, false);
                    try
                    {
                        ChildContainer.this.m_containerAttrs = ChildContainer.this.m_config.getAttributes();
                        ChildContainer.this.parseContainerSettings();
                    }
                    catch(Exception e)
                    {
                        ChildContainer.this.m_context.logMessage("Failed to apply modified container configuration [" + m_configID + "] settings, trace follows...", e, Level.WARNING);
                    }
                }
            };
            m_containerAttrs.registerAttributeChangeHandler(m_context, changeHandler);
        }
    }

    final void setActivationSettings(IAttributeSet activationAttrs)
    {
        m_activationAttrs = activationAttrs;

        parseActivationSettings();
        if (m_activationAttrs != null)
        {
            IAttributeChangeHandler changeHandler = new IAttributeChangeHandler()
            {
                @Override
                public void itemDeleted() { }
                @Override
                public void itemModified(Object value)
                {
                    IAttributeSet daemonAttrs = (IAttributeSet)m_context.getConfiguration(true).getAttributes().getAttribute(IActivationDaemonConstants.CONTAINERS_ATTR);
                    Iterator iterator = daemonAttrs.getAttributes().values().iterator();
                    while (iterator.hasNext())
                    {
                        IAttributeSet childAttrs = (IAttributeSet)iterator.next();
                        Reference ref = (Reference)childAttrs.getAttribute(IActivationDaemonConstants.CONTAINER_REF_ATTR);
                        if (ref.getElementName().equals(m_configID))
                        {
                            ChildContainer.this.m_activationAttrs = childAttrs;
                            ChildContainer.this.parseActivationSettings();
                            if (ChildContainer.this.m_schedules.size() > 0)
                            {
                                ChildContainer.this.m_daemon.notifyScheduler();
                            }
                            break;
                        }
                    }
                }
            };
            m_activationAttrs.registerAttributeChangeHandler(m_context, changeHandler);
        }
    }

    void close()
    {
        m_isClosing = true;
        if (m_containerAttrs != null)
        {
            m_containerAttrs.unregisterAttributeChangeHandler(m_context);
        }
        if (m_activationAttrs != null)
        {
            m_activationAttrs.unregisterAttributeChangeHandler(m_context);
        }
        deactivate(IChildContainerState.STATE_INACTIVE);
    }

    ArrayList getSchedules() { return m_schedules; }

    IChildContainerState getContainerState() { return m_containerState; }

    synchronized void activate(boolean waitForConfig)
    {
        if (waitForConfig)
        {
            while (!m_isClosing && m_containerState == null)
            {
                try { this.wait(); } catch(InterruptedException e) { } // only reason interrupted because were closing
                if (m_isClosing)
                {
                    return;
                }
            }
        }
        if (m_containerState == null)
        {
            throw new IllegalStateException("Container configuration not available");
        }

        m_activationRetries = 0;
        if (m_containerState.getState() != IChildContainerState.STATE_ONLINE)
        {
            m_containerState.clearExitCode();
            m_containerState.setState(IChildContainerState.STATE_OFFLINE);
            m_context.scheduleTask(this, new Date());
        }
    }

    synchronized void deactivate(short endingState)
    {
        if (m_containerState == null)
        {
            return;
        }

        if (m_containerState.getState() == IChildContainerState.STATE_ONLINE)
        {
            internalDeactivateContainer(endingState);
            sendDeactivateNotification();
        }
    }

    boolean isAutoStart() { return m_schedules.size() == 0; }

    boolean isClosing() { return m_isClosing; }

    private synchronized void parseContainerSettings()
    throws Exception
    {
        if (m_containerAttrs != null)
        {
            String domainName = (String)m_containerAttrs.getAttribute(IContainerConstants.DOMAIN_NAME_ATTR);
            if (domainName == null)
            {
                domainName = IContainerConstants.DOMAIN_NAME_DEFAULT;
            }
            m_containerName = new CanonicalName(domainName, (String)m_containerAttrs.getAttribute(IContainerConstants.CONTAINER_NAME_ATTR), "");

            // setup the working directory & the native directory
            if (IContainer.CURRENT_LAUNCHER_VERSION == null)
            {
                m_workingDir =  new File(m_rootWorkingDir + m_containerName.getContainerName());
            }
            else
            {
                File sonicContainer = new File(System.getProperty(IContainer.SONIC_CONTAINERS_DIR_PROPERTY));
                m_workingDir = new File(sonicContainer, m_containerName.getDomainName() + "." + m_containerName.getContainerName());
            }

            if (!(m_workingDir.exists() || m_workingDir.mkdir()))
            {
                throw new RuntimeException("Failed to create container's working directory: " + m_workingDir.getAbsolutePath());
            }

            m_nativeLibrariesDir = new File(m_workingDir, IContainer.DEFAULT_LIBX_DIR);
            if (!(m_nativeLibrariesDir.exists() || m_nativeLibrariesDir.mkdir()))
            {
                throw new RuntimeException("Failed to create container's native library directory: " + m_nativeLibrariesDir.getAbsolutePath());
            }
            m_nativeLibPath = (String)m_containerAttrs.getAttribute(IContainerConstants.CONTAINER_NATIVE_LIBRARY_PATH_ATTR);
            // set an initial state first time through
            if (m_containerState == null)
            {
                m_containerState = new ChildContainerState(m_containerName, m_config.getIdentity());
                m_containerState.setState(IChildContainerState.STATE_OFFLINE);
            }

            m_cacheLockFile = null;
            m_cacheHostDir = null;
            IAttributeSet cacheAttrs = (IAttributeSet)m_containerAttrs.getAttribute(IContainerConstants.CACHE_ATTR);
            if (cacheAttrs == null)
            {
                m_cacheHostDir = IContainerConstants.CACHE_DIRECTORY_DEFAULT;
            }
            else
            {
                m_cachePassword = (String)cacheAttrs.getAttribute(IContainerConstants.PASSWORD_ATTR);
                m_cacheHostDir = (String)cacheAttrs.getAttribute(IContainerConstants.CACHE_DIRECTORY_ATTR);
                if (m_cacheHostDir == null || m_cacheHostDir.length() == 0 || m_cacheHostDir.equals("."))
                {
                    m_cacheHostDir = IContainerConstants.CACHE_DIRECTORY_DEFAULT;
                }
            }
            if (m_cacheHostDir != null)
            {
                if (m_cacheHostDir.startsWith("."))
                {
                    // then its relative to the working dir
                    m_cacheHostDir = m_workingDir.getAbsolutePath() + File.separatorChar + m_cacheHostDir;
                }
                m_cacheLockFile = new File(m_cacheHostDir + File.separatorChar + ConfigCache.CACHE_LOCK_FILE);
            }
        }
    }

    private synchronized void parseActivationSettings()
    {
        if (m_activationAttrs != null)
        {
            setRetryInterval((Long)m_activationAttrs.getAttribute(IActivationDaemonConstants.RETRY_INTERVAL_SECONDS_ATTR));
            setRetryTimes((Integer)m_activationAttrs.getAttribute(IActivationDaemonConstants.RETRY_TIMES_ATTR));
            setSchedules((IAttributeSet)m_activationAttrs.getAttribute(IActivationDaemonConstants.SCHEDULES_ATTR));
        }
    }

    private synchronized void setRetryInterval(Long seconds)
    {
        if (seconds == null)
        {
            m_retryInterval = IActivationDaemonConstants.RETRY_INTERVAL_SECONDS_DEFAULT;
        }
        else
        {
            m_retryInterval = seconds.longValue();
        }
    }

    private synchronized void setRetryTimes(Integer count)
    {
        if (count == null)
        {
            m_retryTimes = IActivationDaemonConstants.RETRY_TIMES_DEFAULT;
        }
        else
        {
            m_retryTimes = count.intValue();
        }
    }

    private synchronized void setSchedules(IAttributeSet schedulesSet)
    {
        m_schedules.clear();

        if (schedulesSet != null)
        {
            Object[] schedules = schedulesSet.getAttributes().values().toArray();

            for (int i = 0; i < schedules.length; i++)
            {
                IAttributeSet scheduleAttrs = (IAttributeSet)schedules[i];

                String scheduleType = (String)scheduleAttrs.getAttribute(IActivationDaemonConstants.ACTION_TYPE_ATTR);
                String occursString = (String)scheduleAttrs.getAttribute(IActivationDaemonConstants.OCCURS_VALUE_ATTR);
                if (occursString == null)
                {
                    occursString = IActivationDaemonConstants.OCCURS_VALUE_DEFAULT;
                }
                String occursType = (String)scheduleAttrs.getAttribute(IActivationDaemonConstants.OCCURS_TYPE_ATTR);
                if (occursType == null)
                {
                    occursType = IActivationDaemonConstants.OCCURS_TYPE_DEFAULT;
                }

                ArrayList occursValues = parseOccursString(occursString, occursType);
                if (occursValues != null)
                {
                    ScheduleInfo scheduleInfo = new ScheduleInfo(scheduleType, occursType, occursValues);
                    m_schedules.add(scheduleInfo);
                }
            }
        }
    }

    private ArrayList parseOccursString(String occursString, String occursType)
    {
        ArrayList result = new ArrayList(0);

        if (occursString.length() == 0 || occursType.length() == 0) //can't be an empty string
        {
            m_context.logMessage("Invalid schedule found for container [" + m_configID + "]; schedule ignored", Level.WARNING);
            return result;
        }

        ArrayList fields = new ArrayList(0);

        StringTokenizer st = new StringTokenizer(occursString, ";");
        while (st.hasMoreTokens())
        {
            fields.add(st.nextToken());
        }

        // validate time field, can be 0:0 or 00:00 (uses 24 hours format)
        st = new StringTokenizer((String)fields.get(0), ":");
        if (st.countTokens() != 2)
        {
            m_context.logMessage("Invalid schedule time format found [" + st.toString() + "] for container [" + m_configID + "]; schedule ignored", Level.WARNING);
            return result;
        }

        ArrayList times = new ArrayList(2);
        while (st.hasMoreTokens())
        {
            times.add(st.nextToken());
        }

        Integer hh = new Integer((String)times.get(0));
        if (hh.intValue() > 23)
        {
            m_context.logMessage("Invalid schedule hour found [" + hh + "] for container [" + m_configID + "]; schedule ignored", Level.WARNING);
            return result;
        }
        result.add(hh);

        Integer mm = new Integer((String)times.get(1));
        if (mm.intValue() > 60)
        {
            m_context.logMessage("Invalid schedule minutes found [" + mm + "] for container [" + m_configID + "]; schedule ignored", Level.WARNING);
            return result;
        }
        result.add(mm);

        if (fields.size() == 2) // DOW (value from 1 to 7 are valid) or DOM (value from 1 to 31 are valid)
        {
            Integer day = new Integer((String)fields.get(1));
            if (occursType.equalsIgnoreCase("WEEKLY"))
            {
                if ((day.intValue() > 7)  || (day.intValue() < 1))
                {
                    m_context.logMessage("Invalid schedule day of week found [" + day + "] for container [" + m_configID + "]; schedule ignored", Level.WARNING);
                    return result;
                }
            }
            if (occursType.equalsIgnoreCase("MONTHLY"))
            {
                if ((day.intValue() > 31)  || (day.intValue() < 1))
                {
                    m_context.logMessage("Invalid schedule day of month found [" + day + "] for container [" + m_configID + "]; schedule ignored", Level.WARNING);
                    return result;
                }
            }
            result.add(day);
        }

        if (fields.size() == 3) // DOM (value from 1 to 31 are valid) and MOY (value from 0 to 11 are valid)
        {
            if (!occursType.equalsIgnoreCase("YEARLY"))
            {
                m_context.logMessage("Invalid schedule found for container [" + m_configID + "]; schedule ignored", Level.WARNING);
                return result;
            }

            Integer dom = new Integer((String)fields.get(1));
            if ((dom.intValue() > 31)  || (dom.intValue() < 1))
            {
                m_context.logMessage("Invalid schedule day of month found [" + dom + "] for container [" + m_configID + "]; schedule ignored", Level.WARNING);
                return result;
            }
            result.add(dom);

            Integer moy = new Integer((String)fields.get(2));
            moy = new Integer(moy.intValue() - 1); //In the configuration months are 1 to 12; in java.util.Calendar 0 to 11
            if ((moy.intValue() > 11)  || (moy.intValue() < 0))
            {
                m_context.logMessage("Invalid schedule month of year found [" + moy + "] for container [" + m_configID + "]; schedule ignored", Level.WARNING);
                return result;
            }
            result.add(moy);
        }

        return result;
    }

    String getConfigID() { return m_configID; }

    /**
     * Get the containers configuration as it contains the information we need to setup a working
     * directory for the container
     */
    private void getConfiguration()
    {
        Throwable lastException = null;

        // loop until we get the configuration or this child container is closed
        while (!m_isClosing)
        {
            try
            {
                try
                {
                    m_config = m_context.getConfiguration(m_configID, false);
                    if (m_config == null)
                    {
                        m_context.logMessage("Retrying to get container configuration [" + m_configID + "] due to unknown configuration", Level.WARNING);
                    }
                }
                catch(Throwable e)
                {
                    if (lastException == null || !(lastException.getClass().isInstance(e)))
                    {
                        m_context.logMessage("Retrying to get container configuration [" + m_configID + "] due to failure, trace follows...", e, Level.WARNING);
                    }
                    lastException = e;
                }

                // if we've now got the configuration, go ahaead an configure the container
                if (m_config != null)
                {
                    try
                    {
                        setContainerSettings(m_config.getAttributes());

                        synchronized(this)
                        {
                            this.notifyAll();
                        }
                        break;
                    }
                    catch(Exception e)
                    {
                        if (lastException == null || !(lastException.getClass().isInstance(e)))
                        {
                            m_context.logMessage("Retrying to get container configuration [" + m_configID + "] due to setup failure, trace follows...", e, Level.WARNING);
                        }
                        lastException = e;
                    }
                }
            }
            finally
            {
                if (m_containerState == null && !m_isClosing) // container state will be null until configured
                {
                    synchronized(this)
                    {
                        try { this.wait(5000); } catch(InterruptedException e) { } // backoff until situation fixed or the daemon is stopped
                    }
                }
            }
        }
    }

    @Override
    public synchronized void run()
    {
        try
        {
            if (m_containerState.getState() != IChildContainerState.STATE_ONLINE)
            {
                internalActivateContainer();
                if (m_isRunning)
                {
                    startStreamProcessors();
                    new Monitor().start();
                }
            }
        }
        catch (Exception e)
        {
            m_context.logMessage(m_containerName.getCanonicalName() + ": Container activation failure, trace follows...", e, Level.SEVERE);
        }
    }

    private void internalActivateContainer()
    throws Exception
    {
        ConfigCache.cleanupCorruptCache(m_cacheLockFile.getParentFile(), (ILogger)m_context);

        ArrayList cmdArray =
            m_daemon.getContainer().prepareCacheForActivatedContainer(m_context.getComponentName().getComponentName(),
                                                                      m_containerName.getContainerName(),
                                                                      m_configID,
                                                                      m_workingDir.getAbsolutePath(),
                                                                      m_cacheHostDir,
                                                                      m_cachePassword,
                                                                      m_activationProps);


        m_containerState.clearExitCode();
        m_context.logMessage(m_containerName.getCanonicalName() + ": Container activation initiated...", Level.INFO);

        // any additional tracing ?
        if ((m_daemon.getTraceMask().intValue() & ActivationDaemon.TRACE_CHILDPROCESS_EXEC) > 0)
        {
            StringBuffer cmd = new StringBuffer("Container execution command args:\n");
            for (int i = 0; i < cmdArray.size(); i++)
            {
                cmd.append("\n[").append((String)cmdArray.get(i) + ']');
            }
            cmd.append('\n');
            m_context.logMessage(cmd.toString(), Level.TRACE);
        }

        // get the native environment. If there are no changes to it, it will return null,
        // so the child process will inherit the parent's environment in that case
        String[] env = getNativeEnvironment();
        //now exec the process and check it did not die right away !
        m_containerProcess = Runtime.getRuntime().exec((String[])cmdArray.toArray(IEmptyArray.EMPTY_STRING_ARRAY), env, m_workingDir);
        try
        {
            m_containerState.setExitCode((short)m_containerProcess.exitValue());
        }
        catch(IllegalThreadStateException ex) // means the process is running
        {
            m_isRunning = true;
            m_containerStdin = m_containerProcess.getOutputStream();
            m_containerState.setState(IChildContainerState.STATE_ONLINE);
            sendActivateNotification();
        }
        catch(IllegalArgumentException e)
        {
            m_containerState.setExitCode((short)IContainerExitCodes.UNSPECIFIED_FAILURE_EXIT_CODE);
            m_context.logMessage(m_containerName.getCanonicalName() + ": Container activation failure, unexpected exit code=" + (short)m_containerProcess.exitValue() + " (unknown)", Level.SEVERE);
        }
    }


    private synchronized void internalDeactivateContainer(short endingState)
    {
        short state = m_containerState.getState();
        if (state == IChildContainerState.STATE_UNKNOWN || state == IChildContainerState.STATE_OFFLINE)
         {
            return; // nothing to stop
        }

        m_context.logMessage(m_containerName.getCanonicalName() + ": Container deactivation initiated...", Level.INFO);

        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(m_containerStdin));
        try
        {
            writer.write("shutdown" + LINE_SEPARATOR);
            writer.flush();
        }
        catch(IOException e) { } // can only assume the process has gone on its own

        // wait up to 5 minutes for the container to be shutdown .. after
        // that if the container process is still alive kill it
        long startDeactivateTime = System.currentTimeMillis();
        while (System.currentTimeMillis() - startDeactivateTime < MAX_DEACTIVATION_TIME)
        {
            try
            {
            	m_containerProcess.exitValue();
                break; // the container process is done, stop waiting
            }
            catch (IllegalThreadStateException e)
            {
            	try
            	{
            	    wait(500);
            	}
            	catch (InterruptedException interrupted)
            	{
            		break; // the monitor thread did a notifyAll. We're done
            	}
            }
        }
        try
        {
            m_containerProcess.exitValue(); // use to detect if process still running
        }
        catch(IllegalThreadStateException e)
        {
            // means the process is still alive! - so destroy it
            m_containerProcess.destroy();
            m_context.logMessage(m_containerName.getCanonicalName() + ": Container deactivation incomplete after " + (MAX_DEACTIVATION_TIME / 60000) +
                                 "minutes, container process destroyed", Level.SEVERE);
        }

        m_containerState.setState(endingState);
    }



    private String[] getNativeEnvironment() throws Exception
    {
    	String OS = System.getProperty("os.name").toLowerCase();
    	// get a copy of the ADs environment.
        ArrayList envList = m_daemon.getNativeEnvironment();
        if (envList != null && m_nativeLibPath != null)
        {
            for (int i=0; i< envList.size(); i++)
            {
        	    String varSetting = (String)envList.get(i);
        	    String newSetting;
        	    newSetting = checkPathVariables(varSetting);
        	    if (newSetting != null)
                {
                    envList.set(i, newSetting);
                }
            }
        }
        //unlikely that there was no path setting prior to this process, but if so, set it here
        if (envList != null && m_nativeLibPath != null &&
        		!m_ldLibraryPathSet && (OS.startsWith("linux") || OS.startsWith("sunos")))
        {

        	String setting = "LD_LIBRARY_PATH=" + this.m_nativeLibPath;
        	//m_context.logMessage("Resetting LD_LIBRARY_PATH to " + setting, Level.TRACE);
        	envList.add(setting);
        }

        if (envList != null && m_nativeLibPath != null &&
        		!m_pathSet && OS.startsWith("windows"))
        {
        	String setting = "PATH=" + this.m_nativeLibPath;
        	//m_context.logMessage("Resetting PATH to " + setting, Level.TRACE);
        	envList.add(setting);
        }
        if (envList != null && m_nativeLibPath != null &&
        		!m_shLibPathSet && OS.startsWith("hp"))
        {
        	String setting = "SHLIB_PATH=" + this.m_nativeLibPath;
        	//m_context.logMessage("Resetting SHLIB_PATH to " + setting, Level.TRACE);
        	envList.add(setting);
        }

        if (envList != null && m_nativeLibPath != null &&
        		!m_libPathSet && OS.startsWith("aix"))
        {
        	String setting = "LIBPATH=" + this.m_nativeLibPath;
        	//m_context.logMessage("Resetting LIBPATH to " + setting, Level.TRACE);
        	envList.add(setting);
        }

        String[] env = null;
        if (envList != null)
        {
        	env = new String[envList.size()];
            envList.toArray(env);
        }
        return env;
    }

    // Add the setting of the CONTAINER_NATIVE_LIBRARY_PATH to the Path (on Windows),
    // LD_LIBRARY_PATH on Linux and Solaris, SHLIB_PATH on HP and LIBPATH on AIX.
    // We assume m_nativeLibPath is not null if we get here.

    private String checkPathVariables(String varSetting)
    {
    	String OS = System.getProperty("os.name").toLowerCase();
    	//m_context.logMessage("ChildContainer.checkPathVariables OS == " + System.getProperty("os.name") , Level.TRACE);
    	String pathSeparator = System.getProperty("path.separator");
    	String newSetting = null;
    	// Windows
    	if (varSetting.toLowerCase().startsWith("path=") && OS.startsWith("windows"))
    	{
    		newSetting = varSetting.substring(0, 5)+ this.m_nativeLibPath + pathSeparator + varSetting.substring(5);
    		m_pathSet = true;
    		//m_context.logMessage("ChildContainer set PATH to " + newSetting, Level.TRACE);
    	}
    	// Linux And Solaris
    	else if (varSetting.toLowerCase().startsWith("ld_library_path=") &&
    			(OS.startsWith("linux") || OS.startsWith("sunos")))
    	{
    		newSetting = varSetting.substring(0, 16)+ this.m_nativeLibPath + pathSeparator + varSetting.substring(16);
    		//m_context.logMessage("ChildContainer.checkPathVariables set LD_LIBRARY_PATH: " + newSetting, Level.TRACE);
    		m_ldLibraryPathSet = true;
    	}
    	else if (varSetting.toLowerCase().startsWith("shlib_path=") && OS.startsWith("hp"))
    	{
    		newSetting = varSetting.substring(0, 11)+ this.m_nativeLibPath + pathSeparator + varSetting.substring(11);
    		//m_context.logMessage("ChildContainer.checkPathVariables set SHLIB_PATH: " + newSetting, Level.TRACE);
    		m_shLibPathSet = true;
    	}
    	else if (varSetting.toLowerCase().startsWith("libpath=") && OS.startsWith("aix"))
    	{
    		newSetting = varSetting.substring(0, 8)+ this.m_nativeLibPath + pathSeparator + varSetting.substring(8);
    		//m_context.logMessage("ChildContainer.checkPathVariables set LIBPATH: " + newSetting, Level.TRACE);
    		m_libPathSet = true;
    	}
    	return newSetting;
    }

    private void startStreamProcessors()
    {
        m_containerStdout = new OutputReader(m_containerProcess.getInputStream(), true);
        m_containerStderr = new OutputReader(m_containerProcess.getErrorStream(), false);

        Thread readerThread = null;

        readerThread = new Thread(m_containerStdout, ChildContainer.this.m_context.getComponentName().getComponentName() + " - Child Container stdout Reader: " + m_containerName.getCanonicalName());
        readerThread.setDaemon(true);
        readerThread.start();

        readerThread = new Thread(m_containerStderr, ChildContainer.this.m_context.getComponentName().getComponentName() + " - Child Container stderr Reader: " + m_containerName.getCanonicalName());
        readerThread.setDaemon(true);
        readerThread.start();

        m_inputFlusher = new InputFlusher(m_containerStdin, "container " + m_containerName.getCanonicalName());
        Thread flusherThread = new Thread(m_inputFlusher, ChildContainer.this.m_context.getComponentName().getComponentName() + " - Child Container input flusher: " + m_containerName.getCanonicalName());
        flusherThread.setDaemon(true);
        flusherThread.start();
    }

    private void monitorContainer()
    {
        try
        {
            short exitCode = (short)m_containerProcess.waitFor();
            synchronized(this)
            {
                notifyAll();
            }
            try
            {
                m_containerState.setExitCode(exitCode);
                m_context.logMessage(m_containerName.getCanonicalName() + ": Container deactivated/shutdown, exit code=" + exitCode + " (" + IContainerExitCodes.EXIT_CODE_TEXTS[exitCode] + ')', Level.INFO);
            }
            catch(IllegalArgumentException e)
            {
                m_containerState.setExitCode((short)IContainerExitCodes.UNSPECIFIED_FAILURE_EXIT_CODE);
                m_context.logMessage(m_containerName.getCanonicalName() + ": Container deactivated/shutdown, unexpected exit code=" + exitCode + " (unknown)", Level.WARNING);
            }
            m_isRunning = false;

            short state = m_containerState.getState();
            if (state == IChildContainerState.STATE_ONLINE)
            {
                m_containerState.setState(IChildContainerState.STATE_OFFLINE);
            }

            if (m_daemon.getState().shortValue() == IComponentState.STATE_OFFLINE)
             {
                return; // monitoring can be terminated at this point
            }

            switch(m_containerState.getExitCode().intValue())
            {
                // normal exit
                case 0:
                    break;
                    // container restart requested
                case IContainerExitCodes.CONTAINER_RESTART_EXIT_CODE:
                {
                    m_context.logMessage(m_containerName.getCanonicalName() + ": Container initiated restart", Level.INFO);
                    Runnable activator = new Runnable()
                    {
                        @Override
                        public void run()
                        {
                            ChildContainer.this.activate(false);
                        }
                    };
                    m_context.scheduleTask(activator, new Date(System.currentTimeMillis() + 2000));
                    break;
                }
                // container cache failure
                case IContainerExitCodes.CACHE_FAILURE_EXIT_CODE:
                {
                    sendFailureNotification();

                    // can't retry if cache has failed
                    if (m_retryTimes > 0)
                    {
                        m_context.logMessage(m_containerName.getCanonicalName() + ": Container activation retry not initiated due to previous cache failure", Level.WARNING);
                        m_activationRetries = 0; // reset counter
                        return;
                    }
                    break;
                }
                // all other exit codes
                default:
                {
                    sendFailureNotification();

                    // start retry ?
                    if (m_retryTimes > 0) // where O indicates no failure rules applies
                    {
                        if (m_activationRetries < m_retryTimes)
                        {
                            // the restart will have already been logged, so just return
                            if (m_containerState.getExitCode().intValue() == IContainerExitCodes.CONTAINER_RESTART_EXIT_CODE)
                            {
                                m_activationRetries = 0; // reset counter
                                return;
                            }

                            Thread.sleep(m_retryInterval * 1000);

                            // if the schedule indicates that we are not in an active schedule phase
                            // then don't bother retrying anymore
                            if (!isExpectedToBeActive())
                            {
                                m_activationRetries = 0; // reset counter
                                return;
                            }

                            m_context.logMessage(m_containerName.getCanonicalName() + ": Container activation retry initiated...", Level.INFO);

                            if (m_retryTimes != -1)
                            {
                                // retry forever
                                m_activationRetries++;
                            }

                            m_context.scheduleTask(this, new Date(System.currentTimeMillis()));
                        }
                    }
                    break;
                }
            }
        }
        catch (InterruptedException ie) {}
    }

    private synchronized boolean isExpectedToBeActive()
    {
        if (m_schedules.isEmpty())
         {
            return true; // no schedule so effectively always expected to be active
        }

        for (int i = 0; i < m_schedules.size(); i++)
        {
            ScheduleInfo scheduleInfo = (ScheduleInfo)m_schedules.get(i);
            if (scheduleInfo.getScheduleType() == 0)
            {
                if (scheduleInfo.getPerformActionAtDate().getTime() <= System.currentTimeMillis())
                {
                    return true;
                }
            }
        }

        return false;
    }

    private void sendActivateNotification()
    {

        INotification notification =
            m_context.createNotification(INotification.SYSTEM_CATEGORY,
                                         INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
                                         ActivationDaemon.ACTIVATE_NOTIFICATION_TYPE,
                                         Level.INFO);

        notification.setLogType(INotification.INFORMATION_TYPE);
        notification.setAttribute("Container", m_containerName.getCanonicalName());
        m_context.sendNotification(notification);
    }

    private void sendDeactivateNotification()
    {
        INotification notification =
            m_context.createNotification(INotification.SYSTEM_CATEGORY,
                                         INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
                                         ActivationDaemon.DEACTIVATE_NOTIFICATION_TYPE,
                                         Level.INFO);

        notification.setLogType(INotification.INFORMATION_TYPE);
        notification.setAttribute("Container", m_containerName.getCanonicalName());
        m_context.sendNotification(notification);
    }


    void sendFailureNotification()
    {
        INotification notification =
           m_context.createNotification(INotification.SYSTEM_CATEGORY,
                                        INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
                                        ActivationDaemon.FAILURE_NOTIFICATION_TYPE,
                                        Level.SEVERE);
        notification.setLogType(INotification.ERROR_TYPE);
        notification.setAttribute("Container", m_containerName.getCanonicalName());
        notification.setAttribute("ExitCode", m_containerState.getExitCode().toString());
        m_context.sendNotification(notification);
    }

    private final class Monitor
    extends Thread
    {
        private Monitor()
        {
            super(ChildContainer.this.m_context.getComponentName().getComponentName() + " - Child Container Monitor: " + ChildContainer.this.m_containerName.getCanonicalName());
            setDaemon(true);
        }

        @Override
        public void run() { ChildContainer.this.monitorContainer(); }
    }

    private final class InputFlusher
    implements Runnable
    {
    	OutputStream m_input;
    	String m_flusherName;

    	private InputFlusher(OutputStream stream, String flusherName)
    	{
    		m_input = stream;
    		m_flusherName = flusherName;
    	}
    	@Override
        public void run()
    	{
    		while (ChildContainer.this.m_isRunning)
    		{
    			try
    			{
    				//System.err.println("ChildContainer flushing child's input");
    				m_input.write((System.getProperty("line.separator")).getBytes());
    			    m_input.flush();
    			}
    			catch (IOException ioE) // When the container is shutting down, the pipes
    			                       // will close and so we'll stop flushing the input
    			{
    			    break;
    			}

    			try
    			{
    			    Thread.sleep(2000);
    			}
    			catch (InterruptedException interrupted)
    			{
    				break;
    			}
    		}
    	}
    }



    private final class OutputReader
    implements Runnable
    {
        InputStream stream;
        boolean isStdOut = false;

        private OutputReader(InputStream stream, boolean isStdOut)
        {
            this.stream = stream;
            this.isStdOut = isStdOut;
        }


        @Override
        public void run()
        {
            BufferedReader reader = new BufferedReader(new InputStreamReader(this.stream));

            final ArrayList traceBuffer = new ArrayList();
            Thread traceThread = new Thread()
            {
                @Override
                public void run()
                {
                    outerLoop:
                    while (ChildContainer.this.m_isRunning && !isInterrupted())
                    {
                        synchronized(traceBuffer)
                        {
                            if (traceBuffer.size() < 2)
                            {
                                try { traceBuffer.wait(500); } catch(InterruptedException e) { }
                            }
                                // is there something to log?
                            if (!traceBuffer.isEmpty())
                            {
                                StringBuffer sb = new StringBuffer(isStdOut ? "[stdout]" : "[stderr]");
                                sb.append(' ').append(ChildContainer.this.m_containerName.getCanonicalName());

                                int initialSize = traceBuffer.size();
                                for (int i = 0; i < initialSize; i++)
                                {
                                    String traceLine = (String)traceBuffer.get(0);
                                    if (i > 0 && traceLine.startsWith("["))
                                    {
                                        break;
                                    }
                                    traceBuffer.remove(0);
                                    if (traceLine.length() == 0 && i == 0)
                                    {
                                        continue outerLoop;
                                    }
                                    sb.append(LINE_SEPARATOR).append(">>").append(traceLine);
                                }
                                ChildContainer.this.m_context.logMessage(sb.toString(), Level.TRACE);
                            }
                        }
                    }
                }
            };
            traceThread.setDaemon(true);
            traceThread.start();

            whileLoop:
            while (ChildContainer.this.m_isRunning)
            {
                try
                {
                    String input = reader.readLine();

                    // sometimes we get null so filter out
                    if (input == null)
                    {
                        try { Thread.sleep(250); } catch(Exception e) { }
                        continue;
                    }

                    // if its the prompt line, filter it out
                    if (input.startsWith(ChildContainer.this.m_containerName.getCanonicalName() + '>'))
                    {
                        continue;
                    }

                    // strip out startup signal
                    if (input.length() > 0)
                    {
                        if (input.charAt(0) == IContainer.STARTUP_SIGNAL_CHAR ||
                           (input.length() > 1 && input.charAt(0) == '*' && input.charAt(1) == IContainer.STARTUP_SIGNAL_CHAR))
                        {
                            ChildContainer.this.m_context.logMessage(ChildContainer.this.m_containerName.getCanonicalName() + ": ...activation complete", Level.INFO);
                            input = input.substring(1);
                            ChildContainer.this.m_activationRetries = 0; // reset the retries as we have success
                        }
                        else
                        {
                            // strip out shutdown signals
                            if (input.charAt(0) == IContainer.SHUTDOWN_SIGNAL_CHAR ||
                               (input.length() > 1 && input.charAt(0) == '*' && input.charAt(1) == IContainer.SHUTDOWN_SIGNAL_CHAR))
                            {
                                int i = 1;
                                for (; i < input.length(); i++)
                                {
                                    if (input.charAt(i) == IContainer.SHUTDOWN_SIGNAL_CHAR)
                                    {
                                        if (i == input.length() - 1)
                                        {
                                            continue whileLoop;
                                        }
                                        input = input.substring(i + 1);
                                        break;
                                    }
                                }
                            }
                        }
                    }

                    // add to container state
                    if (!input.startsWith(ChildContainer.this.m_containerName.getCanonicalName() + '>'))
                    {
                        ChildContainer.this.m_containerState.addLogLine(input);
                    }

                    // do we log on the local console
                    int traceMask = m_daemon.getTraceMask().intValue();
                    if (((traceMask & ActivationDaemon.TRACE_CHILDPROCESS_STDOUT) > 0  && this.isStdOut) ||
                        ((traceMask & ActivationDaemon.TRACE_CHILDPROCESS_STDERR) > 0 && !this.isStdOut))
                    {
                        synchronized(traceBuffer)
                        {
                            traceBuffer.add(input);
                            if (input.startsWith("["))
                            {
                                traceBuffer.notifyAll();
                            }
                        }
                    }
                }
                catch(IOException e)
                {
                    synchronized(traceThread)
                    {
                        if (traceThread.isAlive())
                        {
                            traceThread.interrupt();
                        }
                    }
                    ChildContainer.this.m_context.logMessage(ChildContainer.this.m_containerName + ": Failed to record stdin/stderr, trace follows...", e, Level.WARNING);
                    break;
                }
            }
        }
    }
}
