package com.sonicsw.mf.framework.daemon;

import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;

import com.sonicsw.mf.common.runtime.IChildContainerState;
import com.sonicsw.mf.framework.IFrameworkComponentContext;

class Scheduler
extends Thread
{
    private IFrameworkComponentContext m_context;
    private HashMap m_childContainers;
    private Object m_schedulerLockObj = new Object();

    public static final long HOURLY = (long)60*60*1000;
    public static final long DAILY = 24*HOURLY;
    public static final long WEEKLY = 7*DAILY;
    public static final long MONTHLY = -1;
    public static final long YEARLY = -2;

    Scheduler(IFrameworkComponentContext context, HashMap childContainers)
    {
        super(context.getComponentName().getComponentName() + " - Child Container Activation Scheduler");
        m_context = context;
        m_childContainers = childContainers;
        setDaemon(true);
    }

    private void runSchedules()
    {
        long currentTime = System.currentTimeMillis();
        
        Object[] childContainers = m_childContainers.values().toArray();

        for (int i = 0; i < childContainers.length; i++)
        {
            ChildContainer child = (ChildContainer)childContainers[i];
            if (child.isClosing())
            {
                continue;
            }
            IChildContainerState childContainerState = child.getContainerState();
            // check if there is a sttate and whether schedules are disabled
            if (childContainerState == null || childContainerState.getState() != IChildContainerState.STATE_INACTIVE)
            {
                Object[] schedules =  child.getSchedules().toArray();
                for (int j = 0; j < schedules.length; j++)
                {
                    ScheduleInfo info = (ScheduleInfo)schedules[j];

                    if (info.getPerformActionAtDate().getTime() <= currentTime)
                    {
                        boolean start = info.getScheduleType() == ScheduleInfo.START_TYPE;
                        short currentState = childContainerState == null ? IChildContainerState.STATE_UNKNOWN : childContainerState.getState();

                        if (start && currentState != IChildContainerState.STATE_ONLINE)
                        {
                            m_context.scheduleTask(child, info.getPerformActionAtDate());
                        }
                        else
                        {
                           if (!start && currentState == IChildContainerState.STATE_ONLINE)
                        {
                            child.deactivate(IChildContainerState.STATE_OFFLINE);
                        }                   
                        }
                        updateSchedule(info);
                    }
                }
            }
        }
    }

    private void updateSchedule(ScheduleInfo info)
    {
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTime(info.getPerformActionAtDate());
        if (info.getOccursType() == Scheduler.MONTHLY)
        {
            Date nextDate = getCorrectDOM(info); //to handle maximum days in next month
            info.setPerformActionAtDate(nextDate);
        }
        else if (info.getOccursType() == Scheduler.YEARLY)
        {
             cal.add(Calendar.YEAR, 1);
             info.setPerformActionAtDate(cal.getTime());
        }
        else //weekly or daily
        {
            // CR 51757 changed to add a week or a day to the current schedule, which works around
            // DST dates. For the boundary condition when a scheduled date happens to be
            // on the lost hour of DST start, Calendar will pick the next hour or the previous hour. 
            // Schedules after that would then use the wrong hour, which we deal with by calling
            // ShceduleInfo.getBackToDesiredHour
            if (info.getOccursType() == Scheduler.WEEKLY)
            {
                cal.add(Calendar.WEEK_OF_YEAR, 1);
            }
            else
            {
                cal.add(Calendar.DAY_OF_YEAR, 1);
            }
            info.setPerformActionAtDate(new Date(cal.getTimeInMillis()));
        }
        info.getBackToDesiredHour();
    }
    
    // can't use Calendar.MONTH to change next day for monthly interval
    // when last day of month is entered as scheduled date of month,
    // Such as was scheduled January 31, next month is February(can be only 28 0r 29),
    // thus it will be scheduled for February 28(or 29 depends on year),then next scheduled month is
    // March, where it will be scheduled for 31 ext...

    private Date getCorrectDOM(ScheduleInfo info)
    {
        GregorianCalendar cal = new GregorianCalendar();
        int origDate = info.getOriginalSchDate();
        int maxForNextMonth = info.getMaxForNextMonth();

        cal.setTime(info.getPerformActionAtDate());

        if(maxForNextMonth < origDate)
        {
            cal.set(Calendar.DAY_OF_MONTH,maxForNextMonth);
            cal.add(Calendar.MONTH, 1);
        }
        else
        {
            cal.set(Calendar.DAY_OF_MONTH, 1);
            cal.add(Calendar.MONTH, 1);
            cal.set(Calendar.DAY_OF_MONTH,origDate);
        }
        return cal.getTime();
    }


    public void changed()
    {
        synchronized (m_schedulerLockObj)
        {
            m_schedulerLockObj.notifyAll();
        }
    }

    @Override
    public void run()
    {
        synchronized (m_schedulerLockObj)
        {
            while (!isInterrupted())
            {
                runSchedules();

                // run again at the start of the next minute - this poll only needs to occur each minute because
                // that's the level of granularity we have in schedules - if changes occur they will notify
                // the scheduler to run outside of the minute cycle
                Calendar date = new GregorianCalendar();
                date.set(Calendar.MILLISECOND, 0);
                long delta = (60 - date.get(Calendar.SECOND)) * 1000;
                try
                {
                    if (delta > 0)
                    {
                        m_schedulerLockObj.wait(delta);
                    }
                }
                catch(InterruptedException ie)
                {
                    return; // we were interrupted to close this thread out on activation daemon stop()
                }
            }
        }
    }
}