package com.sonicsw.mf.framework.directory;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Hashtable;

import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.mgmtapi.config.constants.IBackupDirectoryServiceConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IDirectoryServiceConstants;

/**
 * Attaches a BLOB to the given DS config ID.
 * Requires a XML configuration file for the DS in order to bootstrap itself.
 */
public final class DirectoryBlobAttacher
{
    private IDirectoryService m_ds;

    public static void main(String[] args)  throws Exception
    {
        if (args.length < 3 || args[0].equals("?"))
        {
            printUsage();
        }

        try
        {
            new DirectoryBlobAttacher(args);
        }
        catch(Exception e) { e.printStackTrace(); }
    }

    private DirectoryBlobAttacher(String[] args)
    throws Exception
    {
        // get the xml that described the DS configuration and create the DS
        String dsXML = getXMLStringFromFile(args[0]);
        m_ds = createDSInstance(dsXML);

        // get the element we want to attach the BLOB to
        IDirElement element = m_ds.getElement(args[1], true);

        // create a stream to the BLOB
        File blob = new File(args[2]);
        if (!blob.isFile() || !blob.canRead())
        {
            throw new IOException("Cannot read BLOB: " + args[2]);
        }
        FileInputStream fis = new FileInputStream(blob);

        // now place in the DS
        System.out.println("Attaching BLOB in directory store ...");
        m_ds.attachBlob(element.doneUpdate(), fis, null);

        // now report whats in there
        String contents = m_ds.exportDirectoryToXML("/");
        if (System.getProperty("verbose", "false").equals("true"))
        {
            System.out.println("Reporting contents ...");
            System.out.println();
            System.out.println(contents);
            System.out.println();
        }
        m_ds.close();
        System.out.println("Done.");

        System.exit(0);
    }

    private static void printUsage()
    {
        System.out.println();
        System.out.println("Usage: -Dverbose={[false]|true} com.sonicsw.mf.framework.directory.storage.DirectoryBlobSeeder <ds.xml> <seeddata.xml> <blob>");
        System.out.println();
        System.out.println("Where: <ds.xml>  Directory Service configuration.");
        System.out.println("       <element> The configuration element against which the binary");
        System.out.println("                 file will be stored.");
        System.out.println("       <blob>    The path to the binary file to be stored with the");
        System.out.println("                 element.");

        System.exit(1);
    }

    private String getXMLStringFromFile(String xmlFilename)
    throws Exception
    {
        FileInputStream fis = new FileInputStream(xmlFilename);
        byte[] bytes = new byte[fis.available()];
        fis.read(bytes);

        return new String(bytes);
    }


    private IDirectoryService createDSInstance(String dsXML)
    throws Exception
    {
        IDirElement dsConfig = ElementFactory.importElementFromXML(dsXML, null, IDirectoryServiceConstants.DS_TYPE);
        if (dsConfig == null)
        {
            dsConfig = ElementFactory.importElementFromXML(dsXML, null, IBackupDirectoryServiceConstants.DS_TYPE);
        }

        IAttributeSet dsAttributes = dsConfig.getAttributes();
        String hostDir = (String)dsAttributes.getAttribute(IDirectoryServiceConstants.HOST_DIRECTORY_ATTR);

        String domainName = (String)dsAttributes.getAttribute(IDirectoryServiceConstants.DOMAIN_NAME_ATTR);
        Object tmp = dsAttributes.getAttribute(IDirectoryServiceConstants.FILE_SYSTEM_STORAGE_ATTR);

        if (domainName == null || tmp == null)
        {
            throw new MFRuntimeException("Bad Directory Service Configuration - must contain DOMAIN_NAME and FILE_SYSTEM_STORAGE.");
        }

        if (!(tmp instanceof IAttributeSet))
        {
            throw new MFRuntimeException("Bad Directory Service Configuration - FILE_SYSTEM_STORAGE must be an attribute set.");
        }

        IAttributeSet fsStorage = (IAttributeSet)tmp;
        String hostDirDepricated = (String)fsStorage.getAttribute(IDirectoryServiceConstants.HOST_DIRECTORY_ATTR);
        String password = (String)fsStorage.getAttribute(IDirectoryServiceConstants.PASSWORD_ATTR);

        // Creates the directory service object
        Hashtable directoryEnv = new Hashtable();
        directoryEnv.put(IDirectoryService.STORAGE_TYPE_ATTRIBUTE, IDirectoryService.PSE_STORAGE);
        if (hostDirDepricated != null)
        {
            directoryEnv.put(IFSStorage.FS_HOST_DIRECTORY_ATTRIBUTE, hostDirDepricated);
        }
        if (hostDir != null)
        {
            directoryEnv.put(IFSStorage.HOST_DIRECTORY_ATTRIBUTE, hostDir);
        }
        if ((password != null) && (password.length() != 0))
        {
            directoryEnv.put(IFSStorage.PASSWORD_ATTRIBUTE, password);
        }
        DirectoryServiceFactory factory = new DirectoryServiceFactory(directoryEnv);
        return factory.createDirectoryService(domainName);
    }
}
