package com.sonicsw.mf.framework.directory;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;

import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IIdentity;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.agent.ContainerUtil;
import com.sonicsw.mf.framework.directory.impl.DirectoryService;
import com.sonicsw.mf.framework.directory.storage.pse.PSEStorage;
import com.sonicsw.mf.mgmtapi.config.constants.IBackupDirectoryServiceConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IDirectoryServiceConstants;

/**
 * Seeds a Directory Service store to XML
 */
public final class DirectoryDump
{
    private IDirectoryService m_ds;
    private static final String FILES_DUMP_SUFFIX = "_files";

    public static void main(String[] args)  throws Exception
    {
        if (args.length < 2 || args[0].equals("?"))
        {
            printUsage();
        }

        try
        {
            System.setProperty("DSDUMP", "true");
            new DirectoryDump(args);
        }
        catch(Exception e) { e.printStackTrace(); }
    }

    private DirectoryDump(String[] args)
    throws Exception
    {
        try
        {
            // get the xml that described the DS configuration and create the DS
            m_ds = createDSInstance(args[0]);

            // now place in the DS
            
            boolean findBlobsDir = false;
            String dumpFile = null;
            String blobsDir = null;
            File blobsDirFile = null;
            for (int i=1; i<args.length; i++)
            {
                if (findBlobsDir)
                {
                    findBlobsDir = false;
                    blobsDir = args[i];
                }
                else
                {
                    if (args[i].equals("-files"))
                    {
                        findBlobsDir = true;
                    }
                    else
                    {
                        dumpFile = args[i];
                    }
                }

            }
            // check the directory where we're dumping the blobs before dumping anything
            if (blobsDir != null)
            {
                blobsDirFile = new File(blobsDir);
                if (!blobsDirFile.exists())
                {
                    throw new DirectoryServiceException("Directory " + blobsDir + " doesn't exist");
                }
                blobsDirFile = new File(blobsDir, m_ds.getDomain() + FILES_DUMP_SUFFIX);
                if (blobsDirFile.exists())
                {
                    throw new DirectoryServiceException("Directory " + blobsDirFile.getCanonicalPath() + " already exists. Please remove the directory " + blobsDirFile.getName() + " and try again");
                }
               
            }
            String contents = m_ds.exportDirectoryToXML("/");
            
            if (dumpFile != null)
            {
                System.out.println("Dumping configurations:");
                dumpXMLStringToFile(contents, dumpFile);
                System.out.println("Done.");
            }
            if (blobsDirFile != null) // dump the blobs
            {
                System.out.println("Dumping files:");
                dumpBlobs(blobsDirFile, "/");
                System.out.println("Done.");
            }
            reportCorrupt(contents);
        }
        finally
        {
            if (m_ds != null)
            {
                m_ds.close();
            }
        }
        

        System.exit(0);
    }

    private static void printUsage()
    {
        System.out.println();
        System.out.println("Usage: com.sonicsw.mf.framework.directory.storage.DirectoryDump <ds.xml> <out.xml> -files <files_directory>");
        System.out.println();
        System.out.println("Where: <ds.xml>          Directory Service configuration.");
        System.out.println("       <out.xml>         The Directory Service content in XML format.");
        System.out.println("       <files_directory> The directory to write the files stored in the Directory Service. ");
        System.out.println("                         This utility creates the <domain_name>_files subdirectory to dump the files into.");

        System.exit(1);
    }

    private void dumpXMLStringToFile(String xmlString, String dumpFilename)
    throws Exception
    {
        FileOutputStream fos = new FileOutputStream(dumpFilename);
        fos.write(xmlString.getBytes());
        fos.close();
    }

    private void dumpBlobs(File blobDirectory, String dsDir) throws Exception
    {
        IIdentity[] elIds = m_ds.listAll(dsDir);
        
        for (int i=0; i<elIds.length; i++)
        {
            if (elIds[i] instanceof IDirIdentity)
            {
                dumpBlobs(blobDirectory, elIds[i].getName());
            }
            else
            {
                String blobName = elIds[i].getName();
                // if we're reporting corrupt elements, and this element is already tagged
                // as corrupt, getBlob might fail, so skip it
                if (PSEStorage.hasCorruptElements() && PSEStorage.getCorruptIds().keySet().contains(blobName))
                {
                    continue;
                }
                IBlob blob = m_ds.getBlob(blobName, false);
                if (blob != null && blob.getBlobStream() != null)
                {
                    File dumpBlobDir = new File(blobDirectory, blobName.substring(0, blobName.lastIndexOf(DSComponent.MF_DIR_SEPARATOR_STRING)));
                    dumpBlobDir.mkdirs();
                    File dumpBlobFile = new File(blobDirectory, blobName);
                    FileOutputStream dumpStream = new FileOutputStream(dumpBlobFile);
                    InputStream blobStream = blob.getBlobStream();
                    BufferedOutputStream bufOut = new BufferedOutputStream(dumpStream, IBlob.BLOB_CHUNK_SIZE);
                    BufferedInputStream bufIn = new BufferedInputStream(blobStream, IBlob.BLOB_CHUNK_SIZE);
                    int byteRead = bufIn.read();
                    while (byteRead != -1)
                    {
                        bufOut.write(byteRead);
                        byteRead = bufIn.read();
                    }
                    blobStream.close();
                    bufOut.close();
                }
            }
        }
    }

    private IDirectoryService createDSInstance(String dsXMLFile)
    throws Exception
    {

        IElement[] dsConfigs = ContainerUtil.importConfigurations(new File(dsXMLFile), IContainer.PASSWORD);
        IDirElement dsConfig = null;
        for (int i = 0; i < dsConfigs.length; i++)
        {
            String elType = dsConfigs[i].getIdentity().getType();
            if (elType.equals(IDirectoryServiceConstants.DS_TYPE) || elType.equals(IBackupDirectoryServiceConstants.DS_TYPE))
            {
                dsConfig = (IDirElement)dsConfigs[i];
            }
        }

        if (dsConfig == null)
        {
            throw new Exception("File " + dsXMLFile + " does not contain valid Directory Service boot information");
        }

        IAttributeSet dsAttributes = dsConfig.getAttributes();
        String hostDir = (String)dsAttributes.getAttribute(IDirectoryServiceConstants.HOST_DIRECTORY_ATTR);

        String domainName = (String)dsAttributes.getAttribute(IDirectoryServiceConstants.DOMAIN_NAME_ATTR);
        if (domainName == null)
        {
            domainName = IContainerConstants.DOMAIN_NAME_DEFAULT;
        }

        Object tmp = dsAttributes.getAttribute(IDirectoryServiceConstants.FILE_SYSTEM_STORAGE_ATTR);

        if (domainName == null || tmp == null)
        {
            throw new MFRuntimeException("Bad Directory Service Configuration - must contain DOMAIN_NAME and FILE_SYSTEM_STORAGE.");
        }

        if (!(tmp instanceof IAttributeSet))
        {
            throw new MFRuntimeException("Bad Directory Service Configuration - FILE_SYSTEM_STORAGE must be an attribute set.");
        }



        IAttributeSet fsStorage = (IAttributeSet)tmp;

        String hostDirDepricated = (String)fsStorage.getAttribute(IDirectoryServiceConstants.HOST_DIRECTORY_ATTR);
        String password = (String)fsStorage.getAttribute(IDirectoryServiceConstants.PASSWORD_ATTR);

        // Creates the directory service object
        Hashtable directoryEnv = new Hashtable();
        directoryEnv.put(IDirectoryService.STORAGE_TYPE_ATTRIBUTE, IDirectoryService.PSE_STORAGE);
        if (hostDirDepricated != null)
        {
            directoryEnv.put(IFSStorage.FS_HOST_DIRECTORY_ATTRIBUTE, hostDirDepricated);
        }
        if (hostDir != null)
        {
            directoryEnv.put(IFSStorage.HOST_DIRECTORY_ATTRIBUTE, hostDir);
        }
        if ((password != null) && (password.length() != 0))
        {
            directoryEnv.put(IFSStorage.PASSWORD_ATTRIBUTE, password);
        }
        DirectoryServiceFactory factory = new DirectoryServiceFactory(directoryEnv);
        return factory.createDirectoryService(domainName);
    }
    
    private void reportCorrupt(String dumpedXml)
    {
        if (PSEStorage.hasCorruptElements())
        {
            //report a corrupt id cache differently, since the dump doesn't write it out
            // but they can still fix a corrupt id cache by reloading the dumped ds
            HashMap corruptData = PSEStorage.getCorruptIds();
            boolean idCacheCorrupt = corruptData.containsKey(DirectoryService.IDCACHE_ELEMENT);
            boolean onlyIdCacheIsCorrupt = idCacheCorrupt && (corruptData.size() == 1);
            System.out.println();
            if (!onlyIdCacheIsCorrupt)
            {
                int problemIndex = 0;
                System.out.println("dsAdmin dump skipped over the following elements because they were found to be corrupt. Creating a new Directory Service using the dsadmin load option will remove the elements from the Directory Service: ");
                System.out.println();
                Iterator stringIT = corruptData.keySet().iterator();
                while (stringIT.hasNext())
                {
                    String elId = (String)stringIT.next();
                    // skip the id cache. It is reported in a different way below
                    if (!elId.equals(DirectoryService.IDCACHE_ELEMENT))
                    {
                        System.out.println("**** Problem #" + ++problemIndex + " ****");
                        Object message = corruptData.get(elId);
                        printCorruptId(elId);
                        System.out.println();
                        printStackTrace((Object[])message);
                        if (elId.startsWith("/"))
                        {
                            System.out.println();
                            findAndPrintReferences(elId, dumpedXml);
                        }
                        if (stringIT.hasNext())
                        {
                            System.out.println();
                        }
                    }
                }
            }
            if (onlyIdCacheIsCorrupt)
            {
                System.out.println("The id cache was corrupt. Creating a new Directory Service using the dsadmin load option will recreate the id cache.");
            }
            else if (idCacheCorrupt)
            {
                System.out.println("The id cache was also corrupt and will be recreated when you use the dsadmin load option to create a new Directory Service");
            } 
            System.out.println();
        }
    }
    
    private void findAndPrintReferences(String elId, String dumpedXml)
    {
        final String ELEMENT_ID = "ElementID name=";
        // special case config types. They are referenced in every element of its
        // type, which can be a long list.
        if (elId.startsWith("/mx/configTypes"))
        {
            System.out.println("The element " + elId + " is referenced in all the elements of that type");
            return;
        }
        String searchId = getSearchId(elId);
        boolean done = false;
        ArrayList referencedIn = new ArrayList();
        int searchStart = 0;
        while (!done)
        {
            int foundLocation = dumpedXml.indexOf(searchId, searchStart);
            if (foundLocation > -1)
            {
                // go back to find the element name of the element that references elId
                int elementIdLocation = dumpedXml.lastIndexOf(ELEMENT_ID, foundLocation) + 16;
                // we assume this location exists
                int endIdLocation = dumpedXml.indexOf('\"', elementIdLocation);
                String referenceId = dumpedXml.substring(elementIdLocation, endIdLocation);
                if (!referencedIn.contains(referenceId) && (!referenceId.equals(DirectoryService.VIEW_ELEMENT)))
                {
                    referencedIn.add(referenceId);
                }
                searchStart = foundLocation + elId.length();
            }
            else
            {
                done = true;
            }
        }
        if (!referencedIn.isEmpty())
        {
            System.out.println("Element " + elId + " is referenced by the following elements:");
            Iterator idsIt = referencedIn.iterator();
            while (idsIt.hasNext())
            {
                String id = (String)idsIt.next();
                System.out.print(id);
                try
                {
                    String logicalId = m_ds.storageToLogical(id);
                    if (logicalId != null)
                    {
                        System.out.print("(" + logicalId + ")");
                    }
                }
                catch (DirectoryServiceException dirE) {} //couldn't get the logical name, nothing to print
                System.out.println();
            }
        }
        else
        {
            System.out.println(elId + " is not referenced by other elements in the Directory Service");
        }
    }
    
     // References to ESB services, processes and endpoints use the basename of the id instead of the 
    // whole id, and the same base name will appear in the dump many times as the value of different attributes.
    // To find the elements that reference processes and services, you look for name="service_ref" value="MyServiceOrMyProcess".
    // Endpoint references are of the form name="endpoint_ref" value="endpointBaseName"
    private String getSearchId(String id)
    {
        final String SERVICE_PREFIX = "/xqServices/";
        final String PROCESS_PREFIX = "/xqProcesses/";
        final String ENDPOINT_PREFIX = "/xqEndpoints/";
        final String CONNECTION_PREFIX = "/xqConnections/";
        final String CONNECTION_TYPE_PREFIX = "/xqConnectionTypes/";
        final String SERVICE_PROCESS_REF = "name=\"service_ref\" value=\"";
        final String ENDPOINT_REF = "name=\"endpoint_ref\" value=\"";
        final String CONNECTION_REF = "name=\"connection_ref\" value=\"";
        final String CONNECTION_TYPE_REF = "name=\"type_ref\" value=\"";
        
        if (id.startsWith(SERVICE_PREFIX) || id.startsWith(PROCESS_PREFIX))
        {
            return SERVICE_PROCESS_REF + id.substring(id.lastIndexOf('/') + 1) + "\"";
        }
        else if (id.startsWith(ENDPOINT_PREFIX))
        {
            return ENDPOINT_REF+ id.substring(id.lastIndexOf('/') + 1) + "\"";
        }
        else if (id.startsWith(CONNECTION_PREFIX))
        {
            return CONNECTION_REF + id.substring(id.lastIndexOf('/') + 1) + "\"";
        }
        else if (id.startsWith(CONNECTION_TYPE_PREFIX))
        {
            return CONNECTION_TYPE_REF + id.substring(id.lastIndexOf('/') + 1) + "\"";
        }
        else
        {
            return id;
        }
    }
    
    private void printStackTrace(Object[] info)
    {
        System.out.println(info[0]); // the first object is the actual exception or a string
        System.out.println("Exception stack trace:");
        StackTraceElement[] stackTrace = (StackTraceElement[])info[1]; // the second element is the stack trace
        for (int i=0; i<stackTrace.length; i++)
        {
            System.out.println(stackTrace[i].toString());
        }
    }
    
    private void printCorruptId(String elId)
    {
        if (elId.startsWith("/"))
        {
            // if there is a logical name for the id, make it part of the message
            try
            {
                String logicalId = m_ds.storageToLogical(elId);
                if (logicalId != null)
                {
                    System.out.println(elId + "(" + logicalId + ")");
                }
                else
                {
                    System.out.println(elId);
                } 
            }
            catch (DirectoryServiceException dirE) 
            {
                System.out.println(elId); 
            }
        }
        else
        {
            System.out.println("UNKNOWN element");
        }
    }
}
