package com.sonicsw.mf.framework.directory;

import java.io.File;
import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.Hashtable;

import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.agent.ContainerUtil;
import com.sonicsw.mf.mgmtapi.config.constants.IBackupDirectoryServiceConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IDirectoryServiceConstants;

/**
 * Seeds a Directory Service store with configuration data from a given file.
 * Requires a XML configuration file for the DS in order to bootstrap itself.
 */
public final class DirectorySeeder
{
    private IDirectoryService m_ds;
    private static final String FILES_DUMP_SUFFIX = "_files";

    public static void main(String[] args)  throws Exception
    {
        if (args.length < 2 || args[0].equals("?"))
        {
            printUsage();
        }

        try
        {
            new DirectorySeeder(args);
        }
        catch(Exception e) { e.printStackTrace(); }
    }

    private DirectorySeeder(String[] args)
    throws Exception
    {
        try
        {
            // get the xml that described the DS configuration and create the DS
            m_ds = createDSInstance(args[0]);
            ArrayList xmlFiles = new ArrayList();
            // now place in the DS
            System.out.println("Seeding directory store:");
            boolean parseBlobArg = false;
            String blobDirectory = null;
            File blobsSubDir = null;
            for (int i = 1; i < args.length; i++)
            {
                if (parseBlobArg)
                {
                    blobDirectory = args[i];
                    parseBlobArg = false;
                    continue;
                }
                if (args[i].equals("-files"))
                {
                    parseBlobArg = true;
                    continue;
                }
                xmlFiles.add(args[i]);
                
            }

            // check that the blob directory is correct before continuing
            if (blobDirectory != null)
            {
                blobsSubDir = new File(blobDirectory, m_ds.getDomain() + FILES_DUMP_SUFFIX);
                if (!blobsSubDir.exists())
                {
                    throw new DirectoryServiceException("Files directory " + blobsSubDir.getAbsolutePath() + " does not exist.");
                }
            }
            String[] xmlFilesArray = new String[xmlFiles.size()];
            xmlFiles.toArray(xmlFilesArray);
            for (int i=0; i< xmlFilesArray.length; i++)
            {
                String file = xmlFilesArray[i];
                System.out.print("- " + file + " .");
                // get the seed information
                String seedXML = getXMLStringFromFile(file);
                System.out.print('.');
                // now place in the DS
                m_ds.importFromXML(seedXML);
                System.out.println('.');
            }

            // now report what's in there
            if (System.getProperty("verbose", "false").equals("true"))
            {
                String contents = m_ds.exportDirectoryToXML("/");
                System.out.println("Reporting contents:");
                System.out.println();
                System.out.println(contents);
                System.out.println();
            }
            
            if (blobsSubDir != null)
            {
                System.out.println("Loading files...");
                loadBlobs(blobsSubDir, "");
                System.out.println("...done");
            }
        }
        finally
        {
            if (m_ds != null)
            {
                m_ds.close();
            }
        }
        System.out.println("Done.");

        System.exit(0);
    }

    private static void printUsage()
    {
        System.out.println();
        System.out.println("Usage: -Dverbose={[false]|true} com.sonicsw.mf.framework.directory.storage.DirectorySeeder <ds.xml> <seed.xml> -files <files_directory>");
        System.out.println();
        System.out.println("Where: <ds.xml>            Directory Service configuration.");
        System.out.println("       <seeddata.xml>      The configuration information to be used to seed the");
        System.out.println("                           Directory Service storage. Multiple seed files may");
        System.out.println("                           be specified as additional arguments.");
        System.out.println("       <files_directory>   The directory to read files from in order to import them into");
        System.out.println("                           This utility loads the files from the <domain_name>_files subdirectory. ");
        System.out.println("                           the Directory Service. Files are exported using the -files");
        System.out.println("                           option with the DirectoryDump command");

        System.exit(1);
    }

    private String getXMLStringFromFile(String xmlFilename)
    throws Exception
    {
        FileInputStream fis = new FileInputStream(xmlFilename);
        byte[] bytes = new byte[fis.available()];
        fis.read(bytes);

        return new String(bytes);
    }
    
    private void loadBlobs(File blobRootDirectory, String subDir) throws Exception
    {
        
        File blobDirFile = new File(blobRootDirectory, subDir);
        String blobRootDirectoryString = blobRootDirectory.getCanonicalPath();
        File[] children = blobDirFile.listFiles();
        for (int i=0; i< children.length; i++)
        {
            File child = children[i];
            String childCanPath = child.getCanonicalPath();
            String childDsName = childCanPath.substring(blobRootDirectoryString.length());
            childDsName = childDsName.replace(File.separatorChar, IMFDirectories.MF_DIR_SEPARATOR);
            if (child.isDirectory())
            {
                loadBlobs(blobRootDirectory, childDsName);
            }
            else
            {                
                IDirElement blobEl = m_ds.getElement(childDsName, true);
                if (blobEl != null)  // in case there's some junk in the directory
                {
                    FileInputStream blobStream = new FileInputStream(child);                
                    m_ds.attachBlob(blobEl.doneUpdate(), blobStream, null);      
                }
            }
        }
    }


    private IDirectoryService createDSInstance(String dsXMLFile)
    throws Exception
    {

        IElement[] dsConfigs = ContainerUtil.importConfigurations(new File(dsXMLFile), IContainer.PASSWORD);
        IDirElement dsConfig = null;
        for (int i = 0; i < dsConfigs.length; i++)
        {
            String elType = dsConfigs[i].getIdentity().getType();
            if (elType.equals(IDirectoryServiceConstants.DS_TYPE) || elType.equals(IBackupDirectoryServiceConstants.DS_TYPE))
            {
                dsConfig = (IDirElement)dsConfigs[i];
            }
        }

        if (dsConfig == null)
        {
            throw new Exception("File " + dsXMLFile + " does not contain valid Directory Service boot information");
        }


        IAttributeSet dsAttributes = dsConfig.getAttributes();
        String hostDir = (String)dsAttributes.getAttribute(IDirectoryServiceConstants.HOST_DIRECTORY_ATTR);

        String domainName = (String)dsAttributes.getAttribute(IDirectoryServiceConstants.DOMAIN_NAME_ATTR);
        if (domainName == null)
        {
            domainName = IContainerConstants.DOMAIN_NAME_DEFAULT;
        }
        Object tmp = dsAttributes.getAttribute(IDirectoryServiceConstants.FILE_SYSTEM_STORAGE_ATTR);

        if (domainName == null || tmp == null)
        {
            throw new MFRuntimeException("Bad Directory Service Configuration - must contain DOMAIN_NAME and FILE_SYSTEM_STORAGE.");
        }

        if (!(tmp instanceof IAttributeSet))
        {
            throw new MFRuntimeException("Bad Directory Service Configuration - FILE_SYSTEM_STORAGE must be an attribute set.");
        }



        IAttributeSet fsStorage = (IAttributeSet)tmp;

        String hostDirDepricated = (String)fsStorage.getAttribute(IDirectoryServiceConstants.HOST_DIRECTORY_ATTR);
        String password = (String)fsStorage.getAttribute(IDirectoryServiceConstants.PASSWORD_ATTR);

        // Creates the directory service object
        Hashtable directoryEnv = new Hashtable();
        directoryEnv.put(IDirectoryService.STORAGE_TYPE_ATTRIBUTE, IDirectoryService.PSE_STORAGE);
        if (hostDirDepricated != null)
        {
            directoryEnv.put(IFSStorage.FS_HOST_DIRECTORY_ATTRIBUTE, hostDirDepricated);
        }
        if (hostDir != null)
        {
            directoryEnv.put(IFSStorage.HOST_DIRECTORY_ATTRIBUTE, hostDir);
        }
        if ((password != null) && (password.length() != 0))
        {
            directoryEnv.put(IFSStorage.PASSWORD_ATTRIBUTE, password);
        }
        DirectoryServiceFactory factory = new DirectoryServiceFactory(directoryEnv);
        return factory.createDirectoryService(domainName);
    }
}
