/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.framework.directory;

import java.util.Hashtable;

import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.framework.directory.impl.DirectoryService;

/**
 * A DirectoryServiceFactory object creates IDirectoryService objects on top of a file-system storage or an a JNDI compliant
 * directory. The type of storage and its parameter are determined by the hashtable properties passes to the factory's
 * constructor. The createDirectoryService() method creates a new domainName directory in the storage if it does not
 * exist yet, or otherwise, creates an IDirectoryService object on top the existing domain configuration. 
 */
public final class DirectoryServiceFactory
{
  private Hashtable m_startup_parameters;

 /**
  * Constructs a DirectoryServiceFactory object. The properties are validated only  
  * when a method to create a IDirectoryService object is called.
  *
  * @param properties the properties that determine the type of storage and storage specific parameters
  *
  */
  public DirectoryServiceFactory(Hashtable properties)
  {
      m_startup_parameters = properties;
  }

 /**
  * Creates a directory service object for a specific domain. A new, empty, domain is created
  * if that domain does not exist yet.
  *
  * @param domainName the name of the domain
  * @param logger the logger the DS should use to log information
  *
  * @return a directory service object
  *
  * @exception DirectoryServiceException if cannot connect to the underlying storage or access the domain
  *            or create a new one
  */
  public IDirectoryService createDirectoryService(String domainName, com.sonicsw.mf.common.ILogger logger)
        throws DirectoryServiceException 
  {
      return new DirectoryService(m_startup_parameters, domainName, logger);
  }

 /**
  * Creates a directory service object with a default logger. A new, empty, domain is created
  * if that domain does not exist yet.
  *
  * @param domainName the name of the domain
  *
  * @return a directory service object
  *
  * @exception DirectoryServiceException if cannot connect to the underlying storage or access the domain
  *            or create a new one
  */
  public IDirectoryService createDirectoryService(String domainName)
        throws DirectoryServiceException
  {
      return new DirectoryService(m_startup_parameters, domainName, null);
  }

 /**
  * Creates a directory service object in a container. An exception is thrown if that domain does not exist yet.
  *
  * @param domainName the name of the domain
  * @param logger the logger the DS should use to log information
  *
  * @return a directory service object
  *
  * @exception DirectoryServiceException if cannot connect to the underlying storage or the domain does not exist.
  */
  public IDirectoryService createDirectoryServiceInContainer(String domainName, com.sonicsw.mf.common.ILogger logger,
                                                             IClassLoaderUtility loader, IFrameworkComponentContext context)
        throws DirectoryServiceException
  {
      return new DirectoryService(m_startup_parameters, domainName, logger, false, loader, context);
  }



}
