package com.sonicsw.mf.framework.directory;

import java.util.Date;
import java.util.HashMap;

import com.sonicsw.mf.framework.IFrameworkComponentContext;

/**
 * The DS allows internal clients to acquire and hold on to named, domain wide locks. Such locks are
 * leased for a requested period after which the lock will be released.
 * <p>
 * The DS creates a DistrubutedLockManager to manage the leasing of domain wide locks.
 */
class DistributedLockManager
{
    private DSComponent m_dsComponent;
    private IFrameworkComponentContext m_context;

    private HashMap m_leasedLocks = new HashMap();;

    DistributedLockManager(DSComponent dsComponent, IFrameworkComponentContext context)
    {
        m_dsComponent = dsComponent;
        m_context = context;
    }

    void cleanup()
    {
        m_leasedLocks.clear();
        m_dsComponent = null;
        m_context = null;
        m_leasedLocks = null;
    }

    /**
     * Acquire the named lock for the given duration; the lock is acquired against the given key.
     * <p>
     * This facility is used by e.g. a fault-tolerant AM in order to determine exclusive control and hence
     * the FT AM can be the active AM.
     *
     * @param lockName      The name of the lock to be acquired.
     * @param key           The key under which the lock is to be held (this should be unique within
     *                      the domain).
     * @param leaseDuration The period in seconds that the DS will hold this named lock against the given key.
     *
     * @return true if the lock has been held against the given key or false if the lock is already held against a different key.
     *
     * @see #releaseLock(String, String)
     */
    boolean leaseLock(String lockName, String key, long leaseDuration)
    {
        if (lockName == null)
        {
            throw new IllegalArgumentException("Lock name cannot be null");
        }
        if (lockName.length() == 0)
        {
            throw new IllegalArgumentException("Lock name cannot be an empty string");
        }
        if (key == null)
        {
            throw new IllegalArgumentException("Lease key cannot be null");
        }
        if (key.length() == 0)
        {
            throw new IllegalArgumentException("Lease key cannot be an empty string");
        }
        if (leaseDuration < 5000)
        {
            throw new IllegalArgumentException("Lease duration cannot be < 5 seconds");
        }

        synchronized(m_leasedLocks)
        {
            DistributedLock lock = (DistributedLock)m_leasedLocks.get(lockName);
            if (lock == null)
            {
                lock = new DistributedLock(lockName, key);
                m_leasedLocks.put(lockName, lock);
            }

            // if the lock is leased under another key then don't allow it to be leased
            if (!lock.key.equals(key))
            {
                return false;
            }

            lock.lease(leaseDuration);
            return true;
        }
    }


    /**
     * Release a previously acquired lock prior to lease expiration.
     *
     * @param lockName      The name of the lock to be released.
     * @param key           The key under which the lock was held (this should be unique within
     *                      the domain).
     *
     * @see #leaseLock(String, String, Integer)
     */
    void releaseLock(String lockName, String key)
    {
        if (lockName == null)
        {
            throw new IllegalArgumentException("Lock name cannot be null");
        }
        if (lockName.length() == 0)
        {
            throw new IllegalArgumentException("Lock name cannot be an empty string");
        }
        if (key == null)
        {
            throw new IllegalArgumentException("Lease key cannot be null");
        }
        if (key.length() == 0)
        {
            throw new IllegalArgumentException("Lease key cannot be an empty string");
        }

        // cleanup would have caused the loack table to be removed
        if (m_leasedLocks == null)
        {
            return;
        }

        synchronized(m_leasedLocks)
        {
            DistributedLock lock = (DistributedLock)m_leasedLocks.get(lockName);

            if (lock == null)
            {
                return;
            }
            if (!lock.key.equals(key))
            {
                return;
            }

            m_leasedLocks.remove(lockName);
        }
    }

    private class DistributedLock
    {
        private String lockName;
        private String key;
        private long expirationTime;

        private DistributedLock(String lockName, String key)
        {
            this.lockName = lockName;
            this.key = key;
        }

        private void lease(long duration)
        {
            this.expirationTime = System.currentTimeMillis() + duration;

            DistributedLockManager.this.m_context.scheduleTask(new LeaseExpirer(this), new Date(this.expirationTime));
        }
    }

    private class LeaseExpirer
    implements Runnable
    {
        private DistributedLock lock;

        private LeaseExpirer(DistributedLock lock)
        {
            this.lock = lock;
        }

        @Override
        public void run()
        {
            synchronized(DistributedLockManager.this.m_leasedLocks)
            {
                DistributedLock actualLock = (DistributedLock)DistributedLockManager.this.m_leasedLocks.get(this.lock.lockName);
                if (actualLock != null && actualLock == this.lock && this.lock.expirationTime <= System.currentTimeMillis())
                {
                    DistributedLockManager.this.releaseLock(this.lock.lockName, this.lock.key);
                }
            }
        }
    }
}
