/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.framework.directory;
import com.sonicsw.mf.common.IDSImport;
import com.sonicsw.mf.common.IDirectoryCacheService;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.view.INamingListener;
import com.sonicsw.mf.framework.elementversion.IArrayElementNotificationListener;
import com.sonicsw.mf.framework.elementversion.INotificationConsumer;

/**
 * This is the interface of the directory service used locally - not through a container.
 */

public interface IContainerlessDS extends IDSImport, IDirectoryCacheService
{
 /**
  * Force all the updates to storage and release all the resources.
  *
  * @exception DirectoryServiceException if there is a problem accessing the underlying storage
  *
  */
  public void close() throws DirectoryServiceException;

/**
  * Close without writing to storage - used when storage is no longer available (it's in STANDBY state for example).
  *
  * @exception DirectoryServiceException if there is a problem accessing the underlying storage
  *
  */
  public void close(boolean force) throws DirectoryServiceException;


 /**
  * Returns the name of the root domain directory.
  *
  * @return the name of the root domain directory
  *
  */
  public String getDomainDirectoryName();

 /**
  * Sets the debugging mask.
  */
  public void setTraceMask(int mask);


 /**
  * Subscribes to receive notifications for any modifications, deletions or additions of elements.
  *
  * @param listener the listener object for handling notifications
  *
  * @return a consumer object (typically used to cancel the subscription)
  *
  *
  */
  public INotificationConsumer subscribeAll(IArrayElementNotificationListener listener);


 /**
  * Get the backup version timestamp.
  *
  * @return the backup version timestamp
  *
  */
  public long getBackupTimestamp();


 /**
  * Regenerate the trigger validation table. Called when the "/_MFLibrary/validators" directory is modified
  */
  public void newTriggerValidator();

 /**
  * Regenerate the DS handlers table. Called when the "/_MFLibrary/ds_handlers" directory is modified
  */
  public void newDSHandlers();

 /**
  * Imports elements - was created primarily for importing users and groups from an external security source.
  * The main difference between importFromList and setElements is that importFromList replaces existing elements
  * and is also capable of deleting obsolete elements (that are not in the list) while setElements create new one
  * and modify existing ones. Also importFromList is designed for efficient bulk updtates and has an option to
  * delete all the elements that are not explicitly imported.
  *
  * @param list the list of imported elements
  * @param deletionDirectories an optional list of packed elements directories. All the elements in these
  *        directories that are not re-imported are deleted. This feature is used by the facility that
  *        reconciles the external security data with the imported data (can be null or empty).
  *
  */
  public void importFromList(IDirElement[] list, String[] deletionDirectories) throws DirectoryServiceException;

 /**
  * Returns the list of external authentication domain descriptors.
  *
  * @return list of external authentication domain descriptors
  */
  public String[] getExternalDomainsDescriptors();

 /**
  * Tells the authentication manager about the new domain descriptor. The authentication manager reconnects
  * to the external resource if new connection parameters are specified.
  *
  * @param element the new domain descriptor
  */
  public void newAuthenticationDescriptor(IElement element);

  /**
  * Subscribes to receive notifications for naming modifications.
  *
  * @param listener the listener object for handling naming notifications
  *
  *
  */
  public void subscribeNaming(INamingListener listener);
}
