/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.framework.directory;

import java.util.ArrayList;
import java.util.HashMap;

import com.sonicsw.mf.common.IDSTransaction;
import com.sonicsw.mf.common.MFException;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IChunkedBlobStreamer;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.INextVersionToken;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.VersionOutofSyncException;
import com.sonicsw.mf.common.security.IManagementPermission;
import com.sonicsw.mf.common.security.InvalidManagementPermissionException;
import com.sonicsw.mf.common.view.IDeltaView;
import com.sonicsw.mf.framework.IFrameworkComponentContext;

/**
 * This is the interface of the directory service used internally by the MF, mostly for keeping the local
 * container cache up-to-date.
 */

public interface IDirectoryMFService extends IChunkedBlobStreamer
{
    public static final String DS_STORAGE_NAME = "data.odb";
    public static final String BACKUP_VERSION_ATTR = "BACKUP_VERSION";

    public static final String TRACE_MASK_ATTRIBUTE = "TRACE_MASK";

    /**
     * The attribute that determines the type of directory service used (currently FS_STORAGE or JNDI_STORAGE).
     */
    public static final String STORAGE_TYPE_ATTRIBUTE = "STORAGE_TYPE_ATTRIBUTE";

    /**
     * The attribute that points to the JNDI environment hash table
     */
    public static final String JNDI_ENVIRONMENT_ATTRIBUTE = "JNDI_ENVIRONMENT_ATTRIBUTE";

    /**
     * The underlying storage is the file-system
     */
    public static final String FS_STORAGE = "FS_STORAGE";
    
    /**
     * The underlying storage is the PSE database
     */
    public static final String PSE_STORAGE = "PSE_STORAGE";

    /**
     * The underlying storage is a JNDI directory
     */
    public static final String JNDI_STORAGE = "JNDI_STORAGE";

    /**
     * If Boolean.TRUE, indicates that the DS is a backup DS that does not allow updates on failover to active
     */
    public static final String IS_RESTRICTED_BACKUP = "IS_RESTRICTED_BACKUP";

    /**
     * If Boolean.TRUE, indicates that the FT DS is failing over from standby
     */
    public static final String IS_FAILOVER = "IS_FAILOVER";

    /**
     * If Boolean.TRUE, indicates that the DS is fault tolerant and is using shared stoarge with its FT partner
     */
    public static final String IS_SHARED_STORAGE = "IS_SHARED_STORAGE";

    /**
     * For testing only: Test external authentication support in a stand-alone mode - when there is no container
     */
    public static final String TEST_AUTHENTICATION_ATTRIBUTE = "TEST_AUTHENTICATION_ATTRIBUTE";

    /**
     * If  Boolean.TRUE, indicates that the DS is opened at the first phase of the ci.Launcher
     */
    public static final String CI_FIRST_PHASE_ATTRIBUTE = "CI_FIRST_PHASE";
    
    /**
     * Property to customize the cache update frequency
     */
    public static final String AUTH_UPDATE_FREQUENCY_PROPERTY = "sonicsw.ds.auth.updateFrequency";

    /**
     * Property to customize the cache refresh frequency
     */
    public static final String AUTH_REFRESH_FREQUENCY_PROPERTY = "sonicsw.ds.auth.refreshFrequency";

    /**
     * Returns true if this path is to be handled by a DS handler. DSComponent use that to determine whether to register
     * the container that made this call as a client for change notifications - handler calls are not registered since the DS
     * doesn't know when the handler's files or elements are modified.
     */
    public boolean isHandlerPath(String path) throws DirectoryServiceException;

    /**
     * Returns a list of elements that were modified relative to a list of element identities (note that the identity object contains the element's version). If an element in the central repository has version greater than
     * version in the identities list then this element is included in the list of returned elements.
     * If an element in the identities list does not exit in the central repository then the IDirElement object
     * returned in the list returns isDeleted() = true. Also returns elements that were deleted and maybe recreated
     * (specified in the deletedConfigurations list)
     *
     * @param identities the list of element identities
     * @param deletedConfigurations the list of elements that were deleted in the DS but the container is still interested in
     * @param callerBackupVersion the backup version of the caller
     *
     * @return a a list of the latest versions of the elements. If the caller's backup version mismatches the backup version
     *         of the DS then the last element in the returned list is the DS_VERSION_INFO_PATH (contains the current
     *         backup version).
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or the directory does not exist
     *
     */
    public IDirElement[] getUpdatedList(IElementIdentity[] identities, HashMap deletedConfigurations, long callerBackupVersion)
        throws DirectoryServiceException;

    /**
     * Returns a read-only list of elements that were added to a specific dir relative to a list of element identities.
     *
     * @param identities the list of element identities
     *
     * @return a a list of the newly added elements
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or the directory does not exist
     *
     */
    public IDirElement[] getNewElements(String dirName, IElementIdentity[] identities)
        throws DirectoryServiceException;

    /**
     * This method is intended to superseed calls to getUpdatedList()/getNewElements() with a single bulk operation. It will
     * be called by >= v2.1 containers.
     * <p>
     * Returns an array of two elements:
     * <p>
     * The first element holds a list of elements that were modified relative to a list of element identities
     * (note that the identity object contains the element's version). If an element in the central repository
     * has version greater than version in the identities list then this element is included in the list of returned
     * elements. If an element in the identities list does not exist in the central repository then the IDirElement
     * object returned in the list returns isDeleted() = true. Also returns elements that were deleted and maybe
     * recreated (specified in the deletedConfigurations list). If the map of deletedConfigurations is null, then this
     * is a repeat call for further new elements (second element), and the first element will be set to null.
     * <p>
     * The second element holds an array of read-only elements that were added to the given list of directories. If the
     * dirNames parameter is null, then this is a repeat call and the second element will be set to null.
     * <p>
     * If either element is an empty array (or an element of the second array indicates there is more data to come), then
     * it means the results are not complete due to response size constraints and the caller should call the method again.
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or the directory does not exist
     */
    public IDirElement[][] getCacheElements(long callerBackupVersion, IElementIdentity[] identities, HashMap deletedConfigurations, String dirNames[])
        throws DirectoryServiceException;

    /**
     * Returns a correction map of the storage-to-logical name mapping. Nothing is returned for names that match.
     * null is returned for names that no longer exist. The new logical names is returned for renamed elements.
     * This call is used by containers to reconcile their storage-logical map.
     *
     * @param names the container's storage-to-logical map.
     *
     * @return a a list of map corrections.
     *
     */
    public HashMap getUpdatedLogicalNames(HashMap names);

    /**
     * Clone the elements and translate all names and references to logical
     *
     * @param elements the list of element notifications
     *
     * @return a list with to logical names
     *
     */
    public IBasicElement[] translateElementsToLogical(IBasicElement[] elements);

    /**
     * Return true if IDirectoryFileSystemService methods were called on the DS
     *
     * @return true if IDirectoryFileSystemService methods were called on the DS
     *
     */
    public boolean isFSInterfaceInUse();

    /**
     * Get the element using the logical name (but ID and references contain storage names). Used by the container.
     *
     * @param logicalName the element's logical name
     *
     * @return the element (null if the element does not exist)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or there is an inconsistency
     *            in the logica-storage mapping
     */
    public IElement getElementByLogicalName(String logicalName)
        throws DirectoryServiceException;

    /**
     * Get elements using their logical names (but ID and references contain storage names). Used by the container.
     *
     * @param logicalNames the elements' logical names
     *
     * @return the elements (null if the element does not exist)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or there is an inconsistency
     *            in the logica-storage mapping
     */
    public IElement[] getElementsByLogicalNames(String[] logicalNames)
        throws DirectoryServiceException;

    /**
     * Return the full storage-to-logical map to initiate the client cache
     *
     * @return the full storage-to-logical map.
     *
     */
    public HashMap getStorageToLogicalMap();

    /**
     * Stores part of a BLOB in the DS storage. Copies IBlob.BLOB_CHUNK_SIZE bytes from blob into storage. The element
     * remains marked as INCOMPLETE until all the pieces of the blob have been stored. This method is invoked by the
     * DirectoryServiceProxy and is not part of the public interface.
     *
     * @param element the element the BLOB is attached to. Element can be a new IDirElement or IDeltaDirElement object returned
     * from IDirElement.doneUpdate()
     * @param blob the bytes that make up part of the BLOB
     * @param offset the beginning index in blob where the method will start storing from
     * @param last true if this is the last piece of the blob being stored.
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     * @exception VersionOutofSyncException if there is a conflict in applying the delta or modifying the view because of a previous update by
     *                                      another application
     *
     */

    public void appendFSBlob(IBasicElement element, byte[] blob, int offset, boolean last)
      throws DirectoryServiceException, VersionOutofSyncException;


    /**
     * Stores part of a BLOB in the DS storage. Copies IBlob.BLOB_CHUNK_SIZE bytes from blob into storage. The element
     * remains marked as INCOMPLETE until all the pieces of the blob have been stored. This method is invoked by the
     * DirectoryServiceProxy and is not part of the public interface.
     *
     * @param element the element the BLOB is attached to. Element can be a new IDirElement or IDeltaDirElement object returned
     * from IDirElement.doneUpdate()
     * @param blob the bytes that make up part of the BLOB
     * @param offset the beginning index in blob where the method will start storing from
     * @param last true if this is the last piece of the blob being stored.
     * @param view modification to the view (can be null)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     * @exception VersionOutofSyncException if there is a conflict in applying the delta or modifying the view because of a previous update by
     *                                      another application
     *
     */

    public void appendBlob(IBasicElement element, byte[] blob, int offset, boolean last, IDeltaView view)
        throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Return the management permissions for the given paths of the particular type (configure or manage)
     * Used internally by DSComponent to pass the PermissionsManager to DirectoryService. The
     * PermissionsManager allows us to figure out whether a principal is a user or a group. It also allows
     * for permissions check of the calling pricipal. Will throw a runtime ConfigurePermissionDeniedException
     * if the calling user does not have IConfigurePermissionBit.ALLOW_READ permission on any of the paths.
     * @param paths
     *            The logical names of folders or configurations (for "configure" permissions) or folders, containers or
     *            components for "manage" permissions
     * @param type
     *            "manage" or "configure", for manage or configure permissions
     * @param manager permissions manager to get the principal type to return in the IManagementPermission arrays
     * @return An array of defined permissions for each path in paths
     * @throws DirectoryServiceException
     *             if a path is invalid, or there's a problem getting the permissions information
     * @see com.sonicsw.mf.common.security.IManagementPermission
     * @see com.sonicsw.mf.common.security.IConfigureScopeBits
     * @see com.sonicsw.mf.common.security.IConfigurePermissionBits
     * @see com.sonicsw.mf.common.security.IManageScopeBits
     * @see com.sonicsw.mf.common.security.IManagePermissionBits
     */
    
    public IManagementPermission[][] getManagementPermissions(String paths[], String type, IFrameworkComponentContext context, boolean checkPath)
        throws DirectoryServiceException;
    
    /**
     * Set management permissions. For "manage" permissions, the path of components in containers is
     * specified as /<folder1>/.../<foldern>/<ContainerName>:ID=<ComponentName>.
     * Will throw a runtime ConfigurePermissionDeniedException if the caller does not have
     * IConfigurePermissionBits.ALLOW_SET_PERMISSIONS permission on any of the paths
     * 
     * @param paths
     *            The logical names of folders or configurations (for "configure" permissions) or folders, containers or
     *            components for "manage" permissions
     * @param type
     *            "manage" or "configure", for manage or configure permissions
     * @param permissions
     *            An array of permissions for each one of the paths.
     * @param manager
     *            PermissionsManager to check the IConfigurePermissionBits.ALLOW_SET_PERMISSIONS permission of the caller
     * @throws DirectoryServiceException
     *             if a path is invalid, or there's a problem getting or modifying the permissions information
     * @throws InvalidPermissionException
     *             if a scope specification is incorrect for the path, or a principal
     *             does not exist in the global authentication domain
     * @see com.sonicsw.mf.common.security.IManagementPermission
     * @see com.sonicsw.mf.common.security.IConfigureScopeBits
     * @see com.sonicsw.mf.common.security.IConfigurePermissionBits
     * @see com.sonicsw.mf.common.security.IManageScopeBits
     * @see com.sonicsw.mf.common.security.IManagePermissionBits
     * 
     */
    public void setManagementPermissions(String[] paths, String type, IManagementPermission[][] permissions, IFrameworkComponentContext context)
    throws DirectoryServiceException, InvalidManagementPermissionException;

    /**
     * Remove the manage or configure permissions in the corresponding principals array for each path in paths. 
     * Will throw a ConfigurePermissionDeniedException if the caller does not have IConfigurePermissionBits.ALLOW_SET_PERMISSIONS
     * permission on any of the paths.
     * @param paths The logical names of folders or configurations (for "configure" permissions) or folders, containers or components for "manage" permissions
     * @param type "manage" or "configure", for manage or configure permissions
     * @param principals An array of principals for each path in paths. This method removes the permissions set for each principal
     * @param manager PermissionsManager to check the caller's permissions on the paths
     * @throws DirectoryServiceException if a path is invalid, or there's a problem getting the permissions information
     * @throws InvalidManagementPermissionException if a given principal does not have a permission on the associated path
     * @see com.sonicsw.mf.common.security.IManagementPermission
     * @see com.sonicsw.mf.common.security.IConfigureScopeBits
     * @see com.sonicsw.mf.common.security.IConfigurePermissionBits
     * @see com.sonicsw.mf.common.security.IManageScopeBits
     * @see com.sonicsw.mf.common.security.IManagePermissionBits
     */
    public void removeManagementPermissions(String[] paths, String type, String[][] principals, IFrameworkComponentContext context, boolean checkPaths)
    throws DirectoryServiceException, InvalidManagementPermissionException;
    
    /**
     * Remove all manage or configure permissions for each path in paths.
     * Will throw a ConfigurePermissionDeniedException if the caller does not have IConfigurePermissionBits.ALLOW_SET_PERMISSIONS
     * permission on any of the paths.
     * @param paths The logical names of folders or configurations (for "configure" permissions) or folders, containers or components for "manage" permissions
     * @param type "manage" or "configure", for manage or configure permissions
     * @param manager PermissionsManager to check the caller's permissions on the paths
     * @throws DirectoryServiceException if a path is invalid, or there's a problem getting the permissions information
     * @see com.sonicsw.mf.common.security.IManagementPermission
     * @see com.sonicsw.mf.common.security.IConfigureScopeBits
     * @see com.sonicsw.mf.common.security.IConfigurePermissionBits
     * @see com.sonicsw.mf.common.security.IManageScopeBits
     * @see com.sonicsw.mf.common.security.IManagePermissionBits
     */
    public void removeManagementPermissions(String[] paths, String type, IFrameworkComponentContext context, boolean checkPaths)
    throws DirectoryServiceException;
    
    /**
     * Execute the transactions specified in the IDSTransaction object. Used by DSComponent to
     * pass the permissions manager to the transaction code.
     * @param transaction IDSTransaction created through IDirectoryFileSystemService.createTransaction
     * @param context Component context to pass down to the runtime permissions checker
     * @return INextVersionToken
     * @throws DirectoryServiceException
     */
    public INextVersionToken executeTransaction(IDSTransaction transaction, IFrameworkComponentContext context)
    throws DirectoryServiceException;
    
    /**
     * Returns the type of path in the logical namespace
     * @param path The entity path
     * @return int which specifies the type of path in the logical namespace
     */
    public int getNameSpaceType(String path);
    
    /**
     * Used by DSComponent and internally by DirectoryService to check if
     * permissions checking is enabled.
     * @return true if permissions checking is enabled
     * @throws DirectoryServiceException
     */
    
    public boolean isPermissionsCheckingEnabled() throws DirectoryServiceException;
    
    /** 
     * Return the path to be used when checking permissions. If the path ends in "/", then it's
     * a folder. If it's an element path, we need to check if it's a subelement
     * of a hierarchical element. The path to be used for 
     * permissions checking then it that of the top element. 
     * This is used from DSComponent and from SharedPermissionChecks for ClientTransactions
     * @param path The entity path for which we want to check permissions
     * @return The path to be used for permissions checking, not always the same as path.
     */
    public String getPermissionsPath(String path) throws DirectoryServiceException;
    
    /**
     * Used by DSComponent to set the hierarchical types based on the elements in
     * /mx/hierarchicalTypes. This method is called when the DS is started and whenever a change
     * happens to elements in the directory.
     * @param map Calculated by DSComponent after reading all the elements in /mx/hierarchicalTypes 
     */
    public void setHierarchicalTypes(HashMap map);
    
    /**
     * 
     * @param path The element or folder ID in the DS
     * @return The hierarchical configuration path if the given path is a hierarchical
     * configuration or a subpath of the hierarchical configuration.
     * Hierarchical configurations are treated specially by the Permissions API.
     * @throws DirectoryServiceException
     */
    
    public String getHierarchicalPath(String path) throws DirectoryServiceException;
    
    /**
     * Returns true if the path is a hierarchical configuration (with the suffix) or the parent
     * folder (without the suffix). Hierarchical types are defined by the elements in /mx/hierarchicalTypes.
     * @param path The element or folder path
     * @return true if the path is a hirarchical configuration or the parent folder without the suffix.
     * @throws DirectoryServiceException
     */
    public boolean isHierarchicalPath(String path) throws DirectoryServiceException;
        
    /**
     * Called by DSComponent to return the entire blob and not the first chunk 
     * if the container asking for the blob is a 6.1 container.
     */
    
    public IBlob getEntireBlob(String blobStorageName, boolean forUpdate) throws MFException;
    
    /**
     * Answer true if the directory service is not forwarding notifications to the
     * container. 
     * @param containerID The runtime ID of the container
     * @return true if element change notifications should not be forwarded to
     * the container
     */
    public boolean areNotificationsSuspended(String containerID);
    
    public void resumeChangeNotifications(String containerPath) throws DirectoryServiceException;
    
    /**
     * Lists the folders and elements under path. Each HashMap object in the ArrayList contains the folder or element attributes with
     * _FOLDER_NAME String for folders and _ELEMENT_IDENTITY IElementIdentity for elements. _IS_COMPLEX is true if it's
     * a complex folder. Elements to be returned are filtered by name ending in the given extension.
     * if recurse is true, subfolders are traversed as well to return elements and folders.
     * If the user making the request does not have read permissions on a folder, the method 
     * bypasses the traversal of that folder
     *
     * @param path
     *            the folder to list
     * @param getFolders If true, folders are returned in the ArrayList
     * @param getElements If true, elements are returned in the result
     * @param extension Used a a filter to return elements with a logical name
     * ending in the extension. A null value means no filtering. Folder names are
     * not filtered.
     * @return a list of HashMap objects - one for each listed folder or element
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the path doesn't exit
     *
     */
    public ArrayList<HashMap> listFSAll(String path, boolean getFolders, boolean getElements, String extension)
    throws DirectoryServiceException;
}
