package com.sonicsw.mf.framework.directory;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.HashMap;
import java.util.Iterator;

import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.framework.agent.PBE;
import com.sonicsw.mf.framework.directory.impl.DirectoryService;
import com.sonicsw.mf.framework.directory.impl.TransactionManager;
import com.sonicsw.mf.framework.directory.storage.DSEncryptionException;
import com.sonicsw.mf.framework.directory.storage.IStorage;
import com.sonicsw.mf.framework.directory.storage.fs.FSStorage;

/**
 * Prepares a Directory Service backup copy to go online:
 * - Validates that the copy was made while the Directory Service was in a backup state.
 * - Removes runtime files such as the lock file.
 * - Modifies the backup version.
 */
public final class PrepareDSBackupToOnline
{
    private IStorage m_storage;
    private String m_password = null;

    public static void main(String[] args)  throws Exception
    {
        if (args.length == 0 || args[0].equals("?"))
        {
            printUsage();
        }

        try
        {
            new PrepareDSBackupToOnline(args);
        }
        catch(Exception e) { e.printStackTrace(); }
    }

    private PrepareDSBackupToOnline(String[] args)
    throws Exception
    {
        System.err.println("Validating the directory store...");
        if(args.length == 2)
        {
            m_password = args[1];
        }
        File dsDir = new File (args[0]);

        if (!dsDir.exists())
        {
            System.err.println(args[0] + " directory does not exist.");
            System.exit(1);
        }

        if (!dsDir.isDirectory())
        {
            System.err.println(args[0] + " is not a directory.");
            System.exit(1);
        }

        File dataDir = new File (dsDir, IMFDirectories.MF_DATA_DIR);
        File dataTrDir = new File (dsDir, IMFDirectories.MF_DATA_DIR + "." + FSStorage.TR_FILE_POSTFIX);
        File trDir = new File (dsDir, TransactionManager.TRANSACTION_FILE);

        if (!dataDir.exists() ||
            !dataDir.isDirectory() ||
            !dataTrDir.exists() ||
            !dataTrDir.isDirectory() ||
            !trDir.exists())
        {
            System.err.println(args[0] + " is not a valid Directory Service store.");
            System.exit (1);
        }

        File systemDir = new File (dataDir, IMFDirectories.MF_SYSTEM_DIR);
        if (!systemDir.exists() || !systemDir.isDirectory())
        {
            System.err.println(args[0] + " is not a valid Directory Service store.");
            System.exit (1);
        }


        // Reads the version element to validate version
        ObjectInputStream objectSin = null;
        IElementIdentity versionElementID = null;
        IDirElement versionElement = null;
        File versionFile = new File (systemDir, DirectoryService.VERSION_ELEMENT);
        try
        {
            objectSin = new ObjectInputStream(new BufferedInputStream(new FileInputStream(versionFile)));
            Object inS = objectSin.readObject();

            if ((m_password == null) && (inS instanceof byte[]))
            {
                throw new DSEncryptionException("Storage data is encrypted. Password must be specified.");
            }
            if (m_password != null && inS instanceof HashMap)
            {
                throw new DSEncryptionException("Storage data is not encrypted. Invalid password.");
            }
            if (m_password == null)
            {
                HashMap map = (HashMap)inS;
                versionElementID = (IElementIdentity)getObject(map);
                versionElement = (IDirElement)getObject((HashMap)objectSin.readObject());
            }
            else
            {
                versionElementID = (IElementIdentity)getObject(readEncryptedData(inS));
                versionElement = (IDirElement)getObject(readEncryptedData(objectSin.readObject()));
            }
            IAttributeSet attributes = versionElement.getAttributes();
            Integer dsVersion = (Integer)attributes.getAttribute("VERSION");
            if (dsVersion.intValue() != DirectoryService.DS_STORAGE_STRUCTURE_VERSION)
            {
                  System.err.println("The directory service storage version mismatches the " +
                                      "version of the software.\nStorage version is: " + dsVersion + ". Software version is: " +
                                       DirectoryService.DS_STORAGE_STRUCTURE_VERSION + ".");
                  System.exit (1);
            }

        }
        catch (Exception e)
        {
            System.err.println(e);
            System.exit (1);
        }
        finally
        {
            if (objectSin != null)
            {
                objectSin.close();
            }
        }


        File lockFile = new File (systemDir, DirectoryService.LOCK_ELEMENT);
        File openFile = new File (systemDir, DirectoryService.OPEN_ELEMENT);
        File backupFile = new File (systemDir, DirectoryService.BACKUP_ELEMENT);

        // If the backup copy was online when created (has a LOCK_ELEMENT or a OPEN_ELEMENT) but was
        // not in backup state (no BACKUP_ELEMENT) then it is not a valid backup copy.
        if ((lockFile.exists() || openFile.exists()) && !backupFile.exists())
        {
            System.err.println(args[0] + " is not a valid backup copy instance. The directory Service must be put in a 'backup state' " +
                               "when the copy is created.");
            System.exit (1);
        }

        // Note that if the backup copy was created while the DS was not online then non of the files below exist
        // and the delete() calls are NOOP.
        lockFile.delete();
        openFile.delete();
        backupFile.delete();

        System.err.println("...validation complete.");

        System.err.println("Modifying the backup version...");

        File backupVersionFile = new File (systemDir, DirectoryService.BACKUP_VERSION_ELEMENT);
        backupVersionFile.delete();

        IDirElement backupVersionElement =  ElementFactory.createElement(DirectoryService.BACKUP_VERSION_ELEMENT_PATH,
                                                                         "backup_version", "2.0");
        IAttributeSet attributes = backupVersionElement.getAttributes();
        attributes.setLongAttribute(IDirectoryMFService.BACKUP_VERSION_ATTR, new Long(System.currentTimeMillis()));

        String name = null;
        try
        {
             name = (new EntityName(backupVersionElement.getIdentity().getName())).getBaseName();
        }
        catch (Exception e)
        {
             throw new Error(e.toString()); // Should never happen
        }

        ObjectOutputStream objectSout = new ObjectOutputStream (new BufferedOutputStream (new FileOutputStream(backupVersionFile)));
        if ( m_password != null)
        {
           objectSout.writeObject(convertObjectToBytes(name, backupVersionElement.getIdentity()));
           objectSout.writeObject(convertObjectToBytes(name, backupVersionElement));
        }
        else
        {
            HashMap ids = new HashMap(1,1);
            HashMap elements = new HashMap(1,1);
            ids.put(name, backupVersionElement.getIdentity());
            elements.put(name, backupVersionElement);
            objectSout.writeObject(ids);
            objectSout.writeObject(elements);
        }
        objectSout.close();

        // Delete the subscribers file if corrupt - we might still be modifying this file while creating the backup copy
        // so we have to verify its validity.
        objectSin = null;
        File subscribersFile = new File (systemDir, DirectoryService.SUBSCRIBERS_ELEMENT);
        try
        {
            IElementIdentity subscribersElementID = null;
            IDirElement subscribersElement = null;
            objectSin = new ObjectInputStream(new BufferedInputStream(new FileInputStream(subscribersFile)));
            if (m_password != null)
            {
                subscribersElementID = (IElementIdentity)getObject((HashMap)objectSin.readObject());
                subscribersElement = (IDirElement)getObject((HashMap)objectSin.readObject());
            }
            else
            {
                subscribersElementID = (IElementIdentity)getObject(readEncryptedData(objectSin.readObject()));
                subscribersElement = (IDirElement)getObject(readEncryptedData(objectSin.readObject()));
            }
        }
        catch (Exception e)
        {
            subscribersFile.delete();
        }
        finally
        {
            if (objectSin != null)
            {
                objectSin.close();
            }
        }


        System.err.println("...modification complete - " + args[0] + " is ready for use.");

    }

    private byte[] convertObjectToBytes(String name, Object elmntData) throws Exception
    {
         ByteArrayOutputStream out = new ByteArrayOutputStream();
         ObjectOutputStream objectOut = new ObjectOutputStream(out);
         HashMap data = new HashMap(1,1);
         data.put(name, elmntData);
         objectOut.writeObject(data);

         byte[] encBytes = PBE.encrypt(out.toByteArray(), m_password);
         out.close();

         return encBytes;
    }

    private HashMap readEncryptedData(Object input) throws Exception
    {
       if (input instanceof HashMap)
    {
        throw new DSEncryptionException("Could not backup the Directory Storage because of a wrong or missing encryption password.");
    }

        byte[] bytes = PBE.decrypt((byte[])input, m_password);
        ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        ObjectInputStream object = new ObjectInputStream(in);

        return (HashMap)object.readObject();
    }

    private Object getObject(HashMap map)
    {
        Iterator i = map.keySet().iterator();
        return map.get(i.next());
    }

    private static void printUsage()
    {
        System.err.println();
        System.err.println("Usage: com.sonicsw.mf.framework.directory.storage.PrepareDSBackupToOnline <ds-directory> [<password>]");
        System.err.println();
        System.err.println("Where: <ds-directory>       the name of the directory that contains the backup copy. Required.");
        System.err.println("Where: <password>           the password used to encrypt the Directory Service storage.");
        System.exit(1);
    }

}
