package com.sonicsw.mf.framework.directory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.dirconfig.IDeltaDirElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;
import com.sonicsw.mf.common.view.ILogicalNameSpace;


// Scan the DS and reassign the logical names to all the references of the form /NO_STORAGE:<logical-name>. That causes the DS
// to reassign the correct storage name. Use this utility to fix DS instances where elements were created with references to elements
// that don't yet exist. Note that this should not happen with any tools we provide to customers but only internally with QA tools
// such as the sandbox.
public final class RepairNoStorageReferences
{

   // Scan the entire DS - excluding directory names which contain a string from the exclusion list - and reassign
   // /NO_STORAGE:<logical-name> values. For example, if the list contains "/_MF" then /authentication/domains/0/_MFUSers
   // and /authentication/domains/0/_MFGroups will be excluded.
   public static void repairReferences(IDirectoryFileSystemService ds, String[] exclusionList) throws Exception
   {
       fixReferences((IDirectoryAdminService)ds, "/", exclusionList);
   }

   // Scan the entire directory and reassign /NO_STORAGE:<logical-name> values.
   public static void repairReferences(IDirectoryFileSystemService ds) throws Exception
   {
       fixReferences((IDirectoryAdminService)ds, "/", new String[0]);
   }

   // Scan the given directory and reassign /NO_STORAGE:<logical-name> values.
   public static void repairReferences(IDirectoryFileSystemService ds, String directoryName) throws Exception
   {
       fixReferences((IDirectoryAdminService)ds, directoryName, new String[0]);
   }

   public static void repairReferences(IDirectoryFileSystemService ds, ArrayList elementNameList) throws Exception
   {
       for (int i = 0; i < elementNameList.size(); i++)
    {
        fixReferences(ds, ds.getFSElement((String)elementNameList.get(i), true));
    }
   }
 
   private static void fixReferences(IDirectoryAdminService ds, String dirName, String[] exclusionList) throws Exception
   {
      if (dirName.equals("/_MFSystem"))
    {
        return;
    }
      for (int i = 0; i < exclusionList.length; i++)
    {
        if (dirName.indexOf(exclusionList[i]) != -1)
        {
            return;
        }
    }

      IElementIdentity[] elements = ds.listElements(dirName);
      for (int i = 0; i < elements.length; i++)
      {
          IDirElement element = getLogicalElement((IDirectoryFileSystemService)ds, elements[i].getName());
          if (element == null)
        {
            continue;
        }
          fixReferences((IDirectoryFileSystemService)ds, element);
      }
 
      // Look in sub directories
      IDirIdentity[] directories = ds.listDirectories(dirName);
      for (int i = 0; i < directories.length; i++)
    {
        fixReferences(ds, directories[i].getName(), exclusionList);
    }
   }
 
   private static IDirElement getLogicalElement(IDirectoryFileSystemService ds, String storageName)
   {
       try
       {
           return ds.getFSElement(ds.storageToLogical(storageName), true);
       }
       catch (Exception e)
       {
           return null;
       }
   }

   private static void fixReferences(IDirectoryFileSystemService ds, IDirElement element) throws Exception
   {
       if (fixReferences(element.getAttributes()))
    {
        ds.updateFSElement((IDeltaDirElement)element.doneUpdate());
    }
   }

   static boolean fixReferences(IAttributeSet attSet) throws Exception
   {

       boolean needFix = false;
       HashMap map = attSet.getAttributes();
       Set keys = map.keySet();
       Iterator iter = keys.iterator();
       ArrayList targetList = new ArrayList();
       while( iter.hasNext() )
       {
           String key = (String)iter.next();
           Object value = map.get(key);
           if (value instanceof IAttributeSet)
           {
               if (fixReferences((IAttributeSet)value))
            {
                needFix = true;
            }
           }
           else if (value instanceof IAttributeList)
           {
               if (fixReferences((IAttributeList)value))
            {
                needFix = true;
            }
           }
           else if (value instanceof Reference)
           {
               if (((Reference)value).getElementName().startsWith(ILogicalNameSpace.NO_STORAGE_LABEL))
            {
                targetList.add(key);
            }
           }
        else
        {
            continue;
        }
       }

       for (int i = 0; i < targetList.size(); i++)
       {
           needFix = true;
           String key = (String)targetList.get(i);
           Reference newRef =  removeNoStorageLabel((Reference)attSet.getAttribute(key));
           attSet.setReferenceAttribute(key, newRef);
       }

       return needFix;

   }

   private static Reference removeNoStorageLabel(Reference ref) throws Exception
   {
       String oldRefValue = ref.getElementName();
       return new Reference (oldRefValue.substring(ILogicalNameSpace.NO_STORAGE_LABEL.length()));
   }

   private static boolean fixReferences(IAttributeList attList) throws Exception
   {
       ArrayList targetList = new ArrayList();
       boolean needFix = false;
       for (int i = 0; i < attList.getCount(); i++)
       {
           Object value = attList.getItem(i);
           if (value instanceof IAttributeSet)
           {
               if (fixReferences((IAttributeSet)value))
            {
                needFix = true;
            }
           }
           else if (value instanceof IAttributeList)
           {
               if (fixReferences((IAttributeList)value))
            {
                needFix = true;
            }
           }
           else if (value instanceof Reference  && ((Reference)value).getElementName().startsWith(ILogicalNameSpace.NO_STORAGE_LABEL))
        {
            targetList.add(new Integer(i));
        }
        else
        {
            continue;
        }
       }

       for (int i = 0; i < targetList.size(); i++)
       {
           needFix = true;
           int position = ((Integer)targetList.get(i)).intValue();
           Reference newRef =  removeNoStorageLabel((Reference)attList.getItem(position));
           attList.setNewReferenceItem(position, newRef);
       }
       return needFix;

   }



}
