package com.sonicsw.mf.framework.directory;

/*
 * Permission checks for DS operations that can be executed from inside a client transaction.
 * This code is used both by the ClientTransaction class and DSComponent
 */

import java.util.HashMap;

import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.security.IConfigurePermissionBits;
import com.sonicsw.mf.common.view.ILogicalNameSpace;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.framework.IPermissionsManager;

public class SharedPermissionChecks

{
    private IPermissionsManager m_manager;
    private IDirectoryService m_ds;
    private IFrameworkComponentContext m_context;
    
    private static String DOMAIN_LOGICAL_NAME = "/_Default";
    
    public SharedPermissionChecks(IFrameworkComponentContext context, IDirectoryService ds)
    {
        if (context != null)
        {
            m_manager = context.getPermissionsManager();
        }
        m_ds = ds;
        m_context = context;
    }
    
    public void updateFSElementCheck(String elementName) throws DirectoryServiceException
    {
    	if (elementName.equals(DOMAIN_LOGICAL_NAME))
        {
            // perform the same check as DSComponent.setDomainElement
    		configurePermissionCheck(DSComponent.MF_DIR_SEPARATOR_STRING, false, IConfigurePermissionBits.ALLOW_WRITE);
        }
        else
        {
            configurePermissionWriteCheck(elementName);
        }
    }
    
    public void setMetaAttributesCheck(String path) throws DirectoryServiceException
    {
        String parentPath = "";
        try
        {
            EntityName eName = new EntityName(path);
            parentPath = eName.getParent();
            if (!parentPath.endsWith(DSComponent.MF_DIR_SEPARATOR_STRING))
            {
                parentPath = parentPath + DSComponent.MF_DIR_SEPARATOR_STRING;
            }
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Unable to check permissions for path " + path + ": " + e.toString());
        }
        configurePermissionWriteCheck(parentPath);       
    }
    
    public void createFSElementsCheck(IDirElement[] elements) throws DirectoryServiceException
    {
        // Assumes the elements don't exist, so the check will delegate to the parent folder. 
        // If they exist, the call to create them will fail anyway
        for (int i=0; i< elements.length; i++)
        {
            String elName = elements[i].getIdentity().getName();
            EntityName elEName;
            try
            {
                elEName = new EntityName(elName);
            }
            catch (Exception e)
            {
                throw new DirectoryServiceException("Unable to check permissions while creating elements: " + e.toString());
            }
            String parentFolder = elEName.getParent();
            if (elName.endsWith("_Default") || elName.endsWith("_MFDomainDescriptor") || elName.endsWith("_MQPolicyDescriptor"))
            {
                try
                {
                    EntityName parentEName = new EntityName(parentFolder);
                    parentFolder = parentEName.getParent();
                }
                catch (Exception e)
                {
                    throw new DirectoryServiceException("Unable to check permissions while creating elements: " + e.toString());
                }
            }
            if (!parentFolder.equals("/"))
            {
                parentFolder += IMFDirectories.MF_DIR_SEPARATOR;
            }
            configurePermissionWriteCheck(parentFolder);
        }
    }
    
    
    public void createFolderCheck(String folderPath, boolean existingOk)
        throws DirectoryServiceException
    {
        EntityName folderPathEName = null;
        HashMap test = m_ds.getMetaAttributes(folderPath);
        if ((test != null) && existingOk)
        {
            String folderName = folderPath;
            if (!folderName.endsWith(DSComponent.MF_DIR_SEPARATOR_STRING))
            {
                folderName = folderName + DSComponent.MF_DIR_SEPARATOR_STRING;
            }
            configurePermissionReadCheck(folderName);
        }
        else
        {
            String parentFolder = "";
            try
            {
                folderPathEName = new EntityName(folderPath);
                parentFolder = folderPathEName.getParent();
            }
            catch (ConfigException e)
            {
                throw new DirectoryServiceException("Error while checking permissions to createFolder: " + e.toString());
            }
            if (!parentFolder.equals("/"))
            {
                parentFolder += IMFDirectories.MF_DIR_SEPARATOR;
            }

            configurePermissionWriteCheck(parentFolder);
        }
    }
    
    public void deleteFolderCheck(String folderPath) throws DirectoryServiceException
    {
        String checkFolder = (folderPath.endsWith("/") ? folderPath : folderPath + IMFDirectories.MF_DIR_SEPARATOR);
        String hierarchicalPath = m_ds.getHierarchicalPath(folderPath);
        // if you're trying to delete a folder under the hierarchical configuration,
        // you want to check the write permission of the hierarchical configuration
        if (hierarchicalPath != null)
        {
            // The SMC will delete hierarchical configs by putting a DeleteFolder
            // in the transaction. The folder name will be the hierarchical config name,
            // without the trailing slash. If this is the folder being deleted, check
            // the delete permission of the path.
            if (hierarchicalPath.equals(folderPath))
            {
                configurePermissionDeleteCheck(hierarchicalPath);
            }
            else
            {
                configurePermissionWriteCheck(hierarchicalPath);
            }
        }
        else
        {
            configurePermissionDeleteCheck(checkFolder);
        }
    }
    
    public void attachFSBlobCheck(String blobName) throws DirectoryServiceException
    {
        configurePermissionWriteCheck(blobName);
    }
    
    public void appendFSBlobCheck(String blobName) throws DirectoryServiceException
    {
        configurePermissionWriteCheck(blobName);    
    }
    
    public void subclassFSElementCheck(String superClassName, String newElementName)
        throws DirectoryServiceException
    {
        configurePermissionReadCheck(superClassName);
        EntityName newEName;
        try
        {
            newEName = new EntityName(newElementName);
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Unable to check permissions while subclassing " + superClassName + e.toString());
        }
        String parent = newEName.getParent();
        if (!parent.equals("/"))
        {
            parent += IMFDirectories.MF_DIR_SEPARATOR;
        }
        configurePermissionWriteCheck(parent);
    }
    
    public void cloneFSElementCheck(String elementPath, String newElementPath)
        throws DirectoryServiceException
    {
        configurePermissionReadCheck(elementPath);
        EntityName newEName;
        try
        {
            newEName = new EntityName(newElementPath);
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Unable to check permissions while cloning " + elementPath + e.toString());
        }
        String parent = newEName.getParent();
        if (!parent.equals("/"))
        {
            parent += IMFDirectories.MF_DIR_SEPARATOR;
        }
        configurePermissionWriteCheck(parent);
    }
    
    public void deleteFSElementCheck(String elementName) throws DirectoryServiceException
    {
        String hierarchicalPath = m_ds.getHierarchicalPath(elementName);
        // if you're trying to delete a config under the hierarchical configuration,
        // for instance, a queue under a broker or a user under an authentication domain
        // you want to check the write permission of the hierarchical configuration
        // Deletes of sub configurations that happen as part of a transaction that is deleting the
        // hierarchical configuration don't end up here; they're handled by the transaction code.
        if (hierarchicalPath != null && !m_ds.isHierarchicalPath(elementName))
        {
            configurePermissionWriteCheck(hierarchicalPath);
        }
        else
        {
            configurePermissionDeleteCheck(elementName);
        }
    }
    
    public void renameCheck(String oldName, String newName)
        throws DirectoryServiceException
    {
        if (oldName.equals(newName))
        {
            return;
        }
        //check ALLOW_WRITE on both parent folders - might be the same folder
        EntityName oldEName; 
        EntityName newEName; 
        try
        {
            oldEName = new EntityName(oldName);
            newEName = new EntityName(newName);
        }
        catch (ConfigException e)
        {
            throw new DirectoryServiceException(e.toString());
        }
        String oldParent = oldEName.getParent();
        if (!oldParent.equals("/"))
        {
            oldParent += IMFDirectories.MF_DIR_SEPARATOR;
        }
        String newParent = newEName.getParent();
        if (!newParent.equals("/"))
        {
            newParent += IMFDirectories.MF_DIR_SEPARATOR;
        }
        configurePermissionWriteCheck(oldParent);
        configurePermissionWriteCheck(newParent);
        String oldLeaf = oldEName.getBaseName();
        String newLeaf = newEName.getBaseName();
        if (!oldLeaf.equals(newLeaf))
        {
            int type = m_ds.getNameSpaceType(oldName);
            if (type == ILogicalNameSpace.ELEMENT_TYPE)
            {
                configurePermissionWriteCheck(oldName);
            }
            else if (type == ILogicalNameSpace.FOLDER_TYPE)
            {
                String hierarchicalPath = m_ds.getHierarchicalPath(oldName);
                if (hierarchicalPath != null && hierarchicalPath.equals(oldName))
                {
                    configurePermissionWriteCheck(oldName);
                }
            }
        }
    }
    
    
    public void detachFSBlobCheck(String blobName) throws DirectoryServiceException
    {
        configurePermissionDeleteCheck(blobName);
    }

    public void configurePermissionReadCheck(String path) throws DirectoryServiceException
    {
        configurePermissionCheck(path, true, IConfigurePermissionBits.ALLOW_READ);
    }
    
    public void configurePermissionWriteCheck(String path) throws DirectoryServiceException
    {
        configurePermissionCheck(path, true, IConfigurePermissionBits.ALLOW_WRITE);
    }
    
    public void configurePermissionDeleteCheck(String path) throws DirectoryServiceException
    {
        configurePermissionCheck(path, true, IConfigurePermissionBits.ALLOW_DELETE);
    }
    
    public void configurePermissionCheck(String path, boolean isLogical, int permission) throws DirectoryServiceException
    {
        //the call to getPermissionsPath can trigger the Eclipse handler midway
        // into being seeded. Do not call the PermissionsManager if permissions checking
        // is not enabled
       if (isPermissionsCheckingEnabled())
    {
        m_manager.configurePermissionCheck(m_context, m_ds.getPermissionsPath(path), isLogical, permission);
    }   
    }
    
    public boolean isPermissionsCheckingEnabled()
    {
        return (m_manager != null && m_manager.isPermissionsCheckingEnabled());
    }
    
}
